/**
 *  soif.h - Summary Object Interchange Format (SOIF)
 *
 * Copyright 2003 Sun Microsystems, Inc. All
 * rights reserved. Use of this product is subject
 * to license terms. Federal Acquisitions:
 * Commercial Software -- Government Users
 * Subject to Standard License Terms and
 * Conditions.
 *
 * Sun, Sun Microsystems, the Sun logo, and Sun ONE
 * are trademarks or registered trademarks of Sun Microsystems,
 * Inc. in the United States and other countries.
 *
 *  Portions adapted from original Harvest code: src/common/include/template.h
 *
 *  ----------------------------------------------------------------------
 *  Original Harvest Copyright: http://harvest.cs.colorado.edu/
 *  Copyright(c) 1994, 1995.  All rights reserved.
 *  
 *    The Harvest software was developed by the Internet Research Task
 *    Force Research Group on Resource Discovery(IRTF-RD):
 *  
 *          Mic Bowman of Transarc Corporation.
 *          Peter Danzig of the University of Southern California.
 *          Darren R. Hardy of the University of Colorado at Boulder.
 *          Udi Manber of the University of Arizona.
 *          Michael F. Schwartz of the University of Colorado at Boulder.
 *          Duane Wessels of the University of Colorado at Boulder.
 *
 *    This copyright notice applies to software in the Harvest
 *    ``src/'' directory only.  Users should consult the individual
 *    copyright notices in the ``components/'' subdirectories for
 *    copyright information about other software bundled with the
 *    Harvest source code distribution.
 *  
 *  TERMS OF USE
 *    
 *    The Harvest software may be used and re-distributed without
 *    charge, provided that the software origin and research team are
 *    cited in any use of the system.  Most commonly this is
 *    accomplished by including a link to the Harvest Home Page
 *   (http://harvest.cs.colorado.edu/) from the query page of any
 *    Broker you deploy, as well as in the query result pages.  These
 *    links are generated automatically by the standard Broker
 *    software distribution.
 *    
 *    The Harvest software is provided ``as is'', without express or
 *    implied warranty, and with no support nor obligation to assist
 *    in its use, correction, modification or enhancement.  We assume
 *    no liability with respect to the infringement of copyrights,
 *    trade secrets, or any patents, and are not responsible for
 *    consequential damages.  Proper use of the Harvest software is
 *    entirely the responsibility of the user.
 *  
 *  DERIVATIVE WORKS
 *  
 *    Users may make derivative works from the Harvest software, subject 
 *    to the following constraints:
 *  
 *      - You must include the above copyright notice and these 
 *        accompanying paragraphs in all forms of derivative works, 
 *        and any documentation and other materials related to such 
 *        distribution and use acknowledge that the software was 
 *        developed at the above institutions.
 *  
 *      - You must notify IRTF-RD regarding your distribution of 
 *        the derivative work.
 *  
 *      - You must clearly notify users that your are distributing 
 *        a modified version and not the original Harvest software.
 *  
 *      - Any derivative product is also subject to these copyright 
 *        and use restrictions.
 *  
 *    Note that the Harvest software is NOT in the public domain.  We
 *    retain copyright, as specified above.
 *  
 *  HISTORY OF FREE SOFTWARE STATUS
 *  
 *    Originally we required sites to license the software in cases
 *    where they were going to build commercial products/services
 *    around Harvest.  In June 1995 we changed this policy.  We now
 *    allow people to use the core Harvest software(the code found in
 *    the Harvest ``src/'' directory) for free.  We made this change
 *    in the interest of encouraging the widest possible deployment of
 *    the technology.  The Harvest software is really a reference
 *    implementation of a set of protocols and formats, some of which
 *    we intend to standardize.  We encourage commercial
 *    re-implementations of code complying to this set of standards.  
 *  
 */
#ifndef _RDM_SOIF_H_
#define _RDM_SOIF_H_

#include <stdio.h>		/* for FILE */

#ifdef __cplusplus
extern "C" {
#endif

#ifndef NSAPI_PUBLIC_DEFINES
#define NSAPI_PUBLIC_DEFINES
#ifdef XP_WIN32
#define NSAPI_PUBLIC    __declspec(dllexport)
#define NSAPI_PRIVATE   static
typedef enum { B_FALSE, B_TRUE } boolean_t;
#else
#if defined(HPUX) || defined(OSF1) || defined(Linux)
/* HPUX and OSF1 doesn't have boolean_t? */
typedef enum { B_FALSE, B_TRUE } boolean_t;
#else
#if defined(AIX)
#include <sys/types.h>		/* for boolean_t */
#define B_FALSE 0		
#define B_TRUE  1
#else
#include <sys/types.h>		/* for boolean_t */
#endif /* AIX */
#endif /* HPUX */
#define NSAPI_PUBLIC
#define NSAPI_PRIVATE   static
#endif /* XP_WIN32 */
#endif /* !NSAPI_PUBLIC_DEFINES */

/* SOIF API objects */
typedef struct soif_s 		SOIF;
typedef struct soif_avpair_s 	SOIFAVPair;
typedef struct soif_stream_s 	SOIFStream;
typedef struct soif_buffer_s 	SOIFBuffer;

/** To find out the version of the SOIF library, use SOIF_Version() */
NSAPI_PUBLIC const char *SOIF_Version(void);

/*  ---------------------------------------------------------------------- */

/** 
 *  SOIF object
 *
 *  Associates a URL with a collection of Attribute-Value pairs and a
 *  Schema-Name to identify how to intepret the Attribute-Value pairs.
 *  Supports binary data as values, and multivalued attributes.
 *
 *  Example SOIF:
 *
 *	@DOCUMENT { http://www.iplanet.com/
 *	Title{19}:	Welcome to Sun ONE!
 *	Author{10}:	Mark Twain
 *	}
 */

/**
 *  SOIF_Create() - creates a new SOIF object initialized with the
 *  given Schema-Name and URL.
 *
 *  SOIF_Free() - frees all memory associated with the SOIF.
 */
NSAPI_PUBLIC SOIF *SOIF_Create(const char *schema_name, const char *url);
NSAPI_PUBLIC SOIF *SOIF_Copy(SOIF *);
NSAPI_PUBLIC void SOIF_Free(SOIF *);

struct soif_s {			/* SOIF object */
	char *url;		/* The URL */
	char *schema_name;	/* The Schema-Name */

	/** To access the collection of associated AVPairs, 
	 *  use find() to retrieve the AVPair for the given attribute, or
	 *  use findval() to retrieve the value string for the given attribute.
 	 *  It's required that you use all lowercase attribute names for find*(),
	 *  since only exact attribute name lookups are supported.
	 *
	 *  Example:
	 *	SOIFAVpair *avp = (*s->find)(s, "title");
	 *	printf("Title = %s\n", (*s->findval)(s, "title"));
         */
	SOIFAVPair *(*find)(SOIF *s, const char *attribute);
	const char *(*findval)(SOIF *s, const char *attribute);

	/** To access all of the AVPairs in the collection,
	 *  use apply() to apply the given function to every AVPair object
	 *  in the collection.
 	 *
	 *  Example:
	 *	void print_av(SOIF *s, SOIFAVPair *avp, void *unused)
	 *	{
	 *		printf("%s = %s\n", avp->attribute, avp->value);
	 *	}
	 *
	 *	...
	 *	printf("URL = %s\n", s->url);
	 *	(*s->apply)(s, print_av, NULL);
	 */
	void (*apply)(SOIF *s, 
                      void (*fn)(SOIF *s, SOIFAVPair *avp, void *data), 
                      void *data);

	/** To add or delete AVPairs in the collection,
	 *  use insert() to copy the given attribute-value pair, or
	 *  use insert_avp() to add an existing AVPair object to the collection.
 	 *  Return 0 if successful, or non-zero on error.
	 *
	 *  Example:
	 *	(*s->insert)(s, "Author", "Darren Hardy", 12);
	 *
	 *	SOIFAVPair *avp = 
	 *		SOIFAVPair_Create("Author", "Darren Hardy", 12);
	 *	(*avp->add_value)(avp, "Author", "Santa Claus", 11);
	 *	(*s->insert_avp)(s, avp);
	 *
	 *	(*s->remove)(s, "Author");
	 */
	int (*insert)(SOIF *s, const char *a, const char *v, int vsz);
	int (*insert_avp)(SOIF *s, SOIFAVPair *avp);
	int (*remove)(SOIF *, const char *);

	/** To access the names of all available attributes, then
	 *  use get_attributes() to receive an array of attribute names.
	 *  The returned arrary is malloc'ed, but the attribute names
	 *  themselves are read-only.
	 *
	 *  Example:
	 *	int i;
	 *	char **attrs = (*s->get_attributes)(s);
	 *
	 *	for (i = 0; attrs[i]; i++)
	 *		SOIFAVPair *avp = (*s->find)(s, attrs[i]);
	 *	free(attrs);
	 */
	char **(*get_attributes)(SOIF *);

	void *internal;		/* private data */
};

/** To insert an AVpair with a string as a value, use SOIF_InsertStr().
 *  To replace an existing attribute, use SOIF_Replace().
 * 
 *  Example:
 *	SOIF_InsertStr(s, "Author", "Darren Hardy");
 *	SOIF_ReplaceStr(s, "Author", "Tommy");		-- to overwrite
 */
#define SOIF_InsertStr(s, a, v) \
	((*(s)->insert)((s), (a), (v), strlen(v)))
#define SOIF_ReplaceStr(s, a, v) \
	SOIF_Replace((s), (a), (v), strlen(v))

NSAPI_PUBLIC int SOIF_Replace(SOIF *s, const char *a, const char *v, int vsz);

/** To estimate the number of bytes in the SOIF object, 
 *  use SOIF_GetTotalSize() which adds the size of the URL,
 *  Schema-Name, and all attribute-value pairs in the collection.
 */
NSAPI_PUBLIC int SOIF_GetTotalSize(SOIF *s);		/* everything */
NSAPI_PUBLIC int SOIF_GetAttributeCount(SOIF *s);	/* attr names only */
NSAPI_PUBLIC int SOIF_GetAttributeSize(SOIF *s);	/* attr names only */
NSAPI_PUBLIC int SOIF_GetValueSize(SOIF *s);		/* values only */
NSAPI_PUBLIC int SOIF_GetValueCount(SOIF *s);		/* values only */

/** To merge 2 SOIF objects (perform a Union of their attribute-values),
 *  use SOIF_Merge().  It returns non-zero on error; otherwise, returns
 *  zero and the 'dst' SOIF object contains all of the attribute-vaules
 *  from the 'src' SOIF object.
 *
 *  SOIF_Merge - If the 'dst' object contains the same attribute
 *  as 'src', then dst replaces src.
 *  SOIF_Absorb - If the 'dst' object contains the same attribute
 *  as 'src', then src is ignored.
 */
NSAPI_PUBLIC int SOIF_Merge(SOIF *dst, SOIF *src);
NSAPI_PUBLIC int SOIF_Absorb(SOIF *dst, SOIF *src);

/** Shortcuts */
#define SOIF_Find(_s, _a)		((*(_s)->find)((_s),(_a)))
#define SOIF_Findval(_s, _a)		((*(_s)->findval)((_s),(_a)))
#define SOIF_Remove(_s, _a)		((*(_s)->remove)((_s),(_a)))
#define SOIF_Insert(_s, _a, _v, _vsz)	((*(_s)->insert)((_s),(_a),(_v),(_vsz)))
#define SOIF_InsertAVP(_s, _avp)	((*(_s)->insert_avp)((_s),(_avp)))
#define SOIF_Apply(_s, _fn, _d)		((*(_s)->apply)((_s),(_fn),(_d)))

/*  ---------------------------------------------------------------------- */

/**
 *  Attribute-Value Pair 
 *
 *  Contains an Attribute and an associated value.  The value often is
 *  a simple null-terminated string; however, the value may be binary
 *  data.  Also supports multiple values associated with the attribute.
 */

/**
 *  SOIFAVPair_Create() - creates an AVPair structure initialized 
 *  with the given attribute 'a' & value 'v'.  The value 'v' is a
 *  buffer of 'vsz' bytes.
 *
 *  SOIFAVPair_Free() - frees all memory associated with the given AVPair.
 */
NSAPI_PUBLIC SOIFAVPair *SOIFAVPair_Create(const char *a, const char *v, int vsz);
NSAPI_PUBLIC void SOIFAVPair_Free(SOIFAVPair *avp);

struct soif_avpair_s {		/* AVPair object */
	char *attribute;	/* Attribute string; '\0' terminated */
	char *value;		/* primary value; may be '\0' terminated */
	size_t vsize;		/* # of bytes (8 bits) for primary value */

	/* Multivalued attribute support; otherwise just use primary value */
	char **values;		/* Multiple values for multivalued attributes */
	size_t *vsizes;		/* the sizes for the values */
	int nvalues;		/* number of values associated with attribute */
	int last_slot;		/* last valid slot - array may contain holes */

	/**
	 *  add_value() copies the given value to the list of associated 
	 *  values for the attribute.  Returns the index into values[] 
	 *  array for the new value, or -1 on error.
 	 *
	 *  del_value() deletes the given value when given an index 
	 *  into the values[] array.  Use '0' for single valued attributes.
	 *  May re-index/number the values.
	 *
	 *  Example:
	 *	int n = (*avp->add_value)(avp, "foo", 3);
	 *	(*avp->del_value)(avp, 0);
	 */
	int (*add_value)(SOIFAVPair *avp, const char *v, int vsz);
	void (*del_value)(SOIFAVPair *avp, int nth);

	void *internal;		/* private data */
};
/**
 *  SOIF Attribute Names
 *
 *  SOIF_AttributeCompare() - compares two attribute names.  
 *  Returns 0 (zero) if they are equal, or non-zero if they are different.
 *
 *  SOIF_AttributeCompareMV() - is same as AttributeCompare() but
 *  accounts for multivalued attribute names, like Title-1 vs. Title-35.
 */
NSAPI_PUBLIC int SOIF_AttributeCompare(const char *a1, const char *a2);
NSAPI_PUBLIC int SOIF_AttributeCompareMV(const char *a1, const char *a2); 

/**
 *  SOIF Multivalued Attribute 
 *
 *  SOIF supports the convention of using -### to indicate a multivalued 
 *  attribute.  For example, Title-1, Title-2, Title-3, etc.  
 *  The -### do not need to be sequential positive integers 
 *  (e.g., 1, 2, 3, ...); the sequence can have "holes" in it.
 *
 *  SOIF_IsMVAttribute() - returns NULL is the given attribute is not 
 *  a multivalued attribute; otherwise returns a pointer to where the
 *  multivalued number occurs in the attribute string.
 *
 *  SOIF_MVAttributeParse() - returns the multivalued number of the
 *  given attribute, and strips the 'attribute' string of its -### indicator;
 *  otherwise, returns zero in the case of a normal attribute name.
 */
NSAPI_PUBLIC int SOIF_MVAttributeParse(char *a);
NSAPI_PUBLIC char *SOIF_IsMVAttribute(const char *a);

/*
 *
 *  SOIF_InsertMV() - inserts a new value at index MultiValueNumber for the
 *  given attribute (in non-multivalue form).  The useval flag if set tells
 *  the function to use the given value buffer rather than creating its own.
 *
 *  SOIF_SqueezeMV() - forces a re-numbering to ensure that the multivalue 
 *  indexes are sequentially increasing (e.g., 1, 2, 3,...). Used to fill 
 *  in any holes that might have occurred during SOIF_InsertMV()'s.
 *
 *  Example: to insert values explicitly for author-*
 *	SOIF_InsertMV(s, "author", 1, "John", 4, 0);
 *	SOIF_InsertMV(s, "author", 2, "Kevin", 5, 0);
 *	SOIF_InsertMV(s, "author", 6, "Darren", 6, 0);
 *	SOIF_InsertMV(s, "author", 9, "Tommy", 5, 0);
 *	SOIF_FindvalMV(s, "author", 9) == "Tommy";
 *	SOIF_SqueezeMV(s);
 *	SOIF_FindvalMV(s, "author", 9) == NULL;
 *	SOIF_FindvalMV(s, "author", 4) == "Tommy";
 */

NSAPI_PUBLIC int SOIF_InsertMV(SOIF *s, const char *a, int slot, const char *v, int vsz, int useval);
NSAPI_PUBLIC int SOIF_ReplaceMV(SOIF *s, const char *a, int slot, const char *v, int vsz, int useval);
NSAPI_PUBLIC int SOIF_DeleteMV(SOIF *s, const char *a, int slot);
NSAPI_PUBLIC const char *SOIF_FindvalMV(SOIF *s, const char *a, int slot);
NSAPI_PUBLIC void SOIF_SqueezeMV(SOIF *s);

/**
 *  Handling multiple values
 *
 *  Use SOIFAVPair_IsMV() to determine if the AVPair has multiple values or not.
 *  Use SOIFAVPair_NthValid() to determine if the Nth value is valid or not.
 *  Use SOIFAVPair_NthValue() to access the Nth value.
 *  Use SOIFAVPair_NthVsize() to access the size of the Nth value.
 *
 *  Example:  prints out all string values for given attribute
 *	for (i = 0; i <= avp->last_slot; i++)
 *		if (SOIFAVPair_NthValid(avp, i))
 *			printf("%s = %s\n", avp->attribute, 
 *			       SOIFAVPair_NthValue(avp, i));
 */
#define SOIFAVPair_IsMV(avp)	/* multivalued? */ \
   ((((avp)->nvalues == 1) && ((avp)->last_slot != 0)) || ((avp)->nvalues > 1))
#define SOIFAVPair_NthValid(avp,n) \
   (((n) >= 0) && ((n) <= (avp)->last_slot) && ((avp)->values[n] != NULL))
#define SOIFAVPair_NthValue(avp,n)	((avp)->values[n])
#define SOIFAVPair_NthVsize(avp,n)	((avp)->vsizes[n])

/**
 *  To determine if the given attribute contains the given value,
 *  use SOIF_Contains().  It returns B_TRUE if the value matches
 *  one or more of the values assigned to attribute 'a' in the 
 *  given SOIF 's'.
 */
NSAPI_PUBLIC boolean_t SOIF_Contains(SOIF *s, const char *a, const char *v, size_t vsz);

/**
 *  To rename a given attribute, use SOIF_Rename().
 */
NSAPI_PUBLIC int SOIF_Rename(SOIF *s, const char *old_attr, const char *new_attr);

/*  ---------------------------------------------------------------------- */

/** SOIF Stream Parsing and Printing 
 *
 *  Supports the parsing of many SOIF objects from memory- or disk-based
 *  SOIF streams.  Also supports printing SOIF objects to memory or disk.
 *
 *  To start, you need to specify if you're going to be printing or parsing 
 *  a SOIF stream and if you'll be using a memory- or disk-based stream.
 *
 *  For printing streams,
 *	SOIF_PrintInitFile() - creates a disk-based stream ready for printing.
 *	SOIF_PrintInitStr()  - creates a memory-based stream ready for printing.
 *	SOIF_PrintInitFn()   - creates a generic application-defined stream 
 *                             ready for printing.  The given 'write_fn'
 *			       is used to print the stream.
 *
 *  For parsing streams,
 *	SOIF_ParseInitFile() - creates a disk-based stream ready for parsing.
 *	SOIF_ParseInitStr()  - creates a memory-based stream ready for parsing.
 *
 *  To finish, simply close the stream using SOIFStream_Finish().
 *  Use SOIFStream_SetFinishFn() to trigger the given finish_fn function 
 *  call during SOIFStream_Finish().
 */
NSAPI_PUBLIC SOIFStream *SOIF_PrintInitFile(FILE *disk);
NSAPI_PUBLIC SOIFStream *SOIF_PrintInitStr(SOIFBuffer *memory);
NSAPI_PUBLIC SOIFStream *SOIF_PrintInitFn(int (*write_fn)(void *data, char *buf, int bufsz), void *data);
NSAPI_PUBLIC SOIFStream *SOIF_ParseInitFile(FILE *fp);
NSAPI_PUBLIC SOIFStream *SOIF_ParseInitStr(char *buf, int bufsz);
NSAPI_PUBLIC int SOIFStream_Finish(SOIFStream *);
NSAPI_PUBLIC int SOIFStream_SetFinishFn(SOIFStream *, int (*finish_fn)(SOIFStream *));
NSAPI_PUBLIC int SOIFStream_Rewind(SOIFStream *); /* memory streams only for now */

typedef enum soif_stream_enum_type {
	SOIF_UNDEF, SOIF_PRINT, SOIF_PARSE
} SOIFStreamType;

struct soif_stream_s {			/* Stream object */
	SOIFStreamType type;		/* Stream type */

	/** To parse the next SOIF object in the SOIF stream, 
	 *  use parse() which will parse the next object and return it.
	 *
	 *  To print another SOIF object to the SOIF stream, 
	 *  use print() which will perform the print of given object.
	 *  Returns 0 on success, or non-zero on error.
	 *
	 *  Example:
	 *
	 *	SOIFStream *in = SOIF_ParseInitFile(stdin);
	 *	SOIFStream *out = SOIF_PrintInitFile(stdout);
	 *	SOIF *s;
	 *
	 *	while (!SOIFStream_IsEOS(in)) {
	 *		if ((s = (*in->parse)(in))) {
	 *			(*out->print)(out, s);
	 *			SOIF_Free(s);
	 *		}
	 *	}
	 */
	SOIF *(*parse)(SOIFStream *ss);
	int (*print)(SOIFStream *ss, SOIF *s);

	boolean_t (*is_end_of_stream)(SOIFStream *ss);

	void *internal;		/* private data */
	void *caller_data;	/* hook to be used by caller */
};

/** Shortcuts */
#define SOIFStream_Print(_ss, _s)	((*(_ss)->print)((_ss),(_s)))
#define SOIFStream_Parse(_ss)		((*(_ss)->parse)((_ss)))
#define SOIFStream_IsEOS(s) \
	((!(s)) || ((s) && (*(s)->is_end_of_stream)(s)))
#define SOIFStream_IsPrinting(s) 	((s) && ((s)->type == SOIF_PRINT))
#define SOIFStream_IsParsing(s)		((s) && ((s)->type == SOIF_PARSE))

/** To support targeted parsing/printing, use the attribute filtering
 *  mechanisms in the SOIF stream.  For each SOIF stream object, you
 *  can associate a list of "allowed" attributes.  When printing
 *  a SOIF stream, only the attributes which match the "allowed"
 *  attributes will be printed.  When parsing a SOIF stream, only the
 *  attributes which match the "allowed" attributes will be parsed.
 *  Get/SetDenied is the opposite of the Allowed functionality.
 *  Only Allowed or Denied may be active on the stream at any one time.
 */
NSAPI_PUBLIC boolean_t SOIFStream_IsAllowed(SOIFStream *ss, const char *attribute);
NSAPI_PUBLIC int SOIFStream_SetAllowed(SOIFStream *ss, char *allowed_attrs[]);
NSAPI_PUBLIC int SOIFStream_SetDenied(SOIFStream *ss, char *denied_attrs[]);
NSAPI_PUBLIC char **SOIFStream_GetAllowed(SOIFStream *ss);
NSAPI_PUBLIC char **SOIFStream_GetDenied(SOIFStream *ss);

/*  ---------------------------------------------------------------------- */

/** Simple memory buffer management for printing SOIF streams to memory */

NSAPI_PUBLIC SOIFBuffer *SOIFBuffer_Create(int default_sz);
NSAPI_PUBLIC void SOIFBuffer_Free(SOIFBuffer *sb);

struct soif_buffer_s {			/* memory buffer object */
	char *data;			/* data buffer */
	int nbytes; 			/* valid bytes of data in buffer */
	int current; 			/* current byte in stream */

	/** To manipulate the data in the buffer, 
	 *  use append() which copies n bytes of data into the buffer; or
	 *  use increase() to increase the size of the data buffer itself; or
	 *  use reset() to reset the size of the data buffer and invalidate
	 *  all currently valid data.
	 */
	void (*append)(SOIFBuffer *sb, const char *data, int nbytes);
	void (*increase)(SOIFBuffer *sb, int add_nbytes);
	void (*reset)(SOIFBuffer *sb);

	void *internal;		/* private data */
}; 

/*  ---------------------------------------------------------------------- */

/** Debugging support */
NSAPI_PUBLIC void SOIF_Dump(SOIF *s, const char *fn, const char *msg);

#ifdef __cplusplus
};
#endif

#endif /* _RDM_SOIF_H_ */
