# Gate Specific Code -- BEGIN
#
# Copyright 1999-2004 Sun Microsystems, Inc.  All rights reserved.
# Use is subject to license terms.
#
#ident  "@(#)dns_svc_stop.ksh 1.12     04/01/23 SMI"
# Gate Specific Code -- END
#!/bin/ksh
# Sun Cluster Data Services Builder template version 1.0
#
# Stop method for DNS
#

###############################################################################
# Parse program arguments.
#
function parse_args # [args ...]
{
	typeset opt

	while getopts 'R:G:T:' opt
	do
		case "$opt" in
		R)
		# Name of the DNS resource.
		RESOURCE_NAME=$OPTARG
		;;

		G)
		# Name of the resource group in which the resource is
		# configured.
		RESOURCEGROUP_NAME=$OPTARG
		;;

		T)
		# Name of the resource type.
		RESOURCETYPE_NAME=$OPTARG
		;;

		*)
		logger -p ${SYSLOG_FACILITY}.err \
		-t [$RESOURCETYPE_NAME,$RESOURCEGROUP_NAME,$RESOURCE_NAME] \
		"ERROR: Option $OPTARG unknown"
		exit 1
		;;

		esac
	done

}

##############################################################################
# MAIN
##############################################################################

export PATH=/bin:/usr/bin:/usr/cluster/bin:/usr/sbin:/usr/proc/bin:$PATH

# Obtain the syslog facility to use. This will be used to log the messages.
SYSLOG_FACILITY=`scha_cluster_get -O SYSLOG_FACILITY`

# Parse the arguments that have been passed to this method.
parse_args "$@"

# get the Timeout value allowed for stop method from the RTR file
STOP_TIMEOUT=`scha_resource_get -O STOP_TIMEOUT -R $RESOURCE_NAME \
	-G $RESOURCEGROUP_NAME`

# We will try to wait wait for 80% of the stop_method_timeout value when we
# send a SIGTERM through PMF to the Data service. This is to make sure that
# the application stops in a decent manner. If the application does not
# respond favourably to this then we use SIGKILL to stop the data service
# and this will be done for a 15% of the Stop_method_timeout value. However,
# if the data service has not stopped by now, we conclude that there was a
# Failure in the stop method and exit non-zero. The remaining 5% of the 
# stop method timeout is for other needs. 
((SMOOTH_TIMEOUT=$STOP_TIMEOUT * 80/100))

((HARD_TIMEOUT=$STOP_TIMEOUT * 15/100))

PMF_TAG=$RESOURCEGROUP_NAME,$RESOURCE_NAME,0.svc
SYSLOG_TAG=$RESOURCETYPE_NAME,$RESOURCEGROUP_NAME,$RESOURCE_NAME

# We need to know the full path for the gethostnames utility which resides
# in the directory <RT_BASEDIR>. Get this from the RT_BASEDIR property of the
# resource type.
RT_BASEDIR=`scha_resource_get -O RT_BASEDIR -R $RESOURCE_NAME \
        -G $RESOURCEGROUP_NAME`

hostnames=`$RT_BASEDIR/gethostnames -R $RESOURCE_NAME -G $RESOURCEGROUP_NAME \
        -T $RESOURCETYPE_NAME`

stop_cmd_args=

stop_cmd_prog=`echo $stop_cmd_args | nawk '{print $1}'`

typeset -i SEND_KILL=0

# User added code -- BEGIN vvvvvvvvvvvvvvv
# User added code -- END   ^^^^^^^^^^^^^^^

pmfadm -q $PMF_TAG
if [[ $? == 0 ]]; then
	if [[ -f $stop_cmd_prog && -x $stop_cmd_prog ]]; then
		# User added code -- BEGIN vvvvvvvvvvvvvvv
		# User added code -- END   ^^^^^^^^^^^^^^^

		pmfadm -s $PMF_TAG
		if [[ $? != 0  ]]; then
			logger -p ${SYSLOG_FACILITY}.info \
				-t [$SYSLOG_TAG] \
				"Failed to take DNS out of PMF \
				control; trying to send SIGKILL now"
			SEND_KILL=1
		else
			# User added code -- BEGIN vvvvvvvvvvvvvvv
			# User added code -- END   ^^^^^^^^^^^^^^^

			# execute the user specified stop_cmd using hatimerun
			hatimerun -k KILL -t $SMOOTH_TIMEOUT $stop_cmd_args
			if [[ $? != 0 ]]; then
				logger -p ${SYSLOG_FACILITY}.err \
					-t [$SYSLOG_TAG] \
					"Failed to stop DNS using \
					the custom stop command; trying \
					SIGKILL now."
			fi

			# Regardless of whether the command succeeded or not we
			# send KILL signal to the pmf tag. This will ensure
			# that the process tree goes away if it still exists.
			# If it doesn't exist by then, we return NOERR.
			SEND_KILL=1
		fi
	else
		# User added code -- BEGIN vvvvvvvvvvvvvvv
		# User added code -- END   ^^^^^^^^^^^^^^^

		# Send a SIGTERM signal to the Data service and wait for
		# 80% of the total timeout value.
		pmfadm -s $PMF_TAG -w $SMOOTH_TIMEOUT TERM
		# We compare the exit status of pmfadm to be greater than 2
		# because "exit status = 1" means nametag doesn't exist, which
		# is a OK, for the stop method has to be idempotent.
		if [[ $? -ge 2 ]]; then 
			logger -p ${SYSLOG_FACILITY}.err \
				-t [$SYSLOG_TAG] \
				"Failed to stop DNS with SIGTERM; \
				retry with SIGKILL"
			SEND_KILL=1;
		fi
	fi

	pmfadm -q $PMF_TAG
	if [[ $SEND_KILL == 1 && $? == 0 ]]; then 
		# User added code -- BEGIN vvvvvvvvvvvvvvv
		# User added code -- END   ^^^^^^^^^^^^^^^

		# Since the Data service did not stop with a SIGTERM we will
		# use a SIGKILL now and wait for another 15% of total timeout.
		pmfadm -s $PMF_TAG -w $HARD_TIMEOUT KILL
		# We compare the exit status of pmfadm to be greater than 2
		# because "exit status = 1" means nametag doesn't exist, which
		# is a OK, for the stop method has to be idempotent.
		if [[ $? -ge 2 ]]; then
			logger -p ${SYSLOG_FACILITY}.err -t [$SYSLOG_TAG] \
			"Failed to stop DNS; exiting UNSUCCESFUL"
			exit 1
		fi
	fi
else 
	# The Data service is not running as of now. Log a message and 
	# exit suceess.
	logger -p ${SYSLOG_FACILITY}.err -t [$SYSLOG_TAG] \
		"DNS not already running"

	# Even if the DNS is not running, we exit success, for idempotency of
	# the STOP method.
	exit 0
fi

# User added code -- BEGIN vvvvvvvvvvvvvvv
# User added code -- END   ^^^^^^^^^^^^^^^

# could successfully stop DNS. Log a message and exit success.
logger -p ${SYSLOG_FACILITY}.info -t [$SYSLOG_TAG] "DNS successfully stopped"
exit 0
