/*
 * This is a  DataTranslator plugin which can accept Outlook comma-separated-value (CSV) files
 * and convert them to XML format and vice versa.
 * 
 * When generating CSV, it used James Clarks XML expat parser for tokenization
 *
 */


#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "csDataTranslatorCSV.h"
#include "nspr.h"
#include "nsString.h"
#include "plstr.h"
#include "xmlparse.h"
#include "csICalendarServer.h"
#include "csIMalloc.h"

static csIMalloc * mi = NULL;

static NS_DEFINE_IID(kISupportsIID,             NS_ISUPPORTS_IID);
static NS_DEFINE_IID(kCDataTranslatorCSVCID,    CS_IDATATRANSLATOR_CSV_CID);
static NS_DEFINE_IID(kPluginIID,                CS_IPLUGIN_IID);
static NS_DEFINE_IID(kDataTranslatorIID,        CS_IDATATRANSLATOR_IID);
static NS_DEFINE_IID(kICalendarServerIID,       CS_ICALENDARSERVER_IID);
static NS_DEFINE_IID(kIMallocIID,               CS_IMALLOC_IID);

static int process_startdate(token& t, const char * s, PRInt32 l);
static int process_enddate(token& t, const char * s, PRInt32 l);

/*
 * CSV tags & macros
 */
static const char * NEW_LINE      = "\r\n";
static const char * OPEN_QUOTE    = "\"";
static const char * CLOSE_QUOTE   = "\"";
static const char * COMMA         = ",";
static const char * EMPTY_STRING  = "\"\"";
static const char * FALSE_STRING  = "false";
static const char * TRUE_STRING   = "true";

#define BUF_ALLOC_SIZE 4096
#define CSV_HEADER  "\"Subject\",\"Start Date\",\"Start Time\",\"End Date\",\"End Time\",\"All day event\",\"Description\""
#define NEW_LINE_LENGTH 2

/*
 * XML tags & macros
 */
#define ICALENDAR       "icalendar"
#define VEVENT          "vevent"
#define DESCRIPTION     "description"
#define SUMMARY         "summary"
#define DTSTART         "dtstart"
#define DTEND           "dtend"

#define ELEMENT_IS_ICALENDAR(a)   (0 == PL_strcasecmp(a,ICALENDAR))
#define ELEMENT_IS_COMPONENT(a)   (0 == PL_strcasecmp(a,VEVENT))
#define ELEMENT_IS_DESCRIPTION(a) (0 == PL_strcasecmp(a,DESCRIPTION))
#define ELEMENT_IS_SUMMARY(a)     (0 == PL_strcasecmp(a,SUMMARY))
#define ELEMENT_IS_DTSTART(a)     (0 == PL_strcasecmp(a,DTSTART))
#define ELEMENT_IS_DTEND(a)       (0 == PL_strcasecmp(a,DTEND))

/*
 * token handling macros
 */

#define NULLIFY_TOKEN(t) \
  t.subject = NULL; \
  t.startdate = NULL; \
  t.starttime = NULL; \
  t.enddate = NULL; \
  t.endtime = NULL; \
  t.alldayevent = NULL; \
  t.description = NULL; \
  t.tag = CSV_TAG_NONE;

#define FREE_TOKEN_SUBJECT(t) \
  if (t.subject) PL_strfree(t.subject);

#define FREE_TOKEN_STARTDATE(t) \
  if (t.startdate) PL_strfree(t.startdate);

#define FREE_TOKEN_STARTTIME(t) \
  if (t.starttime) PL_strfree(t.starttime);

#define FREE_TOKEN_ENDDATE(t) \
  if (t.enddate) PL_strfree(t.enddate);

#define FREE_TOKEN_ENDTIME(t) \
  if (t.endtime) PL_strfree(t.endtime);

#define FREE_TOKEN_ALLDAYEVENT(t) \
  if (t.alldayevent) PL_strfree(t.alldayevent);

#define FREE_TOKEN_DESCRIPTION(t) \
  if (t.description) PL_strfree(t.description);

#define RESET_TOKEN(t) \
  FREE_TOKEN_SUBJECT(t) \
  FREE_TOKEN_STARTDATE(t) \
  FREE_TOKEN_STARTTIME(t) \
  FREE_TOKEN_ENDDATE(t) \
  FREE_TOKEN_ENDTIME(t) \
  FREE_TOKEN_ALLDAYEVENT(t) \
  FREE_TOKEN_DESCRIPTION(t) \
  NULLIFY_TOKEN(t)

#define SET_TOKEN_TAG(t,g) \
  t.tag = g

#define WRITE_OPEN_QUOTE(o) \
  o->buf_write(OPEN_QUOTE, PL_strlen(OPEN_QUOTE)) ;

#define WRITE_CLOSE_QUOTE(o) \
  o->buf_write(CLOSE_QUOTE, PL_strlen(CLOSE_QUOTE)) ;

#define WRITE_COMMA(o) \
  o->buf_write(COMMA, PL_strlen(COMMA)) ;

#define WRITE_SUBJECT(t,o) \
  WRITE_OPEN_QUOTE(o); \
  if (t.subject) o->buf_write(t.subject, PL_strlen(t.subject)) ;\
  WRITE_CLOSE_QUOTE(o); \
  WRITE_COMMA(o);

#define WRITE_STARTDATE(t,o) \
  WRITE_OPEN_QUOTE(o); \
  if (t.startdate) o->buf_write(t.startdate, PL_strlen(t.startdate)) ;\
  WRITE_CLOSE_QUOTE(o); \
  WRITE_COMMA(o);

#define WRITE_STARTTIME(t,o) \
  WRITE_OPEN_QUOTE(o); \
  if (t.starttime) o->buf_write(t.starttime, PL_strlen(t.starttime)) ;\
  WRITE_CLOSE_QUOTE(o); \
  WRITE_COMMA(o);

#define WRITE_ENDDATE(t,o) \
  WRITE_OPEN_QUOTE(o); \
  if (t.enddate) o->buf_write(t.enddate, PL_strlen(t.enddate)) ;\
  WRITE_CLOSE_QUOTE(o); \
  WRITE_COMMA(o);

#define WRITE_ENDTIME(t,o) \
  WRITE_OPEN_QUOTE(o); \
  if (t.endtime) o->buf_write(t.endtime, PL_strlen(t.endtime)) ;\
  WRITE_CLOSE_QUOTE(o); \
  WRITE_COMMA(o);

#define WRITE_ALLDAYEVENT(t,o) \
  WRITE_OPEN_QUOTE(o); \
  if (t.alldayevent) o->buf_write(t.alldayevent, PL_strlen(t.alldayevent)) ;\
  WRITE_CLOSE_QUOTE(o); \
  WRITE_COMMA(o);

#define WRITE_DESCRIPTION(t,o) \
  WRITE_OPEN_QUOTE(o); \
  if (t.description) o->buf_write(t.description, PL_strlen(t.description)) ;\
  WRITE_CLOSE_QUOTE(o); \

#define WRITE_TOKEN(t,o) \
  WRITE_SUBJECT(t,o) \
  WRITE_STARTDATE(t,o) \
  WRITE_STARTTIME(t,o) \
  WRITE_ENDDATE(t,o) \
  WRITE_ENDTIME(t,o) \
  WRITE_ALLDAYEVENT(t,o) \
  WRITE_DESCRIPTION(t,o)

#define STORE_TOKEN_DESCRIPTION(t,s,l) \
  { FREE_TOKEN_DESCRIPTION(t) \
  t.description = PL_strndup(s,l);}

#define STORE_TOKEN_SUBJECT(t,s,l) \
  { FREE_TOKEN_SUBJECT(t) \
  t.subject = PL_strndup(s,l);}

#define STORE_TOKEN_STARTDATE(t,s,l) \
  { FREE_TOKEN_STARTDATE(t) \
  process_startdate(t,s,l);}

#define STORE_TOKEN_STARTTIME(t,s,l) \
  { FREE_TOKEN_STARTTIME(t) \
  t.starttime = PL_strndup(s,l);}

#define STORE_TOKEN_ENDDATE(t,s,l) \
  { FREE_TOKEN_ENDDATE(t) \
  process_enddate(t,s,l);}

#define STORE_TOKEN_ENDTIME(t,s,l) \
  { FREE_TOKEN_ENDTIME(t) \
  t.endtime = PL_strndup(s,l);}

#define STORE_TOKEN_ALLDAYEVENT(t,s,l) \
  { FREE_TOKEN_ALLDAYEVENT(t) \
  t.alldayevent = PL_strndup(s,l);}

#define STORE_TOKEN(t,s,l) \
  if (t.tag == CSV_TAG_DESCRIPTION) STORE_TOKEN_DESCRIPTION(t,s,l); \
  if (t.tag == CSV_TAG_SUBJECT) STORE_TOKEN_SUBJECT(t,s,l); \
  if (t.tag == CSV_TAG_STARTDATE) STORE_TOKEN_STARTDATE(t,s,l); \
  if (t.tag == CSV_TAG_STARTTIME) STORE_TOKEN_STARTTIME(t,s,l); \
  if (t.tag == CSV_TAG_ENDDATE) STORE_TOKEN_ENDDATE(t,s,l); \
  if (t.tag == CSV_TAG_ENDTIME) STORE_TOKEN_ENDTIME(t,s,l); \
  if (t.tag == CSV_TAG_ALLDAYEVENT) STORE_TOKEN_ALLDAYEVENT(t,s,l);


/*
 * local callback functions for XML Parser
 */
void startElement(void *userData, const char *name, const char **atts);
void endElement(void *userData, const char *name);
void processingInstruction(void *userData, const char *target, const char *data);
void characterData(void *userData, const char *s, int len);

static PRFileDesc *fd_stdout = NULL;

csDataTranslatorCSV :: csDataTranslatorCSV()
{
  NS_INIT_REFCNT();
  output_buffer = NULL;
  output_size = 0;
  buf_ptr = 0;
  fd_stdout = PR_GetSpecialFD(PR_StandardError);
  debug = PR_FALSE;
  bInComponent = PR_FALSE;

  NULLIFY_TOKEN(cur_token);

}

csDataTranslatorCSV :: ~csDataTranslatorCSV()
{
  if (mi)
    mi->Release();
}

nsresult csDataTranslatorCSV :: QueryInterface(REFNSIID aIID, void** aInstancePtr)
{

  if (NULL == aInstancePtr)
    return NS_ERROR_NULL_POINTER;

  if (aIID.Equals(kDataTranslatorIID))
    *aInstancePtr = (void*)((csIDataTranslator *)this);
  else if (aIID.Equals(kISupportsIID))
    *aInstancePtr = (void*) ((nsISupports*)(csIDataTranslator *)this);
  else if (aIID.Equals(kPluginIID))
    *aInstancePtr = (void*) ((nsISupports*)(csIPlugin *)this);
  else
    return (NS_NOINTERFACE);  

  NS_ADDREF_THIS();
  return NS_OK;
}

NS_IMPL_ADDREF(csDataTranslatorCSV)
NS_IMPL_RELEASE(csDataTranslatorCSV)

nsresult csDataTranslatorCSV :: Init(nsISupports * aServer)
{
  nsresult res = NS_COMFALSE ;
  PRUint32 min, maj;

  csICalendarServer * cs;

  /* Here we QueryInterface for the CalendarServer. If this call succeeds, we will have
     bumped the reference count of the server by 1 
  */
  if (aServer)
    res = aServer->QueryInterface(kICalendarServerIID,(void**)&cs);

  /* If we succeeded in getting a reference to the server, let's check it's version */
  if (NS_SUCCEEDED(res))
  {
    
    cs->GetVersion(maj,min);

    if (min > 0 && maj >= 1)
      res = NS_OK;
    else
      res = NS_COMFALSE;

    /* Release our reference to this server instance */
    cs->Release();
  }

  /* If we are happy about this version of the server, let's get a
     pointer to the malloc interface to use CSAPI's super fast memory allocation
     routines :-) */
  if (NS_OK == res)
    res = aServer->QueryInterface(kIMallocIID,(void**)&mi);

  return res;
}

NS_IMETHODIMP csDataTranslatorCSV :: GetDescription(nsString& aDescription)
{
  aDescription = "The CSV Data Translator Plugin";
  return NS_OK;
}

NS_IMETHODIMP csDataTranslatorCSV :: GetVendorName(nsString& aVendorName)
{
  aVendorName = "Netscape Communications Corporation";
  return NS_OK;
}

NS_IMETHODIMP csDataTranslatorCSV :: GetVersion(PRUint32& aMajorValue, PRUint32& aMinorValue) 
{
  aMajorValue = 2;
  aMinorValue = 0;
  return NS_OK;
}

NS_IMETHODIMP csDataTranslatorCSV :: Translate(char * aInContentType, 
                                               char * aOutContentType, 
                                               char ** aInBuffer, 
                                               char ** aOutBuffer, 
                                               PRInt32 * aInSize,
                                               PRInt32 * aOutSize,
                                               PRInt32 * aReturnCode)
{

  *aOutSize = 0;
  *aReturnCode = 1;

  (void*) aInBuffer;
  (void*) aOutBuffer;

  if ((0 == PL_strcmp(aInContentType,"text/xml")) && (0 == PL_strcmp(aOutContentType,"text/csv")))
  {
    /* Convert from XML to CSV */
    if (*aInBuffer)
    {

      buf_ptr = 0;
      bInComponent = PR_FALSE;

      output_buffer = (char*) PR_Calloc(1,BUF_ALLOC_SIZE);
      output_size = BUF_ALLOC_SIZE;

      *aOutSize = convert_xml_to_csv(aInBuffer, &output_buffer, *aInSize);

      *aOutBuffer = output_buffer;
      *aReturnCode = 0;
    }

  }

  return NS_OK;
}

NS_IMETHODIMP csDataTranslatorCSV :: GetSupportedContentTypes(char ** aSupportedInContentTypes, 
                                                              char ** aSupportedOutContentType,
                                                              char ** aPreferredInContentType,
                                                              PRInt32 * aReturnCode) 
{
  nsresult res = NS_COMFALSE;
  PRBool bSupportsXML = PR_FALSE;

  char ** ep = aSupportedInContentTypes;
  char * cp = *ep;

  while(cp != NULL)
  {
    if (PL_strcmp(cp,"text/xml") == 0)
      bSupportsXML = PR_TRUE;

    ep++;
    cp = *ep;
  }


  if (PR_TRUE == bSupportsXML)
  {
    *aSupportedOutContentType = (char*) calloc(1, (PL_strlen("text/csv")+1)*(sizeof(char)));
    PL_strcpy(*aSupportedOutContentType, "text/csv");

    *aPreferredInContentType = (char*) calloc(1, (PL_strlen("text/xml")+1)*(sizeof(char)));
    PL_strcpy(*aPreferredInContentType, "text/xml");

    res = NS_OK;
  } 

  *aReturnCode = 0;

  return res;
}


NS_IMETHODIMP_(PRInt32) csDataTranslatorCSV :: convert_xml_to_csv(char ** inbuf, char ** outbuf, PRInt32 insize)
{
  (char **) outbuf;

  output_written = 0;

  XML_Parser parser = XML_ParserCreate(NULL);
  mparser = parser;
  XML_SetUserData(parser, this);
  XML_SetElementHandler(parser, startElement, endElement);
  XML_SetCharacterDataHandler(parser, characterData);
  XML_SetProcessingInstructionHandler(parser, processingInstruction);

  /*
   * Before parsing the XML, first write out the CSV headers
   */

  output_csv_header();

  if (!XML_Parse(parser, *inbuf, insize, PR_TRUE)) 
  {
    fprintf(stderr, "XML Parser: Converting to CSV: %s at line %d and column %d\n",
	          XML_ErrorString(XML_GetErrorCode(parser)),
	          XML_GetCurrentLineNumber(parser),
            XML_GetCurrentColumnNumber(parser));
    return 1;
  }

  XML_ParserFree(parser);

  return (output_written);
}

NS_IMETHODIMP csDataTranslatorCSV :: output_csv_header()
{
  buf_write(CSV_HEADER, PL_strlen(CSV_HEADER));
  return NS_OK;
}

NS_IMETHODIMP csDataTranslatorCSV :: buf_write(const char * data, PRInt32 len)
{
  /* realloc? */
  if (len > (output_size - output_written))
  {
    output_size += (len+BUF_ALLOC_SIZE);
    output_buffer = (char*) PR_Realloc((void*)output_buffer,output_size);    
  }

  memcpy(output_buffer+buf_ptr, data, len); 
  buf_ptr += len;
  output_written += len;

  return NS_OK;
}

void startElement(void *userData, const char *name, const char **atts)
{
  csDataTranslatorCSV * obj = (csDataTranslatorCSV*) userData;

  (const char **) atts;

  if (obj->debug)
    PR_fprintf(fd_stdout,"startElement(%d,%d): name = %s\n",XML_GetCurrentLineNumber(obj->mparser), XML_GetCurrentColumnNumber(obj->mparser), name);

  if (ELEMENT_IS_ICALENDAR(name))
    obj->buf_write(NEW_LINE, NEW_LINE_LENGTH);

  if (ELEMENT_IS_COMPONENT(name))
  {
    obj->buf_write(NEW_LINE, NEW_LINE_LENGTH);
    obj->bInComponent = PR_TRUE;
    RESET_TOKEN(obj->cur_token);
  }

  if (ELEMENT_IS_DESCRIPTION(name))
    SET_TOKEN_TAG(obj->cur_token,CSV_TAG_DESCRIPTION);
  else if (ELEMENT_IS_SUMMARY(name))
    SET_TOKEN_TAG(obj->cur_token,CSV_TAG_SUBJECT);
  else if (ELEMENT_IS_DTSTART(name))
    SET_TOKEN_TAG(obj->cur_token,CSV_TAG_STARTDATE);
  else if (ELEMENT_IS_DTEND(name))
    SET_TOKEN_TAG(obj->cur_token,CSV_TAG_ENDDATE);
  else
    SET_TOKEN_TAG(obj->cur_token,CSV_TAG_NONE);
}

void endElement(void *userData, const char *name)
{
  csDataTranslatorCSV * obj = (csDataTranslatorCSV*) userData;

  if (obj->debug)
    PR_fprintf(fd_stdout,"endElement(%d,%d):   name = %s\n",XML_GetCurrentLineNumber(obj->mparser), XML_GetCurrentColumnNumber(obj->mparser), name);

  if (ELEMENT_IS_ICALENDAR(name))
    obj->buf_write(NEW_LINE, NEW_LINE_LENGTH);

  if (ELEMENT_IS_COMPONENT(name))
  {
    obj->bInComponent = PR_FALSE;
    WRITE_TOKEN(obj->cur_token,obj);
    obj->buf_write(NEW_LINE, NEW_LINE_LENGTH);
    RESET_TOKEN(obj->cur_token);
  }

  SET_TOKEN_TAG(obj->cur_token,CSV_TAG_NONE);

}

void processingInstruction(void *userData, const char *target, const char *data)
{
  csDataTranslatorCSV * obj = (csDataTranslatorCSV*) userData;

  if (obj->debug)
    PR_fprintf(fd_stdout,"processingInstruction: target = %s;; data = %s\n",target,data);

  return;
}

void characterData(void *userData, const char *s, int len)
{
  csDataTranslatorCSV * obj = (csDataTranslatorCSV*) userData;

  if (obj->debug)
    PR_fprintf(fd_stdout,"characterData: s = %c",*s);

  STORE_TOKEN(obj->cur_token, s, len);  

  return;
}

static int extract_date(const char * s, PRInt32 l, char * cdate)
{
  char xdate[8];
  PRInt32 rc = 0;

  if (l >= 8)
  {    
    PL_strncpy(xdate,s,8);

    cdate[0] = xdate[4];
    cdate[1] = xdate[5];
    cdate[2] = '/';
    cdate[3] = xdate[6];
    cdate[4] = xdate[7];
    cdate[5] = '/';
    cdate[6] = xdate[0];
    cdate[7] = xdate[1];
    cdate[8] = xdate[2];
    cdate[9] = xdate[3];
    cdate[10] = '\0';

    rc = 1;
  }

  return rc;
}

static int extract_time(const char * s, PRInt32 l, char * ctime)
{
  PRInt32 rc = 0;
  char xtime[6];

  if (l >= 9)
  {
    char t = s[8];

    if ((t == 'T' || t == 't') && (l >= 15))
    {
      PRInt32 i;
      char hour[3];

      PL_strncpy(xtime,s+9,6);

      ctime[0] = xtime[0];
      ctime[1] = xtime[1];
      ctime[2] = ':';
      ctime[3] = xtime[2];
      ctime[4] = xtime[3];
      ctime[5] = ' ';
      ctime[7] = 'M';
      ctime[8] = '\0';

      hour[0] = ctime[0];
      hour[1] = ctime[1];
      hour[3] = '\0';

      i = atoi(hour);

      if (i > 11)
        ctime[6] = 'P';
      else
        ctime[6] = 'A';

      rc = 1;

    }

  }

  return rc;
}

static int process_startdate(token& t, const char * s, PRInt32 l)
{
  char cdate[11];
  char ctime[9];
  PRInt32 rc;

  rc = extract_date(s,l,cdate);

  if (rc)
  {
    t.startdate = PL_strndup(cdate,PL_strlen(cdate));
  }

  rc = extract_time(s,l,ctime);

  if (rc)
  {
    t.starttime = PL_strndup(ctime,PL_strlen(ctime));
    t.alldayevent = PL_strndup(FALSE_STRING,PL_strlen(FALSE_STRING));
  } else {
    t.alldayevent = PL_strndup(TRUE_STRING,PL_strlen(TRUE_STRING));
  }


  return 0;
}

static int process_enddate(token& t, const char * s, PRInt32 l)
{
  char cdate[11];
  char ctime[9];
  PRInt32 rc;

  rc = extract_date(s,l,cdate);

  if (rc)
  {
    t.enddate = PL_strndup(cdate,PL_strlen(cdate));
  }

  rc = extract_time(s,l,ctime);

  if (rc)
  {
    t.endtime = PL_strndup(ctime,PL_strlen(ctime));
  } 
  
  return 0;
}
