#include "nsIFactory.h"
#include "nsISupports.h"
#include "csIPlugin.h"
#include "csIDataTranslator.h"

#include "csDataTranslator.h"

// interfaces
static NS_DEFINE_IID(kISupportsIID,         NS_ISUPPORTS_IID);
static NS_DEFINE_IID(kIFactoryIID,          NS_IFACTORY_IID);
static NS_DEFINE_IID(kIPluginIID,           CS_IPLUGIN_IID);
static NS_DEFINE_IID(kIDataTranslatorIID,   CS_IDATATRANSLATOR_IID);

// implementations
static NS_DEFINE_IID(kCDataTranslator,  CS_IDATATRANSLATOR_CID);

class csDataTranslatorFactory : public nsIFactory
{   
  public:   
    // nsISupports methods   
    NS_IMETHOD QueryInterface(const nsIID &aIID,    
                              void **aResult);   
    NS_IMETHOD_(nsrefcnt) AddRef(void);   
    NS_IMETHOD_(nsrefcnt) Release(void);   

    // nsIFactory methods   
    NS_IMETHOD CreateInstance(nsISupports *aOuter,   
                              const nsIID &aIID,   
                              void **aResult);   

    NS_IMETHOD LockFactory(PRBool aLock);   

    csDataTranslatorFactory(const nsCID &aClass);   
    ~csDataTranslatorFactory();   

  private:   
    nsrefcnt  mRefCnt;   
    nsCID     mClassID;
};   

csDataTranslatorFactory::csDataTranslatorFactory(const nsCID &aClass)   
{   
  mRefCnt = 0;
  mClassID = aClass;
}   

csDataTranslatorFactory::~csDataTranslatorFactory()   
{   
  NS_ASSERTION(mRefCnt == 0, "non-zero refcnt at destruction");   
}   

nsresult csDataTranslatorFactory::QueryInterface(const nsIID &aIID,   
                                                 void **aResult)   
{   
  if (aResult == NULL) {   
    return NS_ERROR_NULL_POINTER;   
  }   

  // Always NULL result, in case of failure   
  *aResult = NULL;   

  if (aIID.Equals(kISupportsIID)) {   
    *aResult = (void *)(nsISupports*)this;   
  } else if (aIID.Equals(kIFactoryIID)) {   
    *aResult = (void *)(nsIFactory*)this;   
  }   

  if (*aResult == NULL) {   
    return NS_NOINTERFACE;   
  }   

  AddRef(); // Increase reference count for caller   
  return NS_OK;   
}   

nsrefcnt csDataTranslatorFactory::AddRef()   
{   
  return ++mRefCnt;   
}   

nsrefcnt csDataTranslatorFactory::Release()   
{   
  if (--mRefCnt == 0) {   
    delete this;   
    return 0; // Don't access mRefCnt after deleting!   
  }   
  return mRefCnt;   
}  

nsresult csDataTranslatorFactory::CreateInstance(nsISupports *aOuter,  
                                                 const nsIID &aIID,  
                                                 void **aResult)  
{  
  (nsISupports*) aOuter;

  if (aResult == NULL) {  
    return NS_ERROR_NULL_POINTER;  
  }  

  *aResult = NULL;  
  
  nsISupports *inst = NULL;

  if (mClassID.Equals(kCDataTranslator) || mClassID.Equals(kIDataTranslatorIID)) {
    inst = (nsISupports *)(csIDataTranslator*)new csDataTranslator();
  } else if (mClassID.Equals(kIPluginIID)) {
    inst = (nsISupports *)(csIPlugin*)new csDataTranslator();
  }

  if (inst == NULL) {  
    return NS_ERROR_OUT_OF_MEMORY;  
  }  

  nsresult res = inst->QueryInterface(aIID, aResult);

  if (res != NS_OK) {  
    // We didn't get the right interface, so clean up  
    delete inst;  
  }  

  return res;  
}  

nsresult csDataTranslatorFactory::LockFactory(PRBool aLock)  
{  
  // Not implemented in simplest case.  
  (PRBool) aLock;
  return NS_OK;
}  

// return the proper factory to the caller
extern "C" NS_EXPORT nsresult NSGetFactory(const nsCID &aClass, 
                                           nsISupports* aServiceMgr,
                                           nsIFactory **aFactory)
{
  (nsISupports*) aServiceMgr;

  if (NULL == aFactory) {
    return NS_ERROR_NULL_POINTER;
  }

  *aFactory = new csDataTranslatorFactory(aClass);

  if (NULL == aFactory) {
    return NS_ERROR_OUT_OF_MEMORY;
  }

  return (*aFactory)->QueryInterface(kIFactoryIID, (void**)aFactory);
}

