#include "nsIFactory.h"
#include "nsISupports.h"
#include "csIPlugin.h"
#include "csIAccessControl.h"
#include "csAccessControl.h"

// interfaces
static NS_DEFINE_IID(kISupportsIID,         NS_ISUPPORTS_IID);
static NS_DEFINE_IID(kIFactoryIID,          NS_IFACTORY_IID);
static NS_DEFINE_IID(kIPluginIID,           CS_IPLUGIN_IID);
static NS_DEFINE_IID(kIAccessControlIID,    CS_IACCESSCONTROL_IID);

// implementations
static NS_DEFINE_IID(kCAccessControl,  CS_IACCESSCONTROL_CID);

class csAccessControlFactory : public nsIFactory
{   
  public:   
    // nsISupports methods   
    NS_IMETHOD QueryInterface(const nsIID &aIID,    
                              void **aResult);   
    NS_IMETHOD_(nsrefcnt) AddRef(void);   
    NS_IMETHOD_(nsrefcnt) Release(void);   

    // nsIFactory methods   
    NS_IMETHOD CreateInstance(nsISupports *aOuter,   
                              const nsIID &aIID,   
                              void **aResult);   

    NS_IMETHOD LockFactory(PRBool aLock);   

    csAccessControlFactory(const nsCID &aClass);   
    ~csAccessControlFactory();   

  private:   
    nsrefcnt  mRefCnt;   
    nsCID     mClassID;
};   

csAccessControlFactory::csAccessControlFactory(const nsCID &aClass)   
{   
  mRefCnt = 0;
  mClassID = aClass;
}   

csAccessControlFactory::~csAccessControlFactory()   
{   
  NS_ASSERTION(mRefCnt == 0, "non-zero refcnt at destruction");   
}   

nsresult csAccessControlFactory::QueryInterface(const nsIID &aIID,   
                                                 void **aResult)   
{   
  if (aResult == NULL) {   
    return NS_ERROR_NULL_POINTER;   
  }   

  // Always NULL result, in case of failure   
  *aResult = NULL;   

  if (aIID.Equals(kISupportsIID)) {   
    *aResult = (void *)(nsISupports*)this;   
  } else if (aIID.Equals(kIFactoryIID)) {   
    *aResult = (void *)(nsIFactory*)this;   
  }   

  if (*aResult == NULL) {   
    return NS_NOINTERFACE;   
  }   

  AddRef(); // Increase reference count for caller   
  return NS_OK;   
}   

nsrefcnt csAccessControlFactory::AddRef()   
{   
  return ++mRefCnt;   
}   

nsrefcnt csAccessControlFactory::Release()   
{   
  if (--mRefCnt == 0) {   
    delete this;   
    return 0; // Don't access mRefCnt after deleting!   
  }   
  return mRefCnt;   
}  

nsresult csAccessControlFactory::CreateInstance(nsISupports *aOuter,  
                                                 const nsIID &aIID,  
                                                 void **aResult)  
{  
  (nsISupports*) aOuter;

  if (aResult == NULL) {  
    return NS_ERROR_NULL_POINTER;  
  }  

  *aResult = NULL;  
  
  nsISupports *inst = NULL;

  if (mClassID.Equals(kCAccessControl) || mClassID.Equals(kIAccessControlIID)) {
    inst = (nsISupports *)(csIAccessControl*)new csAccessControl();
  } else if (mClassID.Equals(kIPluginIID)) {
    inst = (nsISupports *)(csIPlugin*)new csAccessControl();
  }

  if (inst == NULL) {  
    return NS_ERROR_OUT_OF_MEMORY;  
  }  

  nsresult res = inst->QueryInterface(aIID, aResult);

  if (res != NS_OK) {  
    // We didn't get the right interface, so clean up  
    delete inst;  
  }  

  return res;  
}  

nsresult csAccessControlFactory::LockFactory(PRBool aLock)  
{  
  // Not implemented in simplest case.  
  (PRBool) aLock;
  return NS_OK;
}  

// return the proper factory to the caller
extern "C" NS_EXPORT nsresult NSGetFactory(const nsCID &aClass, 
                                           nsISupports* aServiceMgr,
                                           nsIFactory **aFactory)
{
  (nsISupports*) aServiceMgr;

  if (NULL == aFactory) {
    return NS_ERROR_NULL_POINTER;
  }

  *aFactory = new csAccessControlFactory(aClass);

  if (NULL == aFactory) {
    return NS_ERROR_OUT_OF_MEMORY;
  }

  return (*aFactory)->QueryInterface(kIFactoryIID, (void**)aFactory);
}

