#!/bin/sh
# Copyright 12/24/03 Sun Microsystems, Inc. All Rights Reserved.
#pragma ident  "@(#)es-commonsetup.sh	1.5 03/12/24 Sun Microsystems"

#
# validate whether SunMC server has been setup.
#
validate_sunmc_setup() {
    # insure that core is setup properly
    tmp_file="${ESROOT}/sbin/es-device"
    if [ ! -x ${tmp_file} ]; then
        echolog 'Invalid Sun Management Center server setup.'
        echolog 'ERROR: no execute permission ${tmp_file}.'
        exit 1
    fi

    # insure that core is setup properly
    if [ ! -d ${VAROPTDIR} ]; then
        echolog 'Invalid Sun Management Center server setup.'
        tmp_file="${VAROPTDIR}"
        echolog '${tmp_file} does not exist.'
        exit 1
    fi
}

#
# To get a eservice composite configuration file
#
pre_eservice_object() {
    #
    # insure ${VAROPTDIR}/cfg/E*Service-composite.conf exists
    #
    SRC="${ESROOT}/classes/base/console/cfg/eservices-composite.conf"
    TAR="${VAROPTDIR}/cfg/eservices-composite.conf"
    if [ ! -f "${TAR}" ]; then
        /usr/bin/cat "${SRC}" | /usr/bin/sed -e "s|##BASEDIR##|$ESROOT|g" > "${TAR}"
        if [ $? -ne 0 ]; then
             echolog "Replacing installed directory has error."
             exit 1
        fi
    fi

    # the file should exist
    if [ -f "${TAR}" ]; then
        /usr/bin/chmod a=,u=rw "${TAR}"
    else
       tmp_file="${TAR}"
       echolog 'Setup is not able to create ${tmp_file}.'
       exit 1
    fi
}

#
# To Modify a right click menu for eservice object in 
# E*Service family configuration file
#
after_eservice_object() {
    #
    # insure ${VAROPTDIR}/cfg/EService family file exists
    #
    SRC="${VAROPTDIR}/cfg/EServices_Composite_Object_Type-family-j.x"
    if [ ! -f "${SRC}" ]; then
        echolog 'Family file for Service Availability Manager does not exist.'
        exit 1
    fi

    SRCOLD=$SRC.old
    TAR=${SRC}
    /bin/mv ${SRC} ${SRCOLD}
    if [ ! -f "${TAR}" ]; then
        /bin/cat ${SRCOLD} | sed -e "s|pasteinto alarmaction|pasteinto alarmaction modify eserv|" > ${TAR}
        if [ $? -ne 0 ]; then
             echolog "Adding a new menu item for Service Availability Manager has generated error."
             exit 1
        fi

        /bin/echo "commandLabel(modify) = base.console.eservice.Eservice:menuitem.modifyLabel" >> ${TAR} 
        /bin/echo "commandSpec(modify) = launchApp com.sun.symon.base.console.eservice.CeEsObjectMain %nodeName  %targetUrl" >> ${TAR}

        /bin/echo "commandLabel(eserv) = base.console.eservice.Eservice:menuitem.label" >> ${TAR} 
        /bin/echo "commandSpec(eserv) = launchApp com.sun.symon.base.console.eservice.CeEserviceMain %nodeName  %targetUrl" >> ${TAR}
    fi

    # set permission for TAR
    if [ -f "${TAR}" ]; then
        /usr/bin/chmod 444 "${TAR}"
    fi
}

#
#  es-device parse eservice composite object
#
setup_eservice_object() {
    pre_eservice_object

    echolog "Setup for Service Availability Manager in progress, please wait."

    #
    # attempt to remove previous device information, ignore errors
    #
    tmp_file="${ESROOT}/sbin/es-device"
    addon_wrapper "${tmp_file} -d ${VAROPTDIR}/cfg/eservices-composite.conf" > /dev/null 2>&1

    #
    # delete old EServices_Composite_Object_Type-family-j.x
    #
    rm -fr ${VAROPTDIR}/cfg/EServices_Composite_Object_Type-family-j.x.old

    #
    # now add device information, do not suppress errors
    #
    addon_wrapper "${tmp_file} -a ${VAROPTDIR}/cfg/eservices-composite.conf"
}

#
# configure E*Service Object
#
configure_eservice_object() {
    #
    # attempt to remove previous eservice composite object
    #
    tmp_file="${ESROOT}/sbin/es-dt"
    addon_wrapper "${tmp_file} -d -l EServicesObject" > /dev/null 2>&1

    #
    # add composite eservice composite object
    #
    addon_wrapper "${tmp_file} -a -h ${localhost} -p ${agent_port} -n EServices_Composite_Object_Type -o 1.3.6.1.4.1.42.2.12.2.2.78.1 -l EServicesObject" > /dev/null 2>&1
}

#
# add edot module in base-module-d.dat
#
add_edot_module() {
	${BASEDIR}/sbin/es-load-default -a -f ${BASEDIR}/addons/EServices/sbin/edot.minfo
}

#
# setup at Server's side
#
server_setup()
{
    validate_sunmc_setup

    # Add the module info into the database. Do it everytime in case
    # the previous SunMC server layer setup cleaned up the database.
    ${BASEDIR}/lib/sbin/es-add-module ${BASEDIR}/addons/EServices/sbin/modules.info > /dev/null
    if [ $? -ne 0 ] ; then
        echolog "Service Availability Manager setup failed."
        exit 1
    fi
    echolog ""
    # only need to setup server package
    setup_eservice_object

    # Modify E*service family file
    after_eservice_object

    # Add help entries.
    if [ -x ${ESROOT}/sbin/es-chelp ] ; then
   	${ESROOT}/sbin/es-chelp -a eservice-manager-help sam/es-using-1.html#I997547
    	${ESROOT}/sbin/es-chelp -a eservice-modifier-help sam/es-using-1.html#I997597
    	${ESROOT}/sbin/es-chelp -a eservice-filter-help  sam/es-using-1.html#I997408
    fi

    # add line to .license.packs
    addLicensePack "a Service Availability Manager"
}

#
# Process available SAM data. Decide if SAM data migration is possible.
# Ask user for input, if necessary.
# Based on a return value, gui setup will set import_sam_data flag to 1 or 0.
#
# Used by GUI SAM setup.
#
# Return codes:   3 - Core set migrate flag to 1 (=true)
#                     Previous SAM data to be migrated exist.
#                     Do not ask migration question, just proceed to import SAM data.
#                 2 - Agent-only setup. Core MIGRATE_DATA flag is not set.
#                     Previous SAM data to be migrated exist.
#                     Ask user if data needs to be migrated.
#                 0 - Core migrate flag is set to 0 (=false).
#                     AND/OR Previous SAM data to be migrated does not exist.
#                     Do not ask user a question and do not migrate data.
#                     

detect_sam_agent_data_gui ()
{
   upgrade_sam_dir="$VAROPTDIR/upgrade/addons/EService"

   # Preserve data flag can be set by the core.
   # If flag is already set we do not attempt to re-set it, just set the
   # import_sam_data value based on MIGRATE_DATA already set by core.

   if [ ! -z "$MIGRATE_DATA" -a "$MIGRATE_DATA" -eq  1 ] ; then

      if [ ! -d ${upgrade_sam_dir} ] ; then
         return 0
      else
         return 3
      fi
   fi

   if [ -z "$MIGRATE_DATA" -a -d ${upgrade_sam_dir} ] ; then
      # Addon only setup.
      # need to ask user "Do you want to migrate the Service Availability Manager data"
      return 2
   else
      # MIGRATE_DATA set to 0. Meaning that during Core setup
      # User chose not to migrate data.
      return 0
   fi

}

#
# Import agent data, if necessary
# Used by GUI and command line SAM setup.
#
detect_sam_agent_data ()
{

   import_sam_data=0
   upgrade_sam_dir="$VAROPTDIR/upgrade/addons/EService"

   # Preserve data flag can be set by the core.
   # If flag is already set we do not attempt to re-set it, just set the
   # import_sam_data value based on MIGRATE_DATA already set by core.

   if [ ! -z "$MIGRATE_DATA" -a "$MIGRATE_DATA" -eq  1 ] ; then

      if [ ! -d ${upgrade_sam_dir} ] ; then
         echologverbose "There is no previous data to migrate for the Service Availability Manager addon"
      else
         import_sam_data=1
      fi
   fi
   
   if [ -z "$MIGRATE_DATA" -a -d ${upgrade_sam_dir} ] ; then 
      # Addon only setup.
      ask_user "Do you want to migrate the Service Availability Manager data"
      if [ $answer -eq 1 ] ; then
         import_sam_data=1
      fi
         
   fi

}

#
# Function used to migrate SAM data, based on the 
# value of the import_sam_data parameter.
# Input parameters : $1 - import_sam_data.
#                    0 - do not migrate data, just clean up upgrade directory for SAM only.
#                    1 - import SAM data, clean up upgrade directory for SAM only.
#

process_sam_agent_data ()
{

   # Restore SAM data.
   import_sam_data=$1
   upgrade_sam_dir="$VAROPTDIR/upgrade/addons/EService"

   if [ $import_sam_data -eq 1 ]; then

      # Restore SAM related module files and merge base-modules-d.dat.
      /usr/bin/cp -f `/usr/bin/ls -1 ${upgrade_sam_dir}/*  | /usr/bin/grep -v base-modules-d.dat` $VAROPTDIR/cfg 

      # Merge the base-modules-d.dat file
      if [ -f ${upgrade_sam_dir}/base-modules-d.dat ] ; then

          echologverbose 'Merging base-modules-d.dat'

          sam_modules_to_mig="httpST httpSE ftpST ftpSE dnsST dnsSE telnetST telnetSE smtpST smtpSE nisST nisSE ldapST ldapSE imapSE imapST popST popSE calendarST calendarSE edot"

          for each_sam_module in $sam_modules_to_mig
          do

             /usr/bin/grep "^${each_sam_module}" ${upgrade_sam_dir}/base-modules-d.dat > /dev/null 2>&1
             if [ $? -eq 0 ] ; then

                # Check if this module entry is not present in target base-modules-d.dat already.
                /usr/bin/grep "^${each_sam_module}" ${VAROPTDIR}/cfg/base-modules-d.dat > /dev/null 2>&1
                if [ $? -ne 0 ] ; then

                    /usr/bin/grep "^${each_sam_module}" ${upgrade_sam_dir}/base-modules-d.dat >> ${VAROPTDIR}/cfg/base-modules-d.dat 
                    if [ $? -ne 0 ]; then
                       echologverbose 'Failed to merge $2 entry into base-modules-d.dat.' ${each_sam_module}
                    fi
                fi

              fi
           done

        /usr/bin/chmod 600 $VAROPTDIR/cfg/base-modules-d.dat
        
      fi
   fi

   # Clean up SAM upgrade data.
   /usr/bin/rm -rf ${upgrade_sam_dir} > /dev/null 2>&1

}

#
# Setup at agent's side
#
agent_setup()
{
    import_sam_data=$1
    set_xput
    #
    # Get Current machine name
    #
    localhost=`/usr/bin/uname -n`
    agent_port=""
    if [ -f ${VARDIR}/domain-config.x ] ; then
        agent_port=`$XGET ${VARDIR}/domain-config.x agent.snmpPort`
    else
        agent_port=161
    fi

    add_edot_module
    configure_eservice_object

    process_sam_agent_data ${import_sam_data}
    #
    # The proposed change makes it easier for user to modify service 
    # elements through the GUI, rather than doing it through command line. 
    # This shell will not call.  
    #
    /usr/bin/touch ${VAROPTDIR}/cfg/eservicedot.dat
}


######################################################
#
#  Main function
#
######################################################
do_setup()
{
    set_commondirs
    set_basedir
    check_root
    MODULES_INFO=$ESDIR/EServices/modules.info

    # this is needed, for a call to check_root in addon_setup
    PROGNAME="$0"; export PROGNAME
    import_sam_data=$1

    tmp_file="${ESROOT}/addons/EServices/sbin/addon_common.sh"
    if [ -n "${ESROOT}" ]; then
        if [ -r "${tmp_file}" ]; then
            . ${tmp_file}
        else
            /usr/bin/echo "${tmp_file} not readable."
            exit 1
        fi
    else
        echo "Sun Management Center ESROOT not setup."
        exit 1
    fi

    setup_textdomain SUNW_SUNMC_SAM

    echolog  '-----------------------------------------------------------------'
    echolog  'Starting Sun Management Center Service Availability Manager Setup' 
    echolog  '-----------------------------------------------------------------'

    #
    # get all the installed layers
    #
    /usr/bin/pkginfo -qi SUNWesses
    if [ $? -eq 0 ] ; then
        server_setup
    fi

    /usr/bin/pkginfo -qi SUNWesaes
    if [ $? -eq 0 ] ; then
        agent_setup ${import_sam_data}
    fi

    echolog "Setup of Service Availability Manager complete."

}

######################################################
#
#  Source es-common.sh
#
######################################################
. ${BASEDIR}/SUNWsymon/sbin/es-common.sh
