#!/bin/sh
# \
        exec tclsh "$0" ${1+"$@"}
#
# Copyright C 2003 Sun Microsystems, Inc.
# All rights reserved. Use is subject to license terms.
#
#
# Sun, Sun Microsystems, and the Sun logo are trademarks or registered
# trademarks of Sun Microsystems, Inc. in the United States and other
# countries.
#
# Federal Acquisitions: Commercial Software--Government Users Subject to
# Standard License Terms and Conditions
#

# Role : This script
# ----
#    - downgrades/upgrades the DPS part of serverroot and the DPS
#      instances in serverroot (see usage for more details)



#==============================================================================
# For trace margin
#==============================================================================

set Margin(0) ""
set Margin(1) "   "
set Margin(2) "      "
set Margin(3) "         "
set Margin(4) "            "
set Margin(5) "               "
set Margin(6) "                  "
set Margin(7) "                     "
set Margin(8) "                        "
set Margin(9) "                           "


#==============================================================================
# traceProcIn and traceProcOut
#==============================================================================

proc traceProcIn { xresult msg } {
    upvar $xresult result
    global Margin
    set level [info level]
    if {$level < 11} {
       set level [expr $level - 2]
    } else {
       set level 9
    }

    set curMargin $Margin($level)
    append result "$curMargin --> $msg"
}

proc traceProcOut { xresult msg } {
    upvar $xresult result
    global Margin
    set level [info level]
    if {$level < 11} {
       set level [expr $level - 2]
    } else {
       set level 9
    }

    set curMargin $Margin($level)
    append result "$curMargin <-- $msg"
}


#==============================================================================
# assertUpOrDown
# --------------
# Checks that syncType argument value is valid
#==============================================================================

proc assertUpOrDown { syncType {msg [list]} } {

    if [string compare "upgrade" $syncType] {
        if [string compare "downgrade" $syncType] {
            error [lappend msg "syncType option should be upgrade or downgrade, found $syncType."]
        }
    }

    return $syncType
}


#==============================================================================
# noedit
# ------
# Argument where value check is done
#==============================================================================

proc noedit { x {msg [list]} } {
    return $x
}


#==============================================================================
# unscramble
# ----------
# Clarify content of tailor
#==============================================================================

proc unscramble {string} {

    if [string match "{}*" $string] {

        set i 0
        foreach char {A B C D E F G H I J K L M N O P Q R S T U V W X Y Z
                      a b c d e f g h i j k l m n o p q r s t u v w x y z
                      0 1 2 3 4 5 6 7 8 9 + /} {
            set theArray($char) $i
            incr i
        }

        set theGroup 0
        set Bits 18
        foreach char [split [string trim $string "{}"] {}] {
            if {[string compare $char "="]} {
                set theBits $theArray($char)
                set theGroup [expr {$theGroup | ($theBits << $Bits)}]
                if {[incr Bits -6] < 0} {
                    scan [format %06x $theGroup] %2x%2x%2x t1 t2 t3
                    append theResult [format %c%c%c $t1 $t2 $t3]
                    set theGroup 0
                    set Bits 18
                }
            } else {
                scan [format %04x $theGroup] %2x%2x t1 t2
                if {$Bits == 6} {
                    append theResult [format %c $t1]
                } elseif {$Bits == 0} {
                    append theResult [format %c%c $t1 $t2]
                }
                break
            }
        }

    } else {
        set theResult $string
    }
    return $theResult
}


#==============================================================================
# decode_uninstall_dwp
# --------------------
# Clarify content of uninstallContext.tcl
#==============================================================================

proc decode_uninstall_dwp { xName xInstance } {
    upvar $xName x
    upvar $xInstance instance

    set instanceRoot [file join $x(-serverroot) ${instance}]

    set fName [file join $instanceRoot uninstallContext.tcl]
    set 522fName [file join $instanceRoot 522uninstallContext.tcl]

    if [file exists $fName] {

        file rename $fName $522fName

        set 522fFile [open ${522fName} r+ ]
        set fperm  [file attributes ${522fName} -permissions]
        set fgroup [file attributes ${522fName} -group]
        set fowner [file attributes ${522fName} -owner]

        set fFile [open ${fName}  w  $fperm ]

        while {[gets $522fFile line] >= 0} {
            if [regexp {(password)} $line] {
                set dwp [string trim [lindex [split $line "\""] 1] " "]
                puts $fFile "  set x(password) \"[unscramble $dwp]\""
            } else {
                puts $fFile $line
            }
        }
        close $fFile
        file attributes ${fName} -permissions $fperm -group $fgroup -owner $fowner

        close $522fFile
        file delete $522fName
        return "SUCCESS: $fName clarified.\n"
    } else {
        return "WARNING: $fName NOT FOUND.\n"
    }
}


#==============================================================================
# decode_tailor_dwp
# -----------------
# Clarify content of tailor file
#==============================================================================

proc decode_tailor_dwp { xName xInstance } {
    upvar $xName x
    upvar $xInstance instance

    set instanceRoot [file join $x(-serverroot) ${instance}]

    set tailorName [file join $instanceRoot  etc tailor.txt]
    set 522tailorName [file join $instanceRoot  etc 522tailor.txt]

    if [file exists $tailorName] {

        file rename $tailorName $522tailorName

        set 522tailorFile [open ${522tailorName} r+ ]
        set tailorperm  [file attributes ${522tailorName} -permissions]
        set tailorgroup [file attributes ${522tailorName} -group]
        set tailorowner [file attributes ${522tailorName} -owner]

        set tailorFile [open ${tailorName}  w  $tailorperm ]

        set pattern configuration_bind_pw

        while {[gets $522tailorFile line] >= 0} {
            if [regexp $pattern $line] {
                set dwp [string trim [lindex [split $line " "] 1] " "]
                puts $tailorFile "configuration_bind_pw: [unscramble $dwp]"
            } else {
                puts $tailorFile $line
            }
        }
        close $tailorFile
        file attributes ${tailorName} -permissions $tailorperm \
             -group $tailorgroup -owner $tailorowner

        close $522tailorFile
        file delete $522tailorName
        return "SUCCESS: $tailorName clarified.\n"
    } else {
        return "WARNING: $tailorName NOT FOUND.\n"
    }
}


#==============================================================================
# emit
# ----
# changes in utilities files the name of the jar to be used
#==============================================================================

proc emit { xName fName cName } {
    upvar $xName x

    set perm  [file attributes [file join $x(-serverroot) $fName] -permissions]
    set owner [file attributes [file join $x(-serverroot) $fName] -owner]
    set group [file attributes [file join $x(-serverroot) $fName] -group]

    set oName [file join $x(-serverroot) $fName]
    set oFile [open $oName w]

    puts $oFile "#!/bin/sh"
    puts $oFile "#"
    puts $oFile "# PROPRIETARY/CONFIDENTIAL. Use of this product is subject to"
    puts $oFile "# license terms. Copyright C 2001-2003 Sun Microsystems, Inc."
    puts $oFile "# All rights reserved."
    puts $oFile "#"
    puts $oFile "\"[file join $x(-serverroot) bin/base/jre/bin/java]\" -cp \"[file join $x(-serverroot) java/jars/$x(jarName)]:[file join $x(-serverroot) java/ldapjdk.jar]:[file join $x(-serverroot) java/mcc52.jar]\" $cName \"\$@\""

    close $oFile

    file attributes ${oName} -permissions $perm -group $group -owner $owner
}

#==============================================================================
# emitWindows
# ----
# changes in utilities files the name of the jar to be used (for Windows)
#==============================================================================

proc emitWindows { xName fName cName } {
    upvar $xName x

    set oName [file join $x(-serverroot) $fName]
    set oFile [open $oName w]

    puts $oFile "@echo off"
    puts $oFile "rem"
    puts $oFile "rem PROPRIETARY/CONFIDENTIAL. Use of this product is subject to"
    puts $oFile "rem license terms. Copyright C 2001-2003 Sun Microsystems, Inc."
    puts $oFile "rem All rights reserved."
    puts $oFile "rem"
    puts $oFile "\"[file join $x(-serverroot) bin/base/jre/bin/java]\" -cp \"[file join $x(-serverroot) java/jars/$x(jarName)]:[file join $x(-serverroot) java/ldapjdk.jar]:[file join $x(-serverroot) java/mcc52.jar]\" $cName \"\$@\""

    close $oFile

}


#==============================================================================
# upgrade_serverroot_global_3_4
# -----------------------------
# Upgrade ServerRoot data from JES3 to JES4, instance independant.
#==============================================================================

proc upgrade_serverroot_global_3_4 { xName } {

    upvar $xName x
    global tcl_platform
    set rc 0

    # Trace
    traceProcIn  result "upgrade_serverroot_global_3_4\n"

    if [string equal $tcl_platform(platform) unix] {

      set rc [catch {
         set jartarget "target_not_defined"
         set group [file attributes [file join $x(-serverroot) java jars] -group]
         set owner [file attributes [file join $x(-serverroot) java jars] -owner]

         # Create link from serverroot to installdir for new jars
         cd [file join $x(-serverroot) java jars]

         if [catch {file readlink dps52.jar} result] {
               # append result "WARNING: dps522.jar is expected to be a symbolic link: no action.\n"
         } else {
               set jartarget [file dirname $result]
               append result "jartarget: ${jartarget} ${owner}\n"
         }
         if [catch {file readlink dps522.jar} result] {
               # append result "WARNING: dps522.jar is expected to be a symbolic link: no action.\n"
         } else {
               set jartarget [file dirname $result]
               append result "jartarget: ${jartarget} ${owner}\n"
         }
         if [catch {file readlink dps523.jar} result] {
               # append result "WARNING: dps523.jar is expected to be a symbolic link: no action.\n"
         } else {
               set jartarget [file dirname $result]
               append result "jartarget: ${jartarget} ${owner}\n"
         }

         if [string equal $jartarget "target_not_defined"] {
               append result "Found no symlinks for dps52x.jar, cannot create symlinks for dps524.jar"
               set rc 1
         } else {

               cd $x(-cid)/java/jars
               if [ catch {glob dps524*.jar} iList ] {
               } else {
                  if [expr [llength $iList] > 0] {
                     foreach i $iList {
                           # Fix bug 5022131: Cannot update jarfiles if
                           # they have already been downloaded (bug is
                           # meaningful for AS, DS and DPS).
                           # So if the jarfile already exists, delete it first...

                           set src_file [file join ${jartarget} $i]
                           set dst_file [file join $x(-serverroot) java jars $i]

                           if [file exists $dst_file] {
                              append result "remove file: $dst_file\n"
                              set rc [catch {exec rm -f $dst_file} exec_result]
                              append result "rm $dst_file: exec_result = $exec_result\n"
                           }

                           if {$rc == 0} {
                              append result "create symlink $src_file -> $dst_file\n"
                              set rc1 [catch {exec ln -s $src_file $dst_file} exec_result]
                              append result $exec_result
                              set rc2 [catch {exec chown -h  $owner $dst_file} exec_result]
                              append result $exec_result
                              set rc3 [catch {exec chgrp -h  $group $dst_file} exec_result]
                              append result $exec_result
                              set rc [expr $rc1 + $rc2 + $rc3]
                           }
                     }
                  }
               }
	   }
      }]
      if {$rc != 0} {append result "Error occurred when trying to create symlinks for dps524*.jar\n"}
  

    # changes in utilities files the name of the jar to be used
    # following section not adequate for Windows
    if {$rc == 0} {
         set x(jarName) dps524.jar
         set rc1 [catch {emit x bin/dps_utilities/dpsconfig2ldif com.iplanet.idar.util.IDARPrintConfig}]
         set rc2 [catch {emit x bin/dps_utilities/dpsldif2config com.iplanet.idar.task.ImportConfigurationLdif}]
         set rc3 [catch {emit x bin/dps_utilities/migratefromidar50 com.iplanet.idar.util.MigrateConfig3}]
         set rc [expr $rc1 + $rc2 + $rc3]
	}    
  } else {
	# windows     
	if {$rc == 0} {
	    set x(jarName) dps524.jar
	    set rc1 [catch {emitWindows x bin/dps_utilities/dpsconfig2ldif.bat com.iplanet.idar.util.IDARPrintConfig}]
	    set rc2 [catch {emitWindows x bin/dps_utilities/dpsldif2config.bat com.iplanet.idar.task.ImportConfigurationLdif}]
	    set rc3 [catch {emitWindows x bin/dps_utilities/migratefromidar50.bat com.iplanet.idar.util.MigrateConfig3}]
	    set rc [expr $rc1 + $rc2 + $rc3]
	}    
    }


    # Trace
    traceProcOut result "upgrade_serverroot_global_3_4\n"

    # If an error was found then throw an exception to caller
    # otherwise return traces
    if {$rc == 0} {
        return $result
    } else {
        return -code error $result
    }
}


#==============================================================================
# upgrade_serverroot_global_2_3
# -----------------------------
# Upgrade ServerRoot data from JES2 to JES3, instance independant.
#==============================================================================

proc upgrade_serverroot_global_2_3 { xName } {

    upvar $xName x
    global tcl_platform
    set rc 0

    # Trace
    traceProcIn  result "upgrade_serverroot_global_2_3\n"

    if [string equal $tcl_platform(platform) unix] {

      set rc [catch {
         set jartarget "target_not_defined"
         set group [file attributes [file join $x(-serverroot) java jars] -group]
         set owner [file attributes [file join $x(-serverroot) java jars] -owner]

         # Create link from serverroot to installdir for new jars
         cd [file join $x(-serverroot) java jars]

         if [catch {file readlink dps52.jar} result] {
               # append result "WARNING: dps522.jar is expected to be a symbolic link: no action.\n"
         } else {
               set jartarget [file dirname $result]
               append result "jartarget: ${jartarget} ${owner}\n"
         }
         if [catch {file readlink dps522.jar} result] {
               # append result "WARNING: dps522.jar is expected to be a symbolic link: no action.\n"
         } else {
               set jartarget [file dirname $result]
               append result "jartarget: ${jartarget} ${owner}\n"
         }

         if [string equal $jartarget "target_not_defined"] {
               append result "Found no symlinks for dps52x.jar, cannot create symlinks for dps523.jar"
               set rc 1
         } else {

               cd $x(-cid)/java/jars
               if [ catch {glob dps523*.jar} iList ] {
               } else {
                  if [expr [llength $iList] > 0] {
                     foreach i $iList {
                           # Fix bug 5022131: Cannot update jarfiles if
                           # they have already been downloaded (bug is
                           # meaningful for AS, DS and DPS).
                           # So if the jarfile already exists, delete it first...

                           set src_file [file join ${jartarget} $i]
                           set dst_file [file join $x(-serverroot) java jars $i]

                           if [file exists $dst_file] {
                              append result "remove file: $dst_file\n"
                              set rc [catch {exec rm -f $dst_file} exec_result]
                              append result "rm $dst_file: exec_result = $exec_result\n"
                           }

                           if {$rc == 0} {
                              append result "create symlink $src_file -> $dst_file\n"
                              set rc1 [catch {exec ln -s $src_file $dst_file} exec_result]
                              append result $exec_result
                              set rc2 [catch {exec chown -h  $owner $dst_file} exec_result]
                              append result $exec_result
                              set rc3 [catch {exec chgrp -h  $group $dst_file} exec_result]
                              append result $exec_result
                              set rc [expr $rc1 + $rc2 + $rc3]
                           }
                     }
                  }
               }
	   }
      }]
      if {$rc != 0} {append result "Error occurred when trying to create symlinks for dps523*.jar\n"}
  

    # changes in utilities files the name of the jar to be used
    # following section not adequate for Windows
    if {$rc == 0} {
         set x(jarName) dps523.jar
         set rc1 [catch {emit x bin/dps_utilities/dpsconfig2ldif com.iplanet.idar.util.IDARPrintConfig}]
         set rc2 [catch {emit x bin/dps_utilities/dpsldif2config com.iplanet.idar.task.ImportConfigurationLdif}]
         set rc3 [catch {emit x bin/dps_utilities/migratefromidar50 com.iplanet.idar.util.MigrateConfig3}]
         set rc [expr $rc1 + $rc2 + $rc3]
	}    
  } else {
	# windows     
	if {$rc == 0} {
	    set x(jarName) dps523.jar
	    set rc1 [catch {emitWindows x bin/dps_utilities/dpsconfig2ldif.bat com.iplanet.idar.util.IDARPrintConfig}]
	    set rc2 [catch {emitWindows x bin/dps_utilities/dpsldif2config.bat com.iplanet.idar.task.ImportConfigurationLdif}]
	    set rc3 [catch {emitWindows x bin/dps_utilities/migratefromidar50.bat com.iplanet.idar.util.MigrateConfig3}]
	    set rc [expr $rc1 + $rc2 + $rc3]
	}    
    }


    # Trace
    traceProcOut result "upgrade_serverroot_global_2_3\n"

    # If an error was found then throw an exception to caller
    # otherwise return traces
    if {$rc == 0} {
        return $result
    } else {
        return -code error $result
    }
}


#==============================================================================
# upgrade_serverroot_global_1_2
# -----------------------------
# Upgrade ServerRoot data from JES1 to JES2, instance independant.
#==============================================================================

proc upgrade_serverroot_global_1_2 { xName } {
    upvar $xName x
    global tcl_platform

    # Trace
    traceProcIn result "upgrade_serverroot_global_1_2: $x(-serverroot)\n"

    if [string equal $tcl_platform(platform) unix] {

        # Replace symlink by dir hierarchy
        set DPSDIR [file join $x(-serverroot) bin dps]

        if [catch {file readlink ${DPSDIR}} result] {
            append result "WARNING: ${DPSDIR} is expected to be a symbolic link: no action.\n"
            return $result
        } else {
            set target $result
            set group [file attributes [file join $x(-serverroot)] -group]
            set owner [file attributes [file join $x(-serverroot)] -owner]
            append result "Target: ${target} ${owner} ${group}\n"
        }

        # sr/bin/dps is a symlink
        cd [file join $x(-serverroot) bin]
        # rm symlink
        file delete dps
        # creates subdirs instead
        file mkdir [file join $x(-serverroot) bin dps]
        file attributes [file join $x(-serverroot) bin dps] \
             -permissions 0755 -group $group -owner $owner

        file mkdir [file join $x(-serverroot) bin dps server]
        file attributes [file join $x(-serverroot) bin dps server] \
             -permissions 0755 -group $group -owner $owner

        file mkdir [file join $x(-serverroot) bin dps server bin]
        file attributes [file join $x(-serverroot) bin dps server bin] \
             -permissions 0755 -group $group -owner $owner

        cd [file join $x(-serverroot) bin dps]

        exec ln -s [file join ${target} admin] admin
        exec chown -h  $owner admin
        exec chgrp -h  $group admin
        exec ln -s [file join ${target} install] install
        exec chown -h  $owner install
        exec chgrp -h  $group install
        exec ln -s [file join ${target} COPYRIGHT] COPYRIGHT
        exec chown -h  $owner COPYRIGHT
        exec chgrp -h  $group COPYRIGHT
        exec ln -s [file join ${target} license.txt] license.txt
        exec chown -h  $owner license.txt
        exec chgrp -h  $group license.txt

        cd [file join $x(-serverroot) bin dps server]
        exec ln -s [file join ${target} server script] script
        exec chown -h  $owner script
        exec chgrp -h  $group script
        cd [file join $x(-serverroot) bin dps server bin]
        exec ln -s [file join ${target} server bin dpsAsks.txt] dpsAsks.txt
        exec chown -h  $owner dpsAsks.txt
        exec chgrp -h  $group dpsAsks.txt
        exec ln -s [file join ${target} server bin dpsAsks_en.txt] dpsAsks_en.txt
        exec chown -h  $owner dpsAsks_en.txt
        exec chgrp -h  $group dpsAsks_en.txt
        exec ln -s [file join ${target} server bin ldapfwd] ldapfwd
        exec chown -h  $owner ldapfwd
        exec chgrp -h  $group ldapfwd
    }

    # Trace
    traceProcOut result "upgrade_serverroot_global_1_2: SUCCESS\n"
    return $result
}


#==============================================================================
# upgrade_serverroot_instance_2_X  
# -------------------------------
# Upgrade instance specific ServerRoot data, from JES2 to JESX (X>2).
#==============================================================================

proc upgrade_serverroot_instance_2_X { xName xInstance } {

   upvar $xName x
   upvar $xInstance instance
   global tcl_platform

   set instanceRoot [file join $x(-serverroot) ${instance}]

   traceProcIn  result "upgrade_serverroot_instance_2_X\n"

   # Update versions repository with new JES release
   cd [file join $x(-serverroot) shared bin]

   if [string equal $tcl_platform(platform) unix] {
       set rc [catch {exec env LD_LIBRARY_PATH= $x(cmd_to_sync_version) \
        -upgradeNotification -v $x(current_jes_release) -i ${instanceRoot}} cmdresult]
   } else {
       set rc [catch {exec $x(cmd_to_sync_version) \
        -upgradeNotification -v $x(current_jes_release) -i ${instanceRoot}} cmdresult]
   }

   append result "${instance}: -upgradeNotification result:$cmdresult rc:$rc.\n"

   traceProcOut result "upgrade_serverroot_instance_2_X\n"

   # If an error was found then throw an exception to caller
   # otherwise return traces
   if {$rc == 0} {
       return $result
   } else {
       return -code error $result
   }
}


#==============================================================================
# upgrade_serverroot_instance_1_2
# -------------------------------
# Upgrade instance specific ServerRoot data, from JES1 to JES2.
#==============================================================================

proc upgrade_serverroot_instance_1_2 { xName xInstance } {

    upvar $xName x
    upvar $xInstance instance
    global tcl_platform

    # Trace
    traceProcIn result "upgrade_serverroot_instance_1_2 for instance $instance\n"

    set instanceRoot [file join $x(-serverroot) ${instance}]

    cd [file join $x(-serverroot) shared bin]

    cd [file join $x(-serverroot) alias]
    if [catch {glob ${instance}*.db} aliasList] {
       append result "${instance}: no certificate to upgrade.\n"
    } else {
       # Upgrade NSS certificates
       cd [file join $x(-serverroot) shared bin]

       if [string equal $tcl_platform(platform) unix] {
	   set rccert [catch {exec env LD_LIBRARY_PATH= $x(cmd_to_upgrade_sec) \
				  -r $x(-serverroot) -i ${instance} -V 7 -v 8} cmdresult]
       } else {
	   set rccert [catch {exec $x(cmd_to_upgrade_sec) \
				  -r $x(-serverroot) -i ${instance} -V 7 -v 8} cmdresult]
       }

       append result "${instance}: $x(cmd_to_upgrade_sec) result:$cmdresult "
       append result "rc:$rccert expected:0=SUCCESS.\n"
    }

    # No upgrade of tailor encoding for existing instance

    # Notification is now performed in upgrade_serverroot_instance_2_X
    ### cd [file join $x(-serverroot) shared bin]
    ### set rc [catch {exec env LD_LIBRARY_PATH= $x(cmd_to_sync_version) \
    ###     -upgradeNotification -v $x(current_jes_release) -i ${instanceRoot}} cmdresult]
    ### append result "${instance}: -upgradeNotification result:$cmdresult rc:$rc.\n"

    # Trace
    traceProcOut result "upgrade_serverroot_instance_1_2: SUCCESS\n"
    return $result
}


#==============================================================================
# upgrade_serverroot_2_3
# ----------------------
# Upgrade ServerRoot data, from JES2 to JES3
# 2 steps :
#   - global job (not "instance by instance")
#   - instance by instance job
#==============================================================================

proc upgrade_serverroot_2_3 { xName } {

   upvar $xName x
   traceProcIn  result "upgrade_serverroot_2_3\n"

    # common part of the job
    set rc [catch {upgrade_serverroot_global_2_3 x} upgrade_result]
    append result $upgrade_result

    # instance by instance part of the job
    # 08/06/2005 This is done at upgrading 3_4
    #if {$rc == 0} {
    #   cd $x(-serverroot)
    #  if [catch {glob dps-* } iList] {
    #     append result "No instances.\n"
    #  } else {
    #     append result "Instances list : $iList\n"
    #     if [expr [llength $iList] > 0] {
    #           foreach i $iList {
    #              set x(instance) [file tail "$i"]
    #              set lrc [catch {upgrade_serverroot_instance_2_X x x(instance)} upgrade_result]
    #              if {$lrc == 0} {
    #                 append result "upgrade_serverroot_instance_2_X $x(instance) OK\n$upgrade_result"
    #              } else {
    #                 append result "upgrade_serverroot_instance_2_X $x(instance) NOK\n$upgrade_result"
    #                 set rc $lrc
    #              }
    #           }
    #     }
    #  }
    #}

   traceProcOut result "upgrade_serverroot_2_3\n"

   # If an error was found then throw an exception to caller
   # otherwise return traces
   if {$rc == 0} {
       return $result
   } else {
       return -code error $result
   }
}


#==============================================================================
# upgrade_serverroot_3_4
# ----------------------
# Upgrade ServerRoot data, from JES3 to JES4
# 2 steps :
#   - global job (not "instance by instance")
#   - instance by instance job
#==============================================================================

proc upgrade_serverroot_3_4 { xName } {

   upvar $xName x
   traceProcIn  result "upgrade_serverroot_3_4\n"

    # common part of the job
    set rc [catch {upgrade_serverroot_global_3_4 x} upgrade_result]
    append result $upgrade_result

    # instance by instance part of the job
    if {$rc == 0} {
       cd $x(-serverroot)
      if [catch {glob dps-* } iList] {
         append result "No instances.\n"
      } else {
         append result "Instances list : $iList\n"
         if [expr [llength $iList] > 0] {
               foreach i $iList {
                  set x(instance) [file tail "$i"]
                  set lrc [catch {upgrade_serverroot_instance_2_X x x(instance)} upgrade_result]
                  if {$lrc == 0} {
                     append result "upgrade_serverroot_instance_2_X $x(instance) OK\n$upgrade_result"
                  } else {
                     append result "upgrade_serverroot_instance_2_X $x(instance) NOK\n$upgrade_result"
                     set rc $lrc
                  }
               }
         }
      }
    }

   traceProcOut result "upgrade_serverroot_3_4\n"

   # If an error was found then throw an exception to caller
   # otherwise return traces
   if {$rc == 0} {
       return $result
   } else {
       return -code error $result
   }
}


#==============================================================================
# upgrade_serverroot_1_2
# ----------------------
# Upgrade ServerRoot data, from JES1 (DPS52) to JES2 (DPS52patch2)
# 2 steps :
#   - global job (not "instance by instance")
#   - instance by instance job
#==============================================================================

proc upgrade_serverroot_1_2 { xName } {
    upvar $xName x

    # Trace
    traceProcIn result "upgrade_serverroot_1_2\n"

    # common part of the job
    append result [upgrade_serverroot_global_1_2 x]

    # instance by instance part of the job

    cd $x(-serverroot)
    if [catch {glob dps-* } iList] {
        append result "No instances.\n"
    } else {
        append result "Instances list : $iList\n"
        if [expr [llength $iList] > 0] {
            foreach i $iList {
                set x(instance) [file tail "$i"]
                set lrc [upgrade_serverroot_instance_1_2 x x(instance)]
                append result "$lrc"
            }
        }
    }

    # Trace
    traceProcOut result "upgrade_serverroot_1_2\n"
    return $result
}


#==============================================================================
# downgrade_serverroot_global_3_2
# -------------------------------
# Downgrade instance independant ServerRoot data, from JES3 to JES2
#==============================================================================

proc downgrade_serverroot_global_3_2 { xName } {

   upvar $xName x
   traceProcIn  result "downgrade_serverroot_global_3_2\n"
   traceProcOut result "downgrade_serverroot_global_3_2\n"
   return $result
}


#==============================================================================
# downgrade_serverroot_global_2_1
# -------------------------------
# Downgrade instance independant ServerRoot data, from JES2 to JES1
#==============================================================================

proc downgrade_serverroot_global_2_1 { xName } {
    upvar $xName x
    global tcl_platform

    traceProcIn result "downgrade_serverroot_global_2_1\n"

    if [string equal $tcl_platform(platform) unix] {

        # Replace dir hierarchy by symlink
        set DPSDIR [file join $x(-serverroot) bin dps]

        # SR/bin/dps must be a directory and not a symlink
        if [catch {file readlink ${DPSDIR}} result ] {
            if [catch {file isdirectory ${DPSDIR}} result ] {
                append result "${DPSDIR} is expected to exist as a directory: no common update needed.\n"
                return $result
            }
        }

        # isdirectory ${DPSDIR}/server/bin
        if [catch {file isdirectory [file join ${DPSDIR} server bin]} result ] {
            error "[file join ${DPSDIR} server bin] is expected to exist as a directory."
        }

        cd [file join ${DPSDIR} server bin]

        # rm symlink dpsAsks.txt
        set fn dpsAsks.txt
        if [catch {file readlink ${fn}} result ] {
            error "[file join ${DPSDIR} server bin ${fn}] is expected to exist as a symbolic link."
        }
        file delete $fn

        # rm symlink dpsAsks_en.txt
        set fn dpsAsks_en.txt
        if [catch {file readlink ${fn}} result ] {
            error "[file join ${DPSDIR} server bin ${fn}] is expected to exist as a symbolic link."
        }
        file delete $fn

        # rm symlink ldapfwd
        set fn ldapfwd
        if [catch {file readlink ${fn}} result ] {
            error "[file join ${DPSDIR} server bin ${fn}] is expected to exist as a symbolic link."
        }
        file delete $fn

        # rm dir ${DPSDIR}/server/bin
        cd [file join ${DPSDIR} server]
        file delete bin
    }

    # rm symlink  script
    cd [file join ${DPSDIR} server]
    if [catch {file readlink script} lresult] {
        error "[file join ${DPSDIR} server script] is expected to exist as a symbolic link."
    } else {
        # keep track of the directory containing binaries
        set target [file dirname [file dirname $lresult]]
        append result "Target: ${target}\n"
    }
    file delete script

    cd ${DPSDIR}
    # rm dir ${DPSDIR}/server
    if [catch {file isdirectory server} result] {
        error "[file join ${DPSDIR} server] is expected to exist as a directory."
    }
    file delete server

    # rm symlink admin
    if [catch {file readlink admin} result] {
        error "[file join ${DPSDIR} admin] is expected to exist as a symbolic link"
    }
    file delete admin

    # rm symlink install
    if [catch {file readlink install} result] {
        error "[file join ${DPSDIR} install] is expected to exist as a symbolic link"
    }
    file delete install

    # rm symlink COPYRIGHT
    if [catch {file readlink COPYRIGHT} result] {
        error "[file join ${DPSDIR} COPYRIGHT] is expected to exist as a symbolic link"
    }
    file delete COPYRIGHT

    # rm symlink license.txt
    if [catch {file readlink license.txt} result] {
        error "[file join ${DPSDIR} license.txt] is expected to exist as a symbolic link"
    }
    file delete license.txt

    # replace dir dps by symlink
    cd [file join $x(-serverroot) bin]
    file delete dps
    set cmd [list exec ln -s ${target} ${DPSDIR}]
    if [catch $cmd result] {
        error "$cmd : $result"
    }
    exec chown -h  [file attributes [file join $x(-serverroot)] -owner] ${DPSDIR}
    exec chgrp -h  [file attributes [file join $x(-serverroot)] -group] ${DPSDIR}

    traceProcOut result "downgrade_serverroot_global_2_1: SUCCESS\n"
    return $result
}


#==============================================================================
# downgrade_serverroot_instance_4_3
# ---------------------------------
# Downgrade instance specific ServerRoot data from JES4 to JES3
#==============================================================================

proc downgrade_serverroot_instance_4_3 { xName xInstance } {

    upvar $xName x
    upvar $xInstance instance
    set instanceRoot [file join $x(-serverroot) ${instance}]

    # Trace
    traceProcIn  result "downgrade_serverroot_instance_4_3\n"

    # rm symlink to new jars
    # also eventually delete icon file that may have been downloaded by the console
    cd [file join $x(-serverroot) java jars]
    if [ catch {glob dps524*} iList ] {
    } else {
        append result "Will delete : $iList\n"
        if [expr [llength $iList] > 0] {
            foreach i $iList {
                catch {file delete $i} lresult
            }
        }
    }

    # changes in utilities files the name of the jar to be used
    set x(jarName) dps523.jar
    emit x bin/dps_utilities/dpsconfig2ldif com.iplanet.idar.util.IDARPrintConfig
    emit x bin/dps_utilities/dpsldif2config com.iplanet.idar.task.ImportConfigurationLdif
    emit x bin/dps_utilities/migratefromidar50 com.iplanet.idar.util.MigrateConfig3

    # Downgrade notification to update versions repository
    cd [file join $x(-serverroot) shared bin]
    set rc [catch {exec env LD_LIBRARY_PATH= $x(cmd_to_sync_version) -downgradeNotification \
        -i ${instanceRoot}} cmdresult]
    append result "${instance}: -downgradeNotification result=$cmdresult rc=$rc.\n"

    # Trace
    traceProcOut result "downgrade_serverroot_instance_4_3\n"

   # If an error was found then throw an exception to caller
   # otherwise return traces
   if {$rc == 0} {
       return $result
   } else {
       return -code error $result
   }
}


#==============================================================================
# downgrade_serverroot_instance_3_2
# ---------------------------------
# Downgrade instance specific ServerRoot data from JES3 to JES2
#==============================================================================

proc downgrade_serverroot_instance_3_2 { xName xInstance } {

    upvar $xName x
    upvar $xInstance instance
    set instanceRoot [file join $x(-serverroot) ${instance}]

    # Trace
    traceProcIn  result "downgrade_serverroot_instance_3_2\n"

    # rm symlink to new jars
    # also eventually delete icon file that may have been downloaded by the console
    cd [file join $x(-serverroot) java jars]
    if [ catch {glob dps523*} iList ] {
    } else {
        append result "Will delete : $iList\n"
        if [expr [llength $iList] > 0] {
            foreach i $iList {
                catch {file delete $i} lresult
            }
        }
    }

    # changes in utilities files the name of the jar to be used
    set x(jarName) dps522.jar
    emit x bin/dps_utilities/dpsconfig2ldif com.iplanet.idar.util.IDARPrintConfig
    emit x bin/dps_utilities/dpsldif2config com.iplanet.idar.task.ImportConfigurationLdif
    emit x bin/dps_utilities/migratefromidar50 com.iplanet.idar.util.MigrateConfig3

    # Downgrade notification to update versions repository
    cd [file join $x(-serverroot) shared bin]
    set rc [catch {exec env LD_LIBRARY_PATH= $x(cmd_to_sync_version) -downgradeNotification \
        -i ${instanceRoot}} cmdresult]
    append result "${instance}: -downgradeNotification result=$cmdresult rc=$rc.\n"

    # Trace
    traceProcOut result "downgrade_serverroot_instance_3_2\n"

   # If an error was found then throw an exception to caller
   # otherwise return traces
   if {$rc == 0} {
       return $result
   } else {
       return -code error $result
   }
}


#==============================================================================
# downgrade_serverroot_instance_3_2
# ---------------------------------
# Downgrade instance specific ServerRoot data from JES3 to JES2
#==============================================================================

proc downgrade_serverroot_instance_3_2 { xName xInstance } {

    upvar $xName x
    upvar $xInstance instance
    set instanceRoot [file join $x(-serverroot) ${instance}]

    # Trace
    traceProcIn  result "downgrade_serverroot_instance_3_2\n"

    # rm symlink to new jars
    # also eventually delete icon file that may have been downloaded by the console
    cd [file join $x(-serverroot) java jars]
    if [ catch {glob dps523*} iList ] {
    } else {
        append result "Will delete : $iList\n"
        if [expr [llength $iList] > 0] {
            foreach i $iList {
                catch {file delete $i} lresult
            }
        }
    }

    # changes in utilities files the name of the jar to be used
    set x(jarName) dps522.jar
    emit x bin/dps_utilities/dpsconfig2ldif com.iplanet.idar.util.IDARPrintConfig
    emit x bin/dps_utilities/dpsldif2config com.iplanet.idar.task.ImportConfigurationLdif
    emit x bin/dps_utilities/migratefromidar50 com.iplanet.idar.util.MigrateConfig3

    # Downgrade notification to update versions repository
    cd [file join $x(-serverroot) shared bin]
    set rc [catch {exec env LD_LIBRARY_PATH= $x(cmd_to_sync_version) -downgradeNotification \
        -i ${instanceRoot}} cmdresult]
    append result "${instance}: -downgradeNotification result=$cmdresult rc=$rc.\n"

    # Trace
    traceProcOut result "downgrade_serverroot_instance_3_2\n"

   # If an error was found then throw an exception to caller
   # otherwise return traces
   if {$rc == 0} {
       return $result
   } else {
       return -code error $result
   }
}


#==============================================================================
# downgrade_serverroot_instance_2_1
# ---------------------------------
# Downgrade instance specific ServerRoot data from JES2 to JES1
#==============================================================================

proc downgrade_serverroot_instance_2_1 { xName xInstance } {
    upvar $xName x
    upvar $xInstance instance

    traceProcIn result "downgrade_serverroot_instance_2_1: $instance\n"

    set instanceRoot [file join $x(-serverroot) ${instance}]
    set backout_from_version $x(current_jes_release)

    cd [file join $x(-serverroot) shared bin]
    set rc [catch {exec env LD_LIBRARY_PATH= $x(cmd_to_sync_version) -isDowngradePossible  \
                   -v ${backout_from_version} -i ${instanceRoot}} cmdresult]
    append result "${instance}: -isDowngradePossible result:$cmdresult rc:$rc expected:0=Yes.\n"

    if { $rc == 0 } {

        cd [file join $x(-serverroot) shared bin]
        set rc [catch {exec env LD_LIBRARY_PATH= $x(cmd_to_sync_version) -getVersionToDowngradeTo \
                -v ${backout_from_version} -i ${instanceRoot}} cmdresult]
        append result "${instance}: -getVersionToDowngradeTo result:$cmdresult "
        append result "rc:$rc expected:1=RTM.\n"

        if { $rc == 1 } {

            cd [file join $x(-serverroot) alias]
            if [catch {glob ${instance}*.db} aliasList] {
                append result "${instance}: no certificate to be manually downgraded.\n"
            } else {
                # Warning to manually downgrade NSS certificates
                puts stdout "Reverting to prepatch certificate database for ${instance}."
                puts stdout "Certificates added after patching the server must be downgraded manually."
                puts stdout "See 'Installation Guide' or patch README for instructions."
            }

            # Downgrade encoding in tailor and uninstallcontext files
            append result [decode_tailor_dwp x  instance]
            append result [decode_uninstall_dwp x  instance]

            cd [file join $x(-serverroot) shared bin]
            set rc [catch {exec env LD_LIBRARY_PATH= $x(cmd_to_sync_version) -downgradeNotification \
                               -i ${instanceRoot}} cmdresult]
            append result "${instance}: -downgradeNotification result:$cmdresult rc:$rc.\n"

            # Remove the upgrade directory
            set rc [catch [file delete [file join ${instanceRoot} upgrade versions.conf]] lresult]
            append result "${instance}: delete [file join ${instanceRoot} upgrade versions.conf] "
            append result "result:$lresult rc:$rc.\n"

            set rc [catch [file delete [file join ${instanceRoot} upgrade]] lresult]
            append result "${instance}: delete [file join ${instanceRoot} upgrade] "
            append result "result:$lresult rc:$rc.\n"

            append result "SUCCESS: ${instance} downgraded.\n"
        }
    }

    traceProcOut result "downgrade_serverroot_instance_2_1\n"
    return $result
}


#==============================================================================
# downgrade_serverroot_4_3
# ---------------------------------
# Downgrade from JES4 to JES3
# 2 steps :
#   - global job (not "instance by instance")
#   - instance by instance job
#==============================================================================

proc downgrade_serverroot_4_3 { xName } {

   upvar $xName x
   set rc 0
   traceProcIn  result "downgrade_serverroot_4_3\n"

    # No common part of the job

    # instance by instance part of the job
    cd $x(-serverroot)
    if [catch {glob dps-* } iList] {
        append result "No instances.\n"
    } else {
        if [expr [llength $iList] > 0] {
            foreach i $iList {
                set x(instance) [file tail "$i"]
                set _rc [catch {downgrade_serverroot_instance_4_3 x x(instance)} _result]
                append result $_result
                if {$_rc > 0} {
                   set rc $_rc
                }
            }
        }
    }

   traceProcOut result "downgrade_serverroot_4_3\n"

   # If an error was found then throw an exception to caller
   # otherwise return traces
   if {$rc == 0} {
       return $result
   } else {
       return -code error $result
   }
}

#==============================================================================
# 
# ---------------------------------
# Downgrade from JES3 to JES2
# 2 steps :
#   - global job (not "instance by instance")
#   - instance by instance job
#==============================================================================

proc downgrade_serverroot_3_2 { xName } {

   upvar $xName x
   set rc 0
   traceProcIn  result "downgrade_serverroot_3_2\n"

    # common part of the job
    append result [downgrade_serverroot_global_3_2 x]

    # instance by instance part of the job
    cd $x(-serverroot)
    if [catch {glob dps-* } iList] {
        append result "No instances.\n"
    } else {
        if [expr [llength $iList] > 0] {
            foreach i $iList {
                set x(instance) [file tail "$i"]
                set _rc [catch {downgrade_serverroot_instance_3_2 x x(instance)} _result]
                append result $_result
                if {$_rc > 0} {
                   set rc $_rc
                }
            }
        }
    }

   traceProcOut result "downgrade_serverroot_3_2\n"

   # If an error was found then throw an exception to caller
   # otherwise return traces
   if {$rc == 0} {
       return $result
   } else {
       return -code error $result
   }
}


#==============================================================================
# downgrade_serverroot_instance_2_1
# ---------------------------------
# Downgrade from JES2 to JES1
# 2 steps :
#   - global job (not "instance by instance")
#   - instance by instance job
#==============================================================================

proc downgrade_serverroot_2_1 { xName } {
    upvar $xName x

    traceProcIn result "downgrade_serverroot_2_1\n"

    # common part of the job
    append result [downgrade_serverroot_global_2_1 x]

    # instance by instance part of the job
    cd $x(-serverroot)
    if [catch {glob dps-* } iList] {
        append result "No instances.\n"
    } else {
        if [expr [llength $iList] > 0] {
            foreach i $iList {
                set x(instance) [file tail "$i"]
                append result [downgrade_serverroot_instance_2_1 x x(instance)]
            }
        }
    }

    traceProcOut result "downgrade_serverroot_2_1\n"
    return $result
}


#==============================================================================
# find_one_dps_instance
# ---------------------
# Find one DPS instance, ie one subdir whose name is like <ServerRoot>/dps-*
# The instance name, dps-*, is stored in $x(one_dps_instance). If no instance
# is found then instance name value is "no instance found"
#==============================================================================

proc find_one_dps_instance {xNames} {

    upvar $xNames x

    # Trace
    traceProcIn result "find_one_dps_instance: ServerRoot = $x(-serverroot)\n"

    # Search for DPS instance name
    cd $x(-serverroot)
    if [catch {glob dps-* } iList] {
        set one_dps_instance "no instance found"
    } else {
        set one_dps_instance [lindex $iList 0]
    }

    # Store instance name found
    set x(one_dps_instance) [file join $x(-serverroot) $one_dps_instance]

    # Trace
    traceProcOut result "find_one_dps_instance: instance = $x(one_dps_instance)\n"

    # All is ok, return traces
    return $result
}


#==============================================================================
# is_upgrade_required
# -------------------
#   Check whether ServerRoot data upgrade is needed or not.
#   Upgrade is needed when current data version is lower than current binary
#   version (see current_jes_release)
#==============================================================================

proc is_upgrade_required {xNames} {

    upvar $xNames x
    global errorInfo
    global tcl_platform

    set x(upgrade_is_required) no_defined
    set x(current_version)     not_defined

    # Trace
    traceProcIn result "is_upgrade_required...\n"

    # Check whether upgrade is needed or not
    cd [file join $x(-serverroot) shared bin]

    if [string equal $tcl_platform(platform) unix] {
	set rc [catch {exec env} _res_]
	append result "==> env = \n$_res_\n"
	set rc [catch {exec env LD_LIBRARY_PATH= ldd $x(cmd_to_sync_version)} _res_]
	append result "==> ldd result:\n$_res_\n"
	set rc [catch {exec env LD_LIBRARY_PATH= $x(cmd_to_sync_version) \
			   -getVersionToUpgrade -v $x(current_jes_release) -i $x(one_dps_instance) \
		       } x(current_version)]
    } else {
	set rc [catch {exec $x(cmd_to_sync_version) \
			   -getVersionToUpgrade -v $x(current_jes_release) -i $x(one_dps_instance) \
		       } x(current_version)]
    }

    if {$rc != 0} {
       append result $errorInfo "\n"
       append result "rc = $rc\n"
    } else {
       switch $x(current_version) {
          0 {
             set x(upgrade_is_required) no
          }
          -1 {
             set x(upgrade_is_required) no
          }
          1 {
             set x(upgrade_is_required) yes

             # No versions.conf file found: on Linux the first version
             # is 5.2.2

	      if [string equal $tcl_platform(platform) unix] { 		  
		  set platform [ exec uname -s ]
		  if [string compare "Linux" $platform] {
		  } else {
		      set x(current_version) 50202
		  }
	      } else {
		  set x(current_version) 50202
	      }	  
          }
          default {
             set x(upgrade_is_required) yes
          }
       }
    }

    # Trace
    if {$rc == 0} {
       append result "is_upgrade_required: current_version = $x(current_version)\n"
    }
    traceProcOut result "is_upgrade_required: $x(upgrade_is_required)\n"

    # If an error was found then throw an exception to caller
    # otherwise return traces
    if {$rc == 0} {
        return $result
    } else {
        return -code error $result
    }
}


#==============================================================================
# is_downgrade_required
# -------------------
#   Check whether ServerRoot data downgrade is needed or not.
#   Downgrade is needed when previous data version is higher or equal current
#   binary version (see current_jes_release)
#==============================================================================

proc is_downgrade_required {xNames} {

    upvar $xNames x
    global errorInfo

    set x(downgrade_is_required) no_defined
    set x(current_version)       not_defined

    # Trace
    traceProcIn result "is_downgrade_required\n"

    # Check whether downgrade is needed or not
    cd [file join $x(-serverroot) shared bin]
    set rc [catch {exec env LD_LIBRARY_PATH= $x(cmd_to_sync_version) \
       -getVersionToDowngradeTo -v $x(current_jes_release) -i $x(one_dps_instance) \
       } x(previous_version)]

    if {$rc != 0} {
       append result $errorInfo "\n"
       append result "rc = $rc\n"
    } else {
       switch $x(previous_version) {
          0 {
             set x(downgrade_is_required) no
          }
          -1 {
             set x(downgrade_is_required) no
          }
          default {
             set x(downgrade_is_required) yes
          }
       }
    }

    # Trace
    if {$rc == 0} {
       append result "is_downgrade_required: previous_version = \[$x(previous_version)\]\n"
    }
    traceProcOut result "is_downgrade_required: $x(downgrade_is_required)\n"

    # If an error was found then throw an exception to caller
    # otherwise return traces
    if {$rc == 0} {
        return $result
    } else {
        return -code error $result
    }
}


#==============================================================================
# upgrade_server_root
# -------------------
#   Upgrade ServerRoot data.
#==============================================================================

proc upgrade_server_root {xNames} {

    upvar $xNames x

    # Trace
    traceProcIn result "upgrade_server_root\n"

    #
    # is upgrade required?
    # --------------------
    # Call is_upgrade_required; as a result:
    # x(upgrade_is_required) = yes means upgrade is needed
    # and x(current_version) = ServerRoot data version 
    #
    set rc [catch {is_upgrade_required x} upgrade_result]
    append result $upgrade_result

    if {$rc == 0} {
       if {$x(upgrade_is_required) == "yes"} {
            switch $x(current_version) {
                1 {
                    # Upgrade from JES1
                    set rc [catch {upgrade_serverroot_1_2 x} upgrade_result]
                    append result $upgrade_result
                    if { $rc == 0} {
                       set rc [catch {upgrade_serverroot_2_3 x} upgrade_result]
                       append result $upgrade_result
                       if { $rc == 0 } {
                          set rc [catch {upgrade_serverroot_3_4 x} upgrade_result]
                          append result $upgrade_result
                       }
                    }
                }
                50202 {
                    # Upgrade from JES2
                    set rc [catch {upgrade_serverroot_2_3 x} upgrade_result]
                    append result $upgrade_result
                    if { $rc == 0} {
                       set rc [catch {upgrade_serverroot_3_4 x} upgrade_result]
                       append result $upgrade_result
                    }
                }
                50203 {
                    # Upgrade from JES3
                    set rc [catch {upgrade_serverroot_3_4 x} upgrade_result]
                    append result $upgrade_result
                }
                50204 {
                    # Data already upgraded to JES4, nothing to do
                    append result "Current version is 4 (JES4), nothing to do\n"
                }
                default {
                    # Mmmm, probably a bug somewhere!
                    append result "Current data version cannot be $x(current_version)\n"
                    set rc 1
                }
            }
        }
    }

    # Trace
    traceProcOut result "upgrade_server_root\n"

    # If an error was found then throw an exception to caller
    # otherwise return traces
    if {$rc == 0} {
        return $result
    } else {
        return -code error $result
    }
}


#==============================================================================
# downgrade_server_root
# ---------------------
#   Downgrade ServerRoot data.
#==============================================================================

proc downgrade_server_root {xNames} {

    upvar $xNames x

    # Trace
    traceProcIn result "downgrade_server_root\n"

    #
    # is downgrade required?
    # ----------------------
    # Call is_downgrade_required; as a result:
    # x(downgrade_is_required) = yes means downgrade is needed
    # and x(previous_version) = version target for downgrade
    #
    set rc [catch {is_downgrade_required x} downgrade_result]
    append result $downgrade_result
    if {$rc == 0} {
        if {$x(downgrade_is_required) == "yes"} {
            switch $x(previous_version) {
                1 {
                     # Need to downgrade to JES1
                     append result [downgrade_serverroot_4_3 x]
                     append result [downgrade_serverroot_3_2 x]
                     append result [downgrade_serverroot_2_1 x]
                  }
                50202 {
                     # Need to downgrade to JES2
                     append result [downgrade_serverroot_4_3 x]
                     append result [downgrade_serverroot_3_2 x]
                  }
                50203 {
                     # Need to downgrade to JES3
                     append result [downgrade_serverroot_4_3 x]
                  }
                50204 {
                     # Previous version cannot be JES4!
                     append result "Previous data version cannot be 4 (JES4)\n"
                     set rc 1
                  }
                default {
                     # Mmmm, probably a bug somewhere!
                     append result "Unknown previous data version ($x(previous_version))\n"
                     set rc 1
                  }
            }
        }
    }

    # Trace
    traceProcOut result "downgrade_server_root\n"

    # If an error was found then throw an exception to caller
    # otherwise return traces
    if {$rc == 0} {
        return $result
    } else {
        return -code error $result
    }
}


#==============================================================================
# syncserverroot
# --------------
# Downgrade or Upgrade according to syncType
#==============================================================================

proc syncserverroot { xName } {

    upvar $xName x

    # Trace
    traceProcIn result "syncserverroot: syncType = $x(-syncType)\n"

    # Search at least one DPS instance...
    set rc [catch {find_one_dps_instance x} find_result]
    append result $find_result

    # ... and upgrade or downgrade ServerRoot data
    if {$rc == 0} {
        # Trace
        append result "syncserverroot: instance = $x(one_dps_instance)\n"
        if {$x(one_dps_instance) == "no instance found"} {
            #
            # No DPS instance means no data to upgrade or downgrade
            # -> nothing to do
            #
            append result "No instance found, nothing to do.\n"
        } else {
            # Trace
            append result "syncserverroot: syncType = $x(-syncType)\n"
            switch $x(-syncType) {
                upgrade {
                    set rc [catch {upgrade_server_root x} upgrade_result]
                    append result $upgrade_result
                }
                downgrade {
                    set rc [catch {downgrade_server_root x} downgrade_result]
                    append result $downgrade_result
                }
            }
        }
    }

    # Trace
    traceProcOut result "syncserverroot: syncType = $x(-syncType)\n"

    # If an error was found then through an exception to caller
    # otherwise return traces
    if {$rc == 0} {
        return $result
    } else {
        return -code error $result
    }
}


#==============================================================================
#
# main
#
#==============================================================================


# edit array : valued options => have to check value
set edit(-cid) "noedit"
set edit(-syncType) assertUpOrDown
set edit(-serverroot) "noedit"

set defParm(-cid) -serverroot

foreach i [array names edit] {
    # add to parms, list of allowed params
    lappend parms $i
    set x($i) ""
}

# knob array : option
set knob(-verbose) [expr 0]
foreach i [array names knob] {
    # add to parms, list of allowed params
    lappend parms $i
    set x($i) $knob($i)
}

#
set optional [list -cid]
set required [list -serverroot]
set errmsg [list]
set mode ""
set usage [expr 0]

# from argv, sets errmsg, x with name or 0
foreach i $argv {
    if [string length $mode] {
        if [string length $x($mode)] {
            lappend errmsg [list "Parameter $mode stipulated multiple times."]
        } else {
            # run the edit defined for that param
            if [catch {$edit($mode) $i [list "For parameter $mode:"]} result] {
                lappend errmsg $result
                set x($mode) $i
            } else {
                set x($mode) $result
            }
        }
        set mode ""
    } else {
        if [expr [lsearch $parms $i] < 0] {
            lappend errmsg [list "Unknown parm $i."]
            incr usage
        } else {
            if [info exist knob($i)] {
                set x($i) [expr $knob($i) == 0]
            } else {
                set mode $i
            }
        }
    }
}

# no default value

set missing [list]

foreach i $required {
    if [string length $x($i)] {
    } else {
        lappend missing $i
    }
}

if [llength $missing] {
    lappend errmsg [list "Required parameter(s) missing: $missing."]
    set usage [expr 1]
}

if [llength $errmsg] {
} else {
    foreach i $parms {
        if [expr [lsearch $optional $i] < 0] {
            if [string length $x($i)] {
            } else {
                lappend errmsg [list "Parameter $i is required and missing."]
                incr usage
            }
        }
    }
}

if {$usage} {
    lappend errmsg [list \
        "Usage:" \
        "sync-dps.tcl  -syncType synctype -serverroot path \[-cid installdir\]" \
        "\t\t\[-verbose\]" \
        "" \
        ]
}

if [llength $errmsg] {
    foreach i $errmsg {
        foreach j $i {
            puts stderr $j
        }
    }
    exit 1
}

set result "Updating serverroot for DPS instances."

# set x(tclsh) [file join $x(-cid) bin tcl8.2 [cmd tclsh]]

# Define the current JES release: DPS 5.2.4 = JES4
set x(current_jes_release) 50204

# Get the path to subcommands that will help updating
set x(cmd_to_sync_version) [file join $x(-serverroot) shared bin sync-version]
append result "cmd_to_sync_version $x(cmd_to_sync_version)\n"

set x(cmd_to_upgrade_sec) [file join $x(-serverroot) shared bin cert-convert]
append result "cmd_to_upgrade_sec $x(cmd_to_upgrade_sec)\n"

# Do the job...
set rc [catch {syncserverroot x} result]

# ...report and exit
if {$rc != 0} {
    puts stderr $result
    puts stderr "ERROR : $x(-serverroot) update failure."
    exit 1
} else {
    if {$x(-verbose)} {
        append result "SUCCESS: sync-dps.tcl script.\n"
        puts stdout $result
        puts stdout ""
        # to get traces with -verbose --> puts stderr $result
        # to get traces with -verbose --> puts stderr ""
    }
    exit 0
}


