#!/bin/ksh
#
# cci_wrapper.sh -- Wrapper script for the configUtility command.
#
# This script is expected to only be invoked via a symbolic link named
# to be one of the valid commands that this script supports.
#
# usage: xxx [-DEBUG] [-NORUN] parameter ...
#
# The precise set and ordering of parameters depends on the target
# command name.
#
# Copyright 2001-2003 Sun Microsystems, Inc. All rights reserved.
#
# Return Value:
#   0 - Success
#   1 - Usage problem or other problem.
#   5 - Retry might be worth doing to make it succeed.
###

# The list of commands supported by this script
#
# Each of the names below is normally the name of a symbolic link
# that points to this script.
#
REGARPCMD="reg_arp"
REGESMOMCMD="reg_esmom"
REGSCANNERCMD="reg_scanner"
REGTARGETCMD="reg_target"
SYNCESMOMCMD="sync_esmom"
UNREGARPCMD="unreg_arp"
UNREGESMOMCMD="unreg_esmom"
UNREGSCANNERCMD="unreg_scanner"
UNREGTARGETCMD="unreg_target"


: ${ESM_BASE:=`/usr/bin/pkgparam SUNWstm ESM_BASE 2>/dev/null`}
if [ -z "$ESM_BASE" ] ; then
    echo "Sorry, but ESM_BASE can not be resolved!" >&2
    exit 1
fi

: ${ESM_BIN:=$ESM_BASE/bin}

# Change "false" to "true" to enable extra debug output
: ${CCI_DEBUG:=/usr/bin/false}


#-------------------------------------------------------------------------------
# Utility Routine Section.
#-------------------------------------------------------------------------------

cci_debug () {
	$CCI_DEBUG && echo "$*" >&2
}
	
#-------------------------------------------------------------------------------
# Routine to get the ESMID for host
#
#
getHostESMId ()
{
	ESMID_VER=1
	ESMID_TAG=$ESMID_VER~host
	ESMID_TYPE=hostid

	hid=`hostid`
	vendor=`uname -p`
	if [ "$vendor" = "sparc" ]
	then
    	vendor=sun
	else
    	vendor=unknown
	fi

	ostype=`uname -s`
	if [ "$ostype" = "SunOS" ]
	then
		ostype=solaris
	else
		ostype=unknown
	fi	

	esmid=$ESMID_TAG~$vendor~$ostype~$ESMID_TYPE~$hid

	
	echo $esmid
}
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# Routine to get the ESMID for cluster
#
#
getClusterESMId ()
{
	ESMID_VER=1
	ESMID_TAG_VERITAS=$ESMID_VER~cluster
	
	#  Discovery uses its own algorithm to determine the ESMID since it cannot
	#  access the hex value on a sun cluster -- if the ip subnet address cannot
	#  be determined, then just use the regular cluster esmid algorithm.
	ESMID_TAG_SUN_DISCOVERY=temp~$ESMID_VER~cluster
	ESMID_TAG_SUN=$ESMID_VER~cluster
    _clusteralias=$1
    _clustertype=$2
    _sunclusterbin=/usr/cluster/bin
	
	cci_debug " cluster alias = $_clusteralias"
	cci_debug " cluster type = $_clustertype"

    if [ "$_clustertype" = "veritas" ]; then
        # get mac address of lowest llt node - lltstat -nvv
        CLUS_ID=`/sbin/lltstat -nvv | \
                awk '/^                            / && NF == 3 {print $3}' | \
                line`
        esmid=${ESMID_TAG_VERITAS}~veritas~vcs~custom~${_clusteralias}~${CLUS_ID}
		cci_debug "cluster id = $CLUS_ID"
		cci_debug "esm id = $esmid"
    else
		# This is NOT the esmid for a sun cluster since discovery can not often 
		# determine the hex value from running /usr/cluster/bin/scconf -p
		if IP_SUBNET=`getLocalNetAddr` ; then
			esmid=${ESMID_TAG_SUN_DISCOVERY}~sun~sc~unknown~${_clusteralias}~${IP_SUBNET}
		else
        	CLUSTER_ID=`${_sunclusterbin}/scconf -p | grep "Cluster ID:" | awk '{ print $3 }'`
        	esmid=${ESMID_TAG_SUN}~sun~sc~custom~${_clusteralias}~${CLUSTER_ID}			
		fi

		cci_debug "ip subnet = $IP_SUBNET"
		cci_debug "esm id = $esmid"
    fi
	
	echo "$esmid"
}
#-------------------------------------------------------------------------------


#-------------------------------------------------------------------------------
# Routine to return the IP Address of a host. If it can't be resolve, the
#  value passed in as the host name will be returned.
#
getIpAddr ()
{
    _hostname=$1
    _ipaddr=`$ESM_BIN/ResolveName -n $_hostname -i`
    [ $? -eq 0 ] || _ipaddr=$_hostname
    echo $_ipaddr
}

#-------------------------------------------------------------------------------
# Routine to fully resolve a host name or return what was passed in.
#
getFullHostName ()
{
    _hostname=$1
    _frhostname=`$ESM_BIN/ResolveName -n $_hostname`
    [ $? -eq 0 ] || _frhostname=$_hostname
    echo $_frhostname
}

#-------------------------------------------------------------------------------
# Routine to fully resolve the network address for the local host.
#
getLocalNetAddr ()
{
    interface=`/usr/bin/netstat -i                 | \
               /usr/bin/grep \`/usr/bin/hostname\` | \
               /usr/bin/head -1                    | \
               /usr/bin/awk '{ print $1 }'`

    _ifentry=`/usr/sbin/ifconfig $interface`

    _addr=${_ifentry##*inet }
    _addr=${_addr%% netmask*}

    _mask=${_ifentry##*netmask }
    _mask=${_mask%% broadcast*}

	# getnetaddr.pl outputs the net address to stdout
	#
    $ESM_BIN/getnetaddr.pl $_addr $_mask
	
	# Return the last exit value
}

#-------------------------------------------------------------------------------
# Ensure the proper base environment exists, and setup additional environment.
#
fatal ()
{
    if [ -n "$*" ] ; then el_log 1 "${CMD}: $*" ; fi
    exit 1
}


############################################################
# Find cURL location
#
# Currently, this looks for:
#   a) the curl executable in $ESM_BIN
#   b) the curl library (libcurl.so.2.0.2) in $ESM_BASE/lib
#
# Both of these can be found in Accurev at <dev_env_int>/curl
#
############################################################
findCURL() {

    # specify where the curl executable lives
    if [ ! -n "${CURLHOME}" ]; then
        if [ -f $ESM_BIN/curl ]; then
            CURL_HOME=$ESM_BIN/curl
            export CURL_HOME
        else
            echo "The install appears corrupted as no curl can be found."
            exit 1
        fi
    fi

    # specify where the curl libraries live
    # these are needed by the curl executable
    if [ ! -n "${LD_LIBRARY_PATH}" ]; then
        LD_LIBRARY_PATH="$ESM_BASE/lib"
    else
        LD_LIBRARY_PATH="$LD_LIBRARY_PATH:$ESM_BASE/lib"
    fi
    export LD_LIBRARY_PATH

}



############################################################
# Description:
# This sets cURL related options that we want passed on the
# command line.
#
setCurlEnvVars() {

    # indicates what version of SSL to use.  Valid SSL versions
    # are:  --sslv2 or --sslv3
    if [ ! -n "${SSL_VERSION_FLAG}" ]; then
        SSL_VERSION_FLAG=--sslv3
        export SSL_VERSION_FLAG
    fi

    # Valid certificate flags are:
    #   -k/--insecure = insecure mode
    #   --cert <certificate[:password]>  = this tells curl to use
    #          the specified cert file when getting a file with
    #          HTTPS.  The cert must be in PEM format.
    #   --cacert <CA certificate> = (HTTPS)  Tells  curl  to  use
    #          the specified certificate file to verify the peer.
    #          The file may contain  multiple CA certificates.
    #          The certificate(s) must be in PEM format.
    #   --capath <CA cert path> = (HTTPS)  Tells  curl  to  use
    #          the specified certificate directory to verify the
    #          peer. The certificates must  be in  PEM  format,
    if [ ! -n "${CERT_FLAGS}" ]; then
        CERT_FLAGS=-k
        export CERT_FLAGS
    fi


    # Buffering flags
    # by default returned data gets buffered.  The -N option will
    # allow us to disable that output buffering.
    # (for now lets not set it.)
    if [ ! -n "${CERT_FLAGS}" ]; then
        BUFF_FLAG=
        export BUFF_FLAG
    fi

    # error handling flags
    if [ ! -n "${ERROR_OUTPUT_FLAG}" ]; then
        ERROR_OUTPUT_FLAG="-S -s"
        export ERROR_OUTPUT_FLAG
    fi

}



#############################################################
# This takes the passed in args and puts them in an expected
# format that cURL looks for in order to perform get
# request.
#
# cURL takes the following args:
#      http://www.foo.com arg1=foo arg2=foo
# and will make a post request by putting them into the following
# format:   -G -d "arg1=foo&arg2=foo" http://www.foo.com
#
#############################################################
createGETRequest() {

    REQUEST="-G -d"
    POST_ARGS=
    i=0

    # loop thru args and make a big REQUEST statement for cURL
    for arg in $@
    do
        i=`expr $i + 1`

        if [ "$i" -eq 1 ]; then
            # the first arg in a url is separated with a ?
            LAST_ARG=$arg
        else
            if [ "$i" -eq $# ]; then
                # dont add an & after the last arg
                POST_ARGS=$POST_ARGS$arg
            else
                # all but first arg in a url are separated with a &
                POST_ARGS=$POST_ARGS$arg\&
            fi
        fi
    done

    REQUEST="-G -d \"$POST_ARGS\" $LAST_ARG"

    export REQUEST

    # execute our curl statement
    execCurl
}



#############################################################
# This takes the passed in args and puts them in an expected
# format that cURL looks for in order to perform a post
# request.
#
# cURL takes the following args:
#      http://www.foo.com arg1=foo arg2=foo
# and will make a post request by putting them into the following
# format:    -d "arg1=foo&arg2=foo" http://www.foo.com
#
#############################################################
createPOSTRequest() {

    REQUEST=-d
    POST_ARGS=
    i=0

    # loop thru args and make a big REQUEST statement for cURL
    for arg in $@
    do
        i=`expr $i + 1`

        if [ "$i" -eq 1 ]; then
            # the first arg in a url is separated with a ?
            LAST_ARG=$arg
        else
            if [ "$i" -eq $# ]; then
                # dont add an & after the last arg
                POST_ARGS=$POST_ARGS$arg
            else
                # all but first arg in a url are separated with a &
                POST_ARGS=$POST_ARGS$arg\&
            fi
        fi
    done

    REQUEST="$REQUEST \"$POST_ARGS\" $LAST_ARG"

    export REQUEST

    # execute our curl statement
    execCurl
}



###################################################################
# this will get called by createGETRequest() or createPOSTRequest()
###################################################################
execCurl() {
    RetCode=0

    findCURL
    setCurlEnvVars

    FULLCMD="$CURL_HOME $SSL_VERSION_FLAG $CERT_FLAGS $ERROR_OUTPUT_FLAG $BUFF_FLAG $REQUEST 2>&1"

    $DEBUG && echo "$FULLCMD"

    if $NORUN
    then
      echo
      echo "WARNING: NORUN mode: Command will not be executed."
      echo
    else
      HttpOutput=`eval $FULLCMD`
      RetVal=$?

      if [ $RetVal -ne 0 ] ; then
          echo "Communication to Web Server on ESM Management Station failed."
          echo "  $HttpOutput"
          echo
          RetCode=5
      else
          #
          # Check the HTTP Output for success or failure
          #
          $DEBUG && echo "$RetVal"

          echo $HttpOutput | /usr/bin/egrep -s SUCCESS
          rValue=$?
          if [ $rValue -ne 0 ]; then
             echo
             echo "Web Server Response is:"
             echo
             echo "$HttpOutput"
             RetCode=1
          fi

      fi
    fi

    return $RetCode
}

if [ -z "${ESM_BASE:=`/usr/bin/pkgparam SUNWstm ESM_BASE 2>/dev/null`}" ] ; then
    echo "${CMD}: Unable to resolve ESM_BASE. ESM must be installed." >&2
    exit 1
fi

if . $ESM_BASE/lib/esm_env.sh &&
   . $ESM_LIB/esm_globals.ksh &&
   . $ESM_LIB/esm_lib.ksh &&
   . $ESM_LIB/configvars_lib.ksh
then
    :
else
    echo "${CMD}: Unable to import standard ESM shell libraries." >&2
    exit 1
fi

#-------------------------------------------------------------------------------
#  Main Section
#-------------------------------------------------------------------------------


#-------------------------------------------------------------------------------
# Parse the common components of the command line
#
CMD=`/usr/bin/basename $0`

ESM_SILENT=${ESM_SILENT:-off}

DEBUG=false
if [ "$1" = "-DEBUG" ]; then
   DEBUG=true
   shift
fi

NORUN=false
if [ "$1" = "-NORUN" ] ; then
   DEBUG=true
   NORUN=true
   shift
fi

# The hostname or IP address of the Management Station
_ihserver=`echo $1 | cut -f 1 -d ":"`
_ihport=`echo $1 | cut -f 2 -d ":"`
if [ "$_ihserver" = "$_ihport" ]; then
    _ihport=`el_configvar_get IH_SERVICE_PORT`
fi
MGMTSVR=$_ihserver:$_ihport


#----- NOTE:
# Use the standard el_log logging function from this point forward
# for any info/warning/error messages EXCEPT usage displays; also,
# use the internal "fatal" function when appropriate.


if [ ! -L /dev/urandom ] ; then
    RANDFILE=${ESM_CONFIG}/sd.dat
    export RANDFILE

    if [ ! -f $RANDFILE ] ; then
        fatal "$RANDFILE does not exist.  Unable to communicate via HTTPS."
    fi
fi

#
# Setup THISIPADDR which will be used for usage displays and default values.
#
THISHOST=`/usr/bin/hostname`
THISIPADDR=`getIpAddr $THISHOST` ||
    fatal "Failed to resolve local host IP address."

#-------------------------------------------------------------------------------
# The bulk of the remainder of this script is One Giant Switchblock,
# dispatched based on the command name.
#
case $CMD in

#
# Register ESMOM Command
#
$REGESMOMCMD)

  if [ $# -ne 1 -a $# -ne 5 ]
  then
    echo
    echo "ERROR: Invalid Parameters."
    echo 
    echo "Usage:"
    echo "    $CMD <Mgmt-Server> [<ESMOM-Name> <ESMOM-RMI-Port> <ESMOM-Username> <ESMOM-Password>]"
    echo
    echo "    where <ESMOM-Username> is a user that has write access to the"
    echo '                           \\\\root\\esmv2 namespace. (Typically root)'
    echo "          <ESMOM-Password> is the password for the <ESMOM-Username>"
    echo
    echo "    If the optional parameters are not specified, the values configured"
    echo "    for the ESM Installation will be used (if available)."
    echo
    exit 1
  fi

  # if the parameters were specified, then use them.
  #  otherwise use the default configuration for this machine.
  if [ $# -eq 5 ]; then
    ESMOMNAME=`getIpAddr $2`
    ESMOMRMIPORT=$3
    ESMOMUSRNAME=$4
    ESMOMPASSWD=$5
  else
    ESMOMNAME=$THISIPADDR
    ESMOMRMIPORT=`el_configvar_get ESMOM_RMI_PORT`
    ESMOMUSRNAME=`el_configvar_get ESMOM_USER`
    ESMOMPASSWD=`el_configvar_get ESMOM_PASSWORD`
  fi

  if [ -z "$ESMOMNAME" -o -z "$ESMOMRMIPORT" -o -z "$ESMOMUSRNAME" ]; then
    echo
    echo "ERROR: Unable to determine a default value for <ESMOM-Name>, "
    echo "       <ESMOM-RMI-Port>, and/or <ESMOM-Username> from the local ESM"
    echo "       installation configuration."
    echo 
    echo "       Please specify these values on the command line."
    echo
    echo "Usage: $CMD <Mgmt-Server> <ESMOM-Name> <ESMOM-RMI-Port> <ESMOM-Username> <ESMOM-Password>"
    echo
    exit 1
  fi
   
  CCICMD="jobType=registerESMOM"
  CCIParams="esmomName=$ESMOMNAME esmomURL=$ESMOMNAME:$ESMOMRMIPORT userName=$ESMOMUSRNAME password=$ESMOMPASSWD"
  ;;

#
# Unregister ESMOM Cmd.
#
$UNREGESMOMCMD)
  if [ $# -lt 1 ]
  then
    echo
    echo "ERROR: Invalid Parameters."
    echo 
    echo "Usage: $CMD <Mgmt-Server> [<ESMOM-Name>]"
    echo
    echo "       if <ESMOM-Name> is not specified, \"$THISHOST($THISIPADDR)\" will be used."
    echo 
    exit 1
  fi

  ESMOMNAME=`getIpAddr $2`
  if [ -z "$ESMOMNAME" ]
  then
     ESMOMNAME=$THISIPADDR
  fi

  CCICMD="jobType=unregisterESMOM"
  CCIParams="esmomName=$ESMOMNAME"
  ;;

#
# Register ARP Cmd.
#
$REGARPCMD)

  if [ $# -ne 1 -a $# -ne 3 ]
  then
    echo
    echo "ERROR: Invalid Parameters."
    echo 
    echo "Usage: $CMD <Mgmt-Server> [<ARP-Name> <ARP-Port>]"
    echo
    echo "   If <ARP-Name> and <ARP-Port> are not specified, the default values for"
    echo "   this Agent Station (if configured) will be used."
    echo
    exit 1
  fi

  # if the parameters were specified, then use them.
  #  otherwise use the default configuration for this machine.
  if [ $# -eq 3 ]; then
     ARPNAME=`getIpAddr $2`
     ARPPORT=$3
  else
     ARPNAME=$THISIPADDR
     ARPPORT=`el_configvar_get ARP_HTTPS_PORT`
  fi

  if [ -z "$ARPPORT" -o -z "$ARPNAME" ]; then
    echo
    echo "ERROR: Unable to determine a default value for the <ARP-Name> and/or"
    echo "       <ARP-Port> values from the local Agent Station configuration."
    echo 
    echo "       Please specify these values on the command line."
    echo
    echo "Usage: $CMD <Mgmt-Server> <ARP-Name> <ARP-Port>"
    echo
    exit 1
  fi
   
  CCICMD="jobType=registerARPS"
  CCIParams="arpsName=$ARPNAME arpsURL=https://$ARPNAME:$ARPPORT bugWorkaround=/ userName=nobody password=nothing"
  ;;

#
# UnRegister ARP Cmd.
#
$UNREGARPCMD)

  if [ $# -lt 1 ]
  then
    echo
    echo "ERROR: Invalid Parameters."
    echo 
    echo "Usage: $CMD <Mgmt-Server> [<ARP-Name>]"
    echo
    echo "       if <ARP-Name> is not specified, \"$THISHOST($THISIPADDR)\" will be used."
    echo 
    exit 1
  fi

  if [ ! -z "$2" ]
  then
    ARPNAME=`getIpAddr $2`
  else
    ARPNAME=$THISIPADDR
  fi

  CCICMD="jobType=unregisterARPS"
  CCIParams="arpsName=$ARPNAME"
  ;;

#
# Register Scanner Cmd.
#
$REGSCANNERCMD)
  if [ $# -lt 0 ]
  then
    echo
    echo "ERROR: Invalid Parameters."
    echo 
    echo "Usage: $CMD <Mgmt-Server> *** NEED PARAMETER LIST *** ]"
    echo
    exit 1
  fi

  CCICMD="jobType=registerScanner"
  CCIParams=""

  fatal "Sorry, but command '$CMD' is not yet implemented!"
  ;;

#
# UnRegister Scanner Cmd.
#
$UNREGSCANNERCMD)
  if [ $# -lt 0 ]
  then
    echo
    echo "ERROR: Invalid Parameters."
    echo 
    echo "Usage: $CMD <Mgmt-Server> *** NEED PARAMETER LIST *** ]"
    echo
    exit 1
  fi

  CCICMD="jobType=unregisterScanner"
  CCIParams=""

  fatal "Sorry, but command '$CMD' is not yet implemented!"
  ;;

#
# Register Target Cmd.
#
$REGTARGETCMD)
  if [ $# -ne 5 ]
  then
    echo
    echo "ERROR: Invalid Parameters."
    echo 
    echo "    $CMD <Mgmt-Server> <Asset-Name> <Asset-Type> <Asset-SubType>  <Access-Path>"
    echo
    echo "    where <Asset-Name> is the name of the device to be registered"
    echo "          <Asset-Type> is the type of target "
    echo "          <Asset-SubType> is the target sub-type"
    echo "          <Access-Path> is the location of the remote agent"
    echo 
    echo "    Valid asset-type and asset-subtypes:                                          "
    echo "                 ASSET TYPE               ASSET SUB-TYPE                          "
    echo "                 ----------               --------------                          "
    echo "                 host                     solaris or virtual                      "
    echo "                 array                    3500, 3900, 6120, 6320, 6900, 9900 or T3"
    echo "                 cluster                  sun or veritas                          "
    echo "                 switch                   brocade, mcdata, qlogic or sun          "
    echo
    echo
    exit 1
  fi

  if [ $# -eq 5 ]
  then
    ASSETNAME=$2
    ASSETTYPE=$3
    ASSETSUBTYPE=$4
    ACCESSPATH=`getIpAddr $5`
    shift
  else [ $# -eq 4 ]
    ASSETNAME=$2
    ASSETTYPE=$3
    ASSETSUBTYPE=$4
    ACCESSPATH=$THISIPADDR
  fi  
       
  if $DEBUG ; then
      echo "    Asset Name = '$ASSETNAME'"
      echo "    Asset Type = '$ASSETTYPE'"
      echo "Asset Sub-type = '$ASSETSUBTYPE'"
      echo "   Access Path = '$ACCESSPATH'" 
  fi

  ESM_ASSET_TYPE="esm.common.assettype.$ASSETTYPE"

  # Validate the Asset Type and Subtypes. Also make any tweaks to 
  #  The ESM_ASSET_TYPE and ESM_ASSET_SUB_TYPE values if necessary.
  case "$ASSETTYPE" in

	array)
		case "$ASSETSUBTYPE" in
		  3500|3900|6120|6320|6900|9900|T3)
                        ESM_ASSET_SUB_TYPE="$ESM_ASSET_TYPE.$ASSETSUBTYPE"
			;;
		  *) fatal "Invalid asset sub-type: '$ASSETSUBTYPE'"
		     ;;
		esac
		;;
		
	cluster)
		case "$ASSETSUBTYPE" in
		  sun|veritas)  
			   ESM_ID=`getClusterESMId "$ASSETNAME" "$ASSETSUBTYPE"`
			   #echo "ESM ID from getClusterESMId call = '$ESM_ID'"
               ESM_ASSET_SUB_TYPE="$ESM_ASSET_TYPE.$ASSETSUBTYPE"
			   ;;
                  *) fatal "Invalid asset sub-type: '$ASSETSUBTYPE'"
			   ;;
		esac
		;;
		
	host)
        ESM_ID=`getHostESMId`
		#echo "ESM ID from getHostESMId call = $ESM_ID"
        case "$ASSETSUBTYPE" in
                    solaris|virtual)
                            ESM_ASSET_SUB_TYPE="$ESM_ASSET_TYPE.$ASSETSUBTYPE"
			   ;;
                  *) fatal "Invalid asset sub-type: '$ASSETSUBTYPE'"
			   ;;
		esac
		;;
		
	switch)
		case "$ASSETSUBTYPE" in
		  brocade|mcdata|qlogic|sun)
                        ESM_ASSET_SUB_TYPE="$ESM_ASSET_TYPE.$ASSETSUBTYPE"
			;;
		  *) fatal "Invalid asset sub-type: '$ASSETSUBTYPE'"
		     ;;
		esac
		;;
		
        *)
   		fatal "Manual Registration Not Supported for Asset Type = $ASSETTYPE"
                ;;
  esac
	  
  CCICMD="jobType=registerTarget"
  CCIParams="displayName=$ASSETNAME esmNavAssetType=$ESM_ASSET_TYPE assetSubType=$ESM_ASSET_SUB_TYPE accessPath=$ACCESSPATH esmid=$ESM_ID"
  ;;

#
# UnRegister Target Cmd.
#
$UNREGTARGETCMD)
  if [ $# -ne 4 ]
  then
    echo
    echo "ERROR: Invalid Parameters."
    echo 
    echo "    $CMD <Mgmt-Server> <Asset-Name> <Asset-Type> <Asset-SubType>"
    echo
    echo "    where <Asset-Name> is the name of the device to be unregistered"
    echo "          <Asset-Type> is the type of target "
    echo "          <Asset-SubType> is the target sub-type"
    echo 
    echo "    Valid asset-type and asset-subtypes:                                          "
    echo "                 ASSET TYPE               ASSET SUB-TYPE                          "
    echo "                 ----------               --------------                          "
    echo "                 host                     solaris or virtual                      "
    echo "                 array                    3500, 3900, 6120, 6320, 6900, 9900 or T3"
    echo "                 cluster                  sun or veritas                          "
    echo "                 switch                   brocade, mcdata, qlogic or sun          "
    echo
    echo
    exit 1
  fi

  if [ $# -eq 4 ]
  then
    ASSETNAME=$2
    ASSETTYPE=$3
    ASSETSUBTYPE=$4
    shift
  fi
       
  ESM_ASSET_TYPE="esm.common.assettype.$ASSETTYPE"

  # Validate the Asset Type and Subtypes. Also make any tweaks to 
  #  The ESM_ASSET_TYPE and ESM_ASSET_SUB_TYPE values if necessary.
  case "$ASSETTYPE" in

	array)
		case "$ASSETSUBTYPE" in
		  3500|3900|6120|6320|6900|9900|T3)
		  				ESM_ASSET_SUB_TYPE="$ESM_ASSET_TYPE.$ASSETSUBTYPE"
			;;
		  *) fatal "Invalid asset sub-type: '$ASSETSUBTYPE'"
		     ;;
		esac
		;;
		
	cluster)
		case "$ASSETSUBTYPE" in
		  sun|veritas)
		  				ESM_ASSET_SUB_TYPE="$ESM_ASSET_TYPE.$ASSETSUBTYPE"
			   ;;
                  *) fatal "Invalid asset sub-type: '$ASSETSUBTYPE'"
			   ;;
		esac
		;;
		
	host) 
        case "$ASSETSUBTYPE" in
                    solaris|virtual)
                        ESM_ASSET_SUB_TYPE="$ESM_ASSET_TYPE.$ASSETSUBTYPE"
			   ;;
                  *) fatal "Invalid asset sub-type: '$ASSETSUBTYPE'"
			   ;;
		esac
		;;
		
	switch)
		case "$ASSETSUBTYPE" in
		  brocade|mcdata|qlogic|sun)
                        ESM_ASSET_SUB_TYPE="$ESM_ASSET_TYPE.$ASSETSUBTYPE"
			;;
		  *) fatal "Invalid asset sub-type: '$ASSETSUBTYPE'"
		     ;;
		esac
		;;
		
        *)
   		fatal "Manual Registration Not Supported for Asset Type = $ASSETTYPE"
                ;;

  esac
	  
  CCICMD="jobType=unregisterTarget"
  CCIParams="displayName=$ASSETNAME esmNavAssetType=$ESM_ASSET_TYPE assetSubType=$ESM_ASSET_SUB_TYPE"
  ;;

#
# Sync ESMOM Cmd.
#
$SYNCESMOMCMD)
  if [ $# -lt 1 ]
  then
    echo
    echo "ERROR: Invalid Parameters."
    echo 
    echo "Usage: $CMD <Mgmt-Server> [<ESMOM-Name>]"
    echo
    echo "       if <ESMOM-Name> is not specified, all ESMOMS will be "
    echo "       synchronized."
    echo 
    exit 1
  fi

  ESMOMNAME=
  if [ $# -eq 2 ]; then
    ESMOMNAME=`getIpAddr $2`
  fi  
  CCIParams=
  if [ ! -z "$ESMOMNAME" ]
  then
    CCIParams="esmomName=$ESMOMNAME"
  fi

  CCICMD="jobType=Synchronize"
  ;;

#
# Default Case (Invalid Command)
#
*)
   echo
   echo "ERROR: Invalid command usage."
   echo
   echo "    This command is not intended to be called directly. Please"
   echo "    create symbolic links to this file with one of the valid"
   echo "    commands that this script supports."
   echo 

   exit 1;;

esac

#-------------------------------------------------------------------------------
# Execute the configUtility program to effect the validated command,
# or display the intended command if NORUN is enabled.
#
CCIURL="https://$MGMTSVR/esmih/CCIServlet"

createPOSTRequest $CCIURL $CCICMD $CCIParams
RetCode=$?

if [ $RetCode -eq 0 ]; then
       echo "$CMD Success"
else
       echo "$CMD Failure"
fi

exit $RetCode
