;;;                 Sun Public License Notice
;;;
;;; The contents of this file are subject to the Sun Public License
;;; Version 1.0 (the "License"). You may not use this file except in
;;; compliance with the License. A copy of the License is available at
;;; http://www.sun.com/
;;;
;;; The Original Code is NetBeans. The Initial Developer of the Original
;;; Code is Sun Microsystems, Inc. Portions Copyright 1997-2000 Sun
;;; Microsystems, Inc. All Rights Reserved.

(require 'cl)

(defvar netbeans-root-menu (vector 'netbeans-menu *ide-name*  nil nil nil))
(defvar netbeans-tools-menu nil
  "The name of the %*ide-name*% submenu under which the tools specified by the IDE are listed" )
(make-variable-buffer-local 'netbeans-tools-menu)
(put 'netbeans-tools-menu 'permanent-local t)

(defvar netbeans-start-button
  (vector 'netbeans-button (format "Start %s" *ide-name*) nil
		   t nil 'netbeans-start t))
(defvar netbeans-quit-button
  (vector 'netbeans-button (format "Quit %s" *ide-name*) nil
		  nil nil 'netbeans-quit t))
(defvar netbeans-mount-button
  [netbeans-button "Mount File's Directory" nil
		  nil (and *netbeans-network-connection* (not netbeans-buffer-number)) netbeans-mount-and-open-file t])

(defvar netbeans-toggle-breakpoint-button
  [netbeans-button "Toggle Breakpoint" "C-f8"
		  nil (netbeans-check-toggle-breakpoint-conditions) netbeans-toggle-breakpoint nil])

(defvar netbeans-menu-item-alist nil
  "Alist from action name to the actual MENU-ITEM-PATH.")
(make-variable-buffer-local 'netbeans-menu-item-alist)
(put 'netbeans-menu-item-alist 'permanent-local t)

(defvar netbeans-start-toolbarbutton-sensitive t
  "*If t, %*ide-name*% toolbar button should be enabled.")

(defvar netbeans-toolbar-icon-directory nil
  "The location where netbeans toolbar icons reside")

;; An netbeans-menu object has the form:
;;    [netbeans-menu LABEL ITEMS HANDLE KEYMAP]

;; 'netbeans-menu : symbol, identifies this vector as an netbeans-menu
;;  label       : string, displayed in menu bar
;;  items       : array of menu items
;;  handle      : editor specific handle to menu
;;  keymap      : keymap  holding the keybindings for the menu

(defun netbeans-menu-p (object)
  "Return t if OBJECT is an netbeans-menu."
  (netbeans-debug "In menu:menu-p")
  (and (vectorp object)
       (= (length object) 5)
       (eq (aref object 0) 'netbeans-menu)))

(netbeans-def-member netbeans-menu label 1)
(netbeans-def-member netbeans-menu items 2)
(netbeans-def-member netbeans-menu handle 3)
(netbeans-def-member netbeans-menu keymap 4)

;; An netbeans-button object has the form:
;;    [netbeans-button LABEL ACCELERATOR SENSE SENSESYM CMD]

;; 'netbeans-button : symbol, identifies this vector as an netbeans-button
;;  label         : string, displayed in button
;;  accelerator   : string, key binding to invoke button
;;  sense         : bool, if t, button is enabled [sense default value]
;;  sensesym      : symbol, used to store sense current value.if specified, sense is not needed
;;  cmd           : function, function to call on button up if not nil
;;  included      : bool,  if t, button is displayed when not enabled. if nil, the visibility depends on the value of sensesym. See `current-menubar' for more info.

(defun netbeans-button-p (object)
  "Return t if OBJECT is an netbeans-button."
  (netbeans-debug "In menu:button-p")
  (and (vectorp object)
       (= (length object) 7)
       (eq (aref object 0) 'netbeans-button)))

(netbeans-def-member netbeans-button label 1)
(netbeans-def-member netbeans-button accelerator 2)
(netbeans-def-member netbeans-button sense 3)
(netbeans-def-member netbeans-button sensesym 4)
(netbeans-def-member netbeans-button cmd 5)
(netbeans-def-member netbeans-button included 6)


(defun netbeans-submenu (&rest buttons)
  "Creates a XEmacs submenu from the BUTTONS"
  (netbeans-debug "In menu:submenu")
  (mapcar #'(lambda (button)
	      (netbeans-menuitem-create netbeans-root-menu button))
	  buttons))

(defun netbeans-menu-init ()
  "Initializes the `netbeans-root-menu'. Calls `netbeans-menu-create' for all the open buffers."
  (netbeans-debug "In menu:init")
  (let ((menulist
	 `(
	   ,(netbeans-menuitem-create netbeans-root-menu
					   netbeans-start-button)
	   "----"
	   ,(netbeans-menuitem-create netbeans-root-menu
					   netbeans-mount-button)
	   ,(netbeans-menuitem-create netbeans-root-menu
					   netbeans-toggle-breakpoint-button)
	   ("Tools") ;place holder for Tools menu
	   "----"
	   ,(netbeans-menuitem-create netbeans-root-menu
					   netbeans-quit-button))))

    (setf (netbeans-menu-handle netbeans-root-menu) (list (netbeans-menu-label netbeans-root-menu)))
    (setf (netbeans-menu-items netbeans-root-menu) menulist)

    (save-excursion
      (dolist (buffer (buffer-list))
	(set-buffer buffer)
	(netbeans-menu-create)))))

(defun netbeans-menu-create (&optional buffer)
  "Initializes the `netbeans-tools-menu'. Adds the %*ide-name*% menu to the XEmacs menu bar for BUFFER. If no BUFFER, `current-buffer' is assumed."
  (netbeans-debug "In menu:create")
  (unless buffer
    (setq buffer (current-buffer)))
  (save-excursion
   (set-buffer buffer) 
   (setq netbeans-tools-menu (vector 'netbeans-menu "Tools"  nil nil nil))
   (setf (netbeans-menu-handle netbeans-tools-menu) 
	 (list (netbeans-menu-label netbeans-root-menu) 
	       (netbeans-menu-label netbeans-tools-menu)))
   (when *netbeans-network-connection*
     (netbeans-create-import-class-menuItem))
   (add-submenu
        nil
	(append (list (netbeans-menu-label netbeans-root-menu) :filter 'netbeans-menu-filter) 
		 (netbeans-menu-items netbeans-root-menu)))))


(defun netbeans-menu-remove ()
  "Remove the %*ide-name*% menu from the xemacs menu bar"
  (netbeans-debug "In menu:menu-remove")
  (when (car (find-menu-item current-menubar
			     (list (netbeans-menu-label netbeans-root-menu))))
    (save-excursion
      (dolist (buffer (buffer-list))
	(set-buffer buffer)
	(netbeans-clean-tools-menu)
	(delete-menu-item (netbeans-menu-handle netbeans-root-menu))))))


(defun netbeans-menu-reinit (connected)
  "Reinitialize the menu items and toolbar depending on CONNECTED."
  (netbeans-debug "In menu:reinit-menuItems")
  (netbeans-set-sensitivity netbeans-start-button (not connected))
  (netbeans-set-sensitivity netbeans-quit-button connected)
  (setq netbeans-start-toolbarbutton-sensitive (not connected))
  (netbeans-refresh-toolbar) 					
  (save-excursion
    (dolist (buffer (buffer-list))
      (set-buffer buffer)
      (if (not connected)
	  (netbeans-clean-tools-menu)
	(netbeans-create-import-class-menuItem)))))

(defun netbeans-clean-tools-menu ()
  "Delete the \"%*ide-name*%>Tools\" submenu. Unassociate any keybindings defined by the menu-items in this submenu."
  (if (null netbeans-tools-menu)
      (netbeans-menu-create)
    (setq minor-mode-map-alist(remrassoc (netbeans-menu-keymap netbeans-tools-menu) minor-mode-map-alist))
    (setf (netbeans-menu-keymap netbeans-tools-menu) nil)
    (setf (netbeans-menu-items netbeans-tools-menu) nil))
  (setq  netbeans-menu-item-alist nil))

(defun netbeans-menuitem-create (menu button)
  "Return an xemacs menuitem from BUTTON."
  (netbeans-debug "In menu:menuitem-create")
  ;; First fix up netbeans-button
  (when (not (netbeans-button-sensesym button)) 
    (setf (netbeans-button-sensesym button)
	  (netbeans-button-create-sensesym
	   button (netbeans-menu-label menu)))
    (set (netbeans-button-sensesym button)
	 (netbeans-button-sense button)))
  (let ((menu-item
	 (vector
	  (netbeans-button-label button)
	  (netbeans-button-cmd button)
	  :active (netbeans-button-sensesym button)
	  :included (if (netbeans-button-included button)
			t
		      (netbeans-button-sensesym button)))))
   
    (if (netbeans-button-accelerator button)
	(setq menu-item (vconcat menu-item (vector :keys  (netbeans-button-accelerator button)))))
    menu-item))
     

(defun netbeans-button-create-sensesym (button name-prefix)
  "Return the symbol which when evaluated determines button sense."
  (netbeans-debug "In menu:button-create-sensesym")
  (when (and name-prefix button)
    (intern (concat name-prefix "-"
		    (netbeans-button-label button) "-sense"))))

(defun netbeans-set-sensitivity (button &optional sense)
  "Set the sensitivity of BUTTON to SENSE. If SENSE is nil or omitted, the BUTTON is disabled."
  (netbeans-debug "In menu:set-sensitivity")
  (set (netbeans-button-sensesym button) sense))

(defun netbeans-menu-filter (menu)
  "This is the menu filter for the top-level \"%*ide-name*%\" menu.
It dynamically creates the \"Tools\" submenu"
  (netbeans-debug "In menu:menu-filter %s %s " (listp menu) menu)
  (if (null netbeans-tools-menu)
      (netbeans-menu-create))
  (let ((index -1))
    (dolist (item menu)
      (setq index (1+ index))
      (when (and (listp item) (equal (car item) (netbeans-menu-label netbeans-tools-menu)))
	(setcdr item  (netbeans-menu-items netbeans-tools-menu))
	(return)))
    menu))

(defun netbeans-create-minor-mode-local-keymap ()
  "Creates a keymap which will be buffer-local and adds it to `minor-mode-map-alist'."
  (netbeans-debug "In menu:create-minor-mode-local-keymap")
  (let* ((var-name (intern (format "netbeans-buf%s-local-keymap-var" netbeans-buffer-number)))
	 (var (eval `(defvar ,var-name nil "Internally created variable for adding the buffer-local keymap to the `minor-mode-map-alist' DO NOT CHANGE THE VALUE OF THIS VARIABLE. See also `netbeans-create-minor-mode-local-keymap'")))
	 (keymap (make-sparse-keymap))) 
    (make-variable-buffer-local `(,@var ))
    (put `(,@var ) 'permanent-local t)
    (set `(,@var ) t)
    (setf (netbeans-menu-keymap netbeans-tools-menu) keymap)
    (pushnew (cons `(,@var) keymap) minor-mode-map-alist :test 'equal)))

(defun* netbeans-frob-toolbar ()
  "Replace Debug and Compile toolbar buttons with the `Start %*ide-name*%' button.
   It will try to locate the icons named netbeans-cap-up.xpm, -cap-xx.xpm, -up.xpm and
   -xx.xpm first in `netbeans-toolbar-icon-directory' if not null. Otherwise, it will look in the \"images\" directory where \"netbeans.el\" resides."
  (netbeans-debug "In menu:frob-toolbar")
  (interactive)
  (when (and (featurep 'xpm)
	     (featurep 'toolbar)
	     (consp (specifier-instance default-toolbar)))
    (let ((up (if (not (null netbeans-toolbar-icon-directory))
		  (expand-file-name "netbeans-up.xpm" netbeans-toolbar-icon-directory)
		(netbeans-path-search "images/netbeans-up.xpm" load-path)))
	  (disabled (if (not (null netbeans-toolbar-icon-directory))
			(expand-file-name "netbeans-xx.xpm" netbeans-toolbar-icon-directory)
		      (netbeans-path-search "images/netbeans-xx.xpm" load-path)))
	  (cap-up (if (not (null netbeans-toolbar-icon-directory))
		      (expand-file-name "netbeans-cap-up.xpm" netbeans-toolbar-icon-directory)
		    (netbeans-path-search "images/netbeans-cap-up.xpm" load-path)))
	  (cap-disabled (if (not (null netbeans-toolbar-icon-directory))
			    (expand-file-name "netbeans-cap-xx.xpm" netbeans-toolbar-icon-directory)
			  (netbeans-path-search "images/netbeans-cap-xx.xpm" load-path))))
      (when (null up)
	 (error "Cannot locate the %s toolbar icons. See `netbeans-frob-toolbar' for more details." *ide-name*)
	 (return-from  netbeans-frob-toolbar nil))
      (setq toolbar-netbeans-icon
	    (toolbar-make-button-list up nil disabled cap-up nil cap-disabled)))

    ;; Remove the (non-NetBeans) debug and compile button from the toolbar
    (let ((buttons (specifier-instance default-toolbar)))
      (setq buttons
	    (delete-if
	     (lambda (button)
	       (memq (aref button 1) '(toolbar-debug toolbar-compile)))
	     buttons))

      ;; Add netbeans button, if not already there
      (unless (find-if (lambda (button) (eq (aref button 1) 'netbeans-start))
		       buttons)
	(setq buttons
	      (append buttons
		      (list (vector 'toolbar-netbeans-icon
			 'netbeans-start
			 'netbeans-start-toolbarbutton-sensitive
			 (format "Start %s" *ide-name*)))))

      (set-specifier default-toolbar buttons)))))


(defun netbeans-refresh-toolbar ()
  "Update the default toolbar with changes to netbeans buttons"
  (netbeans-debug "In menu:refresh-toolbar")
  (set-specifier default-toolbar (specifier-instance default-toolbar)))

(defun* netbeans-create-import-class-menuItem ()
  "Creates the \"Import Class...\" menu item under Tools menu if the buffer's major-mode is `jde-mode' or `java-mode'"
  (netbeans-debug "In menu:create-import-class-menuItem")
  (if (not (and  netbeans-buffer-number
		 (or (equal major-mode 'jde-mode)
		     (equal major-mode 'java-mode))))
      (return-from netbeans-create-import-class-menuItem))
  (let ((menu-item (vector "Import Class..."
			   'netbeans-import-java-class
			   :active t
			   :included t
			   :keys  "A-I")))
    (if (not (netbeans-menu-keymap netbeans-tools-menu))
	(netbeans-create-minor-mode-local-keymap))
    (define-key (netbeans-menu-keymap netbeans-tools-menu) [(alt I)] 'netbeans-keyboard-event)
    (setf (netbeans-menu-items netbeans-tools-menu) (list menu-item))))


(defun netbeans-import-java-class ()
  "Callback function for \"Import Class\" menu item in JDE or JAVA major-mode"
  (netbeans-debug "In menu:import-java-class")
  (interactive)
  (netbeans-keyboard-event  (make-event 'key-press '(key ?I modifiers (alt)))))


(defun netbeans-check-toggle-breakpoint-conditions ()
  "Checks whether the required conditions are set to toggle a breakpoint."
  (netbeans-debug "In menu:check-toggle-breakpoint-conditions")
   (if (and  *netbeans-network-connection* netbeans-buffer-number
	     (or (equal major-mode 'jde-mode)
		 (equal major-mode 'java-mode)
		 (equal major-mode 'c++-mode)
		 (equal major-mode 'c-mode)
		 (equal major-mode 'fortran-mode)))
       t
     nil))

(defun netbeans-toggle-breakpoint ()
  "Callback function for toggling breakpoints."
  (netbeans-debug "In menu:toggle-breakpoint")
  (interactive)
  (if (netbeans-check-toggle-breakpoint-conditions)
      (netbeans-keyboard-event  (make-event 'key-press '(key f8 modifiers (shift))))))


;;========
;;EVENTS
;;========
;;EVT_invokeAction
(defun netbeans-invoke-action (menu-item-path action)
  "Create a command function to be invoked when MENU-ITEM-PATH is selected to invoke ACTION. Returns the name of that command function."
  (netbeans-debug "In menu:invoke-action %s" menu-item-path)
  (let ((path-str (format "netbeans-%s-callback" netbeans-buffer-number)))
    (dolist (path menu-item-path)
      (setq path-str (concat path-str "-" path)))
    (eval `(defun ,(intern path-str)
	     ()
	     "Internally created callback function for the menu item. DO NOT USE THIS FUNCTION EXPLICITELY." 
	     (interactive)
	     (netbeans-event netbeans-buffer-number  "invokeAction" ,action)))))

;;========
;;HANDLERS
;;========
;CMD_actionMenuItem
(defun netbeans-cmd-actionMenuItem (bufnum path action enabled keyBinding)
  "Create and add a menu item."
  (netbeans-debug "In menu:cmd-actionMenuItem")
  ;;get rid of "&" mnemonics
  (let ((path-components (split-string-by-char path  ?&)))
    (setq path nil)
    (dolist (component path-components)
      (setq path (concat path component) )))
  (netbeans-with-buffer bufnum
    (let* ((menu-path  (split-string-by-char path ?: ))
	   (menu-item-path (append (netbeans-menu-handle netbeans-tools-menu) (copy-tree menu-path)))
	   callback
	   menu-item)
      (if (equal action "SEP")
	  (progn
	    (setq menu-item "----"
		  menu-path (delete "" menu-path)))
	(let ((name (car (last menu-path))))
	  (setq menu-path (delete name menu-path))
	  (setq callback (netbeans-invoke-action menu-item-path action))
	  (push (list action menu-item-path) netbeans-menu-item-alist)
	  (setq menu-item (vector name
				  callback
				  :active enabled
				  :included t)))
	(when (and keyBinding (not (equal keyBinding "")))
	  (if (not (netbeans-menu-keymap netbeans-tools-menu))
	      (netbeans-create-minor-mode-local-keymap))
	  (if (netbeans-define-key keyBinding callback (netbeans-menu-keymap netbeans-tools-menu))
	      (setq menu-item (vconcat menu-item (vector :keys  keyBinding))))))

      ;;put the menu-item in MENU-LEAF form
      (dolist (path-item  (reverse menu-path))
	(setq menu-item (list path-item menu-item)))
      (let ((list-to-add (netbeans-menu-items netbeans-tools-menu)))
	(if (null list-to-add)
	    (when (not (equal action "SEP")) ;don't add SEP as first menu item
	      (setf (netbeans-menu-items netbeans-tools-menu) (list menu-item)))
	  (dolist (path-item  menu-path)
	    (dolist (tools-list-item list-to-add)
	      (when (and (listp tools-list-item) (equal (car tools-list-item) path-item))
		(setq list-to-add tools-list-item)
		(setq menu-item 
		      (if (listp (cadr menu-item))
			  (cadr menu-item)
			(cdr menu-item)))
		(return))))
	  (cond 
	   ;;add as a vector if there is no remaining parent-menu in menu-item
	   ((and (listp menu-item) (vectorp (car menu-item)))
	    (nconc list-to-add menu-item))
	   ;;add SEP if there is no remaining parent-menu in menu-item
	   ((and (listp menu-item) (equal action "SEP") (= (length menu-item) 1))
	    (nconc list-to-add menu-item))
	   ;;don't add SEP if SEP is the first element in a submenu
	   ((and (listp menu-item) (equal action "SEP") (> (length menu-item) 1))
	    nil)
	   (t
	    (nconc list-to-add  (list menu-item)))))))))
		   
;;CMD_actionSensitivity
(defun netbeans-cmd-actionSensitivity (bufnum action enabled)
  "Set the sensitivity of the menu button invoking the ACTION to ENABLED."
  (netbeans-debug "In menu:cmd-actionSensitivity")
  (netbeans-with-buffer bufnum
    (let ((pair (assoc action netbeans-menu-item-alist)))
      (if (not pair)
	  (error "The menu button for given ACTION is not found.")
	;;init Tools menu in case the buttons are not added yet
	(add-submenu (netbeans-menu-handle netbeans-root-menu) 
		     (append (list (netbeans-menu-label netbeans-tools-menu)) 
			     (netbeans-menu-items netbeans-tools-menu))) 
	(if enabled
	    (enable-menu-item (cadr pair))
	  (disable-menu-item (cadr pair)))))))


;;initilaze the NetBeans menu for all buffers.
(netbeans-menu-init)

(provide 'netbeans-menus)
