;;;                 Sun Public License Notice
;;;
;;; The contents of this file are subject to the Sun Public License
;;; Version 1.0 (the "License"). You may not use this file except in
;;; compliance with the License. A copy of the License is available at
;;; http://www.sun.com/
;;;
;;; The Original Code is NetBeans. The Initial Developer of the Original
;;; Code is Sun Microsystems, Inc. Portions Copyright 1997-2000 Sun
;;; Microsystems, Inc. All Rights Reserved.

(require 'cl)
(require 'netbeans-vars)

(defun netbeans-kill ()
  "Kill %*ide-name*% process `netbeans-app-process' from XEmacs and disconnect from it."
  (interactive)
  (netbeans-debug "in invocation:netbeans-kill")
  (unless (or netbeans-app-process 
	      netbeans-app-started
	      netbeans-nsf-timer)
    (error "Cannot kill %s; it has not been started yet.  Use M-x netbeans-start" *ide-name*))
  (delete-process netbeans-app-process))

(defun netbeans-app-process-sentinel (process status)
  "Handle changes in STATUS to workshop PROCESS."
  (netbeans-debug "in invocation:app-process-sentinel")
  (netbeans-debug "process-status = %s status = %s" (process-status process) status)
  (when (and 
	 (memq (process-status process) '(signal exit closed))
	 netbeans-app-process)
    (netbeans-cleanup-after-exit)
    (setq netbeans-app-process nil)
    (netbeans-log-to-output-buffer (format  "%s terminated..." *ide-name*) t t)))

(defun netbeans-start-cb (proc)
  "Callback function to call after %*ide-name*% is launched from XEmacs."
  (netbeans-debug "in invocation:netbeans-start-cb")
  (setq netbeans-app-started t)
  (setq netbeans-app-process proc)
  (netbeans-start-nsf-timer)
  (set-process-sentinel netbeans-app-process 'netbeans-app-process-sentinel))


(defun netbeans-cleanup-after-exit ()
  "Cleanup after %*ide-name*% exits."
  (netbeans-debug "in invocation:cleanup-after-exit")
  (setq netbeans-app-started nil)
  (netbeans-cancel-nsf-timer)
  (netbeans-disconnect))

(defun netbeans-start-nsf-timer ()
  "Initializes `netbeans-nsf-timer'"
  (netbeans-debug "in invocation:start-nsf-timer")
  (setq netbeans-nsf-timer
	(start-process "nsf-timer" nil "sleep"
		       (int-to-string netbeans-nsf-timer-duration)))
  (set-process-sentinel netbeans-nsf-timer 'netbeans-nsf-timer-expired))

(defun netbeans-nsf-timer-expired (proc status)
  "This function is called when `netbeans-nsf-timer' timeouts or is killed."
  (netbeans-debug "in invocation:nsf-timer-expired")
  (netbeans-debug "process-status = %s status = %s" (process-status proc) status)
  (when (memq (process-status proc) '(exit signal))
    (setq netbeans-nsf-timer nil)
    (netbeans-check-for-startup-file)))

(defun netbeans-check-for-startup-file ()
  "Checks if `netbeans-startup-file' is created. If yes, it loads it. Otherwise it keeps waiting for it."
  (netbeans-debug "in invocation:check-for-startup-file")
  (incf netbeans-nsf-timer-total netbeans-nsf-timer-duration)
  (cond
   (netbeans-nsf-timer-aborted
    (netbeans-log-to-output-buffer (format "%s invocation is aborted. If this is caused by an invalid switch to invocation command, please go to Options>Customize>Emacs>Applications>%s IDE>App Invocation Cmd and edit the arguments." *ide-name* *ide-name*) nil t)
    (setq netbeans-nsf-timer-aborted nil))
   ((file-readable-p netbeans-startup-file)
    (netbeans-load-startup-file)
    (netbeans-cancel-nsf-timer)
    (netbeans-log-to-output-buffer (format "Connected to %s"  *ide-name*) t t))
   ((>= netbeans-nsf-timer-total netbeans-nsf-timer-max)
    (netbeans-cancel-nsf-timer)
    (when netbeans-app-process
      (netbeans-kill))
    (error "Could not start %s. Couldn't load the startup file." *ide-name*))
   (t
    (message "Connecting to %s..." *ide-name*)
    (netbeans-start-nsf-timer))))

(defun netbeans-cancel-nsf-timer ()
  "Cancels the `netbeans-nsf-timer' and stops waiting."
  (netbeans-debug "in invocation:cancel-nsf-timer")
  (setq netbeans-nsf-timer-total 0)
  (when netbeans-nsf-timer
    (setq netbeans-nsf-timer-aborted t)
    (if (eq (process-status netbeans-nsf-timer) 'run)
	(delete-process netbeans-nsf-timer))
    (setq netbeans-nsf-timer nil)))


(defun netbeans-load-startup-file ()
  "Load the file containing code to start the connection between IDE/emacs."
  (netbeans-debug "in invocation:load-startup-file")
  (when (and netbeans-startup-file
	     (stringp netbeans-startup-file)
	     (> (length netbeans-startup-file) 0)
	     (file-exists-p netbeans-startup-file))
    (load netbeans-startup-file nil t)
    (delete-file netbeans-startup-file)
    (setq netbeans-startup-file nil)))

(defun netbeans-start ()
  "Invoke %*ide-name*% from XEmacs"
  (interactive)
  (netbeans-debug "in invocation:netbeans-start")
  (let ((app-command (car netbeans-app-invocation-cmd))
	app-args (cadr netbeans-app-invocation-cmd))
    (when (or netbeans-app-process 
	      netbeans-app-started
	      netbeans-nsf-timer)
      (error "%s has already been started and running - either exit the %s IDE or use M-x netbeans-kill to bring it down" *ide-name* *ide-name*))
    (setq netbeans-startup-file (make-temp-name "/tmp/emacs"))

    (if (not app-command)
	 (setq app-command "runide.sh")
       (if (not (stringp app-command))
	   (error "Cannot start %s, bad value for netbeans-app-invocation-cmd variable" *ide-name*)))

    (let ((startup-arg (concat "-J-Dxemacs.startup.file=" netbeans-startup-file)))
      (if (or (not app-args) (string=  app-args ""))
	  (setq app-args (list startup-arg))
	(setq app-args (split-string (concat app-args " " startup-arg))))
      (netbeans-app-start app-command app-args 'netbeans-start-cb))))

(defun netbeans-app-start (app-name &optional args after-started-cb)
  "Invoke an application from emacs.
   Returns the process of the application."
  (netbeans-debug "in invocation:app-start")
  (unless (stringp app-name)
    (error "Invalid application name specified"))
    (let ((app-path (netbeans-get-app-path app-name))
	  (buf 
	   (if netbeans-show-app-process-output  
	       (generate-new-buffer (format "*%s Process Output*" *ide-name*))
	     nil))
	  app-process)
      (when buf
	(set-window-buffer (frame-selected-window) buf)
	(setq netbeans-output-buffer buf))
      (unless app-path
	(let ((msg (format "Cannot find %s in PATH environment variable" app-name)))
	  (netbeans-log-to-output-buffer msg) 
	  (error msg)))
      (netbeans-debug "PATH for %s: %s" app-name app-path)
      (setq app-process (netbeans-launch-app app-path args buf))
      (when after-started-cb
	(funcall after-started-cb app-process))))

(defun netbeans-get-app-path (app-name)
  "Return the path to the application binary."
  (netbeans-debug "in invocation:get-app-path")
  (cond
   ((eq (aref app-name 0) ?\/)
    app-name)
   ((when running-xemacs)
      (let ((app (concat "../../../"  app-name)))
	(when (file-executable-p app) (return app))))
   ((netbeans-path-search app-name exec-path 'file-executable-p))))


(defun netbeans-launch-app (app-name &optional args buf)
"Launch the APP-NAME with ARGS if given. 
Returns the process of the application.
BUFFER is the buffer or (buffer-name) to associate with the process.
See also: `start-process'"
  (netbeans-debug "in invocation:launch-app")
  (let (app-process)
    (condition-case err
	(progn
          (netbeans-debug "Starting %s %s %s %s" app-name buf app-name (mapconcat (lambda (x) x) args " "))
	  (setq app-process (apply 'start-process app-name buf app-name args))
	  (netbeans-debug "%s started: %s " app-name app-process))
      (file-error
       (error "Could not start  %s %s, check your PATH: %s" app-name args err)))
    app-process))

(defun netbeans-log-to-output-buffer (text &optional show-message debug)
  "Append the text to the `netbeans-output-buffer' if it exists.
If second argument SHOW-MESSAGE is non-nil, it prints a one-line message at the bottom of the frame.
If third  argument DEBUG is non-nil and `netbeans-debugging' is set to true, it prints a one-line message at the bottom of the frame and logs it to the `netbeans-debugging-logfile'."
  (netbeans-debug "log-to-output-buffer %s %s" show-message debug)
  (when netbeans-output-buffer
    (save-excursion
      (set-buffer netbeans-output-buffer)
      (goto-char (point-max))
      (insert "\n[Info] " text)))
  (if (and debug 
	   (eql debug t))
      (netbeans-debug text))
  (if (and show-message 
	   (eql show-message t))
      (message text)))

(provide 'netbeans-invocation)
