;;; -*- eval: (put 'netbeans-with-buffer 'lisp-indent-function 1) -*-

;;;                 Sun Public License Notice
;;;
;;; The contents of this file are subject to the Sun Public License
;;; Version 1.0 (the "License"). You may not use this file except in
;;; compliance with the License. A copy of the License is available at
;;; http://www.sun.com/
;;;
;;; The Original Code is NetBeans. The Initial Developer of the Original
;;; Code is Sun Microsystems, Inc. Portions Copyright 1997-2000 Sun
;;; Microsystems, Inc. All Rights Reserved.

(require 'cl)
(require 'netbeans-vars)
(require 'netbeans-common)
(require 'netbeans-protocol)
(require 'netbeans-events)

(defun netbeans-cmd-create (bufnum &optional filename)
"Associates  a buffer* with the given bufnum and initializes the IDE hookups.
Buffer*:
1) is newly created if no filename given
2) If filename is given, it is the buffer having the contents of FILENAME"
(netbeans-debug "In handler:cmd-create")
(let ((buf 
         (if filename  
             (netbeans-orig-find-file-noselect filename)
           (generate-new-buffer (format "*%s new buffer*" *ide-name*)))))
    (let ((pair (assq bufnum *netbeans-buffers*)))
      (if pair
          (progn
            (message "Warning--replacing buffer for index #%S" bufnum)
            (rplacd pair buf))
        (push (cons bufnum buf) *netbeans-buffers*)))
    (save-excursion
      (set-buffer buf)
      (add-timeout 1 (function netbeans-display-progress-message) (list bufnum 1))
      (setq netbeans-buffer-number bufnum)
      (clear-visited-file-modtime)
      (netbeans-add-idefile-hooks-and-stuff))))

(defun netbeans-cmd-setBufferNumber (bufnum filename)
  "Sets the buffer number of a file opened by Emacs, and makes it
an IDE file"
  (netbeans-debug "In handler:cmd-setBufferNumber")
  (netbeans-cmd-create bufnum  filename))

(defun netbeans-cmd-close (bufnum)
  "Closes the buffer specified by bufnum"
  (netbeans-debug "In handler:cmd-close")
  (netbeans-debug "To close: %S..." bufnum)
  (let ((pair (assq bufnum *netbeans-buffers*)))
    (if pair
        (progn
          (let ((buf (cdr pair)))
            (when (not (buffer-name buf)) 
	      (error "Buffer %S was already dead" bufnum))
            (netbeans-cmd-setVisible bufnum nil t)
            (when (buffer-modified-p buf)
              (save-excursion
                (netbeans-debug "Turning off mod status...")
                (set-buffer buf)
                (set-buffer-modified-p nil)))
            (netbeans-debug "Killing buf... %s" buf)
            (let ((netbeans-suppress-killhooks t))
              (kill-buffer buf)))
          (setq *netbeans-buffers* (delq pair *netbeans-buffers*))
          (netbeans-debug "Finished closing."))
      (netbeans-debug "Buffer %S had already been closed." bufnum))))

(defun netbeans-fun-getLength (bufnum)
(netbeans-debug "In handler:fun-getLength")
  (netbeans-with-buffer bufnum
    (list (buffer-size))))

(defun netbeans-cmd-initDone (bufnum)
"Tells XEmacs that the opening the file and putting it in a buffer is done"
(netbeans-debug "In handler:cmd-initDone")
  (netbeans-with-buffer bufnum
    (netbeans-menu-create)
    ;raise the window
    (netbeans-cmd-setVisible bufnum t)
    (setq netbeans-init-done t)))

(defun netbeans-cmd-startDocumentListen (bufnum)
(netbeans-debug "In handler:cmd-startDocumentListen")
  (netbeans-with-buffer bufnum
    (setq netbeans-doc-listen t)))

(defun netbeans-cmd-stopDocumentListen (bufnum)
(netbeans-debug "In handler:cmd-stopDocumentListen")
  (condition-case nil
      (netbeans-with-buffer bufnum
        (setq netbeans-doc-listen nil))
    (netbeans-nobuffer-error nil)))

(defun netbeans-cmd-startCaretListen (bufnum)
(netbeans-debug "In handler:cmd-startCaretListen")
  (netbeans-with-buffer bufnum
    (setq netbeans-caret-listen t)))

(defun netbeans-cmd-stopCaretListen (bufnum)
(netbeans-debug "In handler:cmd-stopCaretListen")
  (netbeans-with-buffer bufnum
    (setq netbeans-caret-listen nil)))

(defun netbeans-cmd-ignore (bufnum what)
(netbeans-debug "In handler:cmd-ignore")
  (netbeans-debug "Ignoring command %S" what))

(defun netbeans-cmd-setTitle (bufnum title)
  "Sets the buffer's name"
  (netbeans-debug "In handler:cmd-setTitle")
  (netbeans-with-buffer bufnum
    (if (or (not (buffer-name))
	    (not (string-equal title (buffer-name))))
	( rename-buffer title t))
    ;; Set visited name, so that auto-mode-alist etc. will be used in preference to MIME type: 
    (if (not buffer-file-name)
      (setq buffer-file-name title)
      (setq default-directory (file-name-directory buffer-file-name)))
    (netbeans-update-mode)))

;;set filepath
(defun netbeans-cmd-setFullName (bufnum filename)
  "Sets `buffer-file-name'"
  (netbeans-debug "In handler:cmd-setFilename")
  (netbeans-with-buffer bufnum
    (setq buffer-file-name filename)
    (setq default-directory (file-name-directory buffer-file-name))
    (netbeans-update-mode)))

(defun netbeans-fun-remove (bufnum pos length)
(netbeans-debug "In handler:fun-remove")
  (netbeans-with-buffer bufnum
    (when netbeans-doc-pending
      (error "Pending document changes in Emacs were not yet fired when external remove request received"))
    (let ((netbeans-nofire t)
          (inhibit-read-only (or inhibit-read-only (not netbeans-as-user))))
      (cond ((> (+ pos length) (buffer-size)) (list "Buffer pos too large" (+ pos length)))
            (t (condition-case err
                   (progn
                     (delete-region (1+ pos) (+ pos length 1))
		     ;;when an insert occurs, always make the undo-list nil. The user only can undo from IDE's undo list.
		     ;(when (not netbeans-init-done)
		       (setq buffer-undo-list nil));)
                 (netbeans-guarded-error (list "Attempt to remove from a read-only area" (1- (cadr err))))
                 (buffer-read-only (list "Attempt to remove from a read-only area" pos))))))))

(defun netbeans-fun-insert (bufnum pos text)
(netbeans-debug "In handler:fun-insert")
  (netbeans-with-buffer bufnum
    (when netbeans-doc-pending
      (error "Pending document changes in Emacs were not yet fired when external insert request received"))
    (let ((netbeans-nofire t)
          (inhibit-read-only (or inhibit-read-only (not netbeans-as-user))))
      (cond ((> pos (buffer-size)) (list "Buffer pos too large" pos))
            (t (condition-case err
                   (save-excursion
		     (goto-char (1+ pos))
		     (insert text)
		     ;;when an insert occurs, always make the undo-list nil. The user only can undo from IDE's undo list.
		     ;(when (not netbeans-init-done)
		       (setq buffer-undo-list nil));)
                 (netbeans-guarded-error (list "Attempt to insert into read-only area" (1- (cadr err))))
                 (buffer-read-only (list "Attempt to insert into read-only area" pos))))))))

(defun netbeans-fun-getText (bufnum pos length)
(netbeans-debug "In handler:fun-getText")
  (netbeans-with-buffer bufnum
    (cond ((> (+ pos length) (buffer-size)) (list "Buffer pos too large" (+ pos length)))
          (t (list (buffer-substring-no-properties (1+ pos) (+ pos length 1)))))))

(defun netbeans-fun-createPosition (bufnum which pos forward-bias)
(netbeans-debug "In handler:fun-createPosition")
  (netbeans-with-buffer bufnum
    (let ((pair (assq which *netbeans-positions*))
          (marker (make-marker)))
      (set-marker marker (1+ pos))
      (set-marker-insertion-type marker forward-bias)
      (if pair
          (progn
            (message "Warning--replacing position #%S" which)
            (rplacd pair marker))
        (push (cons which marker) *netbeans-positions*))
;;; XXX does not currently trap bad positions
      nil)))

(defun netbeans-fun-lookupPosition (bufnum which)
(netbeans-debug "In handler:fun-lookupPosition")
  (netbeans-with-buffer bufnum
;;; XXX trap missing position
    (list (1- (marker-position (cdr (assq which *netbeans-positions*)))))))

(defun netbeans-cmd-destroyPosition (bufnum which)
(netbeans-debug "In handler:cmd-destroyPosition")
  (netbeans-with-buffer bufnum
    (let ((pair (assq which *netbeans-positions*)))
      (when (not pair) (error "Marker #%S did not exist to destroy!"))
      (setq *netbeans-positions* (delq pair *netbeans-positions*)))))

(defun netbeans-fun-countLines (bufnum)
(netbeans-debug "In handler:fun-countLines")
  (netbeans-with-buffer bufnum
    (list (count-lines 1 (1+ (buffer-size))))))

(defun netbeans-fun-findLineFromOffset (bufnum pos)
(netbeans-debug "In handler:fun-findLineFromOffset")
  (netbeans-with-buffer bufnum
    (list (if (= pos (buffer-size))
              (count-lines 1 (1+ pos))
            (1- (count-lines 1 (+ pos 2)))))))

(defun netbeans-fun-getLineStartOffset (bufnum line)
(netbeans-debug "In handler:fun-getLineStartOffset")
  (netbeans-with-buffer bufnum
    (save-excursion
      (goto-char 1)
      (forward-line line)
      (list (1- (point))))))

(defun netbeans-cmd-setContentType (bufnum type)
(netbeans-debug "In handler:cmd-setContentType")
  (netbeans-with-buffer bufnum
    (setq netbeans-mime-type type)
    (netbeans-update-mode)))

(defun netbeans-cmd-setDot (bufnum pos)
  "Sets the point to POS in the buffer with BUFNUM.
Fronts the buffer if `netbeans-init-done' is t."
  (netbeans-debug "In handler:cmd-setDot")
  (let ((curr (current-buffer))
        (netbeans-nofire t))
    (set-buffer (netbeans-get-buffer bufnum))
    (when netbeans-init-done
      ;front the buffer
      (netbeans-cmd-setVisible bufnum t))
    (unwind-protect
        (progn
          (goto-char (1+ pos))
          (when (and (netbeans-the-mark) (= (mark) (point)))
            (netbeans-deactivate-mark))
          (netbeans-update-dot))
      (if (not netbeans-init-done)
	(set-buffer curr)))))

(defun netbeans-cmd-setMark (bufnum pos)
(netbeans-debug "In handler:cmd-setMark")
  (let ((curr (current-buffer))
        (netbeans-nofire t))
    (set-buffer (netbeans-get-buffer bufnum))
    (unwind-protect
        (progn
          (set-mark (1+ pos))
          (when (and (netbeans-the-mark) (= (mark) (point)))
            (netbeans-deactivate-mark))
          (netbeans-update-mark))
      (set-buffer curr))))

(defun netbeans-fun-getDot (bufnum)
(netbeans-debug "In handler:fun-getDot")
  (netbeans-with-buffer bufnum
    (netbeans-update-dot)
    (list (1- (point)))))

(defun netbeans-fun-getMark (bufnum)
(netbeans-debug "In handler:fun-getMark")
  (netbeans-with-buffer bufnum
    (netbeans-update-mark)
    (list (if (netbeans-the-mark)
              (1- (mark))
            (1- (point))))))

;; Try to find where this buffer is, and show it on an appropriate frame.
(defun netbeans-cmd-setLocAndSize (bufnum x y w h)
  "IGNORES THE POSITION AND SIZE OF THE FRAME. 
If X, Y, W and H are all 0's the buffer is burried, otherwise it is fronted. 
For that functionality use `netbeans-cmd-setVisible' instead."
  (netbeans-debug "In handler:cmd-setLocAndSize")
  ;;; XXX should be much more smart about deiconifying, etc.
  (netbeans-debug "setLocAndSize %S %S %S %S %S" bufnum x y w h)
  (if (and (= x 0) (= y 0) (= w 0) (= h 0))
      (netbeans-cmd-setVisible bufnum nil)
    (progn
      (netbeans-cmd-setVisible bufnum t)
      ;; XXX more annoying than useful:
      ;(let ((the-frame (netbeans-cmd-setVisible t)))
	;(set-frame-position the-frame
			    ;(+ x netbeans-frame-x-adjust)
			    ;(+ y netbeans-frame-y-adjust))
	;(set-frame-width the-frame
			 ;(/ (+ w netbeans-frame-width-adjust (- netbeans-frame-x-adjust)) (frame-char-width the-frame)))
	;(set-frame-height the-frame
			  ;(/ (+ h netbeans-frame-height-adjust (- netbeans-frame-y-adjust)) (frame-char-height the-frame))))
)))


(defun* netbeans-cmd-setVisible (bufnum flag &optional closing-buffer)
  "Fronts the buffer if FLAG is t, otherwise it burries the buffer. If CLOSING-BUFFER is non-nil, burrying the buffer is not done as long as `netbeans-init-done' is true.
Returns nil if the buffer is burried. Otherwise it returns the frame showing the buffer. "
  (netbeans-debug "In handler:cmd-setVisible")
  (let* ((buf (netbeans-get-buffer bufnum))
         (the-win (get-buffer-window buf 'visible)))
    (if (not flag)
        (progn
	  (when (and (not closing-buffer) netbeans-init-done)
	    (netbeans-debug "Ignoring...")
	    (return-from netbeans-cmd-setVisible nil))
          (netbeans-debug "Burying %S..." buf)
          (bury-buffer buf)
          (netbeans-debug "Replacing...")
          (replace-buffer-in-windows buf)
          (netbeans-debug "Replaced. Checking win %s" the-win)
          (when (and the-win (eq the-win (next-window the-win 'no-minibuf 'just-this-frame)))
            (netbeans-debug "Have a win %S and sole one" the-win)
            (netbeans-debug "Will kill frame...")
	    (condition-case err
		(progn
		  (delete-frame (window-frame the-win))
		  (netbeans-debug "Killed."))
	      (error (netbeans-debug "%s" (error-message-string err)))))
	  nil)
      (progn
        (unless the-win
	  (setq the-win (frame-selected-window))
	  (set-window-buffer the-win buf))
	(when netbeans-init-done
	  (pop-to-buffer buf))
        (let ((the-frame (window-frame the-win)))
          (select-frame the-frame)
          (raise-frame the-frame)
	  the-frame)))))

(defun netbeans-cmd-guard (bufnum off len)
(netbeans-debug "In handler:cmd-guard")
  (let ((roff (1+ off)))
    (netbeans-with-buffer bufnum
      (netbeans-debug "netbeans-cmd-guard %s %s %S" roff len (buffer-substring-no-properties roff (+ roff len)))
      (netbeans-guard roff (+ roff len)))))

;;; XXX currently does not seem to be called, but should still be handled...
(defun netbeans-cmd-unguard (bufnum off len)
(netbeans-debug "In handler:cmd-unguard")
  (let ((roff (1+ off)))
    (netbeans-with-buffer bufnum
      (netbeans-debug "netbeans-cmd-unguard %s %s %S" roff len (buffer-substring-no-properties roff (+ roff len)))
      (error "Unguarding not yet implemented"))))

(defun netbeans-cmd-setModified (bufnum mod)
(netbeans-debug "In handler:cmd-setModified")
  (netbeans-with-buffer bufnum
    (set-buffer-modified-p (setq netbeans-is-modified mod))
    (if (not mod)
	(setq buffer-undo-list nil))))

(defun netbeans-cmd-setReadOnly (bufnum flag)
  "Sets the value of `buffer-read-only' to FLAG"
  (netbeans-debug "In handler:cmd-setReadOnly")
  (netbeans-with-buffer bufnum
    (setq buffer-read-only flag)))

(defun netbeans-cmd-startAtomic (bufnum)
(netbeans-debug "In handler:cmd-startAtomic")
  (incf netbeans-atomic-level))

(defun netbeans-cmd-endAtomic (bufnum)
(netbeans-debug "In handler:cmd-endAtomic")
  (decf netbeans-atomic-level))

(defun netbeans-cmd-setAsUser (bufnum asuser)
(netbeans-debug "In handler:cmd-setAsUser")
  (netbeans-with-buffer bufnum
    (setq netbeans-as-user asuser)))

(defun netbeans-cmd-setStyle (bufnum pos name)
(netbeans-debug "In handler:cmd-setStyle")
  (netbeans-with-buffer bufnum
    (let* ((rpos (1+ pos))
           (style (cdr (assoc name '(("breakpoint" . netbeans-breakpoint-style)
                                     ("current" . netbeans-current-style)
                                     ("error" . netbeans-error-style)))))
           replacing
           (bol (save-excursion
                  (goto-char rpos)
                  (beginning-of-line 1)
                  (point)))
           (eol (save-excursion
                  (goto-char rpos)
                  (beginning-of-line 2)
                  (point))))
      (if (boundp 'running-xemacs)
          (let ((extent (extent-at rpos (current-buffer) 'style nil 'after)))
            (unless extent (when style (setq extent (make-extent bol eol))))
            (when extent
              (if style
                  (set-extent-properties extent (symbol-plist style))
                (delete-extent extent))))
        (progn
          (dolist (ov (overlays-at rpos))
            (let ((categ (overlay-get ov 'category)))
              (when (memq categ '(netbeans-breakpoint-style netbeans-current-style netbeans-error-style))
                (setq replacing t)
                (when (not (eq categ style))
                  (if style
                      (overlay-put ov 'category style)
                    (delete-overlay ov))))))
          (unless replacing
            (when style
              (let ((ov (make-overlay bol eol)))
                (overlay-put ov 'category style)))))))))

(defun netbeans-cmd-specialKeys (bufnum keys)
"Maps KEYS specified by IDE to netbeans-keymap. 
The defined keybinding calls netbeans-keyboard-event
to deliver the keyCommand to the IDE"
  (netbeans-debug "In handler:cmd-specialKeys")
  ;;;ignore bufnum for this command
  (if (stringp keys)
      (setq keys (split-string keys)))
  (dolist (key keys) 
    (netbeans-define-key key 'netbeans-keyboard-event))

  ;HACK Set Sh-F8 explicitely. It is getting set unintentionally when setting f8 above.
  (define-key netbeans-keymap  [(control f8)] 'netbeans-toggle-breakpoint)
)


(provide 'netbeans-handler)
