;;;                 Sun Public License Notice
;;;
;;; The contents of this file are subject to the Sun Public License
;;; Version 1.0 (the "License"). You may not use this file except in
;;; compliance with the License. A copy of the License is available at
;;; http://www.sun.com/
;;;
;;; The Original Code is NetBeans. The Initial Developer of the Original
;;; Code is Sun Microsystems, Inc. Portions Copyright 1997-2000 Sun
;;; Microsystems, Inc. All Rights Reserved.

(require 'cl)
(require 'netbeans-vars)


(defun netbeans-get-buffer (bufnum)
  (netbeans-debug "In common:get-buffer")
  (let ((pair (assq bufnum *netbeans-buffers*)))
    (when (not pair) (signal 'netbeans-nobuffer-error (list bufnum)))
    (cdr pair)))

(defmacro netbeans-with-buffer (bufnum &rest body)
"Sets the buffer specified by BUFNUM  and evaluates BODY"
 (netbeans-debug "In common:with-buffer")
  `(save-excursion
     (set-buffer (netbeans-get-buffer ,bufnum))
     ,@body))

(defun netbeans-update-mode ()
"Updates the mode of the buffer and turns on font lock. Currently this function is called but not used actively for setting mode."
 (netbeans-debug "In common:update-mode")
  (condition-case err
      (progn
        (normal-mode)
        (when (eq major-mode 'fundamental-mode)
          (let ((mode (cdr (assoc netbeans-mime-type netbeans-modes-by-mime-type))))
	    (netbeans-debug "mode = %s" mode)
            (if (and mode (fboundp mode))
                (funcall mode)))))
    ;; Errors setting the mode are probably not important enough
    ;; to be dealt with harshly (throwing remote exceptions etc.).
    (error (message "%s" (error-message-string err))))

  ;;;reinitialize the hooks again(needed after filename change)
  (netbeans-add-idefile-hooks-and-stuff)

  ;;;turn on font locking for syntax highlighting
  (turn-on-font-lock)
  ;;(when font-lock-mode (font-lock-fontify-buffer))
 )

(defun netbeans-add-idefile-hooks-and-stuff ()
  "This function adds hooks and stuff for IDE owned files"
  (netbeans-debug "In common:add-idefile-hooks-and-stuff")
  (make-local-hook 'after-change-functions)
  (add-hook 'after-change-functions 'netbeans-after-change nil t)
  (make-local-hook 'before-change-functions)
  (add-hook 'before-change-functions 'netbeans-before-change nil t)
  (make-local-hook 'kill-buffer-hook)
  (add-hook 'kill-buffer-hook 'netbeans-kill-hook t t)
  (make-local-hook 'post-command-hook)
  (add-hook 'post-command-hook 'netbeans-undo-hook nil t)
  (make-local-variable 'revert-buffer-function)
  (setq revert-buffer-function 'netbeans-revert-ide-buffer)
  ;; Does not look like it is possible to have dot + mark work properly
  ;; unless this mode is turned on:
  (if (boundp 'running-xemacs)
      (progn
        (make-local-variable 'zmacs-regions)
        (setq zmacs-regions t))
    (progn
      (make-local-variable 'transient-mark-mode)
      (transient-mark-mode 1))))

(defun netbeans-add-nonide-hooks-and-stuff ()
  "This function adds hooks and stuff for non-IDE owned files"
  (netbeans-debug "In common:add-nonide-hooks-and-stuff")
  (setq netbeans-nonide-fileId (buffer-file-name))
  (netbeans-add-fileModified-hooks)
  (make-local-hook 'kill-buffer-hook)
  (add-hook 'kill-buffer-hook 'netbeans-kill-hook t t))

(defun netbeans-remove-hooks-and-stuff ()
  "Remove all of the hooks and clear the variables associated with netbeans in this buffer"
  (netbeans-debug "In common:remove-hooks-and-stuff")
  (if (not netbeans-buffer-number)
      ;;;if XEmacs file
      (progn
	(setq netbeans-nonide-fileId nil)
	(netbeans-remove-fileModified-hooks))
    ;;;if IDE file
    (setq netbeans-suppress-killhooks nil)
    (remove-hook 'after-change-functions 'netbeans-after-change t)
    (remove-hook 'before-change-functions 'netbeans-before-change t)
    (remove-hook 'post-command-hook 'netbeans-undo-hook t)
    (setq revert-buffer-function nil)
    (netbeans-ballooneval-quit))
  (remove-hook 'kill-buffer-hook 'netbeans-kill-hook t)
  (setq netbeans-buffer-number nil))

(defun netbeans-update-dot ()
 (netbeans-debug "In common:update-dot")
  (let* ((dot (1- (point)))
         (mark (1- (or (netbeans-the-mark) (point)))))
    (if (not netbeans-caret-last)
        (setq netbeans-caret-last (cons dot mark))
      (when (/= dot (car netbeans-caret-last))
        (setq netbeans-caret-last (cons dot (cdr netbeans-caret-last)))))))

(defun netbeans-update-mark ()
 (netbeans-debug "In common:update-mark")
  (let* ((dot (1- (point)))
         (mark (1- (or (netbeans-the-mark) (point)))))
    (if (not netbeans-caret-last)
        (setq netbeans-caret-last (cons dot mark))
      (when (/= mark (cdr netbeans-caret-last))
        (setq netbeans-caret-last (cons (car netbeans-caret-last) mark))))))

(defun netbeans-check-guard (overlay after beg end &optional len)
 (netbeans-debug "In common:check-guard")
  (when (and (not inhibit-read-only) (not after))
    (signal 'netbeans-guarded-error (list beg))))

(defun netbeans-debug (the-format &rest args)
  (when netbeans-debugging
    (let ((text (apply 'format the-format args)))
      (message "[%s] %s" *ide-name* text)
      (when netbeans-debugging-logfile
        (write-region (concat text "\n") nil netbeans-debugging-logfile t 'nomessage)))))

(defun netbeans-make-idle-timer (name time func)
 (netbeans-debug "In common:make-idle-timer")
  (if (boundp 'running-xemacs)
      (start-itimer name func time time t)
    (run-with-idle-timer time t func)))

(defun netbeans-cancel-timer (timer)
 (netbeans-debug "In common:cancel-timer")
  (if (boundp 'running-xemacs)
      (delete-itimer timer)
    (cancel-timer timer)))

(defun netbeans-the-mark ()
 (netbeans-debug "In common:the-mark")
  (if (boundp 'running-xemacs)
      (mark)
    (if mark-active (mark))))

(defun netbeans-deactivate-mark ()
 (netbeans-debug "In common:deactivate-mark")
  (if (boundp 'running-xemacs)
      (zmacs-deactivate-region)
    (deactivate-mark)))

(defun netbeans-process-guarded-text (&optional buffer)
  "Checks if the buffer has any guarded text and if it does, it sets the extent for it.
If no BUFFER, `current-buffer' is assumed."
 (netbeans-debug "in common:process-guarded-text")
 (unless buffer
   (setq buffer (current-buffer)))
 (save-excursion
   (set-buffer buffer) 
   (goto-char (point-min))
   (let ((regexp "//GEN-\\(FIRST\\|LAST\\)"))
     (while (re-search-forward regexp nil t)
       (netbeans-debug "Found guarded text with GEN_FIRST//GEN_LAST")
       (netbeans-guard (point-at-bol) (point-at-eol))))
   (goto-char (point-min))
   (let ((regexp "//GEN-BEGIN")
	 bol
	 eol)
     (while (re-search-forward regexp nil t)
       (netbeans-debug "Found guarded text with GEN_BEGIN")
       (setq bol (point-at-bol))
       (if  (re-search-forward "//GEN-END" nil t)
	   (progn
	     (setq eol (point-at-eol))
	     (netbeans-guard bol eol))
	 (error "Cannot parse guarded blocks. The tags are not set correctly"))))
   (goto-char (point-min))))
       
(defun netbeans-guard (begin end)
  "Guard the text starting at position BEGIN and ending at position END"
  (netbeans-debug "In common:guard")
  (if (boundp 'running-xemacs)
      (let ((ext (make-extent begin end)))
	(set-extent-properties ext (symbol-plist 'netbeans-guarded)))
    (let ((ov (make-overlay begin end  (current-buffer) t nil)))
      (overlay-put ov 'category 'netbeans-guarded))))

(defmacro netbeans-def-member (struct-name member-name index)
  "Poor man's defstruct"
  (let ((struct-member-symbol
	 (intern (format "%s-%s" struct-name member-name))))
    `(progn
       (defun ,struct-member-symbol (,struct-name)
	 ,(format "Return the %s of %s."
		  member-name (upcase (symbol-name struct-name)))
	 (aref ,struct-name ,index))
       (defsetf ,struct-member-symbol (,struct-name) (store)
	 (list 'aset ,struct-name ,index store)))))

(defun netbeans-what-buffer-line (&optional pos)
  "Given the position POS return the  physical line in the buffer. If POS is not given, `point' is used."
  (netbeans-debug "In annotations:what-buffer-line")
  (save-excursion
    (when pos
      (goto-char pos))
    (beginning-of-line)
    (1+ (count-lines 1 (point)))))

(defun netbeans-path-search (file path &optional test-function)
  " Search PATH, a list of directory names, for FILE.
    Return the element of PATH that contains FILE concatenated with FILE, or nil if not found.
    TEST-FUNCTION is the symbol to test the required FILE. By default it is 'file-exists-p, and only checks if the FILE exists in PATH. You can give other TEST-FUNCTION's instead like 'file-executable-p or 'file-readable-p which takes FILENAME as an argument for more detailed searches. "
  (netbeans-debug "in common:path-search")
  (if (not test-function)
      (setq test-function 'file-exists-p))
  (while (and path (not (apply test-function (list (expand-file-name file (car path))))))
    (setq path (cdr path)))
  (when (car path)
    (expand-file-name file (car path))))

(defun* netbeans-define-key (key def &optional keymap)
"Maps the IDE defined KEY to XEmacs keys and defines key sequence KEY, in KEYMAP, as DEF. See `define-key' for more info. The default value for KEYMAP is `netbeans-keymap'
Returns t, if the key is successfully defined."
  ;(netbeans-debug "In common:define-key")
  (when (not (stringp key))
    (error "Key needs to be specified in a string."))
  (let* ((splitterString "-")
	 (index (string-match splitterString key)) 
	 (keystroke "[(" )
	 modifiers
	 keysym)
    (if index
	(setq modifiers (upcase (substring key 0 index))
	      keysym (downcase (substring key (+ 1 index))))
      (setq modifiers nil
	    keysym (downcase key)))

    (while modifiers
      (let ((case-fold-search t)
	    (modifier-char  (string-to-char modifiers))
	    (modifier-string nil))
	(cond
	 ((char-equal modifier-char  ?M) (setq modifier-string "meta"))
	 ((char-equal modifier-char  ?S) (setq modifier-string "shift"))
	 ((char-equal modifier-char  ?A) (setq modifier-string "alt"))
	 ((char-equal modifier-char  ?C) (setq modifier-string "control")))
	(setq keystroke (concat keystroke modifier-string " " ))
	(if (= (length modifiers) 1)
	    (setq modifiers nil)
	  (setq modifiers (substring modifiers 1)))))
    (setq keystroke (concat keystroke keysym ")]"))
    (netbeans-debug "Adding keystroke %s..." keystroke)
    (when def
      (when (not keymap)
	(setq keymap netbeans-keymap))
      (condition-case err
	  (define-key keymap (car (read-from-string keystroke)) def)
	(error (progn
		 (message "Warning: Could not add %s as a keybinding.=> %s" keystroke (error-message-string err))
		 (return-from netbeans-define-key nil))))))
  t)

(defun netbeans-display-progress-message (args)
  "Display the progress message of opening a %*ide-name*% file. ARGS is a list of:
 1) the buffer number for which to display this message 
 2) number of seconds elapsed showing this message."
  (let ((bufnum (car args))
	(secs (cadr args)))
    (netbeans-with-buffer bufnum
      (if netbeans-init-done
	  (progn
	    (lmessage 'progress "Opening %s  buffer:  %s... done. It may take some time to front the buffer." *ide-name* (buffer-name))
	    (add-timeout 2 (lambda (x) (clear-message nil)) nil))
	(lmessage 'progress "Opening %s  buffer:  %s...%s"
	  *ide-name*
	  (buffer-name)
	  (make-string secs ?.))
	(if (< secs 10) ;;update display at most for 10 secs
	    (add-timeout 1 (function netbeans-display-progress-message) (list bufnum (1+ secs)))
	  (display-warning 'netbeans (message "Unable to detect the successful opening of the %s  buffer:  %s... Please either look under the Buffers menu to see if the buffer is there or reopen the file from %s  Explorer." *ide-name* (buffer-name) *ide-name* )))))))

;;for rebranding the documentation
(defun netbeans-brand-ide  ()
  (if (and (stringp ad-return-value) (stringp *ide-name*))
      (let ((cur ad-return-value)
	    acc
	    (beg 0))
	(while (string-match "%\\\*ide-name\\\*%" cur beg)
	  (setq acc (cons (substring cur beg (match-beginning 0)) acc)
		beg (match-end 0)))
	(setq ad-return-value
	      (mapconcat 'identity
			 (reverse (cons (substring cur beg) acc))
			 *ide-name*)))))

(defadvice documentation (after marketing-advice last activate compile)
  (netbeans-brand-ide))


(defadvice documentation-property (after marketing-advice last activate compile)
  (netbeans-brand-ide))


(provide 'netbeans-common)
