;;;                 Sun Public License Notice
;;;
;;; The contents of this file are subject to the Sun Public License
;;; Version 1.0 (the "License"). You may not use this file except in
;;; compliance with the License. A copy of the License is available at
;;; http://www.sun.com/
;;;
;;; The Original Code is NetBeans. The Initial Developer of the Original
;;; Code is Sun Microsystems, Inc. Portions Copyright 1997-2000 Sun
;;; Microsystems, Inc. All Rights Reserved.

(require 'cl)

;;;
;;;Balloon evaluation variables
;;;
(defvar netbeans-balloon-eval-delay 600
  "*Delay for the Balloon Expression Evaluation, in milliseconds.")

;; Whether or not balloon evaluation is currently on. This is the
;; default value.
(defcustom netbeans-balloon-evaluation nil
  "*If t, %*ide-name*% balloon expression evaluation is on."
  :set (lambda (symbol value)
	 (progn 
	   (set-default symbol value)
	   (netbeans-balloon-evaluation-value-changed)))
  :initialize 'custom-initialize-default
  :type 'boolean
  :group 'netbeans-ide)

(defvar netbeans-balloon-eval-buffer nil
  "Buffer pointed to for the last balloon evaluation timeout.")

(defvar netbeans-balloon-eval-cpos nil
  "Buffer position pointed to for the last balloon evaluation timeout.")

(defvar netbeans-balloon-eval-modeline-point nil
  "Modeline point pointed to for the last balloon evaluation timeout.")

(defvar netbeans-balloon-eval-timeout-id nil
  "Timeout-id for the last current balloon evaluation timer.")

(defvar netbeans-balloon-sequence-number 0
  "Indicates the sequence number of the last balloonEval event sent to the IDE.")

(defvar netbeans-balloon-cancel-sequences 0
  "Indicates to ignore balloon evaluations with lesser sequence numbers.")

;; What type of evaluation we want: 0 to evaluate an expression; "1"
;; to dereference the expression and evaluate the resulting
;; expression, and "2" to evaluate the type of the expression
(defvar netbeans-balloon-eval-type 0
  "Balloon Expression Evaluation type: 0:evaluate 1:dereference 2:type.")
;;;
;;;END Balloon evaluation variables
;;;


(defun netbeans-ballooneval-mode-motion-hook (event)
  "Called whenever the mouse position changes in this buffer"
  (netbeans-debug "in balloon:netbeans-ballooneval-mode-motion-hook")
  (when netbeans-balloon-evaluation
    (hide-balloon-help)
    (let* ((buffer (event-buffer event))
	   (modeline-point (and buffer (event-modeline-position event)))
	   (cpos (and buffer (event-point event))))
      ;; If the position has changed, delete the current timer and
      ;; set a new timeout
      (if (not (and cpos netbeans-balloon-eval-cpos
		    (= cpos netbeans-balloon-eval-cpos)))
	  (progn
	    (when netbeans-balloon-eval-timeout-id
	      (disable-timeout netbeans-balloon-eval-timeout-id)
	      (setq netbeans-balloon-eval-timeout-id nil))
	    (netbeans-debug "Creating new balloonEval request")
	    (setq netbeans-balloon-cancel-sequences netbeans-balloon-sequence-number)
	    (setq netbeans-balloon-eval-cpos cpos)
	    (setq netbeans-balloon-eval-buffer buffer)
	    (setq netbeans-balloon-eval-modeline-point modeline-point)
	    (setq netbeans-balloon-eval-timeout-id
		  (add-timeout (/ netbeans-balloon-eval-delay 1000.0)
			       (function netbeans-balloon-eval-display)
			       nil)))))))

(defun netbeans-ballooneval-start ()
  "Start %*ide-name*%'s balloon evaluation feature in this buffer."
  (interactive)
  (netbeans-debug "in balloon:netbeans-ballooneval-start ")
  (netbeans-ballooneval-remove-hooks)
  (netbeans-ballooneval-add-hooks))

(defun netbeans-ballooneval-quit ()
  "Quit %*ide-name*%'s balloon evaluation feature in this buffer."
  (interactive)
  (netbeans-debug "in balloon:netbeans-ballooneval-quit")
  (progn
    (netbeans-ballooneval-remove-hooks)
    (when netbeans-balloon-eval-timeout-id
      (disable-timeout netbeans-balloon-eval-timeout-id)
      (setq netbeans-balloon-eval-timeout-id nil))))

(defun netbeans-ballooneval-expr ()
  "For Balloon Expression Evaluation, evaluate the pointed to expression."
  (interactive)
  (netbeans-debug "in balloon:netbeans-ballooneval-expr")
  (setq netbeans-balloon-eval-type 0))

(defun netbeans-ballooneval-deref-expr ()
  "For Balloon Expression Evaluation, dereference the pointed to expression."
  (interactive)
  (netbeans-debug "in balloon:netbeans-ballooneval-deref-expr")
  (setq netbeans-balloon-eval-type 1))

(defun netbeans-ballooneval-type ()
  "For Balloon Expression Evaluation, evaluate the pointed to expression's type."
  (interactive)
  (netbeans-debug "in balloon:netbeans-ballooneval-type")
  (setq netbeans-balloon-eval-type 2))


(defun netbeans-ballooneval-add-hooks ()
  "For Balloon Expression Evaluation, add hooks."
  (netbeans-debug "in balloon:netbeans-ballooneval-add-hooks")
  (make-local-hook 'pre-command-hook)
  (add-hook 'pre-command-hook 'netbeans-ballooneval-hide-balloon-help nil t)
  (add-hook 'mode-motion-hook 'netbeans-ballooneval-mode-motion-hook)
  (make-local-hook 'mouse-leave-frame-hook)
  (add-hook 'mouse-leave-frame-hook 'netbeans-ballooneval-hide-balloon-help nil t)
  (make-local-hook 'deselect-frame-hook)
  (add-hook 'deselect-frame-hook 'netbeans-ballooneval-hide-balloon-help nil t))

(defun netbeans-ballooneval-remove-hooks ()
  "For Balloon Expression Evaluation, remove hooks which are already added."
  (netbeans-debug "in balloon:netbeans-ballooneval-remove-hooks")
  (remove-hook 'pre-command-hook 'netbeans-ballooneval-hide-balloon-help t)
  (remove-hook 'mode-motion-hook 'netbeans-ballooneval-mode-motion-hook)
  (remove-hook 'mouse-leave-frame-hook 'netbeans-ballooneval-hide-balloon-help t)
  (remove-hook 'deselect-frame-hook 'netbeans-ballooneval-hide-balloon-help t))

(defun netbeans-ballooneval-hide-balloon-help (&optional event)
  "Hides the balloon help, if it is already shown"
  (netbeans-debug "in balloon:netbeans-ballooneval-hide-balloon-help %s " this-command)
  (unless (eq this-command 'show-balloon-help)
    (hide-balloon-help)
    (setq netbeans-balloon-cancel-sequences (+ 1 netbeans-balloon-sequence-number))))

(defun netbeans-balloon-evaluation-value-changed ()
  "Callback function to be called whenever the value of `netbeans-balloon-evaluation' is changed. Initializes all IDE buffers according to its value."
  (netbeans-debug "In balloon:balloon-evaluation-value-changed")
  (save-excursion
    (dolist (pair *netbeans-buffers*)
      (set-buffer (cdr pair))
      (if netbeans-balloon-evaluation
	  (netbeans-ballooneval-start)
	(netbeans-ballooneval-hide-balloon-help)
        (netbeans-ballooneval-quit)))))

;;========
;;EVENTS
;;========
;;; EVT_balloonEval
(defun netbeans-balloon-eval-display (&rest ignored)
  "Called whenever the timer expires: send balloon request to the debugger"
  (netbeans-debug "in events:netbeans-balloon-eval-display")
  (block netbeans-balloon-eval-display-block
    (when ( > netbeans-balloon-cancel-sequences netbeans-balloon-sequence-number)
      (netbeans-debug "Canceling the event for balloonEval")
      (return-from netbeans-balloon-eval-display-block))
    (let* ((start 0)
	   (cpos netbeans-balloon-eval-cpos)
	   (modeline-point netbeans-balloon-eval-modeline-point)
	   (end nil))
      (if (and cpos (not modeline-point))
	  (save-excursion
	    (set-buffer netbeans-balloon-eval-buffer)
	    (goto-char cpos)
	    (setq start (progn (skip-syntax-backward "w_") (point)))
	    (setq end (progn (skip-syntax-forward "w_") (point)))
	    (setq netbeans-balloon-sequence-number (+ 1 netbeans-balloon-sequence-number))
	    (let ((length (- end start)))
	      (when (> length 0)
		(netbeans-event netbeans-buffer-number  "balloonEval"
				(1- start) ; the index should be 0-based
				length
				netbeans-balloon-eval-type)
                ;;;XXXtest
		;(add-timeout 1.5 'netbeans-cmd-balloonResult 
			     ;(format "word[%s] = %s " length 
				     ;(buffer-substring start end)))
	    )))))))
;;;END  EVT_balloonEval

;;========
;;HANDLERS
;;========
;CMD_balloonResult
(defun netbeans-cmd-balloonResult (bufnum msg) ;;;XXXtest (bufnum msg)
  "Post a %*ide-name*% balloon evaluation result."
  (netbeans-debug "in handler:netbeans-cmd-balloonResult")
  (if (and 
       (> netbeans-balloon-sequence-number netbeans-balloon-cancel-sequences)
       (eq (current-buffer) netbeans-balloon-eval-buffer))
      (progn
	(netbeans-debug "showing result")
	(show-balloon-help msg))))

;CMD_enableBalloonEval
(defun netbeans-cmd-enableBalloonEval (bufnum)
  "Enables balloon evaluation in the buffer"
  (netbeans-debug "in balloon:netbeans-add-ballooneval")
  (netbeans-with-buffer bufnum
    (when (and running-xemacs netbeans-balloon-evaluation)
      (netbeans-debug "Enabling balloon eval for: %s" (current-buffer))
      (netbeans-ballooneval-start))))

(provide 'netbeans-ballooneval)
