;;;                 Sun Public License Notice
;;;
;;; The contents of this file are subject to the Sun Public License
;;; Version 1.0 (the "License"). You may not use this file except in
;;; compliance with the License. A copy of the License is available at
;;; http://www.sun.com/
;;;
;;; The Original Code is NetBeans. The Initial Developer of the Original
;;; Code is Sun Microsystems, Inc. Portions Copyright 1997-2000 Sun
;;; Microsystems, Inc. All Rights Reserved.

(require 'cl)
(when (or (featurep 'balloon-help) (locate-file "balloon-help.el" load-path))
  (require 'balloon-help))

;;;
;;;variables
;;;
(defcustom netbeans-balloon-help t
  "*Turn Balloon Help minor mode on in each %*ide-name*% buffer if arg is positive. See `balloon-help-minor-mode' for more detail."
  :set (lambda (symbol value)
	 (progn 
	   (set-default symbol value)
	   (netbeans-balloon-help-value-changed)))
  :initialize 'custom-initialize-default
  :type 'boolean
  :group 'netbeans-ide)

(defvar netbeans-anno-type-list nil
  "List of annotation types which describe the visible properties of an annotation.")

(defvar netbeans-added-anno-list nil
  "List of annotations added in this buffer.")
(make-variable-buffer-local 'netbeans-added-anno-list)
(put 'netbeans-added-anno-list 'permanent-local t)
 
(defvar netbeans-iterator-glyph-filename "NetbeansIterator.gif"
  "The filename of the glyph used for displaying the iterator in annotations.")

(defvar netbeans-iterator-glyph nil
  "The glyph used for displaying the iterator.")

(defvar netbeans-anno-glyph-max-width 0
  "The width in pixels of the largests glyph displayed in an annotation")
(make-variable-buffer-local 'netbeans-anno-glyph-max-width)
(put 'netbeans-anno-glyph-max-width 'permanent-local t)

;; We need to use prime numbers here (e.g. 127, 509, ...)
;; to reduce the likelihood of collisions.
(defvar netbeans-anno-vector-chunk  127
  "Size of netbeans-anno-vector chunk")

(defvar netbeans-anno-iterator-keymap
  (let ((map (make-sparse-keymap)))
    (set-keymap-name map 'netbeans-anno-iterator-keymap)
    (define-key map 'button1 'netbeans-anno-iterate)
    map)
  "Keymap used to iterate over  annotations.")


;;A netbeans-anno-type-list object has the form
   ;;[NAME GLYPH FACE TOOLTIP]

   ;;name      : string  the name of the annotation
   ;;glyph     : glyph   the glyph itself
   ;;face      : face    the face of the highlighted line
   ;;tooltip   : string  the string representing the tooltip help
(netbeans-def-member netbeans-anno-type name 0)
(netbeans-def-member netbeans-anno-type glyph 1)
(netbeans-def-member netbeans-anno-type face 2)
(netbeans-def-member netbeans-anno-type tooltip 3)

;;A netbeans-added-anno-list object has the form
   ;;[TYPENUM EXTENT]

   ;;typenum   : integer index into the vector netbeans-anno-type-list
   ;;extent    : extent  the created extent
(netbeans-def-member netbeans-added-anno typenum 0)
(netbeans-def-member netbeans-added-anno extent 1)


(defun netbeans-get-added-anno (serNum)
  "Get the added annotation with SERNUM from `netbeans-added-anno-list'."
  (netbeans-debug "In annotations:get-added-anno")
  (let ((added-anno (aref netbeans-added-anno-list serNum)))
    (unless added-anno
      (error "Cannot find annotation  %s in the added annotations list." serNum))
    added-anno))

(defun netbeans-get-anno-type (typeNum)
 "Get the annotation type with TYPENUM from `netbeans-anno-type-list'."
  (netbeans-debug "In annotations:get-anno-type")
 (let ((anno-type (aref netbeans-anno-type-list typeNum)))
   (unless anno-type
     (error "Cannot find type number %s in predefined annotation types." typeNum))
   anno-type))


(defun netbeans-grow-vector (vector-to-grow max-index chunk-size)
  "Extend VECTOR-TO-GROW to contain MAX-INDEX.  Return extended vector."
  (netbeans-debug "In annotations:grow-vector")
  (vconcat vector-to-grow
	   (make-vector (- (* chunk-size (+ (/ max-index chunk-size) 1))
			   (length vector-to-grow))
			nil)))

(defun netbeans-get-color-instance  (color)
  "Gets the color instance for COLOR. COLOR is given either in an 32-bit integer format or it is 'none. Returns the color instance or nil."
  (netbeans-debug "In annotations:get-color-instance")
  ;(if (and (stringp color)
	   ;(string-equal color "none"))
  (if (eq color 'none)
      (setq color nil)
    (let  ((rgb (logand color (- (expt 2 24) 1) )))
      (setq color (format "#%X" rgb)))
      color))

(defun netbeans-make-anno-face (fgColor bgColor)
  "Makes a %*ide-name*% annotation face from the FGCOLOR and BGCOLOR and returns it."
  (netbeans-debug "In annotations:make-anno-face")
  (let ((face-name (intern (format "netbeans-anno-face-%s-%s" fgColor bgColor)))
	face)		  
    (setq face (make-face face-name))
    (unless (face-differs-from-default-p face)
      (when fgColor
	(set-face-foreground face  fgColor ))
      (when bgColor
	(set-face-background face bgColor )))
    face))

(defun netbeans-balloon-help-value-changed ()
  "Callback function to be called whenever the value of `netbeans-balloon-help' is changed. Initializes all IDE buffers according to its value."
  (netbeans-debug "In anno:balloon-help-value-changed")
  (when (fboundp 'balloon-help-minor-mode)
    (save-excursion
      (dolist (pair *netbeans-buffers*)
	(set-buffer (cdr pair))
	(if netbeans-balloon-help
	    (balloon-help-minor-mode 1)
	  (balloon-help-minor-mode))))))

(defun netbeans-init-annotations ()
  "Initialize the hooks and stuff to display annotations"
  (netbeans-debug "In annotations:init-annotations")
  (if (and (fboundp 'balloon-help-minor-mode) netbeans-balloon-help)
      (balloon-help-minor-mode 1)) ;;for tooltips
  (setq netbeans-iterator-glyph 
	(make-glyph (netbeans-path-search netbeans-iterator-glyph-filename load-path )))
  ;XXX(make-local-hook 'after-change-functions)
  ;(add-hook 'after-change-functions 'netbeans-anno-after-change nil t)
)

(defun netbeans-disable-annotations ()
  "Remove the hooks and stuff to display annotations"
  (netbeans-debug "In annotations:disable-annotations")
  (if (fboundp 'balloon-help-minor-mode)
      (balloon-help-minor-mode)) ;;for tooltips
  ;XXX(remove-specifier left-margin-width 'buffer)
  ;(remove-hook 'after-change-functions 'netbeans-anno-after-change t)
)
		
(defun netbeans-anno-make-extent (start end face glyph tooltip &optional priority)
  "Creates and returns an annotation extent. Make an extent for the range [START, END) in current buffer. See `make-extent' for more details. Sets the face, glyph and tooltip for this extent with the given args.
PRIORITY is the display priority of extent and is defaulted to 1."
  (netbeans-debug "In annotations:anno-make-extent")
  (when (not priority)
    (setq priority 1))
  (let  ((extent (make-extent start end (current-buffer))))
    (set-extent-face extent face)
    (set-extent-priority extent priority)
    (when ( and tooltip (featurep 'balloon-help))
      (set-extent-property extent 'balloon-help (netbeans-create-multiline-text tooltip 40)))
    (when glyph
      (set-extent-begin-glyph extent glyph 'outside-margin)
      (set-glyph-contrib-p glyph nil)
      (set-glyph-face glyph 'default)
      (netbeans-debug "font size %d glyph size %d" (font-width (face-font 'default)) (glyph-width glyph))
      (when (> (glyph-width glyph) netbeans-anno-glyph-max-width)
	(setq netbeans-anno-glyph-max-width (glyph-width glyph))
	(netbeans-calculate-left-margin-width)))
    extent))

(defun netbeans-calculate-left-margin-width ()
  "Calculate the left marigin width to display the annotations."
  (netbeans-debug "In calculate-left-margin-width")
  (let* ((font-w (font-width (face-font 'default)))
	 (req-w (+ (glyph-width netbeans-iterator-glyph) netbeans-anno-glyph-max-width))
	 (width (1+ (/ req-w font-w))))
    (set-specifier left-margin-width width (current-buffer))))

(defun netbeans-hide-anno (serNum)
  "Remove ANNOTATION's face and glyph so that it is invisible. SERNUM specifies the number of annotation."
  (netbeans-debug "In annotations:hide-anno")
  (let* ((added-anno (netbeans-get-added-anno serNum))
	 (extent (netbeans-added-anno-extent added-anno)))
    (when extent
      (set-extent-face extent nil)
      (set-extent-begin-glyph extent nil)
      (set-extent-property extent 'netbeans-annotation-visible nil))))

(defun netbeans-reveal-anno (serNum )
  "Add ANNOTATION's face and glyph so that it is visible. SERNUM specifies the number of annotation."
  (netbeans-debug "In annotations:reveal-anno")
  (let* ((added-anno (netbeans-get-added-anno serNum))
	 (extent     (netbeans-added-anno-extent added-anno))
	 (type       (netbeans-get-anno-type (netbeans-added-anno-typenum   added-anno)))
	 (face       (netbeans-anno-type-face type))
	 (glyph      (netbeans-anno-type-glyph type)))

    (when extent
      (set-extent-face extent face)
      (set-extent-begin-glyph extent glyph 'outside-margin)
      (set-extent-property extent 'netbeans-annotation-visible t))))

(defun netbeans-show-anno-from-list (anno-list &optional serNum)
  "Shows only one annotation from annotation list ANNO-LIST. If SERNUM is given, it makes that annotation visible. Otherwise, it makes the first annotation in ANNO-LIST visible. The other annotations in ANNO-LIST will be invisible. 
Returns the index of the visible anno in ANNO-LIST."
  (netbeans-debug "In annotations:show-anno-from-list")
  ;;just show one anno at a time
  (let ((org-list anno-list)
	visible-anno 
	index)
    (while anno-list
      (let* ((anno (car anno-list))
	     (anno-num (extent-property anno 'netbeans-annotation-ser-num))
	     (visible  (extent-property anno 'netbeans-annotation-visible)))
	(setq anno-list (cdr anno-list))
	(if serNum
	    (if (= anno-num serNum)
		(progn
		  (netbeans-reveal-anno anno-num)
		  ;front the buffer
		  (goto-line line)
		  (netbeans-cmd-setVisible bufnum t)
		  ;;temp index: length of cdr list after the visible anno
		  (setq index (length anno-list)))
	      (netbeans-hide-anno anno-num))
	  (if visible-anno ;;if a visible anno already exists at line
	      (netbeans-hide-anno anno-num)
	    (if visible
		(progn 
		  (setq visible-anno anno)
		  ;;temp index: length of cdr list after the visible anno
		  (setq index (length anno-list))))))))

    ;;restore the value of anno-list
    (setq anno-list org-list)

    ;;if there is already no visible annotation, display the first one in the list
    (if (and (not visible-anno) (not serNum))
	(progn
	  (setq index (length (cdr anno-list)))
	  (netbeans-reveal-anno 
	   (extent-property (car anno-list) 'netbeans-annotation-ser-num))))

    ;; index is (list length - length of cdr list after visible anno -1)
    ;; ex. in list ( a b c d e) if c is visible, then index is:
    ;; [list length =5 ] - [length of (d e) = 2] - 1 = 2
    ;; from the previos loop, index has value of length of cdr list after visible anno
    (setq index (1- (- (length anno-list) index)))
    index))


(defun* netbeans-manage-annos-at-line (line &optional serNum)
  "Reveals the annotation with SERNUM and updates the iterator at line LINE. See `netbeans-show-anno-from-list' and `netbeans-update-iterator' for more details.
Returns t if there are any annotations at LINE. Otherwise returns false."
  (netbeans-debug "In annotations:manage-annos-at-line")
  (let* ((all-list  (netbeans-list-anno-at-line line))
	(anno-list (aref  all-list 0))
	(iter-list (aref all-list 1))
	index)
    (when (not anno-list)
      (return-from netbeans-manage-annos-at-line nil))

    (setq index (netbeans-show-anno-from-list anno-list serNum))
    (netbeans-update-iterator line iter-list anno-list index)
 t))

(defun netbeans-update-iterator (line iter-list anno-list index)
  "Updates the iterator(s) in ITER-LIST to iterate over the annotations in ANNO-LIST at line LINE. If more than one annotations are in ANNO-LIST and there is no iterator in ITER-LIST, it creates a new iterator. If there is one or none annotation, it deletes the iterator, if exists. If more than one iterator exists in ITER-LIST, it deletes all of them, and creates a new one if needed.
INDEX is the index of the visible anno in ANNO-LIST.

Notes: (1)Normally, there should never be more than one iterator in ITER-LIST per line. But thru some cutting/pasting of lines with annotations in the buffer, this may become a case.
(2) ITER-LIST and ANNO-LIST values could have been obtained via the function call (netbeans-list-anno-at-line LINE) instead of taking the values as arguments. But for performance issues this method is preffered."
  (netbeans-debug "In annotations:update-iterator")
  ;;check the iteration glyph
  (let ((num-of-annos (length anno-list))
	(num-of-iters (length iter-list))
	iterator)
    (netbeans-debug "annos=%d iters=%d" num-of-annos num-of-iters)
    (cond 
     ((and (= num-of-annos 1)
	   (= num-of-iters 0))) ;;do nothing
     ((and (> num-of-annos 1)
	   (= num-of-iters 0)) ;;create a new iterator 
      (netbeans-create-iterator  line anno-list index))
     ((and (> num-of-annos 1)
	   (= num-of-iters 1)) ;;update the existing iterator 
      (setq iterator (car iter-list))
      (set-extent-property iterator 'netbeans-iterator-anno-list anno-list)
      (set-extent-property iterator 'netbeans-iterator-visible-anno-index index))
     (t  ;;delete the existing iterators and create a new one
      (while iter-list
	(setq iterator (car iter-list)
	      iter-list (cdr iter-list))
	(delete-extent iterator))
      (if (> num-of-annos 1)
	  (netbeans-create-iterator  line anno-list index ))))))    
   
	    
(defun netbeans-list-anno-at-line (line)
  "Returns a vector of two lists => [ANNO-LIST ITER-LIST]
ANNO-LIST is a list of all the added netbeans-annotations existing at LINE number
ITER-LIST is a list of all the iteration glyphs existing at LINE number. Normally this would be a list of only one variable, but during the editing of the buffer there might be more than one thru deletion or insertion of text. "
  (netbeans-debug "In annotations:list-anno-at-line")
  (let (line-anno-list 
	filtered-anno-list
	iter-list)
    (save-excursion
      (goto-line line)
      (let ((bol (progn (beginning-of-line) (point)))
	    (eol   (progn (end-of-line) (point))))
	(setq line-anno-list (extent-list (current-buffer) bol eol))))
    (while line-anno-list
      (let ((anno (car line-anno-list)))
	(setq line-anno-list (cdr line-anno-list))
	(if (extent-property anno 'netbeans-iterator-anno-list)
	    (setq iter-list (nconc iter-list (list anno))))
	(if (extent-property anno 'netbeans-annotation-ser-num)
	    (setq filtered-anno-list (nconc filtered-anno-list (list anno))))))
    (vector filtered-anno-list iter-list)))
      
(defun netbeans-create-iterator (line anno-list visible-anno-index)
  "Creates an iterator glyph at line LINE to iterate over the annotations in ANNO-LIST.  
VISIBLE-ANNO-INDEX is the index of annotation in ANNO-LIST  which is visible. This is used to smoothly iterate over the list.

Note: ANNO-LIST value could have been obtained via the function call (netbeans-list-anno-at-line LINE) instead of taking the value as argument. But for performance issues this method is preffered."
  (netbeans-debug "In annotation:create-iterator")
  (let (extent)
    (let ((face (make-face (intern "netbeans-iterator-face" )))
	  (glyph netbeans-iterator-glyph)
	  (tooltip (format "Click to iterate over the %d annotations in this line" (length anno-list)))
	  pos)
      (save-excursion
	(goto-line line)
	(setq pos (progn (beginning-of-line) (point))))
      (setq extent (netbeans-anno-make-extent pos pos face glyph tooltip 2)))

    ;;to iterate each time the glyph is clicked
    (set-extent-property extent 'keymap netbeans-anno-iterator-keymap)

    ;;netbeans annotation properties
    ;;the list of the annotations to iterate over
    (set-extent-property extent 'netbeans-iterator-anno-list anno-list)
    ;;the index of the visible anno in netbeans-iterator-anno-list
    (set-extent-property extent 'netbeans-iterator-visible-anno-index visible-anno-index)))


(defun netbeans-anno-iterate (event)
  "This function is called each time the mouse is pressed over an iterator glyph"
  (interactive "e")
  (netbeans-debug "In annotations:anno-iterate")
  (let ((extent (event-glyph-extent event))
	(mouse-down t))
    (while mouse-down
      (setq event (next-event event))
      (if (button-release-event-p event)
	  (setq mouse-down nil)))
    (when (eq extent (event-glyph-extent event))
      (let ((index (extent-property extent 'netbeans-iterator-visible-anno-index))
	    (anno-list (extent-property extent 'netbeans-iterator-anno-list)))
	(netbeans-debug "hiding index %d" index)
	(netbeans-hide-anno (extent-property (nth index anno-list) 'netbeans-annotation-ser-num))
	(setq index ;;calculate next index
	      (if (= (1+ index) (length anno-list))
		  0
		(1+ index)))
	(netbeans-debug "revealing index %d" index)
	(netbeans-reveal-anno (extent-property (nth index anno-list) 'netbeans-annotation-ser-num))   
	(set-extent-property extent 'netbeans-iterator-visible-anno-index index)))))

(defun netbeans-create-multiline-text (text len)
"Return a string of TEXT separated by a new line pattern [\n] every LEN chracter."
 ; (netbeans-debug "In common:netbeans-create-multiline-text")
  (when (stringp text)
    (let ((text-components (split-string text))
	  (counter 0)
	  component-length)
      (setq text nil)
      (dolist (component text-components)
	(setq component-length (length component))
	(if (> (+ counter component-length) len)
	    (progn
	      (setq text (concat text "\n" component)
		    counter component-length))
	 (setq text (concat text " "  component)
		    counter (+ 1 counter component-length)))))
    text))


;;========
;;EVENTS
;;========

;;========
;;HANDLERS
;;========
;;CMD_defineAnnoType
(defun netbeans-cmd-defineAnnoType (bufnum num name tooltip glyph-file fgColor bgColor)
  "Defines the annotation type with the given properties and adds it to `netbeans-anno-type-list'. NUM is the type number."
  (netbeans-debug "In annotations:defineAnnoType")
  ;;;ignore bufnum for this command
  (when (>= num (length netbeans-anno-type-list))
    (setq netbeans-anno-type-list (netbeans-grow-vector netbeans-anno-type-list num netbeans-anno-vector-chunk)))

  (let (face
	glyph
	anno-type)
    (setf face (netbeans-make-anno-face 
		(netbeans-get-color-instance fgColor)
		(netbeans-get-color-instance bgColor)))
    (when glyph-file
      (setf glyph (make-glyph glyph-file)))
    (setq anno-type (vector name glyph face tooltip))
    (aset netbeans-anno-type-list num  anno-type)
    anno-type))

;;CMD_addAnno
(defun netbeans-cmd-addAnno (bufnum serNum typeNum pos len)
  "Add annotation SERNUM of type TYPENUM to the given position
POS. LEN specifies the length of the annotation. -1 means the whole line. Returns the newly added annotation."
  (netbeans-debug "In annotations:cmd-addAnno")
  (netbeans-with-buffer bufnum
    (when (not netbeans-added-anno-list)
      (netbeans-init-annotations))
 
    (let* ((anno-type (netbeans-get-anno-type typeNum))
	   (glyph (netbeans-anno-type-glyph anno-type))
	   (face  (netbeans-anno-type-face  anno-type))
	   (tooltip  (netbeans-anno-type-tooltip  anno-type))
	   anno
	   extent
	   start
	   end)
      (if (= len -1)
	  (progn
	    (save-excursion
	      (goto-char (1+ pos))
	      (setq start (progn (beginning-of-line) (point)))
	      (setq end   (progn (end-of-line) (point)))))
	(setq start (1+ pos))
	(setq end  (+ start len)))
	       
      (setq extent (netbeans-anno-make-extent start end face glyph tooltip))
      ;;netbeans annotation properties
      ;;the serial number of the annotation to which this extent belongs
      (set-extent-property extent 'netbeans-annotation-ser-num serNum)
      ;;whether this annotation is visible or hidden
      (set-extent-property extent 'netbeans-annotation-visible t)

      (setq anno (vector typeNum extent))
      (when (>= serNum (length netbeans-added-anno-list))
	(setq netbeans-added-anno-list (netbeans-grow-vector netbeans-added-anno-list serNum netbeans-anno-vector-chunk)))
      (aset netbeans-added-anno-list serNum  anno)

      (netbeans-manage-annos-at-line (netbeans-what-buffer-line start) serNum)
      anno)))

;;CMD_moveAnnoToFront
(defun* netbeans-cmd-moveAnnoToFront (bufnum serNum)
  "Remove annotation SERNUM and delete it."
  (netbeans-debug "In annotations:cmd-moveAnnoToFront")
  (netbeans-with-buffer bufnum
    (let* ((added-anno (netbeans-get-added-anno serNum))
	   (extent (netbeans-added-anno-extent added-anno))
	   line)
      (when (not extent)
	(return-from netbeans-cmd-moveAnnoToFront))

      (setq line (netbeans-what-buffer-line (extent-start-position extent)))
      (netbeans-manage-annos-at-line line serNum))))

;;CMD_removeAnno
(defun* netbeans-cmd-removeAnno (bufnum serNum)
  "Remove annotation SERNUM and delete it."
  (netbeans-debug "In annotations:cmd-removeAnno")
  (netbeans-with-buffer bufnum
    (let* (( added-anno (netbeans-get-added-anno serNum))
	   (extent (netbeans-added-anno-extent added-anno))
	   line)
      (when (not extent)
	(return-from netbeans-cmd-removeAnno))

      ;XXX(when (not netbeans-added-anno-list)
	;(netbeans-disable-annotations))
      (setq line (netbeans-what-buffer-line (extent-start-position extent)))
      (delete-extent extent)
      (aset netbeans-added-anno-list serNum  nil)
      (netbeans-manage-annos-at-line line))))

(provide 'netbeans-annotations)
