package Inventory;
use strict;
use System;
use Agent::T3;
use Agent::3510;
use Agent::3310;
use Agent::6120;
use Agent::VE;
use Agent::SVE2;
use Logic::SWITCH;

use vars qw($INDY1 $INDY2 $MASER $VMPATH $VM_TO);

$INDY1 = ",3910,3960,6910,6960,3900SL,6910SL,6960SL,";
$INDY2 = ",3975,6965,6975,";
$MASER = ",6320,6920,6970,";
$VMPATH = "/opt/se6x20";
$VM_TO  = 2 * 60;

sub goldenName {
  return System->get_home() . "/DATA/Inventory.golden";
}

sub log {
  my($class, $text) = @_;
  open(O, ">>" . System->get_home(). "/log/Service.log");
  my $date = Util->get_today();
  print O "$date\t$text\n";
  close(O);
}

sub writeInventory {
   my($inv, $file) = @_;
   $file = $inv->goldenName() if (!$file);
   open(O, ">$file");
   require Data::Dumper;
   $Data::Dumper::Indent = 1;
   print O Data::Dumper::Dumper($inv);
   close(O);
   if ($inv->{solution} eq "se2") {
      $inv->run_vmcfg();
   }
}

sub readInventory {
  my($class, $file) = @_;
  $file = $class->goldenName() if (!$file);

  if (open(O, $file)) {
     my @L = <O>; close(O);
     my $VAR1;
     eval "@L";
     return $VAR1;
  }
  return undef;
}

sub getKeys {
  my($inv) = @_;
  my %K;
  my $devs = $inv->{devices};
  foreach my $d (@$devs) {
     $K{$d->{key}} = $d;
  }
  return \%K;
}

sub deviceCount {
  my($inv) = @_;
  my $devs = $inv->{devices};
  return $#$devs + 1;
}


sub getDevices {
  my($inv) = @_;
  return $inv->{devices};
}

#  $inv = Inventory->read();
#  print $inv->toXML();

sub toXML {
  my($inv) = @_;
  
  my $out =<<EOF;
<?xml version=\"1.0\"?>
<!DOCTYPE inventory PUBLIC "-//Sun Microsystems//DTD XML Version 1.0//EN" "/opt/SUNWstade/System/Inventory.dtd">
<inventory>
  <created>$inv->{created}</created>
  <created2>$inv->{created2}</created2>
  <model>$inv->{model}</model>
  <solution>$inv->{solution}</solution>
EOF
  my $devs = $inv->{devices};
  foreach my $d (@$devs) {
     $out .= "  <device type=\"$d->{type}\">\n";
     my $att;
     foreach my $k (sort keys %$d) {
         if (index(",class,key,ip,name,wwn,", ",$k,") >= 0) {
            $out .= "    <$k>$d->{$k}</$k>\n";
         } elsif ($k ne "type" && $k ne "frus") {
            $att .= "    <attribute id=\"$k\">$d->{$k}</attribute>\n";
         }
     }
     $out .= $att;
     my $frus = $d->{frus};
     foreach my $f0 (sort keys %$frus) {
        my $f = $frus->{$f0};
        my $lv = " <loopVersion>$f->{loopVersion}</loopVersion>" if (exists $f->{loopVersion});
        my $cv = " <ctlrVersion>$f->{ctlrVersion}</ctlrVersion>" if (exists $f->{ctlrVersion});
        if (length($f->{serialNo}) > 1) {
          $out .=<<EOF;
          <component type="$d->{type}">
            <fruId>$f0</fruId>
            <fruModel>$f->{model}</fruModel>
            <fruRevision>$f->{revision}</fruRevision>$lv $cv
            <fruSerialNo>$f->{serialNo}</fruSerialNo>
            <fruVendor>$f->{vendor}</fruVendor>
          </component>
EOF
        }
     }
     $out .= "  </device>\n";
    
  }
  $out .= "</inventory>\n";
  return $out;
}


sub findCurrentModel {
  my($class) = @_;
  my $inv = $class->readInventory();
  return $inv->{model};
}

sub readHostLines {
  my($class) = @_;

  if (open(O, "/etc/hosts")) {
    my @host_lines = <O>;
    close(O);
    return \@host_lines;
  } else {
    print "Error: Cannot read /etc/hosts, aborting \n";
    return undef;
  }
}

sub run_vmcfg {
  my($inv) = @_;
  my($renv, $devs, $hosts,$notifs) = PDM::ConfigFile->read();
  my(%F, $errs, $info);

  my $devices = $inv->{devices};
  foreach my $r (@$devices) {
      next if ($r->{type} ne "6120");
      my($err, $rc) = $inv->run_vmcfg0($r->{ipno});
      if ($err) {
         $errs .= "Array $r->{ipno} ERROR: $err $err\n";
      } else {
         $info .= "Array $r->{ipno}: configured for use.\n";
      }
  }
  print $info . $errs;
}

sub run_vmcfg0 {
  my ($class, $ip, $command) = @_;

  $command = "configure" if (!$command);
  my $array_p = System->getPassword("array");

  my ($err, $rc, $error) = Util->run_command("$VMPATH/bin/vmcfg $command -t t4 -i $ip -u root -p $array_p", "vmcfg", $VM_TO, { cache => 0 });
  return ("$err$error" , $rc);
}
  

# SAVE SNAPSHOT TO RASAGENT.conf
#
sub writeConfig {
   my($inv, $mode) = @_;
   my %F;
   # mode is append or write
   my($renv, $devs, $hosts,$notifs) = PDM::ConfigFile->read();
   if ($mode eq "append") {
     foreach my $d (@$devs) {
       $F{$d->{key}} = 1;
     }
   } elsif ($mode eq "write") { # new devs
     $devs = [];
   }
   my $add;
   my $devices = $inv->{devices};
   foreach my $r (@$devices) {
       next if ($F{$r->{key}});
       next if ($r->{type} eq "sp");
       $F{$r->{key}} = 1;
       my $no = $#$devs + 1;
       my $new = {};
       foreach my $el (keys %$r) {
         $new->{$el} = $r->{$el} if (substr($el,0,1) ne "_" && $el ne "frus");
       }
       $new->{_name} = "device" . $no;
       $new->{mgmtLevel} = "C";
       if ($r->{type} eq "t3") {
          $new->{userLabel} = Agent::T3->getUserLabel($new->{ctrl_model});
       }
       $new->{active} = "Y";
       push(@$devs, $new); $add++;
   }
   $renv->{cust_no}  = $inv->{serial_no};
   $renv->{solution} = $inv->{solution};
   $renv->{solution_model} = $inv->{model};
   $renv->{categories} = "switch|switch2|host|message|san";
   $renv->{categories} .= $inv->{solution} eq "se" ? "|3510|t3|sve2|ve" : "|6120|sve2";
   $renv->{t300logfile} = "/var/adm/messages.t3";

   $renv->{customer}    = "NWS_OPS" if (!$renv->{customer});
   $renv->{site_name}   = "ops"     if (!$renv->{site_name});
   $renv->{site_city}   = "Newark"  if (!$renv->{site_city});
   $renv->{site_zip}    = "98765"   if (!$renv->{site_zip});
   $renv->{site_country}= "USA"     if (!$renv->{site_country});
   $renv->{site_contact}= "ops"     if (!$renv->{site_contact});
   $renv->{site_email}  = "root\@localhost"    if (!$renv->{site_email});
   delete $renv->{role};
   if ($inv->{solution} eq "se2") {
      $renv->{t300logfile}= &find_log();
   }elsif ($inv->{solution} eq "se") {
      $renv->{t300logfile}= "/var/adm/messages.t3";
   } else {
      $renv->{t300logfile}= "/var/adm/messages.t300";
   }
   open(O, ">" . System->get_home() . "/DATA/start");
   close(O);

   my %new2;
   my $sekey = $inv->{model} . "." . System->hostid();
   if ($add && !$F{$sekey}) { # add the rack itself
       my $ipno = Util->name2ip();
       my $no = $#$devs + 1;
       $new2{_name}   = "device" . $no;
       $new2{type}     = $inv->{solution};
       $new2{class}   = "storage.$inv->{solution}";
       $new2{name}     = $inv->{model} . "-" . $renv->{hostname};
;
       $new2{ip}       = $ipno;
       $new2{wwn}      = $sekey;
       $new2{key}      = $sekey;
       $new2{model}    = $inv->{model};
       $new2{mgmtLevel}= "DS";
       $new2{ipno}     = $ipno;
       $new2{active}   = "Y";
       push(@$devs, \%new2);
    }
    PDM::ConfigFile->write( $renv, $devs, $hosts, $notifs) if ($add);
}


sub find_log {
  my ($l,$file);
  if (open(OO, "/etc/syslog.conf")) {
     while ($l = <OO>) {
          if ($l =~ /^local7\.\w+\s+(.+)/) {
             $file = $1; last;
          }
     }
     close(OO);
  }
  return $file;
}

# Update snmp info.
# /opt/SUNWstade/snmp/share/snmp/sys.conf
# sysDescr=Sun StorEdge Solution 3900/6900
# sysObjectID=42.2.28.13.2

sub writeSnmpConfig {
   my($inv) = @_;
   my $solution = $inv->{solution};
   my $model    = $inv->{model};

   open(O, ">/opt/SUNWstade/snmp/share/snmp/sys.conf");
   print O "sysDescr=Sun StorEdge Solution $model\n";
   my $last = $solution eq "se" ? "1" : "2";
   print O "sysObjectID=42.2.28.13.$last\n";
   close(O);
}

#======================================================================

sub check_t3 {
  my($rc, $default_type, $trace_out, $golden, $short, $T,$ip, $ipno, $trace,
     $userLabel) = @_;
  my($next);
  my $warn;
  if (!$rc->{key} && !$rc->{wwn} && !$rc->{wwn2} && !$rc->{ctrl_model} ) {
     my $o = "  -> Error: Could not retrieve tokens from this array\n";
     $$trace_out .= $o; print $o;
     #next; 
  }
  if ($default_type ne $rc->{type}) {
     my $o = "  -> Warning : type is '$rc->{type}', should be '$default_type'\n";
     $warn++;
     $$trace_out .= $o; print $o;
  }
  if (!$rc->{key}) {
     my $o = "  -> Error: cannot find the serial# of this array\n";
     $$trace_out .= $o; print $o if ($trace);
  } 
  if (!$rc->{wwn}) {
     my $o = "  -> Error: cannot find the wwn of the first port of this array\n";
     $$trace_out .= $o; print $o if ($trace);
  } 
  if (!$rc->{wwn2}) {
     my $o = "  -> Error: cannot find the wwn of the second port of this array\n";
     $$trace_out .= $o; print $o if ($trace);
  } 
  if (!$rc->{wwn} || !$rc->{key}) {
     $next = 1;
  } else {
      my (%FRUS);
      $T->{report_t3} .= "$ip, ";
      $T->{t3_cnt}++;
      foreach my $x (keys %$rc) {
           if (substr($x,0,4) eq "FRU.") { 
              my ($xx, $id, $name) = split(/\./, $x);
              $name = substr($name,3) if (substr($name,0,3) eq "fru");
              $name = lcfirst($name);
              $FRUS{$id}{$name} = $rc->{$x};
           }
      }
      push(@$golden, bless({
            type => $rc->{type}, class => "storage.$rc->{type}",
            ip   => $ipno,
            ipno => $ipno,
            name => "$short-$ip",
            userLabel => $userLabel,
            key  => $rc->{key},
            units => $rc->{units},
            wwn  => $rc->{wwn},
            wwn2 => $rc->{wwn2},
            sysRevision => $rc->{"SYSTEM.sysRevision"},
            ctrl_model => $rc->{ctrl_model},
            frus   => \%FRUS },'Inv_dev') );

      my $o = "  -> found $rc->{type} ($ipno / $rc->{wwn}/$rc->{wwn2}\n\t\t$rc->{key}) \n";
      $trace_out .= $o; 
      print $o ;
  }
  return $warn;
}

sub check_3510 {
  my($rc, $default_type, $trace_out, $golden, $short, $T,$ip, $ipno, $trace) = @_;
  my($next);
  my $warn;
  if (!$rc->{key}) {
     my $o = "  -> Error: Could not retrieve tokens from this array\n";
     $$trace_out .= $o; print $o;
     #next; 
  }
  if ($default_type ne $rc->{type}) {
     my $o = "  -> Warning : type is '$rc->{type}', should be '$default_type'\n";
     $warn++;
     $$trace_out .= $o; print $o;
  }

  if (!$rc->{key}) {
     $next = 1;
  } else {
      my (%FRUS);
      $T->{report_t3} .= "$ip, ";
      $T->{t3_cnt}++;
      foreach my $x (keys %$rc) {
           if (substr($x,0,4) eq "FRU.") { 
              my ($xx, $id, $name) = split(/\./, $x);
              $name = substr($name,3) if (substr($name,0,3) eq "fru");
              $name = lcfirst($name);
              $FRUS{$id}{$name} = $rc->{$x};
           }
      }
      push(@$golden, bless({
            type  => $rc->{type}, class => "storage.$rc->{type}",
            ip    => $ipno,
            ipno  => $ipno,
            name  => "$short-$ip",
            key   => $rc->{key},
            units => $rc->{units},
            wwn   => $rc->{key},
            wwns  => $rc->{wwns},
            userLabel => $rc->{model},
            frus   => \%FRUS },'Inv_dev') );

      my $o = "  -> found $rc->{model} ($ipno / $rc->{wwns}\n\t\t$rc->{key}) \n";
      $trace_out .= $o; 
      print $o ;
  }
  return $warn;
}



#########################################################################
#  INDY 
#########################################################################

sub readIndy {
 my($class, $short, $PREFIX, $PASS, $trace, $HTML, $TO) = @_;
 my $golden = [];
 my $xml;
 my ( $trace_out, %T);
 my $lines = Inventory->readHostLines();
 $PREFIX = "192.168" if (!$PREFIX);

 foreach my $l (@$lines) {
    chop($l);
    next if (substr($l,0,1) eq "#");
    my ($ipno, $ip, $rest) = split(/\s+/, $l);
    next if (!$ipno);
    next if (substr($ipno,0,length($PREFIX) ) ne $PREFIX);
    my @nos = split(/\./, $ipno);
    my $no = $nos[3];
    print "<br>" if ($HTML);
    print "$ip: \n";
    if ($no >= 20 && $no <= "23") { # vicom
         if (!&ping($ipno)) {
             my $o = "  -> Warning: cannot ping $ipno \n";
             $o .= "     Device is not responding or not present in this configuration\n" ;
             $trace_out .= $o;
             print $o if ($trace);
             next;
         }
         $T{ve_cnt}++;
         my $wwn = Agent::VE->getWWN($ipno, undef, 20, $PASS);
         my $login = ($no <= 21)? "v1":"v2";
	 my $init = "1";
         if($no == 21){
           $init = "2";
         }elsif($no == 22){
           $init = "1";
         }elsif($no == 23){
           $init = "2";
         }

         if (length($wwn) > 10) {
           $T{report_ve} .= "$ip, ";
           push(@$golden, bless( {
                    type => "ve", class => "switch.ve",
                    ip   => $ipno,
                    ipno => $ipno,
                    key  => $wwn,
                    name => "$short-$ip",
                    wwn  => $wwn,
                    login => $login, initiator => $init,
                    frus  => { $ip => {
                                 vendor   => "SUN",
                                 serialNo => "1000",
                                 model    => "VICOM",
                                 revision => "REV", 
                                          }}},'Inv_dev') );
  
           my $o  = "  -> found VE ($ip / $ipno / $wwn) \n";
           $trace_out .= $o; 
           print $o;
  
         } else {
           my $o = "  -> Error: cannot telnet to VEs at $ip/$ipno (remove password if present)\n";
           $trace_out .= $o; print $o;
           $T{err}++; next;
         }
  
    } elsif ($no >= 25 && $no <= 29) { # PSX=1000
         if (!&ping($ipno)) {
             my $o = "  -> Warning: cannot ping $ipno \n";
             $o .= "     Device is not responding or not present in this configuration\n" ;
             $trace_out .= $o;
             print $o if ($trace);
             next;
         }
         $PASS = "pirus" if (!$PASS);
         my $PASS_encoded = Util->encodeAll($PASS);

         $T{ve_cnt}++;
         my $wwn = Agent::SVE2->getWWN( 
                      {ipno => $ipno, login => "root", telnet => $PASS_encoded});

         if (length($wwn) > 10) {
           $T{report_ve} .= "$ip, ";
           push(@$golden, bless( {
                    type => "sve2", class => "switch.sve2",
                    ip   => $ipno,
                    ipno => $ipno,
                    key  => $wwn,
                    name => "$short-$ip",
                    wwn  => $wwn,
               userLabel => 'DSP',
                   login => "root",
                  telnet => $PASS_encoded,
                   frus  => { $ip => {
                                 vendor   => "SUN",
                                 serialNo => $wwn,
                                 model    => "DSP-1x00",
                                 revision => "REV", 
                                          }}},'Inv_dev') );
  
           my $o  = "  -> found SVE ($ip / $ipno / $wwn) \n";
           $trace_out .= $o; 
           print $o;
  
         } else {
           my $o = "  -> Error: cannot telnet to SVEs at $ip/$ipno (remove password if present)\n";
           $trace_out .= $o; print $o;
           $T{err}++; next;
         }
  
    } elsif ($no >= 40 && $no <= 54) { # T3
       if (&ping($ipno)) {
          my $rc = Agent::T3->getWWN($ipno, undef, $TO, 1);
          my $warn = &check_t3($rc, "t3", \$trace_out, $golden, $short, \%T,$ip, $ipno,  $trace, $rc->{userLabel});
          $T{warn} += $warn;
  
       } else {
          my $o = "  -> Warning: cannot ping $ipno \n";
          $o .= "     Device is not responding or not present in this configuration\n";
          $trace_out .= $o; print $o if ($trace);
       }

    } elsif ($no >= 60 && $no <= 75) { # MINNOW fcaXX
       if (&ping($ipno)) {
          my $rc = Agent::3310->getWWN($ipno);
          my $warn = &check_3510($rc, "3510", \$trace_out, $golden, $short, \%T,$ip, $ipno,  $trace);
          $T{warn} += $warn;
  
       } else {
          my $o = "  -> Warning: cannot ping $ipno \n";
          $o .= "     Device is not responding or not present in this configuration\n";
          $trace_out .= $o; print $o if ($trace);
       }
           
    } elsif ($no >= 30 && $no <= 33) { # switch
       if (&ping($ipno)) {
          my($wwn, $type, $ports, $rev) = &findSwitch($ipno);
          if (!$wwn) {
             my $o =  "  -> Error: cannot find the wwn of $ipno \n";
             $trace_out .= $o; print $o;
             $T{err}++;
          } else {
              if (!$T{sw_ports}) {
                $T{sw_ports} = $ports;
                if ($T{sw_ports} != 16 && $T{sw_ports} != 8) {
                    print "  -> switch $ipno, invalid # ports: $T{sw_ports}\n";
                    $T{err}++;
                }
              }
              $T{report_sw} .= "$ip, ";
              $T{sw_cnt}++;
              push(@$golden,bless( {
                    type => $type, class => "switch.$type",
                    ip   => $ipno,
                    ipno => $ipno,
                    name => "$short-$ip",
                    key  => $wwn,
                    wwn  => $wwn,
                    ports => $T{sw_ports},
                    frus  => { $ip => {
                                 vendor   => "SUN",
                                 serialNo => $wwn,
                                 model    => "SUN-$type",
                                 revision => $rev, 
                                      }}}, 'Inv_dev') );
              my $o = "  -> found $type ($ipno / $wwn) \n";
              $trace_out .= $o; 
              print $o;
          }
  
       } else {
          my $o = "  -> Warning: cannot ping $ipno \n";
          $o .= "     Device is not responding or not present in this configuration\n";
          $trace_out .= $o; print $o if ($trace);
       }
     }
   }
   my $date = Util->get_today();
   my $date2 = `date`;
   chop($date2);
   my $INV = { created => $date, 
           created2 => $date2, 
           solution => "se",
              stats => \%T,
            devices => $golden,
          };
   bless($INV, 'Inventory');
   return ($trace_out, $INV);
}

#  MODEL : requested model
#
sub validateIndy {
 my($inv, $MODEL, $trace) = @_;

 my ($prefix, $suffix, $trace_out, $model);
 my $T = $inv->{stats};

  if (substr($MODEL,0,2) eq "39" && $T->{ve_cnt} > 0) {
    my $o = "  -> Error: Found $T->{ve_cnt} ve, but model is $MODEL!\n";
    $trace_out .= $o; print $o;
    $T->{err}++;
  }
  
  if (($MODEL eq "6910" && $T->{ve_cnt} != 2) ||
      ($MODEL eq "6960" && $T->{ve_cnt} != 4)) { 
    my $o = "  -> Error: Wrong number of VE($T->{ve_cnt}) found for $MODEL! \n";
    $trace_out .= $o; print $o;
    $T->{err}++;
  }
  if ($T->{t3_cnt} == 0) {
    my $o = "  -> Warning: No Storage Array found! \n";
    $trace_out .= $o; print $o;
  }
  
  if ($T->{sw_ports} != 16 && substr($MODEL,2,2) eq "60") {
    my $o = "  -> Error: Model $MODEL need 16 port switches! \n";
    $trace_out .= $o; print $o;
    $T->{err}++;
  }
  
  if ($T->{sw_ports} != 8 && substr($MODEL,2,2) eq "10") {
    my $o =  "  -> Error: Model $MODEL need 8 port switches! \n";
    $trace_out .= $o; print $o;
    $T->{err}++;
  }
  if (index($INDY2, $MODEL) >= 0) {
     $model = $MODEL;
     

  } elsif (index($INDY1, $MODEL) >= 0) {
     my $sw1 = 2;
     my $sw2 = 4;
     if ($MODEL =~ /SL/) {
        $sw1 -= 2;
        $sw2 -= 2;
     }
     if ((substr($MODEL,0,2) eq "39" && $T->{sw_cnt} != $sw1) ||
         (substr($MODEL,0,2) eq "69" && $T->{sw_cnt} != $sw2)) {
       my $o = "  -> Error: Wrong number of switches ($T->{sw_cnt}) for a $MODEL \n";
       $trace_out .= $o; print $o;
       $T->{err}++;
     
     }
     
     $suffix = "XX";
     if ($T->{sw_ports} == 16) {
        $suffix = "60";
     } elsif ($T->{sw_ports} == 8) {
        $suffix = "10";
     } elsif ($T->{sw_cnt} == 0) {
        $suffix = "00";
     }
     my $sl;
     if ($T->{ve_cnt}) {
       $prefix = "69";
       $sl = "SL" if ($T->{sw_cnt} == 2);
     } else {
       $prefix = "39";
       $sl = "SL" if ($T->{sw_cnt} == 0);
     }
     $model = $prefix . $suffix . $sl;
        
     if ($MODEL && $MODEL ne $model) {
        my $o = "ERRORS: Model proposed ($MODEL) is different than discovered model ($model)\n";
        $trace_out .= $o; print $o;
        $T->{err}++;
     }
  }


  $inv->{model} = $model;
  my $w_info = "|Warn(s) : $T->{warn}\n" if ($T->{warn});
  $T->{sw_cnt} += 0;
  $T->{ve_cnt} += 0;
  my $summary =<<EOF;
  
=====================================
|     Configuration Summary         |
-------------------------------------
|  Model : $model, Requested = $MODEL
|  Switch: $T->{sw_cnt}, ports=$T->{sw_ports}
|        : $T->{report_sw}
|  Arrays: $T->{t3_cnt} 
|        : $T->{report_t3}
|  SVEs  : $T->{ve_cnt} 
|        : $T->{report_ve}
$w_info
=====================================
EOF

  return ($T->{err},$trace_out, $summary);
}

#########################################################################
#  MASERATI
#########################################################################

sub readMaserati {
  my($class, $short, $PREFIX, $trace, $HTML, $TO) = @_;
  my(%T, $trace_out);
  my $devcnt   = 1;
  my $golden = [];
  my $xml;
  $PREFIX = "192.168" if (!$PREFIX);
  my $lines = Inventory->readHostLines();

  foreach my $l (@$lines) {
    chop($l);
    last if ($l eq "#DONE");
    next if (substr($l,0,1) eq "#");
    my ($ipno, $ip, $rest) = split(/\s+/, $l);
    next if (!$ipno);
    next if (substr($ipno,0,length($PREFIX) ) ne $PREFIX);
  
    my @X = split(/\./, $ipno);
    my $no = $X[3];
    my $version;
    print "<br>" if ($HTML);
    print "$ip:\n";

    if ($X[2] == 0 && $no == 2) {  # sp
       my $key = "host:" . System->hostname();
       open(O, "/etc/motd"); $version = <O>; close(O);
       my $rev = $version;
       $rev =~ /Revision (.*)/; $rev = $1;
       my $hostinfo = System->hostinfo();
       my $model = System->model();
       push(@$golden, bless({
            type => "sp", class => "host.sp",
            ip   => $ip,
            ipno => $ipno,
            name => "$short-$ip",
            key  => $key,
            wwn  => $key,
            hostinfo => $hostinfo,
            frus  => { $ip => {
                         vendor   => "SUN",
                         serialNo => $key,
                         model    => $model,
                         revision => $rev, 
                              }}}, 'Inv_dev') );
  
    } elsif ($no >= 40 && $no <= 65) { # T4  CC
       if (&ping($ipno)) {
          my $rc = Agent::6120->getWWN($ipno, undef, $TO, 1);
	  if($rc->{err}){
	     print "ERROR on $ipno: $rc->{err}\n";
	     $trace_out .= $rc->{err};
	  }else{
             my $warn = &check_t3($rc, "6120", \$trace_out, $golden, $short, \%T,$ip, $ipno, $trace, "Array");
             $T{warn} += $warn;
	  }
  
       } else {
          my $o = "  -> Warning: cannot ping $ipno \n";
          $o .= "     Device is not responding or not present in this configuration\n";
          $trace_out .= $o; print $o if ($trace);
       }
           
    } elsif ($no >= 300 && $no <= 33) { # switch
       if (&ping($ipno)) {
          my($wwn, $type, $ports, $rev) = &findSwitch($ipno);
          if (!$wwn) {
             my $o =  "  -> Error: cannot find the wwn of $ipno \n";
             $trace_out .= $o; print $o;
             $T{err}++;
          } else {
              if (!$T{sw_ports}) {
                $T{sw_ports} = $ports;
                if ($T{sw_ports} != 16 && $T{sw_ports} != 8) {
                    print "  -> switch $ipno, invalid # ports: $T{sw_ports}\n";
                    $T{err}++;
                }
              }
              $T{report_sw} .= "$ip, ";
              $T{sw_cnt}++;
              push(@$golden, bless({
                    type => $type, class => "switch.$type",
                    ip   => $ipno,
                    ipno => $ipno,
                    name => "$short-$ip",
                    key  => $wwn,
                    wwn  => $wwn,
                    ports => $T{sw_ports},
                    frus  => { $ip => {
                                 vendor   => "SUN",
                                 serialNo => $wwn,
                                 model    => "SUN-$type",
                                 revision => $rev, 
                                      }}}, 'Inv_dev') );
              my $o = "  -> found $type ($ipno / $wwn) \n";
              $trace_out .= $o; 
              print $o;
         }
  
       } else {
          my $o = "  -> Warning: cannot ping $ipno \n";
          $o .= "     Device is not responding or not present in this configuration\n";
          $trace_out .= $o; print $o if ($trace);
       }
     }
   }
   my $date = Util->get_today();
   my $date2 = `date`;
   chop($date2);
   my $INV = { created => $date, 
           created2 => $date2, 
           solution => "se2",
              stats => \%T,
           devices  => $golden,
          };
   bless($INV, 'Inventory');
   return ($trace_out, $INV);
}


sub validateMaserati {
  my($inv, $MODEL, $trace) = @_;
  my ($prefix, $suffix, $trace_out, $model);

  my $T = $inv->{stats};

  if ($MODEL eq "6320" && $T->{sve_cnt} > 0) {
    my $o = "  -> Error: Found $T->{sve_cnt} ve, but model is $MODEL!\n";
    $trace_out .= $o; print $o;
    $T->{err}++;
  }
  
  if (($MODEL eq "6920" && $T->{sve_cnt} != 2) ||
      ($MODEL eq "6970" && $T->{sve_cnt} != 4)) { 
    my $o = "  -> Error: Wrong number of VE($T->{sve_cnt}) found for $MODEL! \n";
    $trace_out .= $o; print $o;
    $T->{err}++;
  }
  if ($T->{t3_cnt} == 0) {
    my $o = "  -> Error: No 6120 found! \n";
    $trace_out .= $o; print $o;
    $T->{err}++;
  }
  
  if ($T->{sw_ports} == 16 && $MODEL ne "6920") {
    my $o = "  -> Error: Model $MODEL need 8 port switches! \n";
    $trace_out .= $o; print $o;
    $T->{err}++;
  }
  
  if ($T->{sw_ports} == 8 && index("6320,6970", $MODEL) >= 0) {
    my $o = "  -> Error: Model $MODEL need 16 port switches! \n";
    $trace_out .= $o; print $o;
    $T->{err}++;
  }
  if ($MODEL ne "6320" && $T->{sw_cnt} != 2) {
    my $o = "  -> Error: Wrong number of switches ($T->{sw_cnt}) for a $MODEL \n";
    $trace_out .= $o; print $o;
    $T->{err}++;
  
  }
  my $sl;
  if ($T->{sve_cnt} == 0) {
    $model = "6320";

  } elsif ($T->{sve_cnt} == 2) {
    $model = "6920";

  } elsif ($T->{sve_cnt} == 4) {
    $model = "6970";
  }
     
  if ($MODEL && $MODEL ne $model) {
     my $o = "ERRORS: Model proposed ($MODEL) is different than discovered model ($model)\n";
     $trace_out .= $o; print $o;
     $T->{err}++;
  }
  $inv->{model} = $model;
  $T->{sw_cnt} += 0;
  $T->{sve_cnt} += 0;
  my $w_info = "|Warn(s) : $T->{warn}\n" if ($T->{warn});
  my $summary =<<EOF;

=====================================
|     Configuration Summary         |
-------------------------------------
|  Model : $model, Requested = $MODEL
|  Switch: $T->{sw_cnt}, ports=$T->{sw_ports}
|        : $T->{report_sw}
|  Arrays: $T->{t3_cnt}
|        : $T->{report_t3}
|  SVEs  : $T->{sve_cnt}
|        : $T->{report_sve}
$w_info
=====================================
EOF

  return ($T->{err}, $trace_out, $summary);
}


# type, wwn, #ports, revision


# returns ($wwn , $type, $ports, $rev)

sub findSwitch {
  my($ip) = @_;
  my($wwn, $rev, $ports);
  my $type = "switch";

  my $DIR = System->get_home() ;
  my @lines = `$DIR/bin/sanbox version $ip`;
  foreach my $l (@lines) {
    if ($l =~ /WWN:\s+([0-9a-f]+)/) {
      $wwn = $1;
    } elsif ($l =~ /HW:\s+(.*)/) {
      $rev = $1;
    }
  }
  
  my @L = `$DIR/bin/sanbox chassis_status $ip`;

  foreach my $l (@L) {
    if ($l =~ /Ports:\s+(\d+)/) {
       $ports = $1;
    } 
  }

  @L = `$DIR/bin/sanbox get_zone $ip sl all`;
  if ("@L" !~ /Port: \d/) {
     $type = 'switch2';
  }

  return ($wwn, $type, $ports, $rev);
}

sub ping {
  my($ip) = @_;

  open(PING, "/usr/sbin/ping $ip 3|");
  my $ans = <PING>; close(PING);
  if ($ans =~ /is alive/) {
     return 1;
  } else {
     return 0;
  }
}

package Inv_dev;


sub frus {
   my($dev) = @_;
   return $dev->{frus};
}

sub info {
   my($dev, $val) = @_;
   if (exists $dev->{$val}) {
      return $dev->{$val};
   }
   return undef;
}


1;
