#
# Copyright (c) 2001 by Sun Microsystems, Inc.
# All rights reserved.
#
#pragma ident   "@(#)Network.pm 1.34     02/11/15 SMI"
#
# Pubic Network (NAFO) class

package Cluster::Network;
use strict;
use POSIX;
use Cluster::Common;
use Sun::Solaris::Utils qw(gettext);
use vars qw(@ISA $VERSION);
$VERSION = '1.00';
@ISA = qw(Cluster::Common);

# Internationalization
# action=
;# gettext("Switch")
;# gettext("Add")
;# gettext("Remove")
;# gettext("Create")
;# gettext("Delete")

##############################################################################
#
# Class constructor
#
##############################################################################

sub new {
    my $proto = shift;
    my $class = ref($proto) || $proto;
    my $self  = {};
    bless ($self, $class);
    return $self;
}

##############################################################################
#
# Class Variables
#
##############################################################################

# File names and location
my $NAFOIMAGE = '/images/nafo_16.gif';
my $TELNETIMAGE = '/images/telnet-win-icon.gif';
my $VIEW = '/cgi-bin/network/network_view.pl';
my $CONFIG = '/cgi-bin/network/network_config.pl';
my $EXEC = "/cgi-bin/network/execute_pnmset.pl";
my $PNMSTAT = '/usr/cluster/bin/pnmstat';
my $PNMSET = '/usr/cluster/bin/pnmset';

# Instantiate cluster objects
my $runcommand  = new Cluster::RunCommand;

# Global Variable
my @NAFOGROUP_LIST;

my @status_icons = (
	["OK", "Online", "ok"], # gettext("Online")
	["DOUBT", "Doubt", "critical"], # gettext("Doubt")
	["DOWN", "Down", "critical"], # gettext("Down")
	["NETWORK DOWN", "Network down", "critical"], # gettext("Network down")
	["NOT MONITORING", "Not monitoring", "minor"] # gettext("Not monitoring")
);

##############################################################################
#
# Class methods
#
##############################################################################

# Return the nafo related jscript
sub network_jscript() {
;# /* TRANSLATION_NOTE The following 3 strings are Javascript expressions */
	return "function verify_newnafogroup() {
		var adps = trim(document.inputs.adp.value)
		var nafogroup = trim(document.inputs.nafogroup.value)
		if (!check_value('".gettext("NAFO Group name")."', nafogroup)) {
			return false
		}
		if (!verify_name(nafogroup, 'nafo')) {
                        alert(".gettext("'Invalid NAFO group name -- ' + nafogroup + '.  Please make sure it starts with nafo and ends with digit(s).'").")
			return false
		}
		var adplist = adps.split(\" \")
		for (var i = 0; i < adplist.length; i++) {
			if (!verify_adapter(adplist[i])) {
				return false
			}
		}
		document.inputs.submit()
		return true
	}

	// verify the given item name is started with the given pattern
	// and ended with digits
	function verify_name(itemvalue, pattern) {
		// verify the item name is started with pattern
                if (itemvalue.indexOf(pattern) != 0) {
                        return false
                }
		// verify the item name is ended with digits
		if (itemvalue.length == pattern.length) {
			return false;
		}
		for (var i = pattern.length; i < itemvalue.length; i++) {
			var digit = itemvalue.charAt(i);
			if (digit < \"0\" || digit > \"9\") {
				return false
			}
                }
		return true
	}
	function trim(itemvalue) {
		// Get rid of leading space(s)
		for (var i = 0; i < itemvalue.length; i++) {
                        var ch = itemvalue.charAt(i);
			if (ch != \" \") {
				break
			}
		}
		// Get rid of trailing space(s)
		for (var j = itemvalue.length - 1; j > -1; j--) {
                        var ch = itemvalue.charAt(j);
			if (ch != \" \") {
				break
			}
		}
		var newvalue = '';
		for (var k = i; k <= j; k++) {
			newvalue += itemvalue.charAt(k)
		}
		return newvalue
	} 
		
	function check_value(itemname, itemvalue) { 
		if (itemvalue.length <= 0) {
			alert(".gettext("'Please input ' + itemname").");
                        return false
                }
		if (itemvalue.indexOf(\" \") >= 0) {
			alert(".gettext("'No space is allowed in ' + itemname").");
			return false
		} 
		return true
	}
	function verify_adapter(adapter) {
		var adp = trim(adapter) 
		if (!check_value('".gettext("Adapter(s) name")."', adp)) {
			return false
		}
		if (!(verify_name(adp, 'qfe') || verify_name( adp, 'sci') || verify_name(adp, 'hme') || verify_name(adp, 'ge') || verify_name(adp, 'wrsm') || verify_name(adp, 'eri') || verify_name(adp, 'ce') || verify_name(adp, 'bge'))) {
			alert(".gettext("'Invalid adapter name -- ' + adp + '.  Please make sure it starts with qfe, hme, sci, ge, eri, wrsm, ce, or bge and ends with digit(s).'").")
			return false
		}
		return true
	} 
	function config_adapter_submit(func, node, nafo) {
		self.location.href = '$CONFIG\?step=2&func=' + func + '&adp=' + document.inputs.adp.options[document.inputs.adp.selectedIndex].value + '&nafogroup=' + nafo + '&node=' + node

	}";
}

sub network_header {
	my ($self, $q, $headername, $node, $nafogroup) = @_;
  
	# Start the action bar table
        $q->start_action_bar_table($headername, undef,
	    {ALIGN => "right", WIDTH => "99%"});

	# Print the telnet link to the node
	$q->telnet_link($node, $self->get_hostdomain());
	
	# Print the middle of the action table
        $q->mid_action_bar_table();

	# Space the actions menu over
	print "&nbsp;&nbsp";

	# Create the actions menu links        
	my @links = (gettext("Switch Active Adapter..."),
		"$CONFIG\?step=1&func=Switch&node=$node&nafogroup=$nafogroup",

		gettext("Add Adapter to Group..."),
		"$CONFIG\?step=1&func=Add&node=$node&nafogroup=$nafogroup",

		gettext("Remove Adapter from Group..."),
		"$CONFIG\?step=1&func=Remove&nafogroup=$nafogroup&node=$node",

		gettext("---------------------"),
		"$VIEW?node=$node",

		gettext("Create NAFO Group..."),
		"$CONFIG\?step=1&node=$node&func=Create",

		gettext("---------------------"),
		"$VIEW?node=$node",
		
		gettext("Delete NAFO Group..."),
		"$CONFIG\?step=1&node=$node&nafogroup=$nafogroup&func=Delete"
		);

	my @action_helpMarkers = ("networkpm-switch-adapter",
			          "networkpm-add-adapter",
			          "networkpm-remove-adapter",
			             "separator",
			          "networkpm-create-nafo",
			             "separator",
			          "networkpm-delete-nafo");
	$q->load_action_helpMarkers(\@action_helpMarkers);

	# Print the action bar table
        $q->actions_menu(gettext("Action Menu"), \@links);

        # End the action bar table
        $q->end_action_bar_table();
}

# Display list of nafo groups on the given node
sub nafogroups_table($$$)
{
	my ($self, $q, $nodename) = @_;
	if (!$self->is_node_online($q, $nodename)) {
		return;
	}

	# Set the headers nafo group table 
        my @table_headers = (gettext("Name"), "30%",
			     "&nbsp;", "14",
			     gettext("Status"), "30%",
			     gettext("Active Adapter"), "40%");

	# Determine the number of columns from the headers
        my $numcols = ($#table_headers + 1) / 2;

	# Start the table
        $q->start_prop_table(sprintf(gettext("NAFO Groups - %s"), $nodename),
			\@table_headers, $NAFOIMAGE);
	
	if ($#NAFOGROUP_LIST < 0) {
		# Not Found
                # Start a table row and cell with the colspan of the table
                $q->start_prop_tr();
                $q->start_prop_td({ COLSPAN => 4 });

                # Print the message in table-label-text style
                $q->start_table_text('table-label-text');
                print sprintf(gettext(
		    "No NAFO groups have been configured on node %s"),
		    $nodename);
                $q->end_table_text();

                # End the cell and row
                $q->end_prop_td();
                $q->end_prop_tr();
                # End the table
                $q->end_prop_table();
	}

	for (my $i = 0; $i <= $#NAFOGROUP_LIST; $i++) {
		# Start a table row
                $q->start_prop_tr();

                # Print a table cell for the nafo name
                $q->start_prop_td();
                $q->table_link_text($NAFOGROUP_LIST[$i]{'name'},
                        $NAFOGROUP_LIST[$i]{'name'},
                        "$VIEW?node=$nodename&nafogroup=$NAFOGROUP_LIST[$i]{'name'}");
                $q->end_prop_td();

		# Print a table cell for the nafo's status
                $q->start_prop_td();
		$q->table_status_text($NAFOGROUP_LIST[$i]{'status'},
		    \@status_icons);
                $q->end_prop_td();

		# Print a table cell for the nafo's active adapter
                $q->start_prop_td();
		$q->start_table_text('table-normal-text');
                print $NAFOGROUP_LIST[$i]{'act_adp'};
		$q->end_table_text();
                $q->end_prop_td();

		$q->end_prop_tr();

		if ($i < $#NAFOGROUP_LIST) {
			$q->line_row($numcols);
		}
	}
		
	# End the node status table
        $q->end_prop_table();
}

# Print a nafo group's property
sub nafo_table($$$$)
{
	my ($self, $q, $node, $nafo) = @_;
	if (!$self->is_node_online($q, $node)) {
		return;
	}

	# Set the headers for the Quorum Device Properties table
        my @table_headers = (gettext("Property"), "50%",
				"&nbsp;", 14,
                                gettext("Value"), "50%");

	# Determine the number of columns from the headers
        my $numcols = ($#table_headers + 1) / 2;

	# Start the table
        $q->start_prop_table(sprintf(gettext("NAFO Group Properties - %s"),
	    $nafo), \@table_headers, "$NAFOIMAGE");

	# Find specified nafo from nafogroup_list
	my $i = $self->find_item_by_name(\@NAFOGROUP_LIST, $nafo);
	if ($i == -1) {
                # Not Found
                # Start a table row and cell with the colspan of the table
                $q->start_prop_tr();
                $q->start_prop_td({ COLSPAN => 3 });

                # Print the message in table-label-text style
                $q->start_table_text('table-label-text');
                print $q->sprintfn(
		    gettext("NAFO Group %1 has Not been Configured on Node %2"),
		    $nafo, $node);
                $q->end_table_text();

                # End the cell and row
                $q->end_prop_td();
                $q->end_prop_tr();
                # End the table
                $q->end_prop_table();
		return;
	}

	$q->start_prop_tr();
        $q->start_prop_td();
        $q->start_table_text('table-label-text');
        print gettext("Name:");
        $q->end_table_text();
        $q->end_prop_td();
        $q->start_prop_td();
	$q->table_status_text($NAFOGROUP_LIST[$i]{'name'});
        $q->end_prop_td();
        $q->end_prop_tr();
        $q->line_row($numcols);

	$q->start_prop_tr();
        $q->start_prop_td();
        $q->start_table_text('table-label-text');
        print gettext("Status:");
        $q->end_table_text();
        $q->end_prop_td();
        $q->start_prop_td();
	$q->table_status_text($NAFOGROUP_LIST[$i]{'status'}, \@status_icons);
        $q->end_prop_td();
        $q->end_prop_tr();
        $q->line_row($numcols);

	$q->start_prop_tr();
        $q->start_prop_td();
        $q->start_table_text('table-label-text');
        print gettext("Active Adapter:");
        $q->end_table_text();
        $q->end_prop_td();
        $q->start_prop_td();
	$q->table_status_text($NAFOGROUP_LIST[$i]{'act_adp'});
        $q->end_prop_td();
        $q->end_prop_tr();
        $q->line_row($numcols);

	$q->start_prop_tr();
        $q->start_prop_td();
        $q->start_table_text('table-label-text');
        print gettext("Adapter List:");
        $q->end_table_text();
        $q->end_prop_td();
        $q->start_prop_td();
	$q->table_status_text($NAFOGROUP_LIST[$i]{'adapters'});
        $q->end_prop_td();
        $q->end_prop_tr();
        $q->line_row($numcols);

	$q->start_prop_tr();
        $q->start_prop_td();
        $q->start_table_text('table-label-text');
        print gettext("Last Failover Duration:");
        $q->end_table_text();
        $q->end_prop_td();
        $q->start_prop_td();
	$q->table_status_text($NAFOGROUP_LIST[$i]{'fo_time'});
        $q->end_prop_td();
        $q->end_prop_tr();

	# End the table
        $q->end_prop_table();
}

# check if the given node is in cluster mode
sub is_node_online($$$)
{
	my ($self, $q, $nodename) = @_;

	my @node_list = $self->get_nodelist();

	for (my $i = 0; $i <= $#node_list; $i++) {
		if (($node_list[$i] eq $nodename) && (lc($self->get_nodestate($nodename)) eq "online")) {
			# cleanup the existing NAFOGROUP_LIST
			splice(@NAFOGROUP_LIST, 0, $#NAFOGROUP_LIST + 1);
			
			# Fill the NAFOGROUP_LIST in advance in case the
			# node goes soon after
			my $ret;
			($ret, @NAFOGROUP_LIST) = $self->get_nafogrouplist($nodename);
			if ($ret == 0) {
				return 1;
			} else {
				# something else went wrong (maybe node goes
				# down while retrieving nafogroup info)
				last;
			}
		}
	}
	# not found or not online
	print "<form>";
	$q->start_message_table("warning");
	print $q->start_p({ class => "message-header-text" });
	print sprintf(gettext(
	    "Unable to retrieve and configure NAFO group(s) on Node %s!"),
	    $nodename);
	print $q->end_p();
	print $q->start_p({ class => "message-description-text" });
	print sprintf(gettext("Node %s might not be in cluster mode at this time or the cluster manager web server is not started on that node."), $nodename);
	print $q->end_p();
	$q->end_message_table();
	$q->buttons_table(sprintf(gettext("  View Node %s Information  "),
	    $nodename),
	    "self.location.href=\"/cgi-bin/node/node_info.pl\?node=$nodename\"");
	print "</form>";	
	return 0;
}

# Get nafo group on a certain node
sub get_nafogrouplist {
	my ($self, $nodename) = @_;
	my (@nafo_list, $i, @parts, $line);
	
	#my ($ret, @output) = `$PNMSTAT -h $nodename -l`;
	my ($ret, @output) = $runcommand->pnmstat("-h $nodename -l");
	if ($ret == 0) {
		# skip the first line	
		for ($i = 1; $i <= $#output; $i++) {
			@parts = split /(\S+)/, $output[$i];
			$nafo_list[$i - 1]{'name'} = $parts[1];
			$nafo_list[$i - 1]{'adapters'} = $parts[3];
			$nafo_list[$i - 1]{'status'} = $parts[5];
			$nafo_list[$i - 1]{'fo_time'} = $parts[7];
			if ($nafo_list[$i - 1]{'fo_time'} eq "NEVER") {
			    $nafo_list[$i - 1]{'fo_time'} = gettext("NEVER");
			}
			$nafo_list[$i - 1]{'act_adp'} = $parts[9];
		}
	}
	return ($ret, @nafo_list);
}

# add new adapters into existing nafo group	
sub add_adapter($$$$$$)
{
	my ($self, $q, $step, $node, $nafogroup, $adp) = @_;

	if (!$self->is_node_online($q, $node)) {
                return;
        }
	
	if ($step == 1) {	
		if ($#NAFOGROUP_LIST < 0) {
                        $q->print_title([gettext("Nodes"),
				"/cgi-bin/node/node_status.pl",
			    $node, 
				"/cgi-bin/node/node_info.pl?node=$node",
			    gettext("NAFO Groups"), "$VIEW?node=$node",
			    gettext("Add Adapter to NAFO Group"),
		"$CONFIG\?step=1&func=Add&node=$node&nafogroup=$nafogroup",
			    gettext("Warning")]);
                        print "<form>";
                        $q->start_message_table("warning");
                        print $q->start_p({ class => "message-header-text" });
                        print sprintf(gettext("No NAFO group has been Configured on Node %s"), $node);
                        print $q->end_p();
                        $q->end_message_table($q);
                        $q->buttons_table(gettext("  Cancel  "), "self.location.href=\"$VIEW\?node=$node\"" );
                        print "</form>";
                        return;
                }
		
		$q->print_title([gettext("Nodes"),
			"/cgi-bin/node/node_status.pl",
		    $node, 
			"/cgi-bin/node/node_info.pl?node=$node",
		    gettext("NAFO Groups"), "$VIEW?node=$node",
		    gettext("Add Adapter to NAFO Group")]);
		# gather information
                print "<form name=inputs action=$CONFIG>";
                print $q->input({ type => "hidden",
                                        name => "step",
                                        value => 2 });
                print $q->input({ type => "hidden",
                                        name => "node",
                                        value => $node });
                print $q->input({ type => "hidden",
                                        name => "func",
                                        value => "Add" });

		print $q->start_table({ cellpadding => 5 });
                print $q->start_Tr();
                print $q->start_td();
                print $q->start_div({ class => "action-window-label-text" });
                print gettext("NAFO Group:");
                print $q->end_div();
                print $q->end_td();
                print $q->start_td();
		print "<select name=nafogroup>";
                for (my $i = 0; $i <= $#NAFOGROUP_LIST; $i++) {
                print "<option value=$NAFOGROUP_LIST[$i]{'name'}>$NAFOGROUP_LIST[$i]{'name'}";
                }
                print "</select>";
                my $selectidx;
                if (defined $nafogroup) {
                        $selectidx = $self->find_item_by_name(\@NAFOGROUP_LIST,
                                         $nafogroup);
                        if ($selectidx < 0) {
                                $selectidx = 0;
                        }
                }
                print $q->start_script();
                print "document.inputs.nafogroup.selectedIndex=$selectidx";
                print $q->end_script();
                print $q->end_td();
                print $q->end_Tr();

		print $q->start_Tr();
                print $q->start_td();
                print $q->start_div({ class => "action-window-label-text" });
                print gettext("New Adapter:");
                print $q->end_div();
                print $q->end_td();
                print $q->start_td();
                print $q->input({ type=> "text", name => "adp",
                                 size => 10,  maxlength => 20 });
                print $q->end_td();
                print $q->end_Tr();
                print $q->end_table();

                $q->buttons_table(gettext(" Add Adapter "),
                        "if (verify_adapter(document.inputs.adp.value)) document.inputs.submit()",
                        gettext("  Cancel  "),
                        "self.location.href = \"$VIEW\?node=$node\"" );
                print "</form>";
	} else {
		# generate the command
		my $cmd="$PNMSET -c $nafogroup -o add $adp";
		
		# Executing the command
                $self->execute_cmd($q, $cmd, $node,
		    [gettext("Nodes"), "/cgi-bin/node/node_status.pl",
			$node, "/cgi-bin/node/node_info.pl?node=$node",
			gettext("NAFO Groups"), "$VIEW?node=$node",
			gettext("Add Adapter"),
	    "$CONFIG\?step=1&func=Add&node=$node&nafogroup=$nafogroup"]);
	}
}

# Remove an adapter from a nafo group
sub remove_adapter($$$$$$)
{
	my ($self, $q, $step, $node, $nafogroup, $adp) = @_;

	if (!$self->is_node_online($q, $node)) {
                return;
        }

	if ($step == 1) {
                if ($#NAFOGROUP_LIST < 0) {
                        $q->print_title([gettext("Nodes"),
				"/cgi-bin/node/node_status.pl",
			    $node, "/cgi-bin/node/node_info.pl?node=$node",
			    gettext("NAFO Groups"), "$VIEW?node=$node",
			    gettext("Remove Adapter from NAFO Group"),
		"$CONFIG\?step=1&func=Remove&nafogroup=$nafogroup&node=$node",
			    gettext("Warning")]);
                        print "<form>";
                        $q->start_message_table("warning");
                        print $q->start_p({ class => "message-header-text" });
                        print sprintf(gettext("No NAFO group has been Configured on Node %s"), $node);
                        print $q->end_p();
                        $q->end_message_table($q);
                        $q->buttons_table(gettext("  Cancel  "), "self.location.href=\"$VIEW\?node=$node\"" );
                        print "</form>";
                        return;
                }
	
		$q->print_title([gettext("Nodes"),
			"/cgi-bin/node/node_status.pl",
		    $node, "/cgi-bin/node/node_info.pl?node=$node",
		    gettext("NAFO Groups"), "$VIEW?node=$node",
		    gettext("Remove Adapter from NAFO Group")]);
                # gather information
                print "<form name=inputs action=$CONFIG>";
                print $q->input({ type => "hidden",
                                        name => "step",
                                        value => 2 });
                print $q->input({ type => "hidden",
                                        name => "node",
                                        value => $node });
                print $q->input({ type => "hidden",
                                        name => "func",
                                        value => "Remove" });

		print $q->start_table({ cellpadding => 5 });
                print $q->start_Tr();
                print $q->start_td();
                print $q->start_div({ class => "action-window-label-text" });
                print gettext("NAFO Group:");
                print $q->end_div();
                print $q->end_td();
                print $q->start_td();
		my $j = 0;
		my (@nafo_links, $selectindex);
		$selectindex = 0;
		for (my $i = 0; $i <= $#NAFOGROUP_LIST; $i++) {
			if ($NAFOGROUP_LIST[$i]{'name'} eq $nafogroup) {
				$selectindex = $i;
			}
			$nafo_links[$j++] = $NAFOGROUP_LIST[$i]{'name'};
			$nafo_links[$j++] = "$CONFIG\?step=1&node=$node&func=Remove&nafogroup=$NAFOGROUP_LIST[$i]{'name'}";
                 }
		$q->select_menu(\@nafo_links, "nafogroup", $nafo_links[1]);
		print $q->start_script();
		print "document.inputs.nafogroup.selectedIndex=$selectindex";
		print $q->end_script();
		print $q->end_td();
                print $q->end_Tr();

		print $q->start_Tr();
                print $q->start_td();
                print $q->start_div({ class => "action-window-label-text" });
                print gettext("Adapter:");
                print $q->end_div();
                print $q->end_td();
                print $q->start_td();
		print "<select name=adp>";
		my @parts = split /(\:)/, $NAFOGROUP_LIST[$selectindex]{'adapters'}; 
	        for (my $j = 0; $j <= $#parts; $j = $j + 2) {
			 print "<option value=$parts[$j]>$parts[$j]";
                }
		print "</select>";
                print $q->end_td();
                print $q->end_Tr();
                print $q->end_table();

		$q->start_message_table("warning");
                print $q->start_p({ class => "message-description-text" });
                print gettext("This action will permanently remove the adapter from the NAFO group.");
                print $q->end_p();
                $q->end_message_table($q);

		$q->buttons_table(gettext(" Remove Adapter "),
			"config_adapter_submit('Remove', '$node',
				'$NAFOGROUP_LIST[$selectindex]{'name'}')",
                        gettext("  Cancel  "),
                        "self.location.href = \"$VIEW\?node=$node\"" );
                print "</form>";
	} else {
		# generate the command
		my $cmd="$PNMSET -c $nafogroup -o remove $adp";
		
		# Executing the command
                $self->execute_cmd($q, $cmd, $node,
		    [gettext("Nodes"), "/cgi-bin/node/node_status.pl",
			$node, "/cgi-bin/node/node_info.pl?node=$node",
			gettext("NAFO Groups"), "$VIEW?node=$node",
			gettext("Remove Adapter"),
	    "$CONFIG\?step=1&func=Remove&node=$node&nafogroup=$nafogroup"]);
	}
}

# switch active adapter of a nafo group
# XXX Note: it prints almost the same as remove_adapter, use a function?
sub switch_adapter($$$$$$)
{
	my ($self, $q, $step, $node, $nafogroup, $adp) = @_;

	if (!$self->is_node_online($q, $node)) {
                return;
        }

	if ($step == 1) {
                if ($#NAFOGROUP_LIST < 0) {
                        $q->print_title([gettext("Nodes"),
				"/cgi-bin/node/node_status.pl",
			    $node, "/cgi-bin/node/node_info.pl?node=$node",
			    gettext("NAFO Groups"), "$VIEW?node=$node",
			    gettext("Switch Active Adapter"),
		"$CONFIG\?step=1&func=Switch&node=$node&nafogroup=$nafogroup",
			    gettext("Warning")]);
                        print "<form>";
                        $q->start_message_table("warning");
                        print $q->start_p({ class => "message-header-text" });
                        print sprintf(gettext("No NAFO group has been Configured on Node %s"), $node);
                        print $q->end_p();
                        $q->end_message_table($q);
                        $q->buttons_table(gettext("  Cancel  "), "self.location.href=\"$VIEW\?node=$node\"" );
                        print "</form>";
                        return;
                }
	
		$q->print_title([gettext("Nodes"),
			"/cgi-bin/node/node_status.pl",
		    $node, "/cgi-bin/node/node_info.pl?node=$node",
		    gettext("NAFO Groups"), "$VIEW?node=$node",
		    gettext("Switch Active Adapter")]);
                # gather information
                print "<form name=inputs action=$CONFIG>";
                print $q->input({ type => "hidden",
                                        name => "step",
                                        value => 2 });
                print $q->input({ type => "hidden",
                                        name => "node",
                                        value => $node });
                print $q->input({ type => "hidden",
                                        name => "func",
                                        value => "Remove" });

		print $q->start_table({ cellpadding => 5 });
                print $q->start_Tr();
                print $q->start_td();
                print $q->start_div({ class => "action-window-label-text" });
                print gettext("NAFO Group:");
                print $q->end_div();
                print $q->end_td();
                print $q->start_td();
		my $j = 0;
		my (@nafo_links, $selectindex);
		$selectindex = 0;
		for (my $i = 0; $i <= $#NAFOGROUP_LIST; $i++) {
			if ($NAFOGROUP_LIST[$i]{'name'} eq $nafogroup) {
				$selectindex = $i;
			}
			$nafo_links[$j++] = $NAFOGROUP_LIST[$i]{'name'};
			$nafo_links[$j++] = "$CONFIG\?step=1&node=$node&func=Switch&nafogroup=$NAFOGROUP_LIST[$i]{'name'}";
                 }
		$q->select_menu(\@nafo_links, "nafogroup", $nafo_links[1]);
		print $q->start_script();
		print "document.inputs.nafogroup.selectedIndex=$selectindex";
		print $q->end_script();
		print $q->end_td();
                print $q->end_Tr();

		print $q->start_Tr();
                print $q->start_td();
                print $q->start_div({ class => "action-window-label-text" });
                print gettext("Adapter:");
                print $q->end_div();
                print $q->end_td();
                print $q->start_td();
		print "<select name=adp>";
		my @parts = split /(\:)/, $NAFOGROUP_LIST[$selectindex]{'adapters'}; 
	        for (my $j = 0; $j <= $#parts; $j = $j + 2) {
			 print "<option value=$parts[$j]>$parts[$j]";
                }
		print "</select>";
                print $q->end_td();
                print $q->end_Tr();
                print $q->end_table();

		$q->buttons_table(gettext(" Switch Active Adapter "),
			"config_adapter_submit('Switch', '$node',
				'$NAFOGROUP_LIST[$selectindex]{'name'}')",
                        gettext("  Cancel  "),
                        "self.location.href = \"$VIEW\?node=$node\"" );
                print "</form>";
	} else {
		# generate the command
		my $cmd="$PNMSET -c $nafogroup -o switch $adp";
		
		# Executing the command
                $self->execute_cmd($q, $cmd, $node,
		    [gettext("Nodes"), "/cgi-bin/node/node_status.pl",
			$node, "/cgi-bin/node/node_info.pl?node=$node",
			gettext("NAFO Groups"), "$VIEW?node=$node",
			gettext("Switch Adapter"),
	    "$CONFIG\?step=1&func=Switch&node=$node&nafogroup=$nafogroup"]);
	}
}

# Create new nafo group
sub create_nafogroup($$$$$$)
{
	my ($self, $q, $step, $node, $nafogroup, $adp) = @_;

	if (!$self->is_node_online($q, $node)) {
		return;
	}
	if ($step == 1) {
		$q->print_title([gettext("Nodes"),
			"/cgi-bin/node/node_status.pl",
		    $node, "/cgi-bin/node/node_info.pl?node=$node",
		    gettext("NAFO Groups"), "$VIEW?node=$node",
		    gettext("Create NAFO Group")]);
		# gather information
		print "<form name=inputs action=$CONFIG>";
		print $q->input({ type => "hidden",
                                        name => "step",
                                        value => 2 });
		print $q->input({ type => "hidden",
                                        name => "node",
                                        value => $node });
		print $q->input({ type => "hidden",
                                        name => "func",
                                        value => "Create" });

		print $q->start_table({ cellpadding => 5 });
		print $q->start_Tr();
		print $q->start_td();
		print $q->start_div({ class => "action-window-label-text" });
		print gettext("NAFO Group Name:");
		print $q->end_div();
		print $q->end_td();
		print $q->start_td();
		print $q->input({ type=> "text", name => "nafogroup",
                                 size => 10,  maxlength => 20 });
		print $q->end_td();
                print $q->end_Tr();

		print $q->start_Tr();
                print $q->start_td();
                print $q->start_div({ class => "action-window-label-text" });
                print gettext("Adapter(s) in Group:");
                print $q->end_div();
                print $q->end_td();
                print $q->start_td();
                print $q->input({ type=> "text", name => "adp",
                                 size => 20,  maxlength => 40 });
		print gettext(" (separate with space)");
                print $q->end_td();     
                print $q->end_Tr();
		print $q->end_table();

		$q->buttons_table(gettext(" Create NAFO Group "),
                        "return verify_newnafogroup()",
                        #"document.inputs.submit()",
                        gettext("  Cancel  "),
                        "self.location.href = \"$VIEW\?node=$node\"" );
                print "</form>";
	} else {
		# generate the command
		my $cmd="$PNMSET -c $nafogroup -o create $adp";

		# Executing the command
                $self->execute_cmd($q, $cmd, $node,
		    [gettext("Nodes"), "/cgi-bin/node/node_status.pl",
			$node, "/cgi-bin/node/node_info.pl?node=$node",
			gettext("NAFO Groups"), "$VIEW?node=$node",
			gettext("Create Adapter"),
	    "$CONFIG\?step=1&func=Create&node=$node&nafogroup=$nafogroup"]);
	}
}

# delete a nafo group from the cluster
sub delete_nafogroup
{
	my ($self, $q, $step, $node, $nafogroup) = @_;

	if (!$self->is_node_online($q, $node)) {
		return;
	}
	if ($step == 1) {
		if ($#NAFOGROUP_LIST < 0) {
                        $q->print_title([gettext("Nodes"),
				"/cgi-bin/node/node_status.pl",
			    $node, "/cgi-bin/node/node_info.pl?node=$node",
			    gettext("NAFO Groups"), "$VIEW?node=$node",
			    gettext("Delete NAFO Group"),
		"$CONFIG\?step=1&func=Delete&nafogroup=$nafogroup&node=$node",
			    gettext("Warning")]);
                        print "<form>";
                        $q->start_message_table("warning");
                        print $q->start_p({ class => "message-header-text" });
                        print sprintf(gettext("No NAFO group has been Configured on Node %s"), $node);
                        print $q->end_p();
                        $q->end_message_table($q);
                        $q->buttons_table(gettext("  Cancel  "), "self.locati
on.href=\"$VIEW\?node=$node\"" );
                        print "</form>";
			return;
		}
			
		$q->print_title([gettext("Nodes"),
			"/cgi-bin/node/node_status.pl",
		    $node, "/cgi-bin/node/node_info.pl?node=$node",
		    gettext("NAFO Groups"), "$VIEW?node=$node",
		    gettext("Delete NAFO Group")]);
		# gather information
		print "<form name=inputs action=$CONFIG>";
		print $q->input({ type => "hidden",
                                        name => "step",
                                        value => 2 });
		print $q->input({ type => "hidden",
                                        name => "node",
                                        value => $node });
		print $q->input({ type => "hidden",
                                        name => "func",
                                        value => "Delete" });

		print $q->start_table({ cellpadding => 5 });
		print $q->start_Tr();
		print $q->start_td();
		print $q->start_div({ class => "action-window-label-text" });
		print gettext("NAFO Group:");
		print $q->end_div();
		print $q->end_td();
		print $q->start_td();
		print "<select name=nafogroup>";
		for (my $i = 0; $i <= $#NAFOGROUP_LIST; $i++) {
                print "<option value=$NAFOGROUP_LIST[$i]{'name'}>$NAFOGROUP_LIST[$i]{'name'}";
        	}
		print "</select>";
		my $selectidx;
		if (defined $nafogroup) {
			$selectidx = $self->find_item_by_name(\@NAFOGROUP_LIST,
					 $nafogroup);
			if ($selectidx < 0) {
				$selectidx = 0;
			}
		}
		print $q->start_script();
		print "document.inputs.nafogroup.selectedIndex=$selectidx";
               	print $q->end_script();	
		print $q->end_td();
                print $q->end_Tr();

		print $q->end_table();

		$q->start_message_table("warning");
               	print $q->start_p({ class => "message-description-text" });
               	print gettext("This action will permanently delete the selected NAFO group.");
               	print $q->end_p();
               	$q->end_message_table($q);

		$q->buttons_table(gettext(" Delete NAFO Group "),
                        "document.inputs.submit()",
                        gettext("  Cancel  "),
                        "self.location.href = \"$VIEW\?node=$node\"" );
                print "</form>";
	} else {
		# generate the command
		my $cmd="$PNMSET -c $nafogroup -o delete";

		# Executing the command
                $self->execute_cmd($q, $cmd, $node,
		    [gettext("Nodes"), "/cgi-bin/node/node_status.pl",
			$node, "/cgi-bin/node/node_info.pl?node=$node",
			gettext("NAFO Groups"), "$VIEW?node=$node",
			gettext("Delete NAFO Group"),
	    "$CONFIG\?step=1&func=Delete&node=$node&nafogroup=$nafogroup"]);
	}
}

sub execute_cmd($$$$$$)
{
	my ($self, $q, $cmd, $node, $title) = @_;
	
	if (!$self->is_node_online($q, $node)) {
		return;
	}

	# replace space with + in order to pass the $cmd through web server 
	my $execcmd = $cmd;
	$execcmd =~ tr/ /+/;
	my $location = "$EXEC\?cmd=$execcmd";
	
	pop @$title; # Don't need last link
	$q->print_title($title);
	my @output = $self->get_http($node, $self->get_http_port(), $location);
	if ($#output > -1) {
		if (!defined($output[0])) {
			# Communication error
			return;
		}
		# There is error returned, print out error messages
		print "<form>";
                $q->start_message_table("error");
                print $q->start_p({ class => "message-header-text" });
		print gettext("Error");
                print $q->end_p();
                print $q->start_p({ class => "message-description-text" });
                print gettext("The command:");
                print $q->end_p();
                print $q->start_p({ class => "message-command-text" });
                print $cmd;
                print $q->end_p();
                print $q->start_p({ class => "message-description-text" });
                print gettext("Failed with the following error:");
                print $q->end_p();
                print $q->start_p({ class => "message-command-text" });
                print "@output";
                print $q->end_p();
                $q->end_message_table();
		$q->buttons_table(gettext("  < Back  "), "history.back()",
			gettext("  Cancel  "),
			"self.location.href = \"$VIEW\?node=$node\"" );
                print "</form>";
	} else {
		$q->print_title($title);
                print "<form>";
                $q->start_message_table("info");
                print $q->start_p({ class => "message-header-text" });
		print gettext("Operation Completed Successfully");
                print $q->end_p();
                print $q->start_p({ class => "message-description-text" });
                print gettext("The following command completed without error:");
                print $q->end_p();
                print $q->start_p({ class => "message-command-text" });
                print $cmd;
                print $q->end_p();
                $q->end_message_table();
                $q->buttons_table(sprintf(gettext(" View NAFO Group Information on Node %s"), $node),
			"self.location.href= \"$VIEW\?node=$node\"" );
                print "</form>";	
        }	
}

sub get_pnmset {
        return $PNMSET;
}
1;
