#
#  Copyright (c) 2001 Sun Microsystems, Inc, All Rights Reserved.
# SCCS Info:      pragma ident   "@(#)VE.pm 1.2     02/03/22 10:45:28 SMI"
#
#  Perl Module for Virtualization Engine
#
#  This module serves as the main library utility for functions that
#  operate on the Virtualization Engine for the Indy.
#
#  The module also supports functions that can parse the VE output.
#

package SE::VE;

use Exporter;

@ISA		= qw(Exporter);
@EXPORT		= qw( enumerate getSystemData getHTMLOutput printHTMLOutput
		      get_entry $VLUN $INITIATOR $DISKPOOL $ZONE );
$VERSION	= 1.00;

use FileHandle;
use CGI qw/:standard *table/;		# load standard CGI routines
use Getopt::Long;
use CGI::Carp;

my ( $IndyPkg ) = qw( /opt/SUNWsecfg/ );
my ( $BinDir )  = qw( bin/ );
my $Exec    = "showvemap";
my $Enum    = "listavailable";
my $Name    = 'SE::VE';

*INITIATOR	= \'INITIATOR';		# constants define
*DISKPOOL	= \'DISKPOOL';		# constants define
*ZONE		= \'ZONE';		# constants define
*VLUN		= \'VLUN';		# constants define

my $lastVE 	= ();


#  Return a FileHandle to a connection with the V.E.
#
#  Input Arguments:	$ve	the name of the VE
#			$path	(optional - the name of a file)
#
#  Return Value:	a FileHandle object that has the VE connection
#  The filehandle will be read subsequently and parsed
#
sub getConnectionCmd {

    my ( $ve, $path, $live ) = @_ if @_;

    my $fp = new FileHandle;

    my $execPath = $IndyPkg . $BinDir . $Exec;

    my $flag = " -f ";

    if ( $live )
    {
	$flag = " -l ";
    }

    return "$execPath -n $ve $flag";
}


#  Return the list of VEs or the subcomponents of a specific one
#  that are available
#
#  Input Arguments:	$who	(optional) if present, the type of
#				subcomponent, one of $INITIATOR, $ZONE, $VLUN,
#				$DISKPOOL
#			$ve_name  the name of the VE
#			$ve	the data structure that may have been populated
#				in a previous call
#			$path	(optional) the name of a file including the
#				system data output for the VE
#
#  Return Value:	the list of instances matching
#
sub enumerate {

    my $class = shift if ( $_[0] eq $Name );

    return _enumerate ( @_ );
}


sub _enumerate {

    my ( $who, $ve_name, $ve, $path ) = @_ if @_;

    my @retval = ();

    if ( ! ( defined $who && $who ) ) {

	my $fp = new FileHandle;
	my $execPath = $IndyPkg . $BinDir . $Enum;

	if ( -x $execPath ) {
	    $fp->open ("$execPath -v |") 
	    or croak "${Name}::enumerate: Unable to execute $execPath $!\n";
	}

	while ( <$fp> ) {
	    push @retval, $_;
	}
	$fp->close;

	@retval = ( 've0', 've1' ) unless @retval;
    }

    else {
	
	if ( ! ( defined $ve && $ve ) ) {
	    $ve = _getSystemData ( $ve_name, $path, $live );
	}

	if ( $who eq $INITIATOR ) {
	    @retval = getNamesByKey ( $ve, 'cinitiator', 'aname' );
	}
	elsif ( $who eq $ZONE ) { 
	    @retval = getNamesByKey ( $ve, 'dzones', 'aname' );
	}
	elsif ( $who eq $VLUN ) { 
	    @retval = getNamesByKey ( $ve, 'avluns', 'aname' );
	}
	elsif ( $who eq $DISKPOOL ) { 
	    @retval = getNamesByKey ( $ve, 'bdiskpool', 'aname' );
	}
    }

    return @retval;
}


#  Internal subroutine to return the list of names associated with the
#  key passed as an argument
#
sub getNamesByKey {

    my ( $theVE, $ve_key, $rec_key ) = @_ if @_;

    my $recs = $$theVE{$ve_key};

    my $allrecs = $$recs{ref} ;
    my %retval = ();

    for my $rec ( @$allrecs )
    {
	my $val = $$rec{$rec_key};

	if ( defined $val ) {
	    $retval{$val} = $val unless ( defined $retval{$val} ) ;
	}
    }
    return sort keys %retval;
}

#  Return a list of all the records whose field '$key' matches the '$value'
#  passed as argument
#
#  Input Arguments:	$who	the type of information from the VE
#				$INITIATOR, $ZONE, $VLUN, $DISKPOOL
#			$ve_name  the name of the V.E.
#			$key	the key to select the record to compare
#				the value
#			$value	the value to compare
#			$ve	the structure for the VE, a result of
#				a previous invocation to getSystemData.
#			$path	(optional) the pathname for a file that
#				includes the data to be parsed
#
#  Return Value:	a list of references to records that match 
#			(i.e. that $rec{$key} eq $value ).
#
sub get_entry {

    my $class = shift if ( $_[0] eq $Name );

    return _get_entry ( @_ );
}

sub _get_entry {

    my ( $who, $ve_name, $key, $value, $ve, $path ) = @_ if @_;

    my @retval = ();

    if ( ! ( defined $ve && $ve ) ) {
	$ve = _getSystemData ( $ve_name, $path, $live );
    }

    if ( $who eq $INITIATOR ) {
	@retval = getRecordByKey ( $ve, 'cinitiator', $key, $value );
    }
    elsif ( $who eq $ZONE ) { 
	@retval = getRecordByKey ( $ve, 'dzones', $key, $value );
    }
    elsif ( $who eq $VLUN ) { 
	@retval = getRecordByKey ( $ve, 'avluns', $key, $value );
    }
    elsif ( $who eq $DISKPOOL ) { 
	@retval = getRecordByKey ( $ve, 'bdiskpool', $key, $value );
    }
    return @retval;
}


#  Internal subroutine to return the list of names associated with the
#  key passed as an argument
#
sub getRecordByKey {

    my ( $theVE, $ve_key, $rec_key, $value ) = @_ if @_;

    my $hash = $$theVE{$ve_key};

    my $allrecs = $$hash{ref};

    my @retval = ();

    for my $rec ( @$allrecs )
    {
	my $val = $$rec{$rec_key};

	if ( $val ) {
	    push @retval, $rec if ( $value eq $val );
	}
    }
    return @retval;
}


# 
#  Obtain V.E. System Info, assemble into a list of Hashes to be returned
#
#  All those that have a match are returned.
#
#  The function expects a reference to appropriate lists.
#
#  Input Arguments:	$ve_name	the name of the V.E.
#			$path		(optional) the pathname to a file
#					containing VE data
#
#  Return Value:	a reference to a composite data structure that
#			contains the parsed information.
#
sub getSystemData {

    my $class = shift if ( $_[0] eq $Name );

    return _getSystemData ( @_ );
}


sub _getSystemData {

    my ( $ve_name, $path, $live ) = @_ if @_;

    my $section	= '';
    my $version = '';

    #  theVE is the composite object that exposes the whole ve.
    #
    #  fields:	version system lunlist lunstat portlist portmap
    #
    my %theVE = ();

    $lastVE = \%theVE;

    #  vluns is a List of Hashes, where every entry in the list
    #  corresponds to a VLUN
    #
    #  The following are the fields that make up the entry:
    #
    #   fields:		name  disk_pool  active_wwn  MP_drive  target  size
    #
    my @vluns = ();
    my %vlunkeys = (
		     aname  		=> 'Name',
		     bactive_wwn	=> 'Active WWN',
		     cMP_drive		=> 'MP Drive',
		     dvlun_target	=> 'VLUN Target',
		     evlun_name		=> 'VLUN Name',
		     fsize		=> 'Size (GB)',
		     gslic_zones	=> 'SLIC Zones'
		   );

    #  diskpool is a List of Hashes
    #
    #  fields:		name  raid  MP_drive  size  free  n_vluns
    #
    my @diskpool = ();
    my %diskpoolkeys = (
		     aname  		=> 'Name',
		     braid		=> 'Raid Level',
		     cMP_drive		=> 'MP Drive',
		     dsize		=> 'Size (GB)',
		     efree		=> 'Free Size (GB)',
		     fn_vluns		=> 'No of VLUNs'
		   );

    #  initiator is a List of Hashes, the zones field is a list
    #
    #  fields:		name  uid  ve_host  online  ( zones )
    #
    my @initiator = ();
    my %initiatorkeys = (
		     aname  		=> 'Name',
		     buid		=> 'UID',
		     cve_host		=> 'VE Host',
		     donline		=> 'Online',
		     ezones		=> 'SLIC Zones'
		   );

    #  zones is a List of Hashes
    #
    #  fields:		name  hba_wwn  initiator  n_vluns
    #
    my @zones = ();
    my %zoneskeys = (
		     aname  		=> 'Name',
		     bhba_wwn		=> 'HBA WWN',
		     cinitiator		=> 'Initiator',
		     dn_vluns		=> 'No of VLUNs'
		   );

    my $devType = 'Virtualization Engine SubSystem';

    my $ip_addr = '';

    #
    #  Initialize first the VE structure to point at the inner Structs
    #  fields:	version system lunlist lunstat portlist portmap
    #
    $theVE{avluns} 	= {
			     name => $devType . ' Virtual LUNs',
			     ref  => \@vluns,
			     keys => \%vlunkeys
			  };
    $theVE{bdiskpool} 	= {
			     name => $devType . ' Disk Pools', 
			     ref  => \@diskpool,
			     keys => \%diskpoolkeys
			  };
    $theVE{cinitiator} 	= {
			     name => $devType . ' Initiators List',
			     ref  => \@initiator,
			     keys => \%initiatorkeys
			  };
    $theVE{dzones} 	= {
			     name => $devType . ' Zone List',
			     ref  => \@zones,
			     keys => \%zoneskeys
			  };

    my $cmd = getConnectionCmd ( $ve_name, $path, $live );

    my ( $ret, $stdout, $stderr ) = SE::Util->execCmd ( $cmd, "0" );

    if ( $ret )
    {
	SE::Util->printCmdAll ( $cmd, $ret, $stdout, $stderr );
	return;
    }
    
    #  Iterate over the command, and process line by line
    #
    my @lines = split '\n', $stdout;
    for $_ (@lines) {
	
	next if (/^\s*#/o);		# skip comments
	next if (/^\s*$/o);		# skip empty lines
	next if (/^\*\*\*\*\*$/o);	# skip section separators
	next if (/^-----------/o);	# skip title separators

	chomp;

	if (/^VIRTUAL LUN SUMMARY\s*$/o)	# if a header for a section
	{
	    $section = $VLUN;		# the section
	}
	elsif (/^DISKPOOL SUMMARY\s*$/o)
	{
	    $section = $DISKPOOL;		# the section
	}
	elsif (/^INITIATOR SUMMARY\s*$/o)
	{
	    $section = $INITIATOR;		# the section
	}
	elsif (/^ZONE SUMMARY\s*$/o)
	{
	    $section = $ZONES;			# the section
	}

	else {
	    s/^\s+//o;				# remove leading spaces

	    if ( $section eq $VLUN ) {
		#pre 1.1.9 
		#next if (/Diskpool\s+Active Path\s+MP Drive\s+VLUN\s+VLUN/o);
		#from 1.1.9 on
		next if (/Diskpool\s+VLUN Serial\s+MP Drive\s+VLUN\s+VLUN/o);
		next if (/WWN\s+Target\s+Target\s+Name\s+GB/o);

		my $rec = {};
	    	( $rec->{aname}, $rec->{bactive_wwn}, $rec->{cMP_drive},
		  $rec->{dvlun_target}, $rec->{evlun_name}, $rec->{fsize} ) =
			/(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s*/o;
		$rec->{gslic_zones} = $' ? $' : '-';

		#  insert the record to the @vluns list
		#
		push @vluns, $rec if $rec;
	    }

	    elsif ( $section eq $DISKPOOL ) {
		next if (/Diskpool\s+RAID\s+MP Drive\s+Size\s+Free Space/o);
		next if (/Target\s+GB\s+GB\s+VLUNs/o);
	
		my $rec = {};
		( $rec->{aname}, $rec->{braid}, $rec->{cMP_drive}, 
		  $rec->{dsize}, $rec->{efree}, $rec->{fn_vluns} ) =
			    /(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)/o;

		#  insert the record to the @diskpool list
		#
		push @diskpool, $rec if $rec;
	    }
	    elsif ( $section eq $INITIATOR ) {
		next if (/Initiator\s+UID\s+VE Host\s+Online\s+SLIC Zones/o);

		my $rec = {};
		my $zones;
		( $rec->{aname}, $rec->{buid}, $rec->{cve_host},
		  $rec->{donline}, $rec->{ezones} ) =
			    /(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s*/o;
		$rec->{ezones} = $';

		#  insert the record to the @diskpool list
		#
		push @initiator, $rec if $rec;
	    }
	    elsif ( $section eq $ZONES ) {
		next if (/Zone\s+Name\s+HBA\s+WWN\s+Initiator\s+Number/o);

		my $rec = {};
		( $rec->{aname}, $rec->{bhba_wwn}, $rec->{cinitiator},
		  $rec->{dn_vluns} ) =
			    /(\S+)\s+(\S+)\s+(\S+)\s+(\S+)/o;

		#  insert the record to the @zones list
		#
		push @zones, $rec if $rec;
	    }
	}
    }

    return \%theVE;
}


#  Internal method to generate the table header
#
sub doTableHeader {

    my ( $title, $devId ) = @_ if @_;

    return h2 ( $title . ' - ' . $devId ) . "\n<center>\n" .
           start_table ( { -border=>0, -width=>"90%", -cellspacing=>2,
			   -bgcolor=>"white", -cellpadding=>8 },
			   caption ( $title ) ) . "\n";
}


#  Method to generate an HTML table based on List of Hashes passed
#  as argument
#
sub genHTMLLoH {

    my ( $list, $keys ) = @_ if @_;

    my $gen = '';

    my @vals = ();

    #  generate the 'th' for the headers line
    #
    for my $key ( sort keys %$keys ) {

	push @vals, $$keys{$key};
    }

    $gen .= '  ' . Tr ( { -align=>CENTER, -valign=>TOP },
	       [ th ( [ @vals ] ) ] ) . "\n" if @vals;

    for my $rec ( @$list ) {

	my @recvals = ();
	for my $k2 ( sort keys %$keys ) {

	    push @recvals, $$rec{$k2};
	}
	$gen .= '  ' . Tr ( [ td ( [ @recvals ] ) ] ) . "\n" if @recvals;
    }
    return $gen . end_table . "\n</center>\n";
}

#  Method to generate an HTML table based on a List passed
#  as argument. If the second argument is present, it indicates the
#  name of a field that is expected to be a list by itself
#
#  Input Argument:	$list	reference to the list
#
sub genHTMLInitiator {

    my ( $list, $keys, $listKey ) = @_ if @_;

    my $gen = '';

    my @vals = ();
    #  generate the 'th' for the headers line
    #
    for my $key ( sort keys %$keys ) {

	push @vals, $$keys{$key};
    }

    $gen .= '  ' . Tr ( { -align=>CENTER, -valign=>TOP },
	       [ th ( [ @vals ] ) ] ) . "\n";

    for my $rec ( @$list ) {

	@vals = ();
	for my $k2 (sort keys %$rec) {

	    if ( defined $listKey && ( $k2 eq $listKey ) ) {

		my $val = join ',',  @{$$rec{$k2}};
	    }
	    else
	    {
		push @vals, $$rec{$k2};
	    }
	}
	$gen .= '  ' . Tr ( [ td ( [ @vals ] ) ] ) . "\n" if @vals
    }
    return $gen . end_table . "\n</center>\n";
}


#  Externally exposed method to generate the HTML output
#
#  Usage: printHTMLOutput ($title, $ve_name [, $args [, $path] ])
#
#  Input Arguments:	$title		the title to use in the output page
#			$ve_name	the name of the VE instance
#			$args		a list with the names of the components
#					can be omitted or an empty list
#			$path		path with VE data to parse
#
#
sub printHTMLOutput {

    my $class = shift if ( $_[0] eq $Name );

    _printHTMLOutput ( @_ );
}


#  Internal method to do the job
#
sub _printHTMLOutput {

    my ( $title, $ve_name, $args, $path, $live ) = @_ if @_;

    my $ve = _getSystemData ( $ve_name, $path, $live );

    my $out =  _getHTMLOutput ( $ve, $title, $ve_name, $args );

    print STDOUT $out;
}


#  Method to generate HTML output. See 'pod' documentation for more details
#
#  Externally exposed method to generate the HTML output
#
#  Usage: getHTMLOutput ($title, $ve_name [, $args [, $path] ])
#
#  Input Arguments:	$title		the title to use in the output page
#			$ve_name	the name of the VE instance
#			$args		a list with the names of the components
#					can be omitted or an empty list
#			$path		path with VE data to parse
#
sub getHTMLOutput {

    my $class = shift if ( $_[0] eq $Name );

    my ( $title, $ve_name, $args, $path, $live ) = @_ if @_;

    my $ve = _getSystemData ( $ve_name, $path, $live );

    return _getHTMLOutput ( $ve, $title, $ve_name, $args );
}


#  Method to generate HTML output. See 'pod' documentation for more details
#
sub _getHTMLOutput {

    my ( $theVE, $title, $ve_name, $args ) = @_ if @_;
 
    my @args = ();

    if ( ! ( defined $args && $args ) ) { # only the VE structure was passed

	#  set it as if all section have been requested
	#
	@args = sort keys %$theVE;
    }
    else {
	@args = @$args;
    }

    my $out = '';
    if ($main::opt_d) {

	$out = start_html ('StorEdge Device Manager - ' . $title) . "\n";
    }

    for $section (@args) {

	my $refHash = $$theVE{$section};
	my $ref = $$refHash{ref};
	my $keys = $$refHash{keys};
	my $formalName = $$refHash{name};

	$out .= doTableHeader ( $formalName, $ve_name );

	if ( $section eq 'avluns' ) {
	    $out .= genHTMLLoH ( $ref, $keys );
	}
	elsif ( $section eq 'bdiskpool' ) {
	    $out .= genHTMLLoH ( $ref, $keys );
	}
	elsif ( $section eq 'cinitiator' ) {
	    $out .= genHTMLLoH ( $ref, $keys );
	}
	elsif ( $section eq 'dzones' ) {
	    $out .= genHTMLLoH ( $ref, $keys );
	}
    }
    $out .= end_html . "\n" if ( $main::opt_d );
    return $out;
}

1;
__END__

=head1 NAME

SE::VE - Perl Module for Virtualization Engine Management

=head1 SYNOPSIS

Functions and Data Structures for Virtualization Engine Management on the 
Indy Platform.

=head1 DESCRIPTION

The C<VE> module supplies subroutines to be used by the storage management
applications developer to access the functionality necessary to manage
a Virtualization Engine on the Indy.

=head1 OVERVIEW OF SUBROUTINES

The subroutines exposed address two main aspects, the first one is to
access the device and obtain the output generated, parsing it into a 
data structure that can be used.

The second aspect is the ability to expose the data structure parsed
into a presentation context. Currently HTML.

=head2 B<getSystemData ()>

Syntax:

	$valueHash = getSystemData ( $ve_id );

The subroutine will access the Virtualization Engine whose identity has been
passed as an argument.

A reference to a hash on which the results of the system data will be
returned is passed as the first argument.

The output generated will be included as a composite data structure
associated with specific entries in the F<%valueHash>.

Sections parsed include B<version>, B<system>, B<lunlist>, B<lunstat>,
B<portlist>, B<portmap>.

=head2 B<valueHash> Data Structure

The B<valueHash> data structure returned will have the following format.


    %valueHash = (
	vluns  => (
		    {
			name		=> "1mylun" ,
			disk_pool	=> "t3b00" ,
			active_wwn	=> "5002.....4315" ,
			MP_drive	=> "T49152" ,
			vlun_target	=> "T16384" ,
			size		=> "1.0" ,
		    },
		  ),

	diskpool => (
		      {
			name    	=> "t3b00" ,
			raid    	=> "5" ,
			MP_drive	=> "T49152" ,
			size		=> "477.2" ,
			free		=> "421.2" ,
			n_vluns		=> "15" ,
		      },
		    ),

	initiator => ( 
		       {
			  name    	=> "I00001" ,
			  uid    	=> "20000E0 ... 2F80" ,
			  ve_host	=> "V1N" ,
			  online	=> "yes" ,
			  zones		=> ( wst4_c9 ,
					     padma_2 ),
			},
		     ),

	zones => ( 
		   {
		      name    	=> "wst4_c9" ,
		      hba_wwn   => "20000E0 ... 2F80" ,
		      initiator	=> "I00001" ,
		      n_vluns	=> "15" ,
		   },
		 ),
	       );

=head2 B<getHTMLOutput ()>

Syntax:

	getHTMLOutput ( \%valueHash [$section [, $section1 [ ... ]]]);

The subroutine operates on the B<%valueHash> that was passed. It is expected
that the hash has been generated following a previous invocation to the
B<getSystemData()>.

If no $section arguments are passed, then all the V.E. system map is
generated.

Valid sections include B<version>, B<system>, B<lunlist>, B<lunstat>,
B<portlist>, B<portmap>.

Output generated will be in HTML form.

=head1 AUTHOR

David Bautista-Lloyd, Arieh Markel

=cut

