#
#
#  Copyright (c) 2001 Sun Microsystems, Inc, All Rights Reserved.
#
# SCCS Info:      pragma ident   "@(#)T3.pm 1.2     02/03/22 10:45:25 SMI"
#  Perl Module for T3
#
#  This module serves as the main library utility for functions that
#  operate on the T3.
#
#  The module also supports functions that can parse the T3 output.
#

package SE::T3;

use Exporter;
use strict;

@T3::ISA	= qw(Exporter);
@T3::EXPORT     = qw(enumerate getSystemData getHTMLOutput printHTMLOutput);
$T3::VERSION	= 1.00;

use FileHandle;
use CGI qw/:standard *table/;		# load standard CGI routines
use Getopt::Long;
use CGI::Carp;
#use SE::Util;

my ( $IndyPkg ) = qw( /opt/SUNWsecfg/ );
my ( $BinDir )  = qw( bin/ );
my $Exec        = 'showt3';
my $Enum        = 'listavailable';
my $Name        = 'SE::T3';

#  Return a FileHandle to a connection with the T3.
#
#  The filehandle will be read subsequently and parsed
#
sub getConnectionCmd
{
    my ( $t3, $passwd, $path ) = @_ if @_;

    my $fp = new FileHandle;

    my $execPath = $IndyPkg . $BinDir . $Exec;

    return "$execPath -n $t3";
}

#  Return the list of T3's that are available
#
#  Input Arguments:     $path   (optional) the name of a file including the
#                               system data output for the T3
#
#  Return Value:        the list of T3s
#
sub enumerate {

    my $class = shift if ( $_[0] eq $Name );

    return _enumerate ( @_ );
}

sub _enumerate {

    my ( $path ) = @_ if @_;

    my $fp = new FileHandle;

    my $execPath = $IndyPkg . $BinDir . $Enum;

    if ( -x $execPath) {
	$fp->open ("$execPath -t |") 
	    or croak "${Name}::enumerate: Unable to execute $execPath $!\n";
    }

    my @t3 = ();
    while ( <$fp> ) {
	push @t3, $_;
    }
    $fp->close;

    @t3 = ( 't3_0', 't3_1', 't3_2' ) unless @t3;

    return @t3;
}

# 
#  Obtain T3 System Info, assemble into a list of Hashes to be returned
#
#  All those that have a match are returned.
#
#  The function expects a reference to appropriate lists.
#
#  Input Arguments:	$t3_name	the name of the T3
#			$path		(optional) the path to a file
#					containing the T3 data to parse
#
#  Output Arguments:	
#
#  Return Value:	a reference to a composite data structure that
#			contains the parsed information.
#
sub getSystemData {

    my $class = shift if ( $_[0] eq $Name );

    return _getSystemData ( @_ );
}

sub _getSystemData {

    my ( $t3_name, $passwd, $path ) = @_ if @_;

    my $section	= '';

    my $version = '';

    my $insection = '';

    my %theT3 = ();

    #  t3 is the composite object that exposes the whole
    #  t3.
    #
    #  fields:	version system lunlist lunstat portlist portmap
    #

    #  system is a plain hash
    #
    #  fields:	blocksize cachemode mirror mp_support readahead
    #		recon_rate memsize  cachesize
    #
    my %system  = ();

    #  lunlist is a Hash, where every entry corresponds
    #  to one of the volumes
    #
    #  The following are the fields that make up the entry:
    #
    #	key: 		volid	
    #   fields:		capacity  raid   data_vol    stby_vol
    #
    my %lunlist = ();

    #  lunstat is a Hash of Hashes
    #
    #  primary key:	volid
    #  secondary key:	slice
    #  fields:		mounted
    #
    my %lunstat = ();

    #  portlist is a Hash, every entry corresponding to the port
    #
    #  key:		portid
    #  fields:		targetid  addr_type   status   host   wwn
    #
    my %portlist = ();

    #  portmap is a Hash, every entry corresponding to the port
    #
    #  key:		portid
    #  fields:		targetid  addr_type  lun  volume owner access
    #
    my %portmap = ();

    my $devType = 'T3 System';

    my $ip_addr = '';

    #  frustat is a Hash of Hashes
    #
    #  There are four inner hashes:	ctlr, disk, loop, pwr
    #
    my @ctlrfru = ();
    my @diskfru = ();
    my @loopfru = ();
    my @pwrfru = ();

    my %fructlrkeys = ( 
			aname 	=> 'name',
			bstatus	=> 'status',
			cstate	=> 'state',
			drole	=> 'role',
			epartner=> 'partner',
			ftemp	=> 'temp'
		      );
    my %frudiskkeys = ( 
			aname 	=> 'name',
			bstatus	=> 'status',
			cstate	=> 'state',
			drole	=> 'role',
			evolume	=> 'volume',
			fport1	=> 'port 1',
			gport2	=> 'port 2',
			htemp	=> 'temp'
		      );
    my %fruloopkeys = ( 
			aname 	=> 'name',
			bstatus	=> 'status',
			cstate	=> 'state',
			dmode	=> 'mode',
			ecable1	=> 'cable 1',
			fcable2	=> 'cable 2',
			gtemp	=> 'temp'
		      );
    my %frupwrkeys = ( 
			aname 	=> 'name',
			bstatus	=> 'status',
			cstate	=> 'state',
			dsource	=> 'source',
			eoutput	=> 'output',
			fbattery=> 'battery',
			gfan1	=> 'fan 1',
			hfan2	=> 'fan 2',
			itemp	=> 'temp'
		      );

    #
    #  Initialize first the t3 structure to point at the inner Structs
    #  fields:	version system lunlist lunstat portlist portmap
    #
    $theT3{aversion} 	= {
			     name => $devType . ' Version',
			     ref  => \$version
			  };
    $theT3{bsystem} 	= {
			     name => $devType . ' Configuration',
			     ref  => \%system
			  };
    $theT3{clunlist} 	= {
			     name => $devType . ' LUN List',
			     ref  => \%lunlist
			  };
    $theT3{dlunstat} 	= {
			     name => $devType . ' LUN Status',
			     ref  => \%lunstat
			  };
    $theT3{eportlist} 	= {
			     name => $devType . ' Port List',
			     ref  => \%portlist
			  };
    $theT3{fportmap} 	= {
			     name => $devType . ' Port Mappings',
			     ref  => \%portmap
			  };
    $theT3{gfrustat} 	= {
			     name => $devType . ' FRU Status',
			     ctlr => {
					name => ' Controllers',
				 	list => \@ctlrfru,
					keys => \%fructlrkeys
				     },
			     disk => {
					name => ' Disks',
					list => \@diskfru,
					keys => \%frudiskkeys
				     },
			     loop => {
					name => ' Loops',
					list => \@loopfru,
					keys => \%fruloopkeys
				     },
			     pwr => {
					name => ' Power Control Units',
					list => \@pwrfru,
					keys => \%frupwrkeys
				     }
			  };
    $theT3{'0ip_addr'}	= {
			     name => $devType . ' IP Address',
			     ref  => \$ip_addr
			  };

    my $cmd = getConnectionCmd ( $t3_name, $passwd, $path );

    my $prePasswd = $ENV{PASSWD};
    $ENV{PASSWD} = $passwd ;

    my ( $ret, $stdout, $stderr ) = SE::Util->execCmd ( $cmd, "0" );

    $ENV{PASSWD} = $prePasswd;

    if ( $ret )
    {
	SE::Util->printCmdAll ( $cmd, $ret, $stdout, $stderr );
	return;
    }
    
    #  Iterate over the command, and process line by line
    #
    my @lines = split '\n', $stdout;
    for (my $cline = 0; $cline < @lines; $cline++) {

	$_  = $lines[$cline];

	next if (/^\s*#/o);		# skip comments
	next if (/^\s*$/o);		# skip empty lines

	chomp;

	if (/^VERSION :-\s*$/o)		# if a header for a section
	{
	    $section = 'VERSION'		# the section
	} elsif (/^SYSTEM LISTING :-\s*$/o)
	{
	    $section = 'SYSTEM';		# the section
	}
	elsif (/^LUN LIST :-\s*$/o)
	{
	    $section = 'LUNLIST';		# the section
	}
	elsif (/^LUN STATUS :-\s*$/o)
	{
	    $section = 'LUNSTAT';		# the section
	}
	elsif (/^PORT LIST :-\s*$/o)
	{
	    $section = 'PORTLIST';		# the section
	}
	elsif (/^PORT MAP :-\s*$/o) {
	    $section = 'PORTMAP';		# the section
	}
	elsif (/^FRU STAT :-\s*$/o) {
	    $section = 'FRUSTAT';		# the section
	}

	else {
	    s/^\s+//o;

	    if ($section eq 'SYSTEM') {
		my ($prop, $val);

		if (/\s+:\s+/o) {
		    $prop = $`;
		    $val  = $';
		    $system{$prop} = $val;
		}
	    }
	    elsif ($section eq 'LUNLIST') {
		next if (/volume\s+capacity\s+raid\s+data\s+standby/o);

		my $rec = {};
		my $number;
		my $unit;
		( $rec->{volume}, $rec->{capacity}, $number, $unit,
		  $rec->{raid}, $rec->{data}, $rec->{standby} ) =

			    # vol     capac   GB       raid    data    standby
			    #
			    #  vo      53.7 GB          5     u1d1-4    u1d9
			    #
			    /(\S+)\s+((\S+)\s+(\S+))\s+(\S+)\s+(\S+)\s+(\S+)/o;

		#  assign the record to the entry in lunlist
		#
		$lunlist{$rec->{volume}} = $rec if ($rec);
	    }
	    elsif ($section eq 'LUNSTAT') {

		my @list = split;		# @list has the first line
		$cline++;
		$_ = $lines[$cline];		# read the next line
		my @list1 = split;		# @list1 has the second line

		my %vol = ();			# hash to hold the LUNs
		my @slices = ();		# list of hashes for slices
		$vol{volname} = $list[0];
		$vol{slices} = \@slices;
		$vol{mounted} = $list1[0];
		shift @list;			# drop the first item
		for my $i (0 .. $#list) {
		    my %slice = ();
		    $slice{name} = $list[$i];
		    $slices[$i] = \%slice;
		}
		$lunstat{$vol{volname}} = \%vol;
	    }
	    elsif ($section eq 'PORTLIST') {
		next if (/port\s+targetid\s+addr_type\s+status\s+host\s+wwn/o);
	
		my $rec = {};
		( $rec->{port}, $rec->{targetid}, $rec->{addr_type}, 
		  $rec->{status}, $rec->{host}, $rec->{wwn} ) =

			    # port    tid  addtype  status   host   wwn
			    #
			    # u1p1     1     hard   online    sun    5002
			    #
			    /(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)/o;

		$portlist{$rec->{port}} = $rec if ($rec);
	    }
	    elsif ($section eq 'PORTMAP') {
		next if (/port\s+targetid\s+addr_type\s+lun\s+volume\s+owner\s+access/o);

		my $rec = {};
		( $rec->{port}, $rec->{targetid}, $rec->{addr_type},
		  $rec->{lun}, $rec->{volume}, $rec->{owner}, $rec->{access} ) =

		    # port   tid   addtype   lun   volume  owner   acc
		    #
		    # u1p1     1     hard     0      v0     u1    primary
		    #
		    /(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)/o;

		$portmap{$rec->{port}} = $rec if ($rec);
	    }
	    elsif ($section eq 'FRUSTAT') {
		next if (/----\s+----/o);

		if    ( /CTLR\s+STATUS\s+STATE\s+ROLE\s+PARTNER\s+TEMP/o ) {
		    $insection = 'CTLR';
		    next;
		}
		elsif ( /DISK\s+STATUS\s+STATE\s+ROLE\s+PORT1\s+PORT2/o ) {
		    $insection = 'DISK';
		    next;
		}
		elsif ( /LOOP\s+STATUS\s+STATE\s+MODE\s+CABLE1\s+CABLE2/o ) {
		    $insection = 'LOOP';
		    next;
		}
		elsif ( /POWER\s+STATUS\s+STATE\s+SOURCE\s+OUTPUT\s+BATT/o ) {
		    $insection = 'POWER';
		    next;
		}

		if ( $insection eq 'CTLR' ) {

		    my $rec = {}; 

		    ( $rec->{aname}, $rec->{bstatus}, $rec->{cstate},
		      $rec->{drole}, $rec->{epartner}, $rec->{ftemp} ) =

		    # u1ctr   ready   enabled   master  u2ctr  39.0
		    #
		/(\S+)\s+(\S+)\s+(\S+)\s+(master|alt master)\s+(\S+)\s+(\S+)/o;

		    push @ctlrfru, $rec if $rec;
		}

		elsif ( $insection eq 'DISK' ) { 

		    my $rec = {}; 

		    ( $rec->{aname}, $rec->{bstatus}, $rec->{cstate},
		      $rec->{drole}, $rec->{fport1}, $rec->{gport2},
		      $rec->{htemp}, $rec->{evolume} ) =

		    # u1d1  ready  enabled  standby ready ready 40 vol1 
		    #
		/(\S+)\s+(\S+)\s+(\S+)\s+(standby|data disk)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)/o;

		    push @diskfru, $rec if $rec;
		}

		elsif ( $insection eq 'LOOP' ) { 

		    my $rec = {}; 

		    ( $rec->{aname}, $rec->{bstatus}, $rec->{cstate},
		      $rec->{dmode}, $rec->{ecable1}, $rec->{fcable2},
		      $rec->{gtemp}) =

		    # u1l1  ready  enabled  master installed - 35.5
		    #
		    /(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)/o;

		    push @loopfru, $rec if $rec;
		}

		elsif ( $insection eq 'POWER' ) { 

		    my $rec = {}; 

		    ( $rec->{aname}, $rec->{bstatus}, $rec->{cstate},
		      $rec->{dsource}, $rec->{eoutput}, $rec->{fbattery},
		      $rec->{itemp}, $rec->{gfan1}, $rec->{hfan2} ) =

		    # u1pcu1  ready  enabled  line normal normal normal normal
		    #
		    /(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)/o;

		    push @pwrfru, $rec if $rec;
		}
	    }
	    elsif ($section eq 'VERSION') {

		$version .= $_ . "\n";

		( $ip_addr ) = /.+\((\d+\.\d+\.\d+\.\d+)\)/ unless ( $ip_addr );
	    }
	}
    }

    return \%theT3;
}

#  Internal method to generate the table header
#
sub doTableHeader {

    my ( $title, $devId ) = @_ if @_;

    return h2 ( $title . ' - ' . $devId ) . "\n" .
           start_table ( { -border=>0, -width=>"90%", -cellspacing=>2,
			   -bgcolor=>"white", -cellpadding=>8 },
			  caption ( $title ) ) . "\n";
}

#  Method to generate an HTML table based on a single keyed Hash passed
#  as argument
#
sub genHTMLH {

    my ( $hash ) = @_ if @_;

    my $gen = Tr ( { "-align=>CENTER", "-valign=>TOP" },
		    [ th ( [ 'property', 'value' ] ) ]
		    ) . "\n";

    for my $key (sort keys %$hash) {

	$gen .=  '  ' . Tr ( [ td ( [ $key, $$hash{$key} ] ) ] ) . "\n";
    }

    $gen .= end_table . "\n";

    return $gen;
}

#  Method to generate an HTML table based on Hash of Hashes passed
#  as argument
#
sub genHTMLHoH {

    my ( $hash, $primary ) = @_ if @_;

    my $gen = '';

    my $firstTime = 1;

    for my $key (sort keys %$hash) {

	my $inner = $$hash{$key};

	if ( $firstTime ) {		# the first time, generate the
					# column headers based on the keys
	    $firstTime = 0;
	    my @keys = ( $primary );
	    push @keys, sort keys %$inner;

	    $gen .= '  ' . Tr ( { "-align=>CENTER", "-valign=>TOP" },
		       [ th ( [ @keys ] ) ] ) . "\n";
	}

	my @vals = ( $key );
	for my $k2 (sort keys %$inner) {

	    push @vals, $$inner{$k2};
	}
	$gen .= '  ' . Tr ( [ td ( [ @vals ] ) ] ) . "\n";
    }

    $gen .= end_table . "\n";

    return $gen;
}

#  Method to generate an HTML table based on Hash of Lists passed
#  as argument
#
sub genHTMLSliceTable {

    my ( $hash, $primary ) = @_ if @_;


    my $gen = '';

    my $firstTime = 1;

    for my $key (sort keys %$hash) {

	my $inner = $$hash{$key};

	my $volstat = '';
	my $volid   = $key;
	my $volname = $key;
	my @slices  = ();
	my $row	    = ();

	for my $k2 (keys %$inner) {

	    if ( $k2 eq 'volname' ) {
		$volname = $$inner{volname};
	    }

	    elsif ( $k2 eq 'mounted' ) {
		$volstat = $$inner{mounted};
	    }

	    elsif ( $k2 eq 'slices' ) {
		my $slicelist = $$inner{slices};
		for my $entry (@$slicelist) {
		    my $slicehash = $entry;
		    push @slices, $$slicehash{name};
		}
	    }
	}

	#  the first time, we need to calculate the slices column width to
	#  be the number of slices in the volume
	#
	if ( $firstTime ) {

	    $firstTime = 0;
	    my @keys = ( 'vol id', 'mount status', 'slices' );

	    $gen .= '  <TR VALIGN="TOP" ALIGN="CENTER">' . "\n"
			. '    ' . th ( [ @keys[0..1] ] )
			. '    ' . th ( { -colspan=>$#slices+1 }, [ $keys[2] ] )
		    . "\n  </TR>\n";
	}

	$gen .= "  <TR>\n" 
		    . '    ' . td ( [ $volname, $volstat ] )
		    . '    ' . td ( [ @slices ] ) 
		. "\n  </TR>\n";
    }

    $gen .= end_table . "\n";

    return $gen;
}

#  Externally exposed method to generate the HTML output
#
#  Usage: printHTMLOutput ($title, $ve_name [, $args [, $path] ])
#
#  Input Arguments:     $title          the title to use in the output page
#                       $t3_name        the name of the T3 instance
#                       $args           a list with the names of the components
#                                       can be omitted or an empty list
#                       $path           path with T3 data to parse
#
sub printHTMLOutput {

    my $class = shift if ( $_[0] eq $Name );

    return _printHTMLOutput ( @_ );
}

sub _printHTMLOutput {

    my ( $title, $t3_name, $args, $path ) = @_ if @_;

    #my $t3 = _getSystemData ( $t3_name, $passwd, $path );
    my $t3 = _getSystemData ( $t3_name, "", $path );

    my $out =  _getHTMLOutput ( $t3, $title, $t3_name, $args );

    print STDOUT $out;
}

#  Method to generate HTML output. See 'pod' documentation for more details
#
#  Externally exposed method to generate the HTML output
#
#  Usage: getHTMLOutput ($title, $ve_name [, $args [, $path] ])
#
#  Input Arguments:     $title          the title to use in the output page
#                       $ve_name        the name of the T3 instance
#                       $passwd         T3 passwd
#                       $args           a list with the names of the components
#                                       can be omitted or an empty list
#                       $path           path with T3 data to parse
#
sub getHTMLOutput {

    my $class = shift if ( $_[0] eq $Name );

    my ( $title, $t3_name, $passwd, $args, $path ) = @_ if @_;

    my $t3 = _getSystemData ( $t3_name, $passwd, $path );

    return _getHTMLOutput ( $t3, $title, $t3_name, $args );
}

#  Method to generate HTML output. See 'pod' documentation for more details
#
sub _getHTMLOutput {

    my ( $theT3, $title, $t3_name, $args ) = @_ if @_;

    my @args = ();

    if ( ! ( defined $args && $args ) ) {# only the T3 structure was passed

	#  set it as if all section have been requested
	#
	@args = sort keys %$theT3;
    }
    else {
	@args = @$args;
    }

    my $out = '';
    if ($main::opt_d) {

	$out = start_html ('StorEdge Device Manager - ' . $title) . "\n";
    }

    for my $section (@args) {

	my $refHash = $$theT3{$section};
	my $ref = $$refHash{ref};
	my $keys = $$refHash{keys};
	my $formalName = $$refHash{name};

	if ( $section eq '0ip_addr' ) {

	    $t3_name .= " ($$ref)";
	}

	elsif ( $section eq 'aversion' ) {

    	    $out .= h2 ( $formalName . ' - ' . $t3_name ) . "\n" .
		    '<blockquote><pre>' . $$ref . "</pre></blockquote>\n";
	}

	else {
	    if ( $section ne 'gfrustat' ) {
		$out .= doTableHeader ( $formalName, $t3_name );
	    }

	    if ( $section eq 'bsystem' ) {
		$out .= genHTMLH ( $ref );
	    }

	    elsif ( $section eq 'clunlist' ) {
		$out .= genHTMLHoH ( $ref, 'vol id' );
	    }

	    elsif ( $section eq 'dlunstat' ) {
		$out .= genHTMLSliceTable ( $ref, 'vol id' );
	    }

	    elsif ( $section eq 'eportlist' ) {
		$out .= genHTMLHoH ( $ref, 'port id' );
	    }

	    elsif ( $section eq 'fportmap' ) {
		$out .= genHTMLHoH ( $ref, 'port id' );
	    }

	    elsif ( $section eq 'gfrustat' ) {
		$out .= genHTMLFru ( $refHash, $t3_name );
	    }
	}
    }

    $out .= end_html . "\n" if ( $main::opt_d );

    return $out;
}

#  Method to generate an HTML table based on List of Hashes passed
#  as argument
#
sub genHTMLLoH {

    my ( $list, $keys ) = @_ if @_;

    my $gen = '';

    my @vals = ();

    #  generate the 'th' for the headers line
    #
    for my $key ( sort keys %$keys ) {

	push @vals, $$keys{$key};
    }

    $gen .= '  ' . Tr ( { "-align=>CENTER", "-valign=>TOP" },
	       [ th ( [ @vals ] ) ] ) . "\n" if @vals;

    for my $rec ( @$list ) {

	my @recvals = ();
	for my $k2 (sort keys %$keys) {

	    push @recvals, $$rec{$k2};
	}
	$gen .= '  ' . Tr ( [ td ( [ @recvals ] ) ] ) . "\n" if @recvals;
    }

    $gen .= end_table . "\n</center>\n";

    return $gen;
}

#  Generate HTML output for the FRUs
#
#  Input Argument:	the frustat entry on the T3 main hash
#
sub genHTMLFru {

    my ( $refHash, $t3_name ) = @_ if @_;

    my $formalName = $$refHash{name};

    my $out = '';

    for my $frutype ( sort keys %$refHash ) {

	if ( $frutype ne 'name' ) {
	    my $frec    = $$refHash{$frutype};
	    my $fruname = $$frec{name};
	    my $list    = $$frec{list};
	    my $keys    = $$frec{keys};

	    $out .= doTableHeader ( ' FRU Status - ' . $fruname, $t3_name );
	    $out .= genHTMLLoH ( $list, $keys );
	}
    }

    return $out;
}

1;
__END__

=head1 NAME

SE::T3 - Perl Module for T3 Management

=head1 SYNOPSIS

Functions and Data Structures for T3 Management on the Indy Platform.

=head1 DESCRIPTION

The C<T3> module supplies subroutines to be used by the storage management
applications developer to access the functionality necessary to manage
a T3 disk array.

=head1 OVERVIEW OF SUBROUTINES

The subroutines exposed address two main aspects, the first one is to
access the device and obtain the output generated, parsing it into a 
data structure that can be used.

The second aspect is the ability to expose the data structure parsed
into a presentation context. Currently HTML.

=head2 B<getSystemData ()>

Syntax:

	getSystemData ( \%valueHash, $t3_id );

The subroutine will access the T3 whose identity has been passed
as an argument.

A reference to a hash on which the results of the system data will be
returned is passed as the first argument.

The output generated will be included as a composite data structure
associated with specific entries in the F<%valueHash>.

Sections parsed include B<version>, B<system>, B<lunlist>, B<lunstat>,
B<portlist>, B<portmap>.

=head2 B<valueHash> Data Structure

The B<valueHash> data structure returned will have the following format.


    %valueHash = (
	version  => "T300 Release 1.14 2000/07/12 ... " ,
	system   => {
		      blocksize     => "64k",
		      cache	        => "writethrough",
		      cache memsize => "256 MBytes",
		      mirror        => "off",
		      mp_support    => "none",
		      rd_ahead      => "on",
		      recon_rate    => "med",
		      sys memsize   => "32 MBytes"
		    },
	lunlist  => {
		      "v0" => {
				  volume   => "v0" ,
				  capacity => "53.7 GB" ,
				  data     => "u1d1-4" ,
				  raid     => "5" ,
				  standby  => "u1d9"
			      }
		    },
	lunstat  => {
		      "v0" => {
				  volname  => "v0" ,
				  slices   => (
						 {
						    name    => "u1d1",
						    mounted => 0,
						 },
						 {
						    name    => "u1d2",
						    mounted => 0,
						 },
						 {
						    name    => "u1d3",
						    mounted => 0,
						 },
						 {
						    name    => "u1d4",
						    mounted => 0,
						 },
						 {
						    name    => "u1d9",
						    mounted => 0,
						 }
					      )
			      }
		    },
	portlist => { 
		      "u1p1" => {
				  port      => "u1p1"
				  addr_type => "hard",
				  host      => "sun",
				  status    => "online" ,
				  targetid  => "1",
				  wwn       => "50020f230000080e" ,
			      }
		    },
	portmap  => { 
		      "u1p1" => {
				  volume    => "v0" ,
				  access    => "primary",
				  addr_type => "hard",
				  lun       => "0" ,
				  owner     => "u1" ,
				  port      => "u1p1"
				  targetid  => "1",
			      }
		    }
	frustat  => {
		      "controller" => (
					{
					    name 	=> "u1ctr",
					    status	=> "ready",
					    state	=> "enabled",
					    role	=> "master",
					    partner	=> "u2ctr",
					    temp	=> "39.0"
					}, ...
				      ),
		      "disk" => ( 
				  {
				    name 	=> "u1d1",
				    status	=> "ready",
				    state	=> "enabled",
				    role	=> "master",
				    port1	=> "ready",
				    port2	=> "ready",
				    temp	=> "40.0",
				    volume	=> "vol2"
				  }, ...
				),
		      "loop" => ( 
				  {
				    name 	=> "u1d1",
				    status	=> "ready",
				    state	=> "enabled",
				    mode	=> "master",
				    cable1	=> "installed",
				    cable2	=> "-",
				    temp	=> "35.5"
				  }, ...
				),
		      "power" => ( 
				  {
				    name 	=> "u1pcu1",
				    status	=> "ready",
				    state	=> "enabled",
				    source	=> "line",
				    output	=> "normal",
				    battery	=> "normal",
				    fan1	=> "normal",
				    fan2	=> "normal"
				  }, ...
				),
		    }
		 };

=head2 B<getHTMLOutput ()>

Syntax:

	getHTMLOutput ( \%valueHash [$section [, $section1 [ ... ]]]);

The subroutine operates on the B<%valueHash> that was passed. It is expected
that the hash has been generated following a previous invocation to the
B<getSystemData()>.

If no $section arguments are passed, then all the T3 system map is
generated.

Valid sections include B<version>, B<system>, B<lunlist>, B<lunstat>,
B<portlist>, B<portmap>.

Output generated will be in HTML form.

=head1 AUTHOR

David Bautista-Lloyd, Arieh Markel

=cut

