#!/opt/cbe/SUPPORT-PKGS/perl/bin/perl -w
#
# SCCS Info:      pragma ident   "@(#)SW.pm 1.2     02/03/22 10:45:24 SMI"
#
#  Copyright (c) 2001 Sun Microsystems, Inc, All Rights Reserved.
#
#  Perl Module for FC Switch
#
#  This module serves as the main library utility for functions that
#  operate on the FC Switch for the Indy.
#
#  The module also supports functions that can parse the FC Switch output.
#
package SE::SW;

use Exporter;
use strict;

@SW::ISA	= qw(Exporter);
@SW::EXPORT	= qw( enumerate getSystemData getHTMLOutput printHTMLOutput
		      get_entry 
		      $PORTS $VERSION $NAMESERVER $SWINFO
		      $WWNZONE $NSZONE $BCASTZONE );

use FileHandle;
use CGI qw/:standard *table/;		# load standard CGI routines
use Getopt::Long;
use CGI::Carp;
#use SE::Util;

my ( $IndyPkg ) = qw( /opt/SUNWsecfg/ );
my ( $BinDir )  = qw( bin/ );
my $Exec        = 'showswitch';
my $Enum        = 'listavailable';
my $Name        = 'SE::SW';

 #*SWINFO	= \'SWINFO';		# constants define
 #*PORTS	= \'PORT';		# constants define
 #*VERSION	= \'VERSION';		# constants define
 #*NAMESERVER	= \'NAMESERVER';	# constants define
 #*WWNZONE	= \'WWNZONE';		# constants define
 #*NSZONE	= \'NSZONE';		# constants define
 #*BCASTZONE	= \'BCASTZONE';		# constants define

my $lastSW 	= ();


#  Return a FileHandle to a connection with the FC Switch
#
#  The filehandle will be read subsequently and parsed
#
sub getConnectionCmd {

    #@_ >= 1 or croak "usage: ${Name}::getConnection ( sw_id [, datafile])";
    my ( $sw, $path ) = @_ if @_;

    my $execPath = $IndyPkg . $BinDir . $Exec;

    return "$execPath -s $sw";
}

#  Return the list of switches available
#
sub enumerate {

    my $class = shift if ( $_[0] eq $Name );

    return _enumerate ( @_ );
}

#  Inner method to execute the enumeration
#
sub _enumerate {

    my ( $who, $sw_name, $sw, $path ) = @_ if @_;

    my @retval = ();

    if ( ! ( defined $who && $who ) ) {

	my $fp = new FileHandle;
	my $execPath = $IndyPkg . $BinDir . $Enum;

	if ( -x $execPath) {
	    $fp->open ("$execPath -s |") 
		or croak "${Name}::enumerate: Unable to execute $execPath $!\n";
	}
	while ( <$fp> ) {
	    push @retval, $_;
	}
	$fp->close;

	@retval = ( 'sw1a', 'sw1b', 'sw2a', 'sw2b' ) unless @retval;
    }

    else {
	
	if ( ! ( defined $sw && $sw ) ) {
	    $sw = getSystemData ( $sw_name, $path );
	}

	if ( $who eq 'PORTS' ) {
	    @retval = getNamesByKey ( $sw, 'cports', 'aname' );
	}

	elsif ( $who eq 'NAMESERVER' ) { 
	    @retval = getNamesByKey ( $sw, 'dnameserver', 'aname' );
	}

	elsif ( $who eq 'ZONES' ) { 
	    @retval = getNamesByKey ( $sw, 'ezones' );
	}
    }

    return @retval;
}

#  Internal subroutine to return the list of names associated with the
#  key passed as an argument
#
sub getNamesByKey {

    my ( $theSW, $sw_key, $rec_key ) = @_ if @_;

    my @allrecs = $$theSW{$sw_key};

    my @retval = ();

    for my $rec ( @allrecs )
    {
	if ( defined $rec_key ) {
	    my $val = $$rec{$rec_key};
	    push @retval, $val if $val;
	}
	else {
	    for my $key ( sort keys %$rec ) {
		push @retval, $key;
	    }
	}
    }

    return @retval;
}

#  Return a list of all the records whose field '$key' matches the '$value'
#  passed as argument
#
#  Input Arguments:	$who	the type of information from the SW
#				$PORTS, $NAMESERVER, $ZONES
#			$sw_name  the name of the FC Switch
#			$key	the key to select the record to compare
#				the value
#			$value	the value to compare
#			$sw	the structure for the SW, a result of
#				a previous invocation to getSystemData.
#
#  Return Value:	a list of references to records that match 
#			(i.e. that $rec{$key} eq $value ).
#
sub get_entry {

    my $class = shift if ( $_[0] eq $Name );

    return _get_entry ( @_ );
}

#  Internal method to do the job
#
sub _get_entry {

    my ( $who, $sw_name, $key, $value, $sw ) = @_ if @_;

    my @retval = ();

    if ( ! ( defined $sw && $sw ) ) {
	$sw = _getSystemData ( $sw_name );
    }

    if ( $who eq 'SWINFO' ) {
	@retval = getRecordByKey ( $sw, 'aswinfo', $key, $value);
    }
    elsif ( $who eq 'PORTS' ) {
	@retval = getRecordByKey ( $sw, 'cports', $key, $value );
    }
    elsif ( $who eq 'NAMESERVER' ) { 
	@retval = getRecordByKey ( $sw, 'dnameserver', $key, $value );
    }
    elsif ( $who eq 'ZONES' ) { 
	@retval = getRecordByKey ( $sw, 'ezones', $key, $value );
    }
    return @retval;
}

#  Internal subroutine to return the list of names associated with the
#  key passed as an argument
#
sub getRecordByKey {

    my ( $theSW, $sw_key, $rec_key, $value ) = @_ if @_;

    my $hash = $$theSW{$sw_key};

    my $allrecs = $$hash{ref};

    my @retval = ();

	push @retval, $$allrecs{$rec_key};

    # The following commented out code generates a "Not an ARRAY
    # reference at ...<the line below>" error. 
    #jw for my $rec ( @$allrecs )
    #jw{
	#jwmy $val = $$rec{$rec_key};

	#jwif ( $val ) {
	#jw    push @retval, $rec if ( $value eq $val );
	#jw}
    #jw}
    return @retval;
}

# 
#  Obtain FC Switch System Info, assemble into a list of Hashes to be returned
#
#  All those that have a match are returned.
#
#  The function expects a reference to appropriate lists.
#
#  Input Arguments:	$sw_name	the name of the FC Switch
#
#  Output Arguments:	
#
#  Return Value:	a reference to a composite data structure that
#			contains the parsed information.
#
sub getSystemData {
 
    my $class = shift if ( $_[0] eq $Name );

    return _getSystemData ( @_ );
}

#  Inner method to perform the system data parsing
#
sub _getSystemData {

    my ( $sw_name, $path ) = @_ if @_;

    my $section	= '';

    my $version = '';

    #  theSW is the composite object that exposes the whole ve.
    #
    #  fields:	version system lunlist lunstat portlist portmap
    #
    my %theSW = ();

    $lastSW = \%theSW;

    #  sw_info if a Hash that contains the basic properties for the switch
    #  whose name is passed as argument
    #
    #  The following are the fields that make up the entry:
    #
    #   fields:		name nports ip_addr
    #
    my %sw_info = ();
    my %sw_infokeys = (
			 aname  		=> 'name',
			 bnports		=> 'no ports',
			 cip_addr		=> 'IP Address'
		      );

    #  switch version info is a simple list of name/value pairs
    #
    #  fields:		HW
    #
    my @sw_verinfo = ();

    #  ports is a list of Hashes, one per port
    #
    #  fields:		no type adm_state op_state status loop_mode
    #
    my @ports = ();
    my %portkeys = (
		     anumber  		=> 'Port #',
		     btype		=> 'Type',
		     cadm_state		=> 'Admin State',
		     dop_state		=> 'Oper State',
		     estatus		=> 'Status',
		     fmode		=> 'Loop Mode'
		   );

    #  nameserver is a List of Hashes
    #
    #  fields:		port  address type portwwn nodewwn fc-4type
    #
    my @nameserver = ();
    my %nskeys = (
		     aport  		=> 'Port',
		     baddress		=> 'Address',
		     ctype		=> 'Type',
		     dportwwn		=> 'Port WWN',
		     enodewwn		=> 'Node WWN',
		     ffc_4		=> 'FC-4 Types'
		   );

    #  zones is a Hash that maintains the list of zones for each type
    #
    #  keys:		wwnzone nszone bcastzone hardzone
    #
    #  values:		aname, benabled, cports
    #
    my %zones = (
		     awwnzone		=> 0,
		     bnszone		=> 0,
		     cbcastzone		=> 0,
		     dhardzone		=> 0,
		     eslzone		=> 0
		   );
    my %zonekeys = (
		     awwnzone		=> 'World-wide Name Zone',
		     bnszone		=> 'NameServer Zone',
		     cbcastzone		=> 'Broadcast Zone',
		     dhardzone		=> 'Hard Zone',
		     eslzone		=> 'SL Zone'
		   );

    my %perzonekeys = (
			aname		=> 'Id',
			benabled	=> 'Enabled',
			cports		=> 'Ports'
		      );

    my $devType = 'FC Switch SubSystem';

    #
    #  Initialize first the SW structure to point at the inner Structs
    #  fields:	version system lunlist lunstat portlist portmap
    #
    $theSW{aswinfo} 	= {
			     name => $devType . ' Switch Information',
			     ref  => \%sw_info,
			     keys => \%sw_infokeys
			  };
    $theSW{bsw_verinfo} = {
			     name => $devType . ' Version Information',
			     ref  => \@sw_verinfo,
			     keys => ''
			  };
    $theSW{cports} 	= {
			     name => $devType . ' Port Information',
			     ref  => \@ports,
			     keys => \%portkeys
			  };
    $theSW{dnameserver} = {
			     name => $devType . ' Nameserver Information',
			     ref  => \@nameserver,
			     keys => \%nskeys
			  };
    $theSW{ezones} 	= {
			     name => $devType . ' Zone Information',
			     ref  => \%zones,
			     keys => \%zonekeys
			  };

    my $cmd = getConnectionCmd ( $sw_name, $path );

    #my ( $ret, $stdout, $stderr ) = SE::Util->execCmd ( $cmd, "0" );
    my ( $ret, $stdout, $stderr ) = execCmd ( $cmd, "0" );

    if ( $ret )
    {
	SE::Util->printCmdAll ( $cmd, $ret, $stdout, $stderr );
	return;
    }
    
    #  Iterate over the command, and process line by line
    #
    my @lines = split '\n', $stdout;
    for $_ (@lines) {
	next if (/^\s*#/o);		# skip comments
	next if (/^\s*$/o);		# skip empty lines
	next if (/^\*\*\*\*/o);		# skip section separators
	next if (/^----/o);		# skip title separators

	chomp;

	if (/^\s*SWITCH:\s*(\S+)\s*$/o)	# if a header for a section
	{
	    $sw_info{aname} = $1;
	    $section = 'SWINFO';
	}
	elsif (/^\s*Version Information\s*$/o)
	{
	    $section = 'VERSION';		# the section
	}
	elsif (/^\s*Port Status\s*$/o)
	{
	    $section = 'PORTS';			# the section
	}
	elsif (/^\s*Name Server\s*$/o)
	{
	    $section = 'NAMESERVER';		# the section
	}
	elsif (/^\s*World-wide Name Zone\s*$/o)
	{
	    $section = 'awwnzone';		# the section
	    my @zonelist = ();
	    $zones{awwnzone} = {
				 name	=> 'World Wide Name Zone',
				 ref	=> \@zonelist, 
				 keys	=> \%perzonekeys
			       };	
	}
	elsif (/^\s*NameServer Zone\s*$/o)
	{
	    $section = 'bnszone';		# the section
	    my @zonelist = ();
	    $zones{bnszone} = { 
				 name	=> 'Nameserver Zone',
				 ref	=> \@zonelist, 
				 keys	=> \%perzonekeys
			       };	
	}
	elsif (/^\s*Broadcast Zone\s*$/o)
	{
	    $section = 'cbcastzone';		# the section
	    my @zonelist = ();
	    $zones{cbcastzone} = {
				 name	=> 'Broadcast Zone',
				 ref	=> \@zonelist, 
				 keys	=> \%perzonekeys
			      };	
	}
	elsif (/^\s*Hard Zone\s*$/o)
	{
	    $section = 'dhardzone';		# the section
	    my @zonelist = ();
	    $zones{dhardzone} = {
				  name	=> 'Hard Zone',
				  ref	=> \@zonelist, 
				  keys	=> \%perzonekeys
			        };	
	}
	elsif (/^\s*SL Zone\s*$/o)
	{
	    $section = 'eslzone';		# the section
	    my @zonelist = ();
	    $zones{eslzone} = {
				 name	=> 'SL Zone',
				 ref	=> \@zonelist, 
				 keys	=> \%perzonekeys
			      };	
	}

	else {
	    s/^\s+//o;				# remove leading spaces

	    if ( $section eq 'SWINFO' ) {

		if ( /Number of Ports:\s*(\S+)/o ) {
		    $sw_info{bnports} = $1;
		}
		elsif ( /IP Address:\s*(\S+)/o ) {
		    $sw_info{cip_addr} = $1;
		}
		elsif ( /\s*:\s*/o ) {
		    $sw_info{$`} = ( $' ) ? $' : '-';
		}
	    }

	    elsif ( $section eq 'VERSION' ) {

		if ( /\s*:\s*/o ) {
		    my $rec = {};

		    my $key = $`;
		    my $val = $';
		    $rec->{$key} = $val;

		    push @sw_verinfo, $rec;
		}
	    }

	    elsif ( $section eq 'PORTS' ) {
		next if (/Port #\s+Port Type\s+Admin State\s+Oper State\s/o);
	
		my $rec = {};
		( $rec->{anumber}, $rec->{btype}, $rec->{cadm_state}, 
		  $rec->{dop_state}, $rec->{estatus} ) =
			/(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s*/o;
		$rec->{fmode} = ( $' ) ? $' : '-';

		#  insert the record to the @diskpool list
		#
		push @ports, $rec if $rec;
	    }

	    elsif ( $section eq 'NAMESERVER' ) {
		next if (/Port\s+Address\s+Type\s+PortWWN\s+Node WWN/o);

		my $rec = {};
		( $rec->{aport}, $rec->{baddress}, $rec->{ctype},
		  $rec->{dportwwn}, $rec->{enodewwn} ) =
			    /(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s*/o;
		$rec->{ffc_4} = ( $' ) ? $' : '-';

		#  insert the record to the @diskpool list
		#
		push @nameserver, $rec if $rec;
	    }

	    elsif ( $section eq 'awwnzone'   ||
		    $section eq 'bnszone'    ||
		    $section eq 'cbcastzone' ||
		    $section eq 'dhardzone'  ||
		    $section eq 'eslzone' ) {

		my $list = $zones{$section}{ref};

		if ( /Zone\s+total\s*:\s*(\S+)/o )
		{
		    my $rec = {};
		    $rec->{aname} = '-';
		    $rec->{benabled} = '-';
		    $rec->{cports} = '-';

		    push @$list, $rec if $rec;
		}
		elsif ( /Zone:\s*(\S+),\s*Enabled:\s*(\S+)/o )
		{
		    my $rec = {};
		    $rec->{aname} = $1;
		    $rec->{benabled} = $2;
		    $rec->{cports} = '';
		    push @$list, $rec if $rec;
		}
		elsif ( /Port:\s*(\S+)/o )
		{
		    my $rec;
		    for my $elem ( @$list )
		    {
			$rec = $elem;
		    }
		    $rec->{cports} .= " $1";
		}
	    }
	}
    }

    return \%theSW;
}

sub usage {

    print ":INFO: [-f file] [-d]";

}

#  Internal method to generate the table header
#
sub doTableHeader {

    my ( $title, $devId ) = @_ if @_;

    return h2 ( $title . ' - ' . $devId ) . "\n<center>\n" .
           start_table ( { -border=>0, -width=>"90%", -cellspacing=>2,
			   -bgcolor=>"white", -cellpadding=>8 },
			   caption ( $title ) ) . "\n";
}

#  Method to generate an HTML table based on a single keyed Hash passed
#  as argument
#
sub genHTMLH {

    my ( $hash, $keys, $keytitle, $valtitle ) = @_ if @_;

    $keytitle = 'property' unless ( defined $keytitle );
    $valtitle = 'value'    unless ( defined $valtitle );

    my $gen = Tr ( { "-align=>CENTER", "-valign=>TOP" },
		    [ th ( [ $keytitle, $valtitle ] ) ]
		    ) . "\n";

    for my $key ( sort keys %$keys ) {

	$gen .=  '  ' . Tr ( [ td ( [ $$keys{$key}, $$hash{$key} ] ) ] ) . "\n";
    }

    $gen .= end_table . "\n</center>\n";

    return $gen;
}

#  Method to generate an HTML table based on a single keyed Hash passed
#  as argument
#
sub genHTMLLoSH {

    my ( $list, $keys, $keytitle, $valtitle ) = @_ if @_;

    $keytitle = 'property' unless ( defined $keytitle );
    $valtitle = 'value'    unless ( defined $valtitle );

    my $gen = Tr ( { "-align=>CENTER", "-valign=>TOP" },
		    [ th ( [ $keytitle, $valtitle ] ) ]
		    ) . "\n";

    for my $elem ( @$list ) {

	for my $key ( sort keys %$elem ) {

	    $gen .=  '  ' . Tr ( [ td ( [ $key, $$elem{$key} ] ) ] ) . "\n";
	}
    }

    $gen .= end_table . "\n</center>\n";

    return $gen;
}

#  Method to generate an HTML table based on List of Hashes passed
#  as argument
#
sub genHTMLLoH {

    my ( $list, $keys ) = @_ if @_;

    my $gen = '';

    my @vals = ();

    #  generate the 'th' for the headers line
    #
    for my $key ( sort keys %$keys ) {

	push @vals, $$keys{$key};
    }

    $gen .= '  ' . Tr ( { "-align=>CENTER", "-valign=>TOP" },
	       [ th ( [ @vals ] ) ] ) . "\n" if @vals;

    for my $rec ( @$list ) {

	my @recvals = ();
	for my $k2 (sort keys %$keys) {

	    push @recvals, $$rec{$k2};
	}
	$gen .= '  ' . Tr ( [ td ( [ @recvals ] ) ] ) . "\n" if @recvals;
    }

    $gen .= end_table . "\n</center>\n";

    return $gen;
}

#  Method to generate an HTML table based on Hash of Lists passed
#  as argument
#
sub genHTMLHoL {

    my ( $list, $keys, $devId ) = @_ if @_;

    my $gen = '';

    #  generate the 'th' for the headers line
    #
    for my $key ( sort keys %$keys ) {

	my $refHash = $$list{$key};
	my $ref = $$refHash{ref};
	my $keys = $$refHash{keys};
	my $formalName = $$refHash{name};

	$gen .= doTableHeader ( $formalName, $devId );

	$gen .= genHTMLLoH ( $ref, $keys );
    }

    return $gen;
}

#  Externally exposed method to generate the HTML output
#
sub printHTMLOutput {

    my $class = shift if ( $_[0] eq $Name );

    return _printHTMLOutput ( @_ );
}

#  Internal method to generate the HTML output
#
sub _printHTMLOutput {

    my ( $title, $sw_name, $args, $path ) = @_ if @_;

    my $ve = _getSystemData ( $sw_name, $args );

    my $out =  _getHTMLOutput ( $ve, $title, $sw_name, $args );

    print STDOUT $out;
}

#  Method to generate HTML output. See 'pod' documentation for more details
#
#  Externally exposed method to generate the HTML output
#
sub getHTMLOutput {

    my $class = shift if ( $_[0] eq $Name );

    my ( $title, $sw_name, $args, $path ) = @_ if @_;

    my $ve = _getSystemData ( $sw_name, $path );

    return _getHTMLOutput ( $ve, $title, $sw_name, $args );
}

#  Method to generate HTML output. See 'pod' documentation for more details
#
sub _getHTMLOutput {

    my ( $theSW, $title, $sw_name, $args ) = @_ if @_;

    my @args = ();

    if ( ! ( defined $args && $args) ) {# only the SW structure was passed

	#  set it as if all section have been requested
	#
	@args = sort keys %$theSW;
    }
    else {
	@args = @$args;
    }

    my $out = '';
    if ($main::opt_d) {

	$out = start_html ('StorEdge Device Manager - ' . $title) . "\n";
    }


    for my $section (@args) {

	my $refHash = $$theSW{$section};
	my $ref = $$refHash{ref};
	my $keys = $$refHash{keys};
	my $formalName = $$refHash{name};

	if ( $section ne 'ezones' ) {
	    $out .= doTableHeader ( $formalName, $sw_name );
	}

	if ( $section eq 'aswinfo' ) {
	    $out .= genHTMLH ( $ref, $keys );
	}

	elsif ( $section eq 'bsw_verinfo' ) {
	    $out .= genHTMLLoSH ( $ref, $keys );
	}

	elsif ( $section eq 'cports' ) {
	    $out .= genHTMLLoH ( $ref, $keys );
	}

	elsif ( $section eq 'dnameserver' ) {
	    $out .= genHTMLLoH ( $ref, $keys );
	}

	elsif ( $section eq 'ezones' ) {
	    $out .= genHTMLHoL ( $ref, $keys, $sw_name );
	}
    }


    $out .= end_html . "\n" if ( $main::opt_d );

    return $out;
}

1;

__END__

=head1 NAME

SE6990::SW - Perl Module for FC Switch Management

=head1 SYNOPSIS

Functions and Data Structures for FC Switch Management on the 
Indy Platform.

=head1 DESCRIPTION

The C<SW> module supplies subroutines to be used by the storage management
applications developer to access the functionality necessary to manage
a FC Switch on the Indy.

=head1 OVERVIEW OF SUBROUTINES

The subroutines exposed address two main aspects, the first one is to
access the device and obtain the output generated, parsing it into a 
data structure that can be used.

The second aspect is the ability to expose the data structure parsed
into a presentation context. Currently HTML.

=head2 B<getSystemData ()>

Syntax:

	$valueHash = getSystemData ( $sw_id );

The subroutine will access the Switch whose identity has been passed
as an argument.

A reference to a hash on which the results of the system data will be
returned is passed as the first argument.

The output generated will be included as a composite data structure
associated with specific entries in the F<%valueHash>.

Sections parsed include B<version>, B<system>, B<lunlist>, B<lunstat>,
B<portlist>, B<portmap>.

=head2 B<valueHash> Data Structure

The B<valueHash> data structure returned will have the following format.


    %valueHash = (
	swinfo  => {
			aname		=> "sw1a" ,
			bnports		=> "16" ,
			cip_addr	=> "192.168.0.30'
		    },

	sw_verinfo => (
			HW    	=> "e06" ,
			PROM    => "40500" ,
			FLASH	=> "30462" ,
			CHASSIS => "A16",
			NUMBER  => "1",
			fabric  => "1",
			wwn  	=> "10000c0dd00aec6",
			mac  	=> "00c0dd00aec5"
		       ),

	ports => ( 
		   {
		      port    	=> "1",
		      type	=> "F_port",
		      adm_state	=> "online",
		      op_state	=> "offline",
		      status	=> "logged-in" ,
		      mode	=> 
		    },
		 ),

	nameserver => ( 
		       {
			  port    	=> "00" ,
			  address   	=> "104000"
			  type		=> "N",
			  portwwn	=> "2a0006022004185",
			  nodewwn	=> "20000e08b04e70f",
			  fc_4		=> "SCSI_FCP"
		       },
		     ),
	zones => {
		    wwnzone => 0,
		    nszone => 0,
		    bcastzone => 0,
		    hardzone => 0
		 }
	       );

=head2 B<getHTMLOutput ()>

Syntax:

	getHTMLOutput ( \%valueHash [$section [, $section1 [ ... ]]]);

The subroutine operates on the B<%valueHash> that was passed. It is expected
that the hash has been generated following a previous invocation to the
B<getSystemData()>.

If no $section arguments are passed, then all the Switch system map is
generated.

Valid sections include B<version>, B<system>, B<lunlist>, B<lunstat>,
B<portlist>, B<portmap>.

Output generated will be in HTML form.

=head1 AUTHOR

David Bautista-Lloyd, Arieh Markel

=cut

