#! /bin/ksh

#
# Copyright (c) 2000 by Sun Microsystems, Inc.
#

#
# @(#)dsmrmadd.sh 1.9 00/04/18 SMI
#

#
# Invocation: dsmrmadd.sh [-i <interval>];
#

#
# Script which is invoked when the disk failure detection is manual, but the
# initiation of the compensating action, and subsequent re-integration of the
# replaced disk is meant to be automatic. Typical scenario is: this script is
# being run in the background or perhaps as a daemon, after a period of time
# sys admin detects disk failure, removes the failed disk and replaces
# it. When the new disk is detected by this script it initiates compensating
# actions and subsequent re-integration of the replacement disk.
#

. dsmcmn.sh		# include the file which contains utility functions.


# ********************** FUNCTION DEFINITIONS *******************************

#
# Purpose:	prints out usage for this script
# Invocation:	print_usage;
# Assumptions:	none
# Returns:	Nothing
#
print_usage()
{
	echo `gettext $DSM_DOMAIN 'Usage:'`;
	echo "\t$SCRIPT `gettext $DSM_DOMAIN '[-i <interval>]'`";
	
}


# ************************* SCRIPT START **********************************

_interval=300;


#
# Check the cur dir, and cd to the bdm dir if cur dir is not the bdm dir
#
check_cur_dir;


#
# Check that the md.cf and disk.cf files have been created and are not empty
#
check_config_files;


#
# check number of option + arg passed
#
if (( "$#" != 0 && "$#" != 2 ))
then
	Echo "err" "${SCRIPT}: `gettext $DSM_DOMAIN 'Usage Error.'`";
	print_usage;
	exit 1;
fi


#
# check the option and arg passed
#
if (( "$#" == 2 ))
then
	while getopts :i: option
	do
		case "$option" in
		  i) _interval=$OPTARG;
		     if [[ "$_interval" != +([0-9]) ]]
		     then
				Echo "err" "${SCRIPT}: `gettext $DSM_DOMAIN 'Usage Error.'`";
				print_usage;
				exit 1;
		     fi;;
		  \?)   Echo "err" "${SCRIPT}: `gettext $DSM_DOMAIN 'Usage Error.'`";
			print_usage;
			exit 1;
		esac
	done
fi


#
# Will henceforth execute in a perpetual loop
#
debug_echo "${SCRIPT}: DEBUG: Starting the perpetual loop";

while true			# start
do
	debug_echo "${SCRIPT}: DEBUG: `date` Sleeping for $_interval seconds";

	sleep "$_interval";
  
	cat ${DSM_CFG_DIR}/disk.cf |
	while read _line
	do
		_disk=`echo "$_line" | cut -d' ' -f1`;
		_old_serialnum=`echo "$_line" | cut -d' ' -f3`;
    
		check_new_disk $_disk $_old_serialnum;
		_exit_stat="$?";
		if (( _exit_stat == 0 ))
		then
			debug_echo "${SCRIPT}: DEBUG: new disk - serial number\
 $_new_serialnum";
			#
			# Initiate compensating actions only if the following succeeds
			#
			enter_compensating_actions;
			_exit_stat="$?";
			if (( _exit_stat == 0 ))
			then
				dsmrm.sh -d $_disk;
				_exit_stat="$?";
				if (( _exit_stat != 0 ))
				then
					Echo "err" \
"${SCRIPT}: ${_disk}: `gettext $DSM_DOMAIN 'Failure in executing compensating actions. Run dsmrm and dsmadd manually.'`";
					exit 1;
				fi
			fi

			# bugID 4469335: tie status loop interval in dsmadd to the 
			# interval defined here.
			SB_DSMADD_INTERVAL="$_interval" dsmadd.sh -n;
			_exit_stat="$?";
			if (( _exit_stat == 1 || _exit_stat == 2 ))
			then
				Echo "err" \
"${SCRIPT}: ${_disk}: `gettext $DSM_DOMAIN 'Failure in executing reintegrating actions. Run dsmadd manually.'`";
				exit 1;
			else
				continue;
			fi
		fi
	done
done

