#! /bin/ksh

#
# Copyright (c) 2000 by Sun Microsystems, Inc.
#

#
# @(#)dsmadd.sh 1.13 00/06/08 SMI
#

#
# Invocation: dsmadd.sh
#
# Script which re-creates the replicas and submirrors, re-integrates them
# with the other disk, and re-inserts this disk in the boot sequence.
#
# exit codes:-
# 0		success.
# 1		deleted_metadevices empty or non-existent.
# 2		dsmadd.sh invoked when compensating actions have not yet been 
#		completed.
# 3		failure to put VTOC on the replacement disk.
# 4		All replicas/submirrors have not been reintegrated.
# 5		All replicas/submirrors reintegrated, but a maintenance
#		condition found, therefore, reintegrating actions not completed
#

. dsmcmn.sh		# include the file which contains utility functions.

# bugID 4469335
# check for interval definition passed to dsmadd upon invocation. If not set
# set it to the default defined in dsmcmn.sh
if [ -z "$SB_DSMADD_INTERVAL" ]
then
	SB_DSMADD_INTERVAL="$_interval";
fi

# ********************** FUNCTION DEFINITIONS *******************************

#
# Purpose:	Insert this disk (alias) in the boot-device variable in
#		NVRAM.
# Invocation:	insert_boot_device <replaced disk>
#
# Assumption:	disk.cf contains correct boot disk alias.
# Returns:	Nothing.
#
insert_boot_device()
{
	typeset _fd _fd_alias _sd_alias _boot_device _boot_device_list;
	typeset _new_boot_device_list;
	
	_fd="$1";
	
	_boot_device_list=`eeprom boot-device | cut -d'=' -f2`;

	_fd_alias=`grep $_fd ${DSM_CFG_DIR}/disk.cf | cut -d' ' -f2`;
	_sd_alias=`grep -v $_fd ${DSM_CFG_DIR}/disk.cf | cut -d' ' -f2`;

	#
	# Insert only if the disk alias is not present
	#
	if (( `echo $_boot_device_list | grep -c $_fd_alias` == 0 ))
	then
		_new_boot_device_list="";

		for _boot_device in $_boot_device_list
		do
			_new_boot_device_list="${_new_boot_device_list}$_boot_device ";
			if [[ $_boot_device = $_sd_alias ]]
			then
				_new_boot_device_list="${_new_boot_device_list}$_fd_alias ";
			fi
		done

		eeprom "boot-device=$_new_boot_device_list";
	fi
}


# ************************ SCRIPT START ***********************************

#
# Check the cur dir, and cd to the dsm dir if cur dir is not the dsm dir
#
check_cur_dir;

#
# Check that the md.cf and disk.cf files have been created and are not empty
#
check_config_files;


#
# check the option and arg passed
#  n is a hidden argument to suppress a notice
#    it is used when dsmadd.sh is called from other scripts that have verified
#    that the same disk is not being added.
#

_suppress_flag=0;    #default value

while getopts :n option
do
        case "$option" in
          n) 
             # Suppress notice
             _suppress_flag=1;
        esac
done

if (( _suppress_flag == 0 ))
then

           Echo "notice" "${SCRIPT}: `gettext $DSM_DOMAIN \
'Reintegration will be attempted on failed disk if it is reinserted.'`";
fi

#
# check that deleted_metadevices file exists and is not empty
#
if [[ ! -s ${DSM_LOG_DIR}/deleted_metadevices ]]
then
	Echo "err" "${SCRIPT}: `gettext $DSM_DOMAIN \"deleted_metadevices file either doesn't exist or is empty.\"`";
	exit 1;
fi


#
# Extract the disk value from the saved file
#
_disk=`sed '1d' ${DSM_LOG_DIR}/deleted_metadevices | cut -d' ' -f6 | \
       sed 's/..$//' | uniq`;


#
# Check that this script is invoked only when compensating actions have been done
#
if [[ -s ${DSM_LOG_DIR}/dsmlog ]] && \
    (( `grep -c "$_disk: $_compensating_actions_completed" \
	${DSM_LOG_DIR}/dsmlog` == 0 ))
then
	Echo "err" "${SCRIPT}: ${_disk}: `gettext $DSM_DOMAIN 'Compensating Actions have not been completed yet. Run dsmrm.'`";
	exit 2;
fi


# XXX: Correct location for this? or after fmthard below?
cp ${DSM_LOG_DIR}/dsmlog ${DSM_LOG_DIR}/dsmlog_tmp;
echo "`date '+%m/%d/%Y %H:%M:%S'` $_disk: $_reintegrating_actions_initiated" \
    >> ${DSM_LOG_DIR}/dsmlog_tmp;
mv ${DSM_LOG_DIR}/dsmlog_tmp ${DSM_LOG_DIR}/dsmlog;


Echo "notice" "${SCRIPT}: ${_disk}: `gettext $DSM_DOMAIN 'Reintegrating actions initiated.'`";


#
# partition the new disk same as the old disk, using the vtoc information from
# the second disk. 
#
_other_disk=`grep -v $_disk ${DSM_CFG_DIR}/disk.cf | cut -d' ' -f1`;
prtvtoc /dev/rdsk/"$_other_disk"s2 | fmthard -s - /dev/rdsk/"$_disk"s2;
_exit_stat="$?";
if (( _exit_stat != 0 ))
then
	Echo "crit" "${SCRIPT}: ${_disk}: `gettext $DSM_DOMAIN 'Failed to put the VTOC. Please insert disk of correct size/geometry.'`";
	exit 3;
fi


#
# get Resyncing strings from SDS's msg files.
#
# bugID:4343375
# Since the SDS localized msgs are located in the default /usr/lib/locale and
# the bdm messages are in the L10N bdm directory, the TEXTDOMAINDIR environment
# variable needs to be set to default before using gettext on SUNW_MD, but
# changed back to the bdm L10N directory.
TEXTDOMAINDIR=/usr/lib/locale;

_resyncing=`gettext SUNW_MD "Resyncing"`;

TEXTDOMAINDIR=${L10N_DSM_DIR};


#if [[ -s ${DSM_LOG_DIR}/dsmlog ]] && \
#   (( `grep -c "$_disk: $_reintegrating_actions_initiated" ${DSM_LOG_DIR}/dsmlog` == 1 ))
#then
#	cp ${DSM_LOG_DIR}/deleted_metadevices ${DSM_LOG_DIR}/deleted_metadevices_tmp;
#fi

cp ${DSM_LOG_DIR}/deleted_metadevices ${DSM_LOG_DIR}/deleted_metadevices_tmp;


#
# Reading the config info from the deleted_metadevices file, add state
# database replicas on the new disk. Then (and then only) create and attach
# the submirrors (RFE (bugid) 4297612). Resync with the existing databases and
# mirrors is automatic.
#
while (( `wc -l ${DSM_LOG_DIR}/deleted_metadevices_tmp | sed 's/  */ /g' | \
	  cut -d' ' -f2` > 0 ))
do
	_line=`sed -n '1p' ${DSM_LOG_DIR}/deleted_metadevices_tmp`;
	if [ `echo $_line | cut -d' ' -f2` = "-m" ]
	then		
		metainit `echo $_line | cut -d' ' -f3-`;
		_mirror=`echo $_line | cut -d' ' -f1`;
		_submirror=`echo $_line | cut -d' ' -f3`;
		metattach "$_mirror" "$_submirror";
		#
		# Until the mirror has resynced don't recreate the next
		# submirror (RFE (bugid) 4297612)
		#
		while (( `metastat $_mirror | grep -ci "$_resyncing"` > 0 ))
		do
			sleep $SB_DSMADD_INTERVAL;	# BugID 4469335
		done
	else
		metadb -a -c $_line;
	fi
	sed '1d' ${DSM_LOG_DIR}/deleted_metadevices_tmp | \
	tee ${DSM_LOG_DIR}/deleted_metadevices_tmp > /dev/null;
done


#
# check that the submirrors and the state database replicas have indeed been
# created
#
check_stat ${DSM_LOG_DIR}/deleted_metadevices created;
_exit_stat="$?";
if (( _exit_stat != 0 ))
then
	Echo "err" "${SCRIPT}: ${_disk}: `gettext $DSM_DOMAIN 'Failure to create the submirrors and/or state database replicas.'`";
	exit 4;
fi


#
# Now check the health of these replicas and submirrors and then only continue
# with the remainder of reintegrating actions
#
check_sdr $_disk;
_es1="$?";
check_sm $_disk;
_es2="$?";
if (( _es1 != 0 || _es2 != 0 ))
then
    Echo "crit" "${SCRIPT}: ${_disk}: `gettext $DSM_DOMAIN 'Disk failure detected.'`";
    
    #
    # "reintegrating actions not completed" is logged to indicate that all the
    # required replicas and submirrors were recreated and reintegrated BUT one
    # or some of them have gone into maintenance/last erred state.
    #
    cp ${DSM_LOG_DIR}/dsmlog ${DSM_LOG_DIR}/dsmlog_tmp;
    echo "`date '+%m/%d/%Y %H:%M:%S'` \
$_disk: $_reintegrating_actions_not_completed" >> ${DSM_LOG_DIR}/dsmlog_tmp;
    mv ${DSM_LOG_DIR}/dsmlog_tmp ${DSM_LOG_DIR}/dsmlog;

    #
    # Serial number is not updated to facilitate the next round of fault
    # finding. (This disk will be considered a brand new disk). The boot-device
    # list is not modified as well.
    # Note that deleted_metadevices is not deleted so as to help in later
    # recovery process.
    #
    exit 5;
fi


#
# Its now safe to update the serial number in the disk.cf file
#
update_disk_config $_disk ${DSM_CFG_DIR}/disk.cf;


#
# (Re)insert this disk alias into the OBP boot-device list.
#
insert_boot_device "$_disk";


#
# cleanup
#
rm -f ${DSM_LOG_DIR}/deleted_metadevices ${DSM_LOG_DIR}/deleted_metadevices_tmp;


#
# Log "reintegrating actions completed" in dsmlog and syslog.
#
cp ${DSM_LOG_DIR}/dsmlog ${DSM_LOG_DIR}/dsmlog_tmp;
echo "`date '+%m/%d/%Y %H:%M:%S'` \
$_disk: $_reintegrating_actions_completed" >> ${DSM_LOG_DIR}/dsmlog_tmp;
mv ${DSM_LOG_DIR}/dsmlog_tmp ${DSM_LOG_DIR}/dsmlog;

Echo "notice" "${SCRIPT}: ${_disk}: `gettext $DSM_DOMAIN 'Reintegrating actions completed.'`";

exit 0;

