#! /usr/bin/sh
#
# Component : ChorusOS patch
# Module    : bin/subpatchit.sh
# Sypnosis  : script patching a sub-component of ChorusOS
#
# Copyright (c) 1998 by Sun Microsystems, Inc.
# All rights reserved.
#
# ident	"@(#)subpatchit.sh	1.0	98/09/12 SMI"
#


######################################################################
#
# FUNCTIONS
#
######################################################################

    #
    # ERROR
    #     Displays a message and exit 1
    #
ERROR() {
  echo " ** ERROR : $1"
  exit 1
}

    #
    # flush
    #    Black magic, to work around a bug in cygwin32.
    #
flush() {
    if [ "${OS}" = "win32" -a "${DISABLE_CYGWIN32_READ_WORKAROUND}" = "" ]; then
	exec <&2
    fi
}

    # 
    # usage
    #    Displays the script usage and exit 1
    #
usage() {
	PROGNAME=`basename $0`
	echo "\n\t${PROGNAME} [ -F ] -E file\n" >&2
	echo "\t\t-F\tForce the subpatch application\n" >&2
	echo "\t\t-E file\nGive the subpatch environment file\n" >&2
	exit 1
}

    #
    # ask_ok
    #    asks a question and wait y or n
    #    default is y.
    #    if reply = y, return 0
    #    if reply = n, return 1
    #
ask_ok() {
    flush
    while [ "true" ]; do
	$ECHO_N "$1 [y] "
	read answer
	if [ -z "${answer}" ]; then
	    break
	fi
	if [ "${answer}" = "y" ]; then
	    break
	fi
	if [ "${answer}" != "n" ]; then
	    echo "Please answer y or n"
	    continue;
	fi
	return 1
    done
    return 0
}

    #
    # ask_nok
    #    asks a question and wait y or n
    #        default is n.
    #        if reply = y, return 0
    #        if reply = n, return 1
    #
ask_nok() {
    flush
    while [ "true" ]; do
	$ECHO_N "$1 [n] "
	read answer
	if [ -z "${answer}" ]; then
	    break
	fi
	if [ "${answer}" = "n" ]; then
	    break
	fi
	if [ "${answer}" != "y" ]; then
	    echo "Please answer y or n"
	    continue;
	fi
	return 0
    done
    return 1
}

    #
    # check_path
    #      Check the value of a path by verifying it's contents 
    #      point to a readable file.
    #      Exit from program if the readable file is not present.
    #
check_path ()
{
  VARIABLE="$1"
  if [ -z "${2}" ]; then
    DESCRIPTION = ""
  else
    DESCRIPTION="as ${2}"
  fi
  XFILE="$3"

  if [ -z "${XFILE}" ]; then
	if [ -d "${VARIABLE}" ] ; then
		echo "Using ${VARIABLE} ${DESCRIPTION}"
	else
		ERROR "${VARIABLE} is not a directory."
	fi
  else
	if [ -r "${VARIABLE}/${XFILE}" ] ; then
		echo "Using ${VARIABLE} ${DESCRIPTION}"
	else
		ERROR "${VARIABLE}/${XFILE} is not readable."
	fi
  fi
}

    #
    # getpath
    #      Ask a path
    #      If no answer is given, return the default path
    #
getpath () {
    flush
    OK="FALSE"
    while [ "${OK}" = "FALSE" ]; 
    do
      $ECHO_N "$1 [$2] " >&2
      read answer
      if [ -z "${answer}" ]; then
	answer="$2"
      fi
      if [ -d "${answer}" ]; then
        OK="TRUE"
      else
        echo "${answer} is not existing !" >&2
      fi
    done
    echo "${answer}"
}

    #
    # make_subpatch_profile
    #      Create the subpatch profile data
    #      Save in it all _CHORUSOS_XXXX variables
    #
make_subpatch_profile () {

chmod u+w "${SUBPATCH_DIR}/subpatch.env"
if [ $? != 0 ]; then
  ERROR "Can not edit the subpatch environment file."
fi
cat <<EOF > "${SUBPATCH_DIR}/subpatch.env"
#      Subpatch information

_CHORUSOS_PATCH_ID="${_CHORUSOS_PATCH_ID}"
_CHORUSOS_PARTNUMBER="${_CHORUSOS_PARTNUMBER}"
_CHORUSOS_HOST="${_CHORUSOS_HOST}"
_CHORUSOS_HOST_NICKNAME="${_CHORUSOS_HOST_NICKNAME}"
_CHORUSOS_RELEASE="${_CHORUSOS_RELEASE}"
_CHORUSOS_TARGET="${_CHORUSOS_TARGET}"
_CHORUSOS_BANNER="${_CHORUSOS_BANNER}"
_CHORUSOS_SUBPATCH_DIR="${_CHORUSOS_SUBPATCH_DIR}"
_CHORUSOS_SUBPATCH_NAME="${_CHORUSOS_SUBPATCH_NAME}"
_CHORUSOS_SUBPATCH_NICKNAME="${_CHORUSOS_SUBPATCH_NICKNAME}"
_CHORUSOS_SUBPATCH_DATE="${_CHORUSOS_SUBPATCH_DATE}"
_CHORUSOS_SUBPATCH_LASTOP="`date`"
EOF
}

    #
    # default_subpatch_profile_values
    #      Set the profile variables with the default values
    #
default_subpatch_profile_values () {
_CHORUSOS_PATCH_ID="${_CHORUSOS_PATCH_ID:-123456}"
_CHORUSOS_PARTNUMBER="${_CHORUSOS_PARTNUMBER:-CLX320-AAAA}"
_CHORUSOS_HOST="${_CHORUSOS_HOST:-SPARC/Solaris}"
_CHORUSOS_HOST_NICKNAME="${_CHORUSOS_HOST_NICKNAME:-'solaris'}"
_CHORUSOS_RELEASE="${_CHORUSOS_RELEASE:-'3.2'}"
_CHORUSOS_TARGET="${_CHORUSOS_TARGET:-Intel x86}"
_CHORUSOS_BANNER="${_CHORUSOS_BANNER:-ChorusOS r3.2 Binary for Intel x86 (SPARC/Solaris host)}"
_CHORUSOS_SUBPATCH_NICKNAME="${_CHORUSOS_SUBPATCH_NICKNAME:-'ZZZZ'}"
_CHORUSOS_SUBPATCH_DIR="${_CHORUSOS_SUBPATCH_DIR:-'/ChorusOS/ZZZZ'}"
_CHORUSOS_SUBPATCH_DIR=`echo "${_CHORUSOS_SUBPATCH_DIR:-'/ChorusOS/ZZZZ'}" | \
			sed "s=^/ChorusOS/${_CHORUSOS_SUBPATCH_NICKNAME}$=${LAUNCH_DIR}/${_CHORUSOS_SUBPATCH_NICKNAME}="`
_CHORUSOS_SUBPATCH_NAME="${_CHORUSOS_SUBPATCH_NAME:-ChorusOS r3.2}"
_CHORUSOS_SUBPATCH_DATE="${_CHORUSOS_SUBPATCH_DATE:-`date`}"
_CHORUSOS_SUBPATCH_LASTOP="${_CHORUSOS_SUBPATCH_LASTOP:-`date`}"
}

    #
    # host_specific
    #      Find the OS
    #
host_specific () {
  # Find the host type
  UNAME=`uname`
  if [ "${UNAME}" = "Linux" ]; then
    OS=linux
  elif [ "${UNAME}" = "SunOS" ]; then
    case `uname -r` in
	4.*) OS=sunos;;
	5.*) OS=solaris;;
    esac;
  elif [ "${UNAME}" = "UNIX_SV" ]; then
    OS=unixware
  elif [ "${UNAME}" = "HP-UX" ]; then
    OS=hpux
  elif [ "${UNAME}" = "CYGWIN32/NT" ]; then
    OS=win32
  elif [ "${UNAME}" = "CYGWIN32_NT" ]; then
    OS=win32
  fi

  if [ -z "${OS}" ]; then
    ERROR "I have not been able to determine your host type. Sorry..."
  fi

  # Host specifics
  case "${OS}" in
    linux)	ECHO_N="echo -n";
		;;
    sunos)	ECHO_N="echo -n";
		;;
    solaris)	ECHO_N="printf %s";
		;;
    unixware)	ECHO_N="echo -n";
		;;
    svr4)	ECHO_N="echo -n";
		;;
    hpux)      	ECHO_N="printf %s";
		;;
    win32)	ECHO_N="printf %s";
		;;
  esac
}

    #
    # check_os
    #      Verify if the subpatch can be applied on this OS
    #
check_os () {
  # Check the host is the expected one
  if [ "${OS}" != "${_CHORUSOS_HOST_NICKNAME}" ]; then
    ERROR "This install script must be run on a ${_CHORUSOS_HOST} OS. Sorry"
  fi
}

    #
    # hereify
    #      Make all specified paths absolute
    #
HERE=`pwd`
hereify() {
  if [ -z "$1" ]; then
    return
  fi
  if echo "$1" | grep -v "^/" >/dev/null 2>&1; then
    if [ "$1" = "." ]; then
      echo "${HERE}"
    else
      echo "${HERE}/$1"
    fi
  else
    echo "$1"
  fi
}

##############################################################################
#
# BANNERS
#
##############################################################################

BANNER_SUBPATCHIT () {
cat <<EOF

     -- Enter in a subpatch --

EOF
}

BANNER_ENV () {
cat << EOF

     -- Finding subpatch environment --

EOF
}

BANNER_PATCHING () {
cat <<EOF

     -- Applying subpatch on $1 --

EOF
}

BANNER_END() {
cat <<EOF

     -- The subpatch has been successfully applied. --

     -- Exit from the subpatch --
EOF
}


##############################################################################
#
# MAIN
#
##############################################################################


host_specific

FORCE_OPT=""
ENV_FILE=""
while getopts "FE:" opt
do
  case "${opt}" in
  F) 
    FORCE_OPT="-F"
    ;;
  E)
    ENV_FILE="${OPTARG}"
    ;; 
  *) usage ;;

  esac
done

BANNER_SUBPATCHIT

BIN_DIR=`dirname $0`
BIN_DIR=`hereify "${BIN_DIR}"`
CIX="${BIN_DIR}/cix"

BANNER_ENV

if [ -f "${ENV_FILE}" ]; then
	. "${ENV_FILE}"
else
  ERROR "Unable to find the subpatch info ${ENV_FILE}."
fi

SUBPATCH_DIR=`dirname "${ENV_FILE}"`
SUBPATCH_DIR=`hereify "${SUBPATCH_DIR}"`

# Create default values if needed
default_subpatch_profile_values

# Verify the OS
check_os

BANNER_PATCHING "${_CHORUSOS_SUBPATCH_NAME}"

WHERE="${_CHORUSOS_SUBPATCH_DIR}"

if ask_ok "Do you want patch the ${_CHORUSOS_SUBPATCH_NAME}?"; then
  WHERE=`getpath "Where ?" "${WHERE}"`

  _CHORUSOS_SUBPATCH_DIR=`hereify "${WHERE}"`
  check_path "${_CHORUSOS_SUBPATCH_DIR}" "${_CHORUSOS_SUBPATCH_NAME} directory"

  if [ -x "${CIX}" ]; then
    "${CIX}" -n -f "${_CHORUSOS_SUBPATCH_DIR}" -p "${SUBPATCH_DIR}" > /dev/null
    case $? in
      0)
	"${CIX}" ${FORCE_OPT} -f "${_CHORUSOS_SUBPATCH_DIR}" -p "${SUBPATCH_DIR}"
        if [ $? -eq 0 ]
	then
	  echo "${_CHORUSOS_SUBPATCH_DIR} patched !"
	else
	  echo "Inconsistency detected!"
	  ERROR "Please reinstall the product and the patch"
	fi
        ;;
      1)
        echo "Some files are missing !"
	ERROR "Please reinstall the product or the patch."
        ;;
      2)
        echo "Some files can not be handled !"
	ERROR "Please reinstall the product or the patch."
        ;;
      3)
        echo "The patch has been modified !"
	ERROR "Please reinstall the patch."
        ;;
      4)
        echo "The product or the patch have been modified !"
	ERROR "Please reinstall the product or the patch."
        ;;
      *)
        ERROR "Unable to patch ${_CHORUSOS_SUBPATCH_DIR}"
        ;;
    esac
  else
    ERROR "Unable to find cix tool"
  fi

  # Generate profile of environment variables
  make_subpatch_profile

fi

BANNER_END











