// Copyright (c) 03/12/99, by Sun Microsystems, Inc.
// All rights reserved.

// @(#)ThreadMgtByThreadGroup.java 1.4 99/03/12 

// java import
//
import java.lang.Thread;
import java.lang.ThreadGroup;
import java.util.Hashtable;

// jaw import
//
import com.sun.jaw.reference.agent.services.ThreadAllocatorSrvIf;

/**
 * The ThreadMgtByThreadGroup class provides an example of the
 * a thread allocator service dedicated to the management of the Thread 
 * allocated to the adaptors of the agent.
 * The other allocated Thread are grouped in a ThreadGroup name "master thread group"
 * in the following. M-Bean can get the master thread group using the "obTainMasterThreadGroup" method.
 * If a ThreadGroup is specified in the parameter of obtainThread method, then the corresponding
 * Thread constructor is directly called.
 * 
 * This service is in charge of creating one ThreadGroup per adaptor.
 * The key word "adaptor" in the requestorId String is used to determine if
 * the requestor is an adaptor or not.
 * 
 * If the requestorId is null, then a new Thread is returned;
 * else the returned thread is part of the master ThreadGroup.
 */

public class ThreadMgtByThreadGroup implements ThreadAllocatorSrvIf {
    
    /**
     * Constructor the thread allocator service.
     * It instantiates the master thread group.
     */
    public ThreadMgtByThreadGroup () {
        main = new ThreadGroup("Main JDMK Thread Group");
        threadGroupList = new Hashtable();
    }
    
    
    private ThreadGroup GetThreadGroup(String reqId){
        ThreadGroup tg = null;
        if (reqId.indexOf("adaptor") < 0) {
            trace("Not an adaptor");
            tg = main;
        } else {
            trace("Thread for Adaptor: "+reqId);
            tg = (ThreadGroup) threadGroupList.get(reqId);
            if ( tg == null ) {
                tg = new ThreadGroup(main, reqId);
                threadGroupList.put(reqId, tg);
            }
        }
        trace("Return Group = "+tg.toString());
        return tg;
    }
    
    /**
     * Allocates a new Thread object. If the requestor is an adaptor,
     * then the returned Thread is part of the corresponding ThreadGroup.
     *
     * @param resquestorId The requestor's identifier.
     */

    public Thread obtainThread(Object requestorId) {
        trace("ThreadMgtByThreadGroup::obtainThread: max count="+main.activeCount());
        if ( maxThreadCount <= main.activeCount()) {
            trace("ThreadMgtByThreadGroup::obtainThread: max count reached");
            return (Thread) null;
        }
        trace("ThreadMgtByThreadGroup::obtainThread: requestor = ["+requestorId.toString()+"]");
        if (requestorId != null)
            return new Thread (GetThreadGroup(requestorId.toString()), (Runnable)null);
        else 
            return new Thread();
    }

    /**
     * Allocates a new Thread object. If the requestor is an adaptor,
     * then the returned Thread is part of the corresponding ThreadGroup.
     *
     * @param resquestorId The requestor's identifier.
     * @param target The object whose "run" method is called.
     */
    public Thread obtainThread(Object requestorId, Runnable target) {
        trace("ThreadMgtByThreadGroup::obtainThread: max count="+main.activeCount());
        if (maxThreadCount <= main.activeCount()) {
            trace("ThreadMgtByThreadGroup::obtainThread: max count reached");
            return (Thread) null;
        }
        trace("ThreadMgtByThreadGroup::obtainThread: requestor = ["+requestorId.toString()+"]");
        if (requestorId != null)
            return new Thread (GetThreadGroup(requestorId.toString()),target);
        else 
            return new Thread(target);
    }

    /**
     * Allocates a new Thread object. Returns a new Thread(group, target)
     *
     * @param resquestorId The requestor's identifier.
     * @param group the thread group.
     * @param target the object whose "run" method is called.
     * @exception SecurityException if the current thread cannot create a
     * thread in the specified thread group.
     */
    public Thread obtainThread(Object requestorId, ThreadGroup group, Runnable target) {
        trace("ThreadMgtByThreadGroup::obtainThread: max count="+main.activeCount());
        if (maxThreadCount <= main.activeCount()) {
            trace("ThreadMgtByThreadGroup::obtainThread: max count reached");
            return (Thread) null;
        }
        trace("ThreadMgtByThreadGroup::obtainThread: requestor = ["+requestorId.toString()+"]");
        return new Thread(group, target);
    }
    

    /**
     * Allocates a new Thread object. If the requestor is an adaptor,
     * then the returned Thread is part of the corresponding ThreadGroup.
     *
     * @param resquestorId The requestor's identifier.
     * @param name the name of the new thread.
     */
    public Thread obtainThread(Object requestorId, String name) {
        trace("ThreadMgtByThreadGroup::obtainThread: max count="+main.activeCount());
        if (maxThreadCount <= main.activeCount()) {
            trace("ThreadMgtByThreadGroup::obtainThread: max count reached");
            return (Thread) null;
        }
        trace("ThreadMgtByThreadGroup::obtainThread: requestor = ["+requestorId.toString()+"]");
        if (requestorId != null)
            return new Thread (GetThreadGroup(requestorId.toString()), null, name);
        else 
            return new Thread (name);
    }

    /**
     * Allocates a new Thread object. Returns a new Thread(group,name).
     *
     * @param resquestorId The requestor's identifier.
     * @param group the thread group.
     * @param name the name of the new thread.
     * @exception SecurityException if the current thread cannot create a
     * thread in the specified thread group.
     */
    public Thread obtainThread(Object requestorId, ThreadGroup group, String name) {
        trace("ThreadMgtByThreadGroup::obtainThread: max count="+main.activeCount());
        if (maxThreadCount <= main.activeCount()) {
            trace("ThreadMgtByThreadGroup::obtainThread: max count reached");
            return (Thread) null;
        }
        trace("ThreadMgtByThreadGroup::obtainThread: requestor = ["+requestorId.toString()+"]");
        return new Thread(group, name);
    }

    /**
     * Allocates a new Thread object. If the requestor is an adaptor,
     * then the returned Thread is part of the corresponding ThreadGroup.
     *
     * @param resquestorId The requestor's identifier.
     * @param target the object whose "run" method is called.
     * @param name the name of the new thread.
     */
    public Thread obtainThread(Object requestorId, Runnable target, String name) {
        trace("ThreadMgtByThreadGroup::obtainThread: max count="+main.activeCount());
        if (maxThreadCount <= main.activeCount()) {
            trace("ThreadMgtByThreadGroup::obtainThread: max count reached");
            return (Thread) null;
        }
        trace("ThreadMgtByThreadGroup::obtainThread: requestor = ["+requestorId.toString()+"]");
        if (requestorId != null)
            return new Thread (GetThreadGroup(requestorId.toString()), target, name);
        else 
            return new Thread(target, name);
    }

    /**
     * Allocates a new Thread object with  
     *  - requestorId as the requestor identifier
     *  - group as the thread group the thread is assigned to
     *  - target as its run object
     *  - name as its name
     *
     * @param resquestorId The requestor's identifier.
     * @param group the thread group.
     * @param target the object whose "run" method is called.
     * @param name the name of the new thread.
     * @exception SecurityException if the current thread cannot create a
     * thread in the specified thread group.
     */
    public Thread obtainThread(Object requestorId, ThreadGroup group, Runnable target, String name) {
        trace("ThreadMgtByThreadGroup::obtainThread: max count="+main.activeCount());
        if ( maxThreadCount <= main.activeCount()) {
            trace("ThreadMgtByThreadGroup::obtainThread: max countreached");
            return (Thread) null;
        }
        trace("ThreadMgtByThreadGroup::obtainThread: requestor = ["+requestorId.toString()+"]");
        return new Thread(group, target, name);
    }
    
    /**
     * Returns the Name of the master thread group of this Service.
     * @return The name of the master thread group.
     */
    public String getThreadGroupName(){
        return main.getName();
    }
        
    /** 
     * Returns an estimate of the number of active groups in the master thread group.
     * @return the number of active groups in the master thread group.
     */  
    public int getActiveThreadGroupCount(){
        return main.activeGroupCount();
    }
    
    /**
     * Returns an estimate of the number of active threads in the master thread group.
     * @return the number of active threads in the master thread group.
     */
    public int getActiveThreadCount(){
        return main.activeCount();
    }
    
    /** 
     * Returns the list of active group names and the names of thread in those group HTML formatted.
     *
     * Example of one element of the array:
     *  defaultDomain:com.sun.jaw.reference.client.adaptor.AdaptorMO.protocol=html,port=8082
     * 
     * 2 thread(s) active(s):
     *  defaultDomain:com.sun.jaw.reference.client.adaptor.AdaptorMO.protocol=html,port=8082
     *  Thread-2
     * @return the list of active group names
     */  
    
    public String[] getActiveThreadGroupNames(){
        int num = main.activeGroupCount();
        String[] list = new String[num];
        ThreadGroup[] th = new ThreadGroup[num];
        Thread[] tl;
        int thactive = 0;
        
        main.enumerate(th);
        for ( int i = 0; i<num ; i++){
            list[i] = th[i].getName()+"<BR>"+th[i].activeCount()+" thread(s) active(s):";
            thactive = th[i].activeCount();
            tl = new Thread[thactive];
            th[i].enumerate(tl);
            list[i] += "<UL>";
            for (int j = 0 ; j < thactive ; j++)
                list[i] += "<LI>"+tl[j].getName();
            list[i] += "</UL>";
        }
        return list;
    }
    
    
    /** 
     * Returns the master thread group managed by this ThreadAllocator service.
     *
     * @return the master thread group of this service
     */
    
    public ThreadGroup obTainMasterThreadGroup(){
        return main;
    }
    
    /**
     * Returns the maximum number of active thread authorized in JDMK agent.
     * 
     * @return the current value of the "maxThreadCount" property.
     */
    public int getMaxThreadCount() {
        return maxThreadCount;
    }


    /**
     * Set the maximum number of active thread authorized in JDMK agent.
     * 
     * @param <VAR>value</VAR> maximum thread number.
     */
    
    public void setMaxThreadCount(int value) {
        if ( value < main.activeCount()) {
            trace("ThreadMgtByThreadGroup::obtainThread: WARNING: maximum below current number of active thread ("+main.activeCount()+")");
        } else {
            maxThreadCount = value;
        }
    }
    
    /**
     * Display trace messages.
     */
    
    private static void trace(String msg) {
        System.out.println(msg);
    }

    private Hashtable threadGroupList = null;
    ThreadGroup main = null;
    
    private int maxThreadCount = 100;
    
}
