// Copyright (c) 03/12/99, by Sun Microsystems, Inc.
// All rights reserved.

// "@(#)SnmpMibAgentImpl.java 1.5 99/03/12 SMI"

// java imports
//
import java.io.*;
import java.net.*;
import java.util.*;

// jaw imports
//
import com.sun.jaw.reference.agent.cmf.* ;
import com.sun.jaw.reference.common.* ;
import com.sun.jaw.snmp.agent.*;
import com.sun.jaw.snmp.common.*;
import com.sun.jaw.snmp.manager.*;


/**
 * The SnmpMibAgentImpl class provides an example of implementation of an SNMP proxy.
 * It enables an agent to serve remote MIBs (that is MIBs running outside the VM where the agent is running).
 * In fact, it behaves like a manager on the agent side.
 * 
 * You initialize the SnmpMibAgentImpl invoking the initializeProxy method
 * with the following parameters:
 *      - host: hostname of the remote SNMP sub-agent you want to query. 
 *      - port: port number to use.
 *      - strOid: root object identifier of the MIB to query (optional).
 *      - name: MIB name (optional).
 */

public class SnmpMibAgentImpl extends SnmpMibAgent implements Serializable {
  
    // INITIALIZATION
    //===============
    /**
     * Initialization of the SnmpMibAgentImpl with no registration in JDMK.
     * @exception IllegalAccessException The SnmpMibAgentImpl can not be initialized.
     */
    public void init() throws IllegalAccessException {
    }
  
    /**
     * Initialization of the SnmpMibAgentImpl with registration in JDMK.
     * @exception IllegalAccessException The SnmpMibAgentImpl can not be initialized.
     * @exception ServiceNotFoundException A requested service is not supported.
     * @exception InstanceAlreadyExistException A m-bean is already registered in the repository.
     */
    public void initCmf(Framework cmf, ObjectName name) 
        throws IllegalAccessException, ServiceNotFoundException, InstanceAlreadyExistException {
	  
        // Initialize framework information.
        //
        this.cmf= cmf;
        cmf.addObject(this, name);
    }
  
    /**
     * Initialization of the proxy stuff.
     * @param <VAR>h</VAR> The hostname of the remote SNMP sub-agent.
     * @param <VAR>p</VAR> The port number to use.
     * @param <VAR>strOid</VAR> The root object identifier of the MIB.
     * @param <VAR>name</VAR> The name of the MIB.
     * @exception UnknownHostException The hostname can not be resolved.
     * @exception SnmpStatusException An error occurred while accessing a MIB node.
     */
    public void initializeProxy(String h, int p, String strOid, String name) throws UnknownHostException, SnmpStatusException {
	
        host= h;
        port= p;
        oid= strOid;
        mibName= name;
	
        SnmpMain.initializeSNMP();

        // Initialization for SNMP v1 protocol.
        //
        SnmpParameters paramsV1= new SnmpParameters("public", "private");
        paramsV1.setProtocolVersion(SnmpDefinitions.snmpVersionOne);
        SnmpPeer peerV1= new SnmpPeer(host, port);
        peerV1.setSnmpParam(paramsV1);
        sessionV1= new SnmpSession("SnmpMibAgentImpl session V1");
        sessionV1.setDefaultPeer(peerV1);
        // Using SNMP v1 protocol, the error is not fixed but is forwarded to the manager.
        sessionV1.snmpOptions.setPduFixedOnError(false);

        // Initialization for SNMP v2 protocol.
        //
        SnmpParameters paramsV2= new SnmpParameters("public", "private");
        paramsV2.setProtocolVersion(SnmpDefinitions.snmpVersionTwo);
        SnmpPeer peerV2= new SnmpPeer(host, port);
        peerV2.setSnmpParam(paramsV2);
        // If we meet a problem, we don't retry the request using SNMP v2 but we try the request using SNMP v1.
        peerV2.setMaxRetries(0);
        sessionV2= new SnmpSession("SnmpMibAgentImpl session V2");
        sessionV2.setDefaultPeer(peerV2);
        // Using SNMP v2 protocol, the error is fixed.
        sessionV2.snmpOptions.setPduFixedOnError(true);

        // Initialization for SNMP v2 protocol simulated using SNMP v1 protocol.
        //
        sessionV2WithV1= new SnmpSession("SnmpMibAgentImpl session V2 with V1");
        sessionV2WithV1.setDefaultPeer(peerV1);
        // Simulating SNMP v2 with SNMP v1 protocol, the error is fixed.
        sessionV2WithV1.snmpOptions.setPduFixedOnError(true);
    }
  
    /**
     * Initialization of the proxy stuff.
     * @param <VAR>h</VAR> The hostname of the remote SNMP sub-agent.
     * @param <VAR>p</VAR> The port number to use.
     * @param <VAR>strOid</VAR> The root object identifier of the MIB.
     * @exception UnknownHostException The hostname can not be resolved.
     * @exception SnmpStatusException An error occurred while accessing a MIB node.
     */
    public void initializeProxy(String h, int p, String strOid) throws UnknownHostException, SnmpStatusException {
        initializeProxy(h, p, strOid, "SnmpMibAgentImpl");
    }
  
    /**
     * Initialization of the proxy stuff.
     * @param <VAR>h</VAR> The hostname of the remote SNMP sub-agent.
     * @param <VAR>p</VAR> The port number to use.
     * @exception UnknownHostException The hostname can not be resolved.
     * @exception SnmpStatusException An error occurred while accessing a MIB node.
     */
    public void initializeProxy(String h, int p) throws UnknownHostException, SnmpStatusException {
        initializeProxy(h, p, null, "SnmpMibAgentImpl");
    }
  

    // SNMP PROXY STUFF
    //=================
    /**
     * Implement the get method from the abstract SnmpMibAgent class.
     * @param <VAR>list</VAR> The variable list to be retrieved. A vector of SnmpVarBind objects.
     * @param <VAR>version</VAR> The version of the protocol for the requested operation.
     * If the operation failed using version 2 of SNMP, we try it again using version 1.
     * @return The variable list containing returned values.
     * @exception SnmpStatusException  An error occured during the operation.
     */
    public void get(Vector list, int version) throws SnmpStatusException {
			  	  	  
        java.lang.System.out.println("\n>> =====================================================================");
        java.lang.System.out.println(">> SnmpMibAgentImpl: Sending get request to SNMP sub-agent on " + host + " using port " + port);

        // Request using SNMP v1 protocol.
        if (version == SnmpDefinitions.snmpVersionOne) {
            get(list, version, sessionV1);
        }
        // Request using SNMP v2 protocol.
        if (version == SnmpDefinitions.snmpVersionTwo) {
            get(list, version, sessionV2);
        }
    }

    /**
     * Implement the set method from the abstract SnmpMibAgent class.
     * @param <VAR>list</VAR> The variable list to be set. A vector of SnmpVarBind objects.
     * @param <VAR>version</VAR> The version of the protocol for the requested operation.
     * If the operation failed using version 2 of SNMP, we try it again using version 1.
     * @return The variable list containing returned values.
     * @exception SnmpStatusException  An error occured during the operation.
     */
    public void set(Vector list, int version) throws SnmpStatusException {
	
        java.lang.System.out.println("\n>> =====================================================================");
        java.lang.System.out.println(">> SnmpMibAgentImpl: Sending set request to SNMP sub-agent on " + host + " using port " + port);

        // Request using SNMP v1 protocol.
        if (version == SnmpDefinitions.snmpVersionOne) {
            set(list, version, sessionV1);
        }
        // Request using SNMP v2 protocol.
        if (version == SnmpDefinitions.snmpVersionTwo) {
            set(list, version, sessionV2);
        }
    }
  
    /**
     * Implement the getNext method from the abstract SnmpMibAgent class.
     * @param <VAR>list</VAR> The variable list to be retrieved. A vector of SnmpVarBind objects.
     * @param <VAR>version</VAR> The version of the protocol for the requested operation.
     * If the operation failed using version 2 of SNMP, we try it again using version 1.
     * @return The variable list containing returned values.
     * @exception SnmpStatusException  An error occured during the operation.
     */
    public void getNext(Vector list, int version) throws SnmpStatusException {
	
        java.lang.System.out.println("\n>> =====================================================================");
        java.lang.System.out.println(">> SnmpMibAgentImpl: Sending getNext request to SNMP sub-agent on " + host + " using port " + port);

        // Request using SNMP v1 protocol.
        if (version == SnmpDefinitions.snmpVersionOne) {
            getNext(list, version, sessionV1);
        }
        // Request using SNMP v2 protocol.
        if (version == SnmpDefinitions.snmpVersionTwo) {
            getNext(list, version, sessionV2);
        }
    }
    
    /**
     * Implement the getBulk method from the abstract SnmpMibAgent class.
     * @param <VAR>list</VAR> The variable list to be retrieved. A vector of SnmpVarBind objects.
     * @param <VAR>nonRepeat</VAR> The number of variables, starting with the dirst variable in the 
     * variable-bindings, for which a single lexicographic successor is requested.
     * @param <VAR>maxRepeat</VAR> The number of lexicographic successors requested for each of the last R variables.
     * R is the number of variables following the first nonRepeat
     * variables for which multiple lexicographic successors are requested.
     * @param <VAR>version</VAR> The version of the protocol for the requested operation.
     * If the operation failed using version 2 of SNMP, we try it again using version 1.
     * @return The variable list containing returned values.
     * @exception SnmpStatusException  An error occured during the operation.
     */
    public void getBulk(Vector list, int nonRepeat, int maxRepeat, int version) throws SnmpStatusException {
        return;
    }
  
    /**
     * Implement the check method from the abstract SnmpMibAgent class.
     * @param <VAR>list</VAR> The variable list to be set. A vector of SnmpVarBind objects.
     * @exception SnmpStatusException  The operation can not be performed.
     */
    public void check(Vector list) throws SnmpStatusException {
        return;
    }
  
    /**
     * This method allows to specify both the protocol version and the session to use for the get request.
     * 
     * If the request was originally sent using SNMP v1, the session does not fix the error but forwards it to the manager.
     * If the request was sent using SNMP v1 to simulate SNMP v2, the session fixes the error.
     */
    private void get(Vector list, int version, SnmpSession session) throws SnmpStatusException {
			  	  	  
        // Construction of the SnmpVarBindList.
        //
        SnmpVarbindList varbindlist= setVarBindList(list);

        // Request handler.
        //
        try {
            request= session.snmpGet(null, varbindlist);
        } 
        catch (SnmpStatusException e) {
            throw new SnmpStatusException(SnmpDefinitions.snmpRspGenErr, 0);
        }
        java.lang.System.out.println("\nRequest:\n" + request.toString());
	    
        boolean completed= request.waitForCompletion(10000);
        if (completed == false) {
            // If the completion failed using SNMP v1, we give up.
            if (version == SnmpDefinitions.snmpVersionOne) {
                java.lang.System.out.println("\nRequest timed out: check reachability of sub-agent.");
                return;
            }
            // If the completion failed using SNMP v2, we try the request again using SNMP v1.
            if (version == SnmpDefinitions.snmpVersionTwo) {
                java.lang.System.out.println("\n>> SnmpMibAgentImpl: Try to submit the get request using SNMP version 1...");
                get(list, SnmpDefinitions.snmpVersionOne, sessionV2WithV1);
                return;
            }
        }

        int errorStatus= request.getErrorStatus();
        int errorIndex= request.getErrorIndex() + 1;
        if (errorStatus != SnmpDefinitions.snmpRspNoError) {
            java.lang.System.out.println("\nError status= " + SnmpRequest.snmpErrorToString(errorStatus));
            java.lang.System.out.println("Error index = " + errorIndex);
            // If there is an error status using SNMP v1, we throw an exception.
            if (version == SnmpDefinitions.snmpVersionOne) {
                throw new SnmpStatusException(errorStatus, errorIndex);
            }
            // If there is an error status using SNMP v2, we try the request again using SNMP v1.
            if (version == SnmpDefinitions.snmpVersionTwo) {
                java.lang.System.out.println("\n>> SnmpMibAgentImpl: Try to submit the get request using SNMP version 1...");
                get(list, SnmpDefinitions.snmpVersionOne, sessionV2WithV1);
                return;
            }
        }
        result= request.getResponseVbList();
        java.lang.System.out.println("\nResult: \n" + result.vbListToString());

        // Update the list parameter.
        //
        Enumeration l = list.elements();
        for (Enumeration e = result.elements(); e.hasMoreElements();) {
            SnmpVarBind varres= (SnmpVarBind) e.nextElement();
            SnmpVarBind varbind= (SnmpVarBind) l.nextElement();
            varbind.value= varres.value;
        }
    }

    /**
     * This method allows to specify both the protocol version and the session to use for the set request.
     * 
     * If the request was originally sent using SNMP v1, the session does not fix the error but forwards it to the manager.
     * If the request was sent using SNMP v1 to simulate SNMP v2, the session fixes the error.
     */
    private void set(Vector list, int version, SnmpSession session) throws SnmpStatusException {
	
        // Construction of the SnmpVarBindList.
        //
        SnmpVarbindList varbindlist= setVarBindList(list);

        // Request handler.
        //
        try {
            request= session.snmpSet(null, varbindlist);
        } 
        catch (SnmpStatusException e) {
            throw new SnmpStatusException(SnmpDefinitions.snmpRspGenErr, 0);
        }
        java.lang.System.out.println("\nRequest:\n" + request.toString());
	    
        boolean completed= request.waitForCompletion(10000);
        if (completed == false) {
            // If the completion failed using SNMP v1, we give up.
            if (version == SnmpDefinitions.snmpVersionOne) {
                java.lang.System.out.println("\nRequest timed out: check reachability of sub-agent.");
                return;
            }
            // If the completion failed using SNMP v2, we try the request again using SNMP v1.
            if (version == SnmpDefinitions.snmpVersionTwo) {
                java.lang.System.out.println("\n>> SnmpMibAgentImpl: Try to submit the set request using SNMP version 1...");
                set(list, SnmpDefinitions.snmpVersionOne, sessionV2WithV1);
                return;
            }
        }

        int errorStatus= request.getErrorStatus();
        int errorIndex= request.getErrorIndex() + 1;
        if (errorStatus != SnmpDefinitions.snmpRspNoError) {
            java.lang.System.out.println("\nError status= " + SnmpRequest.snmpErrorToString(errorStatus));
            java.lang.System.out.println("Error index = " + errorIndex);
            // If there is an error status using SNMP v1, we throw an exception.
            if (version == SnmpDefinitions.snmpVersionOne) {
                throw new SnmpStatusException(errorStatus, errorIndex);
            }
            // If there is an error status using SNMP v2, we try the request again using SNMP v1.
            if (version == SnmpDefinitions.snmpVersionTwo) {
                java.lang.System.out.println("\n>> SnmpMibAgentImpl: Try to submit the set request using SNMP version 1...");
                set(list, SnmpDefinitions.snmpVersionOne, sessionV2WithV1);
                return;
            }
        }
        result= request.getResponseVbList();
        java.lang.System.out.println("\nResult: \n" + result.vbListToString());

        // Update the list parameter.
        //
        Enumeration l = list.elements();
        for (Enumeration e = result.elements(); e.hasMoreElements();) {
            SnmpVarBind varres= (SnmpVarBind) e.nextElement();
            SnmpVarBind varbind= (SnmpVarBind) l.nextElement();
            varbind.value= varres.value;
        }
    }

    /**
     * This method allows to specify both the protocol version and the session to use for the getNext request.
     * 
     * If the request was originally sent using SNMP v1, the session does not fix the error but forwards it to the manager.
     * If the request was sent using SNMP v1 to simulate SNMP v2, the session fixes the error.
     */
    private void getNext(Vector list, int version, SnmpSession session) throws SnmpStatusException {
	
        // Construction of the SnmpVarBindList.
        //
        SnmpVarbindList varbindlist= setVarBindList(list);

        // Request handler.
        //
        try {
            request= session.snmpGetNext(null, varbindlist);
        } 
        catch (SnmpStatusException e) {
            throw new SnmpStatusException(SnmpDefinitions.snmpRspGenErr, 0);
        }
        java.lang.System.out.println("\nRequest:\n" + request.toString());

        boolean completed= request.waitForCompletion(10000);
        if (completed == false) {
            // If the completion failed using SNMP v1, we give up.
            if (version == SnmpDefinitions.snmpVersionOne) {
                java.lang.System.out.println("\nRequest timed out: check reachability of sub-agent.");
                return;
            }
            // If the completion failed using SNMP v2, we try the request again using SNMP v1.
            if (version == SnmpDefinitions.snmpVersionTwo) {
                java.lang.System.out.println("\n>> SnmpMibAgentImpl: Try to submit the getNext request using SNMP version 1...");
                getNext(list, SnmpDefinitions.snmpVersionOne, sessionV2WithV1);
                return;
            }
        }

        int errorStatus= request.getErrorStatus();
        int errorIndex= request.getErrorIndex() + 1;
        if (errorStatus != SnmpDefinitions.snmpRspNoError) {
            java.lang.System.out.println("\nError status= " + SnmpRequest.snmpErrorToString(errorStatus));
            java.lang.System.out.println("Error index = " + errorIndex);
            // If there is an error status using SNMP v1, we throw an exception.
            if (version == SnmpDefinitions.snmpVersionOne) {
                throw new SnmpStatusException(errorStatus, errorIndex);
            }
            // If there is an error status using SNMP v2, we try the request again using SNMP v1.
            if (version == SnmpDefinitions.snmpVersionTwo) {
                java.lang.System.out.println("\n>> SnmpMibAgentImpl: Try to submit the getNext request using SNMP version 1...");
                getNext(list, SnmpDefinitions.snmpVersionOne, sessionV2WithV1);
                return;
            }
        }
        result= request.getResponseVbList();
        java.lang.System.out.println("\nResult: \n" + result.vbListToString());

        // Update the list parameter.
        //
        Enumeration l = list.elements();
        for (Enumeration e = result.elements(); e.hasMoreElements();) {
            SnmpVarBind varres= (SnmpVarBind) e.nextElement();
            SnmpVarBind varbind= (SnmpVarBind) l.nextElement();
            varbind.oid= varres.oid;
            varbind.value= varres.value;
        }
    }

    /**
     * Return an SnmpVarbindList of the components of the specified vector.
     */
    private SnmpVarbindList setVarBindList(Vector list) throws SnmpStatusException {
	
        SnmpVarbindList varbindlist= new SnmpVarbindList("SnmpMibAgentImpl varbind list");
	
        for (Enumeration e= list.elements(); e.hasMoreElements();) {
            SnmpVarBind varbind= (SnmpVarBind) e.nextElement();
            SnmpVar var= new SnmpVar(varbind.oid);
            var.setSnmpValue(varbind.value);
            varbindlist.addVariable(var);
        }
        return varbindlist;
    }
  
    /**
     * Parse the string oid and return the corresponding root object identifier.
     */
    private long[] resolveOidString(String s) {
	
        StringTokenizer	st= new StringTokenizer(s, ".", false) ;
        int		length= st.countTokens();
	  
        long[] oidArray= new long[length];
    
        // Now extract the ids.
        //
        try {
            for (int i = 0 ; i < length ; i++) {
                try {
                    oidArray[i] = Long.parseLong(st.nextToken()) ;
                }
                catch(NoSuchElementException e) {}
            }
        }
        catch(NumberFormatException e) {
            throw new IllegalArgumentException(s) ;
        }
	
        return oidArray;
    }
  
  
    // GETTER/SETTER
    //==============
    /**
     * Return the root object identifier of the MIB.
     */
    public long[] getRootOid() {
	
        // A string oid has been initialized by the user.
        //
        if (oid != null) {
            rootOid= resolveOidString(oid);
        }
	
        // No initialization of the oid string has been done by the user.
        //
        else {
            try {
                Vector		varBindList= new Vector();
                SnmpOid		varBindOid= new SnmpOid("0.0");
                SnmpVarBind	varBind= new SnmpVarBind(varBindOid, null);
				
                varBindList.addElement(varBind);
                getNext(varBindList, SnmpDefinitions.snmpVersionTwo);
		
                // As the 3 last ids cannot be part of the root oid 
                // (either if the string represent a "simple" variable or a table instance), 
                // we remove them from the string oid.
                //
                String	strOid= varBind.oid.toString();
                int	index= strOid.lastIndexOf('.');
                for (int i = 1; i <= 3; i++) {
                    strOid= strOid.substring(0, index);
                    index= strOid.lastIndexOf('.');
                }
                rootOid= resolveOidString(strOid);
            }
            catch (Exception e) {
                java.lang.System.out.println("\nProblem when resolving the string oid...");
            }
        }
	
        return rootOid;
    }
  
    public String getHost() {
        return host;
    }
    public void setHost(String s) {
        host= s;
    }
  
    public int getPort() {
        return port;
    }
    public void setPort(int i) {
        port= i;
    }
  
    public String getOid() {
        return oid;
    }
    public void setOid(String s) {
        oid= s;
    }
  
  
    // PRIVATE VARIABLES
    //==================
    private String		host= null;
    private int			port= 0;
    private String		oid= null;
  
    private SnmpSession		sessionV1= null;
    private SnmpSession		sessionV2= null;
    private SnmpSession		sessionV2WithV1= null;
  
    private transient long[]	rootOid= null;

    SnmpRequest			request= null;
    SnmpVarbindList		result= null;
} 
