// Copyright (c) 03/12/99, by Sun Microsystems, Inc.
// All rights reserved.

// "@(#)RestartSrv.java 3.3 99/03/12 SMI"

// Java imports
//
import java.lang.*;
import java.util.*;

// Jaw imports
//
import com.sun.jaw.reference.common.*;
import com.sun.jaw.reference.agent.cmf.*;
import com.sun.jaw.reference.agent.services.*;

/**
 * This class is instanciated and managed by the repository. The repository
 * register and unregister m-beans with the service. After the agent has been
 * restarted the repository calls the service to activate all the m-beans 
 * registered for reactivation.
 *
 * @see RestartList
 * @see ActivationFailure
 */
public class RestartSrv
    implements java.io.Serializable {

    /**
     * Default constructor.
     *
     * @param <VAR>cmf</VAR> The current framework.
     */
    public RestartSrv(Framework cmf) {
        super();
        restartList = new RestartList();
        this.cmf = cmf;
    }


    // PUBLIC METHODS
    //---------------

    /**
     * Gets the list of m-beans registred for activation.
     *
     * @return A list containing the registered m-beans.
     */
    public Vector getRestartList() {
        return restartList.getAll();
    }
   
    /**
     * Gets the list of activation failures. The list is populated with
     * ActivationFailure objects giving failure information,
     * after the agent has been restarted.
     *
     * @return A list containing the activation failures.
     */
    public Vector getFailureList() {
        return failureList;
    }

    /**
     * Gets the number of m-beans registered with the Restart service.
     *    
     * @return The number of m-beans registered with the service.
     *
     */
    public int getNbElements() {
        return restartList.size();
    }

    /**
     * Registers the m-bean with the Restart service, for reactivation after
     * the agent has been stopped.
     *
     * @param <VAR>name</VAR> The object name of the m-bean to register.
     */
    public void register(ObjectName name) {
        // Add the m-bean name to the list of m-beans and
        // update the repository.
        //
        if (!restartList.contains(name)) {
            trace("register: name=" + name);
            restartList.add(name);
            sync();
        }
    }

    /**
     * Unregisters the m-bean with the Restart service.
     *
     * @param <VAR>name</VAR> The object name of the m-bean to unregister.
     */
    public void unregister(ObjectName name) {
        // Remove the m-bean name from the list of m-beans and
        // update the repository.
        //
        if (restartList.contains(name)) {
            trace("unregister: name=" + name);
            restartList.remove(name);
            sync();
        }
    }

    /**
     * Verifies whether an m-bean is registered with the Restart service.
     *
     * @param <VAR>name</VAR> The object name of the m-bean.
     *
     * @return True if the m-bean is registered with the Restart service, false 
     * otherwise.
     */   
    public boolean contains(ObjectName name) {
        trace("contains: name=" + name);
        boolean found = restartList.contains(name);
        trace("contains: found=" + found);
        return found;
    }
   
    /**
     * Activates all the m-beans registered with the Restart service.
     */
    public void restartServices() {
        lntrace("restartServices: --- Recover and Activate m-beans Begin ---\n");
      
        // Return if the m-beans has already been recovered and activated.
        //
        if (activated) {
            return;
        }
      
        // Create a list to be populated by activation failures.
        //
        failureList = new Vector();
      
        // Step through the list of m-beans and activate each one.
        //
        for (Enumeration enum = restartList.elements(); enum.hasMoreElements();) {
            // Phase I: Get the object name of the m-bean to recover.
            //
            lntrace("restartServices: Phase I: Get the object name of the m-bean to recover.");
            ObjectName name = (ObjectName) enum.nextElement();
         
            // Phase II: Recover the m-bean from the repository.
            //
            lntrace("restartServices: Phase II: Recover the m-bean from the repository.");
            Object recovered = null;
            try {
                recovered = recover(name);
            } catch (InstanceNotFoundException ee) {
                System.err.println("WARNING: Couldn't find the m-bean=" + name);
                trace("restartServices: exception=" + ee.getMessage());
                failureList.addElement(new ActivationFailure(name, ee));
                continue;
            }

            // Phase III: Activate the recovered m-bean.
            //
            lntrace("restartServices: Phase III: Activate the recovered m-bean.");
            try {
                activate(recovered, false);
            } catch (Exception ee) {
                System.err.println("WARNING: Unable to activate the m-bean=" + name);
                trace("restartServices: exception=" + ee.getMessage());
                failureList.addElement(new ActivationFailure(name, ee));
            }
        }
        activated = true;
      
        lntrace("restartServices: --- Recover and Activate m-beans End ---\n");
    }

    /**
     * Warns the user if the service is removed.
     */
    public void deleteCmf() {
        System.out.println("\nWARNING: Recovery mechanism disabled !\n");
    }


    // PROTECTED METHODS
    //------------------
   
    /**
     * Activates an object that implements the ActivatableIf and
     * is not yet active (or override with the force parameter).
     *
     * @param <VAR>object</VAR> The object to be activated.
     * @param <VAR>force</VAR> Force activation.
     */
    protected void activate(Object object, boolean force) {
       
        trace("activate: object=" + object);
        trace("activate: force=" + force);
      
        // Try to activate the object,
        // if it implements the ActivatableIf and is not already active.
        //
        if ((object instanceof ActivatableIf) && (!(((ActivatableIf) object).isActive()) || (force))) {
            ((ActivatableIf) object).performStart();
        }
      
        trace("activate: ok !");
    }


    // PRIVATE METHODS
    //----------------

    /**
     * Recovers an m-bean from the framework.
     *
     * @param <VAR>name</VAR> The object name of the m-bean to retrieve.
     */
    private Object recover(ObjectName name) 
        throws InstanceNotFoundException {

        trace("recover: name=" + name);
      
        // Check that the reference to the framework is set.
        //
        if (cmf == null) {
            throw new ServiceNotFoundException("Can't find the Framework.");
        }
      
        // Retrieve the m-bean from the framework.
        //
        return cmf.retrieveObject(name);
    }
   
    /**
     * Flush changes to disk.
     */
    private void sync() {
        // Check that the reference to the framework is set.
        //
        if (cmf == null) {
            throw new ServiceNotFoundException("Can't find the Framework.");
        }
      
        // Get the persistent repository.
        //
        MoRepSrvIf rep = cmf.getMoRepSrvIf();
      
        // Update this service.
        //
        try {
            rep.update(this, RESTARTSRV);
        } catch (InstanceNotFoundException ee) {
            System.err.println("ERROR: Failed to update the restart service.");
        }
    }

    /**
     * Display debug messages.
     */  
    private void trace(String message) {
        Debug.print("RestartSrv::" + message);
    }

    /**
     * Display debug messages.
     */  
    private void lntrace(String message) {
        Debug.print("\nRestartSrv::" + message);
    }

    // PRIVATE VARIABLES
    //------------------
   
    /* Reference to the framework */
    private Framework cmf = null;
   
    /* List of m-bean names registered for activation */
    private RestartList restartList = null;
   
    /* A list of activation failures */
    private transient Vector failureList = null;
   
    /* The state of the service */
    private transient boolean activated = false;
   
    /* Static object name of Restart service */
    public final static ObjectName RESTARTSRV = new ObjectName(ServiceName.DOMAIN + ":RestartSrv");
}
