// Copyright (c) 03/12/99, by Sun Microsystems, Inc.
// All rights reserved.

// "@(#)RecoveryRepSrv.java 3.3 99/03/12 SMI"

// Java imports.
//
import java.lang.*;
import java.util.*;

// Jaw imports.
//
import com.sun.jaw.reference.query.*;
import com.sun.jaw.reference.common.*;
import com.sun.jaw.reference.agent.cmf.*;
import com.sun.jaw.reference.agent.services.*;
import com.sun.jaw.impl.agent.services.persistent.*;

/**
 * This class provides a simple implementation of a persistent repository, 
 * coupled with a Restart service to provide a recovery mechanism.
 * As soon as the framework has registered the repository will either
 * retrieve and activate the Restart service or if it is not present register it
 * with the framework enabling recovery. All the m-beans that implements the 
 * ActivatableIf interface and is registered persistently in the repository
 * is recoverable and will be registered with the Restart service.
 *
 * @see RestartSrv
 * @see com.sun.jaw.reference.agent.services.MoRepSrvIf
 * @see com.sun.jaw.reference.agent.services.ActivatableIf
 * @see java.io.ObjectOutputStream
 * @see java.io.Serializable
 * @see java.io.Externalizable
 */
public class RecoveryRepSrv 
    extends PersistentRepSrv {

    // CONSTRUCTORS
    //-------------

    /**
     * Default constructor.
     */
    public RecoveryRepSrv() {
        super();
    }

    // PUBLIC METHODS
    //---------------

    /**
     * Invoked by the core management framework to register an m-bean 
     * with the repository, if the m-bean implements the ActivatableIf 
     * it will also be registered with the Restart service.
     *
     * Note: The m-bean will be persistent.
     *
     * @param <VAR>object</VAR> The m-bean to be registered.
     * @param <VAR>name</VAR> The object name with which the m-bean is 
     * to be registered.
     *
     * @exception InstanceAlreadyExistException The m-bean is already
     * registered in the object repository.
     */
    public void registerDB(Object object, ObjectName name)
        throws InstanceAlreadyExistException {

        // Register the object in the persistent repository.
        //
        super.registerDB(object, name);
      
        // Register the object with the Restart service,
        // if it implements the ActivatableIf interface.
        //
        if (object instanceof ActivatableIf) {
            RestartSrv restartsrv = (RestartSrv) getRestartSrv();
            if ((restartsrv != null)) {
                restartsrv.register(name);
            }
        }
    }
   
    /**
     * Sets the recovery mode of the repository.
     */
    public void setRecoverMode(boolean mode) {
        recover = mode;
    }

    /**
     * Gets the recovery mode of the repository.
     */   
    public boolean getRecoverMode() {
        return recover;
    }


    // PROTECTED METHODS
    //------------------

    /**
     * Removes an m-bean from the database and also from the Restart service.
     *  
     * @param <VAR>name</VAR> The object name of the m-bean to be removed. 
     *
     * @exception InstanceNotFoundException The specified m-bean was not 
     * found in the repository.
     */
    protected void unregisterDB(ObjectName name)
        throws InstanceNotFoundException {
      
        super.unregisterDB(name);
      
        // Unregister the m-bean from the Restart service.
        //
        RestartSrv restartsrv = (RestartSrv) getRestartSrv();
        if (restartsrv != null) {
            restartsrv.unregister(name);
        }
    }
   
    /**
     * Enables the synhronization of m-beans with the framework.
     * This is called as soon as the framework has registered.
     * Retrieve the Restart service, if present, and restart all
     * registered m-beans or register the Restart service with the framework,
     * if not present, to make it manageable.
     *
     * @param <VAR>cmf</VAR> The reference to the framework.
     */
    protected void enableSync(Framework cmf) {
   
        // Enables the synhronization of m-beans with the framework.
        //
        super.enableSync(cmf);
      
        // Get the Restart service,
        // if it was found and recover is true restart the m-beans,
        // else add the Restart service to the framework enabling recovery.
        //
        RestartSrv restartsrv = (RestartSrv) getRestartSrv();
        if (restartsrv != null) {
            if (recover) {
                if (restartsrv != null) {
                    trace("enableSync: ** recovery enabled ! **");
                    restartsrv.restartServices();
                }
            }
        } else {
            setRestartSrv();
            trace("enableSync: ** enable recovery mechanism ! **");
        }
    }
   
    /**
     * Register the Restart service with the framework.
     */
    protected void setRestartSrv() {
   
        trace("setRestartSrv: Add the Restart service to the framework, name=" + RestartSrv.RESTARTSRV);
         
        // Check that the reference to the framework is set.
        //
        if (super.cmf == null) {
            throw new ServiceNotFoundException("Can't find the Framework.");
        }   
      
        // Make the Restart service manageable,
        // by adding it to the framework.
        //
        try {
            cmf.addDBObject(new RestartSrv(cmf), RestartSrv.RESTARTSRV);
        } catch (InstanceAlreadyExistException ee) {
            trace("setRestartSrv: The Restart service is already registered with the framework.");
        }
      
        trace("setRestartSrv: ok !");
    }
   
    /**
     * Get the Restart service from the framework.
     */
    protected Object getRestartSrv() {

        trace("setRestartSrv: Retrieves the Restart service from the framework, name=" + RestartSrv.RESTARTSRV);

        // Check that the reference to the framework is set.
        //
        if (super.cmf == null) {
            throw new ServiceNotFoundException("Can't find the Framework.");
        }
      
        // Get handle on m-bean controlled by the core management framework.
        //
        Vector vec = super.getObject(RestartSrv.RESTARTSRV, null);
        Object object = null;
        if (!vec.isEmpty()) {
            NamedObject named = (NamedObject)vec.firstElement();
            object = named.getObject();
        }
      
        trace("setRestartSrv: ok ! found=" + object);
      
        return object;
    }
   
    // PRIVATE METHODS
    //----------------

    /**
     * Displays debug messages.
     */
    private void trace(String msg) {
        Debug.print("RecoveryRepSrv::" + msg);
    }
   
    // PRIVATE VARIABLES
    //------------------

    /* Recover mode status */
    private boolean recover = true;
}
