// Copyright (c) 03/24/99, by Sun Microsystems, Inc.
// All rights reserved.

// "@(#)CounterBean.java 3.4 99/03/24 SMI"

// Java imports
//
import java.awt.*;
import java.awt.event.*;

// Jaw imports
//
import com.sun.jaw.reference.common.*;
import com.sun.jaw.reference.agent.cmf.*;
import com.sun.jaw.reference.agent.services.*;

/**
 * This class is a simple m-bean that implements a counter.
 * The counter is incremented through the framework to ensure that
 * the repository is updated. To be registered with the Restart service
 * it implements the ActivatableIf interface. The counter is
 * displayed in a small frame on the screen. If the frame is closed the
 * agent will be stopped with the counter.
 * 
 * @see RecoveryRepSrv
 */
public class CounterBean 
    extends Frame 
    implements Runnable, ActivatableIf {


    // CONSTRUCTORS
    //-------------

    /**
     * Default constructor.
     */
    public CounterBean() {
        super("Counter");
    }


    // PUBLIC METHODS
    //---------------

    /**
     * Initializes this m-bean and calls the performStart method
     * to start the counter thread.
     *
     * @param <VAR>cmf</VAR> The core management framework to register the service with.
     * @param <VAR>name</VAR> Object name.
     * @param <VAR>db</VAR> Indicates if persistent storage is required.
     * @param <VAR>list</VAR> The modification list to use for setting up parameters.
     *
     * @exception InstanceAlreadyExistException
     *            The m-bean is already registered in the repository.
     */
    public void initCmf(Framework cmf, ObjectName name, boolean db, ModificationList list)
        throws InstanceAlreadyExistException {

        // Initialise local pointer to the Core Management Framework.
        //
        this.cmf = cmf;
        this.name = name;

        // Phase I: Initialize the counter.
        //
        init();
      
        // Phase II: Start the counter.
        //
        performStart();

        // Phase III: Register the service within the core management framework.
        //
        if (db) {
            cmf.addDBObject(this, name);
        } else {
            cmf.addObject(this, name);
        }      
    }

    /**
     * Calls the performStop method if counter is active.
     */
    public void deleteCmf() {
        if (phase == ACTIVE) {
            performStop();
        }
        dispose();
        counterThread = null;
    }
   
    /**
     * Starts the counter.
     */
    public void performStart() {
        // Check that we are initialized.
        //
        if (phase < INITIALIZE) {
            init();
        }
      
        // Start the counter thread.
        //
        if ((counterThread == null) || (!counterThread.isAlive())) {
            counterThread = new Thread(this);
        }
        counterThread.start();

        // Set the phase state.
        //
        phase = ACTIVE;
    }

    /**
     * Stops the counter.
     */
    public void performStop() {
        // Stop the counter thread.
        //
        if ((counterThread != null) && (counterThread.isAlive())) {
            counterThread.interrupt();
            try {
                counterThread.join();
            } catch (InterruptedException ix) {
                // Ignore...
            }
        }

        // Set the phase state.
        //      
        phase = PASSIVE;
    }
   
    /**
     * Tests if the counter is active.
     *
     * @return Returns true if the counter is active, otherwise false.
     */
    public boolean isActive() {
        return (phase == ACTIVE);
    }
   
    /**
     * Sets the count value of the counter.
     *
     * @param count The new counter value.
     */
    public void setCount(int count) {
        this.count = count;
    }
   
    /**
     * Gets the count value of the counter.
     * 
     * @return A snap shot of the counter value.
     */
    public int getCount() {
        return count;
    }
   
    /**
     * Increments the counter and update the counter display.
     */
    public void run() {
        while ((Thread.currentThread() == counterThread) && (!counterThread.interrupted())) {
            try {
                // Sets the value of the count property.
                // This will cause the repository to be updated
                // with the new value.
                //
                cmf.setValue(name, "Count", new Integer(++count), null);
            
                // Repaints the counter with the new count value.
                //
                repaint();
            
                // Wait a bit.
                //
                Thread.sleep(delay);
            } catch (InterruptedException ee) {
                phase = PASSIVE;
                break;
            } catch (Exception ee) {
                // An error occured..
                // Stop the thread and display the error message.
                //
                phase = PASSIVE;
                System.err.println("WARNING: " + ee.getMessage());
                break;
            }
        }
    }

    /**
     * Initializes the counter.
     */
    public void init() {

        // Setup the frame specifics.
        //
        setSize(width, height); // x,y
        enableEvents(AWTEvent.MOUSE_EVENT_MASK);
        enableEvents(AWTEvent.WINDOW_EVENT_MASK);
        show();

        // Setup the double-buffering to eliminating flashing.
        //
        bufImage = createImage(width, height);
        bufGraph = bufImage.getGraphics();
        bufGraph.setFont(new Font("TimesRoman", Font.BOLD, 18));

        // Set the phase state.
        //
        phase = INITIALIZE;
    }
   
    /**
     * Updates the counter.
     *
     * @param g The graphics context to use for printing.
     */
    public void update(Graphics g) {
        draw(bufGraph);
        paint(g);
    }
   
    /**
     * Draws the counter.
     *
     * @param g The graphics context to use for printing.
     */
    public void draw(Graphics g) {
        g.clearRect(0, 0, width, height);
        g.drawString(String.valueOf(count), 90, 50);
    }
   
    /**
     * Paints the counter.
     *
     * @param g The graphics context to use for printing.
     */
    public void paint(Graphics g) {
        // Check that we are initialized.
        //
        if (phase >= INITIALIZE) {
            g.drawImage(bufImage, 0, 0, this);
        }
    }
   
    /**
     * Processes mouse events occurring on the counter.
     *
     * @param The mouse event.
     */
    public void processMouseEvent(MouseEvent e) {
        if (MouseEvent.MOUSE_PRESSED == e.getID()) {
            if (phase == PASSIVE) {
                performStart();
            } else if (phase == ACTIVE) {
                performStop();
            }
        }
    }
   
    /**
     * Processes events occurring on the counter.
     *
     * @param e The event.
     */
    public void processEvent(AWTEvent e) {
        switch (e.getID()) {
        case Event.WINDOW_ICONIFY:
            performStop();
            break;
        case Event.WINDOW_DEICONIFY:
            performStart();
            break;
        case Event.WINDOW_DESTROY:
            System.exit(0);
            break;
        }  
        super.processEvent(e);
    }


    // PRIVATE VARIABLES
    //------------------

    /* Count property */
    private int count = 0;
   
    /* Frame size */
    private int height = 60;
    private int width = 200;
   
    /* Counter information */
    private int delay = 2000;
    private transient Thread counterThread;
   
    /* Double buffering */
    private transient Image bufImage;
    private transient Graphics bufGraph;
   
    /* M-bean information */
    private Framework cmf = null;
    private ObjectName name = null;
   
    /* Phase states */
    private transient int phase = 0;
    private final int INITIALIZE = 1;
    private final int ACTIVE  = 2;
    private final int PASSIVE = 3;
}
