// Copyright (c) 03/12/99, by Sun Microsystems, Inc.
// All rights reserved.

// "@(#)Agent.java 3.5 99/03/12 SMI"

// Java imports.
//
import	java.util.*;

// Jaw imports.
//
import  com.sun.jaw.impl.common.*;
import	com.sun.jaw.reference.common.*;
import	com.sun.jaw.reference.agent.cmf.*;
import	com.sun.jaw.reference.agent.services.*;
import  com.sun.jaw.impl.agent.services.mlet.*;
import	com.sun.jaw.impl.agent.services.jawdiscovery.*;

/**
 * This class provides an example of an agent that instanciates 
 * a framework initialized with a repository offering recovery.
 * The agent registers some adaptors, services and a counter m-bean with a
 * restart service, provided and managed by the repository.
 * If the agent is stopped and restarted the adaptors, services and
 * counter m-bean will be recovered and reactivated.
 *
 * @see RecoveryRepSrv
 * @see CounterBean
 */
public class Agent 
    implements Runnable {

    // PUBLIC METHODS
    //---------------

    public static void main(String argv[]) {                 
        new Agent().execute(argv);
    }

    /**
     * When an object implementing interface Runnable is used to create a thread,
     * starting the thread causes the object's run method to be called in that
     * separately executing thread.
     */
    public void run() {
        execute(new String[0]);
    }


    // PRIVATE METHODS
    //----------------
  
    /**
     * Initialization phase of the agent.
     */
    private void execute(String argv[]) {

        // Parses the system properties specified at the 
        // command-line to determine the specified trace level.
        //
        Debug.parseDebugProperties();

        try {
            // Create the recovery repository.
            //
            Class repSrv = Class.forName(srvClassRep);
            MoRepSrvIf rep    = (MoRepSrvIf)repSrv.newInstance();

            // Setup the repository information and 
            // configure the repository with it.
            //
            Vector	params = new Vector(2);
            params.insertElementAt(dbHome, 0);
            params.insertElementAt(dbName, 1);
            rep.setConfig(params);

            // Create the Framework and initialize it with the 
            // newly created repository.
            //
            cmf = new Framework((MoRepSrvIf)rep, null);
      
            // Retrieve the Framework domain.
            //
            domain = cmf.getDomain();

            // Add the MetaDataService.
            //
            if (cmf.getMetaDataSrvIf() == null) {
                addObject(srvClassMtd, new ObjectName(domain + ":" + mtdClass), true);
            }

            // Add the MLet service.
            //
            ObjectName mletName = new ObjectName(domain + ":" + mletClass);
            if (!cmf.contains(mletName)) {
                addObject(srvClassMlet, mletName, true);
            }

            // Add the Launcher service.
            //
            ObjectName launcherName = new ObjectName(domain + ":" + launcherClass);
            if (!cmf.contains(launcherName)) {
                addObject(srvClassLauncher, launcherName, true);
            }

            // Add the HTTP adaptor service and check its state.
            //
            ObjectName httpName = new ObjectName(domain + ":" + httpClass + httpKeys);
            if (!cmf.contains(httpName)) {
                addObject(adpClassHttp, httpName, true);
            }
      
            // Add the HTML adaptor service and check its state.
            //
            ObjectName htmlName = new ObjectName(domain + ":" + htmlClass + htmlKeys);
            if (!cmf.contains(htmlName)) {
                addObject(adpClassHtml, htmlName, true);
            }
      
            // Add the RMI adaptor service and check its state.
            //
            ObjectName rmiName= new ObjectName(domain + ":" + rmiClass + rmiKeys);
            if (!cmf.contains(rmiName)) {
                addObject(adpClassRmi, rmiName, true);
            }
      
            // Add the Discovery service and check its state.
            //
            ObjectName discoveryName = new ObjectName(domain + ":" + srvClassDisc);
            if (!cmf.contains(discoveryName)) {
                addObject(srvClassDisc, discoveryName, true);
            }
      
            // Add the CounterBean and check its state.
            //
            ObjectName counterName = new ObjectName(domain + ":" + counterClass);
            if (!cmf.contains(counterName)) {
                addObject(beanClassCounter, counterName, true);
            }

            // Wait for the adaptors to initialize.
            //
            try {
                trace("\nNOTE: Sleeping 2 seconds... ");
                Thread.sleep(2000); // 2 seconds
            } catch (Exception e) {
                // ignore ....
            }
      
            // Do a state check.
            //
            trace("\n*** STATE STATISTICS ***");
            isActivate(httpName, "adaptor HTTP/TCP on port 8081");
            isActivate(htmlName, "adaptor HTML on port 8082");
            isActivate(rmiName, "adaptor RMI on port 1099");
            isActivate(discoveryName, "service DISCOVERY on port 9000");
            isActivate(counterName, "m-bean COUNTER");
            trace("*** STATE STATISTICS ***\n");
      
            // Display message.
            //
            usage();

        } catch (Exception ee) {
            System.err.println("Exception ! " + ee.getMessage());
            Debug.printException(ee);
        }
    }
  
    /**
     * Register an m-bean persistent or non-persistent in the framework.
     */
    private Object addObject(String className, ObjectName localName, boolean db) {
        Object object = null;
        try {
            if (db) {
                trace("ADD " + className + " PERSISTENT to the Framework with the NAME=" + localName + "\n");
                object = cmf.newDBObject(className, localName, null);
            } else {
                trace("ADD " + className + " NON-PERSISTENT to the Framework with the NAME=" + localName + "\n");
                object = cmf.newObject(className, localName, null);
            }
        } catch (Exception ee) {
            System.err.println("Failed to add the object to the Framework ! " + localName);
            Debug.printException(ee);
        }
        return object;
    }

    /**
     * Retrieve an m-bean from the framework and
     * activate it, if it implements the ActivatableIf interface.
     */
    private void isActivate(ObjectName localName, String name) {
        Object object = getObject(localName);
        if ((object != null) && (object instanceof ActivatableIf)) {
            if (((ActivatableIf) object).isActive()) {
                trace("STATE: The " + name + " is ACTIVE");
            } else {
                trace("STATE: The " + name + " is NOT ACTIVE");
            }
        }
    }
  
    /**
     * Retrieve an m-bean from the framework.
     */
    private Object getObject(ObjectName localName) {
        Object object = null;
        try {
            Vector mbeans = cmf.getObject(localName, null);
            if (!mbeans.isEmpty()) {
                object = ((NamedObject)mbeans.firstElement()).getObject();
            }
        } catch (Exception ee) {
            System.err.println("Failed to get the object from the Framework ! " + localName);
            Debug.printException(ee);
        }
        return object;
    }
  

  
    /**
     * Display the help information.
     */
    private void usage() {
        System.out.println("\nusage:");
        System.out.println("------\n");
        System.out.println("  After a few seconds you should see a window with a counter.\n" +
                           "  Stop the agent by either pressing Ctrl+C or closing the counter window.\n" +
                           "  Note the counter value before stopping the agent.\n" +
                           "  After you have stopped the agent, restart it and see it recover\n" +
                           "  to its previous state. Note the counter continues its count\n" +
                           "  from where it was stopped before.\n"
                           );
    }
  
    /**
     * Display debug information.
     */
    private void trace(String message) {
        System.out.println(message);
    }
  
 
    // PRIVATE VARIABLES
    //------------------
  
    /* HTML adaptor information */
    private String adpClassHtml = "com.sun.jaw.impl.adaptor.html.AdaptorServerImpl";
    private String htmlClass    = "com.sun.jaw.impl.adaptor.html.AdaptorServerImplMO";
    private String htmlKeys     = ".protocol=html,port=8082";

    /* HTTP adaptor information */
    private String adpClassHttp = "com.sun.jaw.impl.adaptor.http.AdaptorServerImpl";
    private String httpClass    = ServiceName.APT_HTTP;
    private String httpKeys     = ".protocol=http,port=8081";
  
    /* RMI adaptor information */
    private String adpClassRmi = "com.sun.jaw.impl.adaptor.rmi.AdaptorServerImpl";
    private String rmiClass    = ServiceName.APT_RMI;
    private String rmiKeys     = ".protocol=rmi,port=1099";

    /* Repository service information */
    private String srvClassRep = "RecoveryRepSrv";
    private String dbHome = DefaultPaths.getTmpDir();
    private String dbName = "recovery";  

    /* Metadata service information */
    private String srvClassMtd = "com.sun.jaw.impl.agent.services.light.MetaDataSrv";
    private String mtdClass    = ServiceName.META;
  
    /* Mlet service information */
    private String srvClassMlet = "com.sun.jaw.impl.agent.services.mlet.MLetSrv";
    private String mletClass    = "com.sun.jaw.impl.agent.services.mlet.MLetSrvMO";
  
    /* Launcher service information */
    private String srvClassLauncher = "com.sun.jaw.impl.agent.services.mlet.LauncherSrv";
    private String launcherClass    = "com.sun.jaw.impl.agent.services.mlet.LauncherSrvMO";  
  
    /* Discovery service information */
    private String srvClassDisc = "com.sun.jaw.impl.agent.services.jawdiscovery.DiscoveryResponder";
    
    /* Launcher service information */
    private String beanClassCounter = "CounterBean";
    private String counterClass    = "CounterBeanMO";  
    
    /* Reference to framework */
    private Framework cmf;
  
    /* The default domain of the framework */
    String domain;
}
