// Copyright (c) 03/24/99, by Sun Microsystems, Inc.
// All rights reserved.

// "@(#)build.java 3.4 99/03/24 SMI"

import java.io.*;

/**
 * The "kstat" example runs only on machines with Solaris Os.
 *
 * The class build automatically generates a M-bean for any
 * kernel information that can be accessed using the kstat interface.
 * The class takes as input a module name and instance name and generates
 * the associated Java M-bean. The M-bean will make use of the KernelStat
 * class that allows to access kstat info from Java.
 *
 * In order to start the class, you need to invoke:
 *      java build <module-name> <instance-name>
 * where
 *      - module-name is the module name to access (such as "le")
 *      - instance-name is the specific instance of the module to address (such as "le0")
 *
 * As an example, the program could be invoked as follows in order to generate
 * a M-Bean for accessing kernel information on the ethernet driver "le":
 *      java build le le0
 *
 * In this example, the program will generate a M-bean called "le0.java".
 * Note that in order to work you need to have in your LD_LIBRARY_PATH access
 * to the libstat.so library.
 * In order to ease the generation of M-beans and their associated C-beans a
 * script called "doit" is provided under examples/mo/kstat.
 * The script takes the same input as the build program. But the script invokes
 * the build program, then invokes mogen, then javac for
 * compiling the generated M-bean and the objects created by mogen.
 *
 * Then, if you have a Java Dynamic Management agent supporting the HTML adaptor
 * (for instance the base agent), then you can try to instantiate the newly created bean
 * using a Web browser.
 * To do so, go to the admin page. In the field class name, enter instanceName
 * where instance name could be "le0".
 *
 * @version     3.4     03/24/99
 * @author      Sun Microsystems, Inc
 */
public class build {
 
    private static boolean createFile(String fileName, StringBuffer content) {
        RandomAccessFile file;
    
        //
        // Check if exist
        //
        File f = new File(fileName);
    
        if (f.exists()) {
            String sname = new String(fileName + "~");
            File   f1    = new File(sname);
      
            if (f1.exists()) {
                f1.delete();
            }
      
            f.renameTo(f1);
        }
    
        try {
            file = new RandomAccessFile(fileName, "rw");
            file.writeBytes(content.toString());
            file.close();
        } catch (IOException e) {
            System.err.println("Cannot create " + fileName + "\n" + e.toString());
            return (false);
        }

        System.out.println("Generating: " + fileName);
    
        return (true);
    }
  
    /**
     * Build a better property name
     */
    private static String bestPropName(String pName) {
        StringBuffer name = new StringBuffer();
        char         ch;
        boolean      upper = false;
        //
        // be sure that the name does not contains special characters
        // like % = / + -
        //
    
        for (int i = 0; i < pName.length(); i++) {
            ch = pName.charAt(i);
            switch (ch) {
            case '=':
            case '%':
            case '/':
            case '+':
            case '-':
            case '*':
            case ' ':
            case '\\':
                //
                // Just remove and put a capital letter on next step
                //
                upper = true;
                break;
            default:
                if (upper) {
                    ch = Character.toUpperCase(ch);
                    upper = false;
                }
	
                name.append(ch);
                break;
            }
        }
    
        // name.append(pName);
    
        return (name.toString());
    }
  
    private static boolean buildSources(String mod, String inst) {
   
        final String TAB  = "  ";
        final String TAB2 = "      ";
        KernelStat   ks   = new KernelStat();
        StringBuffer buf  = new StringBuffer();
    
        // System.out.println("building (" + mod + ", " + inst + ")...");
    
        //
        // Package for this particular example
        //
        buf.append("/**\n * ");
        buf.append("Generated by the build class");
        buf.append("\n */\n\n");
    
        buf.append("public class " + inst + " implements java.io.Serializable {\n");
    
        //
        // Add a KernelStat Element
        //
        buf.append(TAB + "private KernelStat ks;\n");
    
        //
        // End of fields
        //
        buf.append("\n");
    
        //
        // Add a contructor
        //
        buf.append(TAB  + "public " + inst + "() {\n" +
                   TAB2 + "ks = new KernelStat(\"" + mod + "\", \"" + 
                   inst + "\");\n" +
                   TAB  + "}\n\n");
    
        //
        // Look all the attributes to build the getter
        //
        String prop = ks.statProperties(mod, inst);
    
        if (prop == null) {
            System.err.println("No properties found for (" + mod + ", " + inst + ")");
            return (false);
        }
    
        // System.out.println("prop = " + prop);
    
        boolean done = false;
        int     index, sep;
        String  elt;
        String  pName, tName, bName;
    
        while (!done) {
            index = prop.indexOf(' ');
      
            if (index >= 0) {
                elt  = prop.substring(0, index);
                prop = prop.substring(index + 1).trim();
            }
            else {
                elt  = prop;
                done = true;
            }
      
            // System.out.println("Processing " + elt);
      
            sep = elt.indexOf(':');
      
            if (sep < 0) {
                System.err.println("Invalid element (" + elt + ")");
                continue;
            }
      
            pName  = elt.substring(0, sep);
            tName  = elt.substring(sep +1);
      
            bName  = bestPropName(pName);
      
            buf.append(TAB   + "public " + tName + " get" + bName + "() {\n");
            buf.append(TAB2  + "return (ks.statGet" + tName + "Value(\"" +
                       pName + "\"));\n");
            buf.append(TAB   + "}\n\n");
 
        }
    
        //
        // End of file
        //
        buf.append("}");
    
        // System.out.println("\nFile:\n" + buf.toString());
    
        return (createFile(inst + ".java", buf));
    }
  
    //
    // Arguments is <module name | - > <instance name | - > <module type>
    //
    public static void main(String args[]) {
        if (args.length == 0) {
            System.err.println("Usage:...");
            System.exit(1);
        }
    
    
        String modName  = null;
        String instName = null;
        int    modType  = -1;
    
        if (! args[0].equals("-")) {
            modName = args[0];
        }
    
        if (args.length > 1) {
            if (! args[1].equals("-")) {
                instName = args[1];
            }
        }
    
        if (args.length > 2) {
            if (args[2].equals("named")) {
                modType = KernelStat.STAT_NAMED;
            }
            else if (args[2].equals("io")) {
                modType = KernelStat.STAT_IO;
            }
            else if (args[2].equals("intr")) {
                modType = KernelStat.STAT_INTR;
            }
            else if (args[2].equals("raw")) {
                modType = KernelStat.STAT_RAW;
            }
            else if (args[2].equals("timer")) {
                modType = KernelStat.STAT_TIMER;
            }
            else {
                System.err.println("Invalid type! " +
                                   "(must be: named, io, intr, raw, timer)");
                System.exit(1);
            }
        }
    
        //
        // Get the module:instance list concerned
        //
        KernelStat ks;
        String     list;
    
        ks   = new KernelStat();
        list = ks.statGetList(modName, instName, modType);
    
        if (list == null) {
            System.err.println("No elements found");
            System.exit(1);
        }
    
        // System.out.println("Found:\n" + list);
    
        boolean done = false;
        int     index, sep;
        String  elt;
    
        while (!done) {
            index = list.indexOf(' ');
      
            if (index >= 0) {
                elt  = list.substring(0, index);
                list = list.substring(index + 1);
            }
            else {
                elt  = list;
                done = true;
            }
      
            // System.out.println("Processing " + elt);
      
            sep = elt.indexOf(':');
      
            if (sep < 0) {
                System.err.println("Invalid element (" + elt + ")");
                continue;
            }
      
            modName  = elt.substring(0, sep);
            instName = elt.substring(sep +1);
      
            if (! buildSources(modName, instName)) {
                System.err.println("cannot build the file for (" +
                                   modName + "," + instName);
            }
        }
    }
}
