// Copyright (c) 03/12/99, by Sun Microsystems, Inc.
// All rights reserved.

// "@(#)Season.java 3.3 99/03/12 SMI"

// java import
import java.util.*;

// jaw import
//
import com.sun.jaw.reference.common.*;
import com.sun.jaw.reference.agent.cmf.*;
import com.sun.jaw.impl.agent.services.scheduler.*;


/**
 *
 * A simple definition of an m-bean.
 * The m-bean shows how to:
 *      - implement the SchedulerListener interface
 *      - use the initCmf method to register the m-bean, insert a new scheduler-event into the scheduler vector 
 *        and start the scheduler-service
 *      - register itself to receive SchedulerEvents
 *      - use the deleteCmf method to stop the scheduler-service and remove itself as a listener
 *
 * The object has two properties :
 *      - "Temp" a read only property
 *      - "State" a read only property
 * 
 * @see  com.sun.jaw.impl.agent.services.scheduler.Scheduler
 *
 */

public class Season implements java.io.Serializable, SchedulerListener {

    // CONSTRUCTOR
    //------------
    public Season() {
	
        // Initialisation.
        //
        state = "initial state";
        temp = new Float(0);
        sch = new Scheduler();
	
        // Register a listener for receiving scheduler events.
        //
        sch.addSchedulerListener(this);
    }

    // INIT/DELETE CMF
    //----------------
    /**
     * Register the m-bean, add m-bean as a listener and start the scheduler service.
     *
     * @param     agent  a reference to the core management framework.
     * @param     name  the object name to use for registering.
     * @param     db.
     * @param     list.
     *
     * @exception java.lang.IllegalAccessException
     *            The method cannot access the class definition of the specified m-bean.
     * @exception com.sun.jaw.reference.common.ServiceNotFoundException
     *            The requested service is not supported.
     * @exception com.sun.jaw.reference.common.InstanceAlreadyExistException
     *            The m-bean is already registered in the repository.
     * @exception IllegalArgumentException
     *            One of the parameters in the call to the method is invalid.
     * @exception InvalidPropertyValueException The value specified for a property is invalid.
     */
    public void initCmf(Framework agent, ObjectName name, boolean db, ModificationList list) throws 
    IllegalAccessException, ServiceNotFoundException, 
	InstanceAlreadyExistException, IllegalArgumentException,
	InvalidPropertyValueException {
	  
        // Creates an ObjectName if it was not given.
        // NOTE:
        //      The ObjectName can be null if the request was made
        //      throught the M-Let service.
        //
        if (name == null)
            name = new ObjectName(agent.getDomain(),this.getClass().getName());
	
        // Register the m-bean.
        //
        if (db) {
            agent.addDBObject(this, name);
        } else {
            agent.addObject(this, name);
        }
	
        // Add events into the season scheduler vector.
        // The scheduler is not active at the initialization.
        //
        performAddSeasonEvent();
    }

    /**
     * Stop the scheduler and remove the m-bean as a listener.
     */
    public void deleteCmf() {
	
        // Stops the scheduler service.
        //
        sch.performStop();

        // Remove a listener for scheduler events.
        //
        sch.removeSchedulerListener(this);

        // Set reference to scheduler to null.
        //
        sch = null;	
    }
  
    // SERVICE IMPLEMENTATION
    //-----------------------
    /**
     * This handle will be called each time the date of an event in the scheduler vector 
     * has been reached.
     * 
     * The event to handle can be one of these :
     *  - EventSpring
     *  - EventSummer
     *  - EventAutumn
     *  - EventWinter
     *
     * @param <VAR>event</VAR> The scheduler event.
     *
     * @see SchedulerEvent
     */
    public void handleScheduler(SchedulerEvent event) {
  
        if (event instanceof EventSpring) {
            temp = new Float(15);
            state = "Spring weather";
        }
        else if (event instanceof EventSummer) {
            temp = new Float(30);
            state = "Summer weather";
        }
        else if (event instanceof EventAutumn) {
            temp = new Float(15);
            state = "Autumn weather";
        }
        else if (event instanceof EventWinter) {
            temp = new Float(0);
            state = "Winter weather";
        }
        else
            return;
    }
  
    // ACTIONS 
    //--------
    /**
     * Inserts the events into the scheduler vector.
     *
     * @exception InvalidPropertyValueException
     *            The periodicity has negative values or
     *			the date event is before the current date.
     */
    private void performAddSeasonEvent() throws 
    InstanceAlreadyExistException, InvalidPropertyValueException {
	
        Date currentDate = new Date();
        Date dSpring = new Date();
        Date dSummer = new Date();
        Date dAutumn = new Date();
        Date dWinter = new Date();
	  
        // Dates of the 4 scheduler events.
        //
        dSpring.setTime(currentDate.getTime() + 2 * Scheduler.ONE_SECOND);
        dSummer.setTime(currentDate.getTime() + 4 * Scheduler.ONE_SECOND);
        dAutumn.setTime(currentDate.getTime() + 6 * Scheduler.ONE_SECOND);
        dWinter.setTime(currentDate.getTime() + 8 * Scheduler.ONE_SECOND);
	
        // Scheduler periods of the 4 scheduler events.
        //
        long periodSpring = 8 * Scheduler.ONE_SECOND;
        long periodSummer = 8 * Scheduler.ONE_SECOND;
        long periodAutumn = 8 * Scheduler.ONE_SECOND;
        long periodWinter = 8 * Scheduler.ONE_SECOND;

        // Scheduler events.
        //
        evtSpring = new EventSpring(sch);
        evtSummer = new EventSummer(sch);
        evtAutumn = new EventAutumn(sch);
        evtWinter = new EventWinter(sch);
	  
        // Add events.
        //
        sch.performAddEvent((SchedulerEvent)evtSpring, dSpring, periodSpring);
        sch.performAddEvent((SchedulerEvent)evtSummer, dSummer, periodSummer);
        sch.performAddEvent((SchedulerEvent)evtAutumn, dAutumn, periodAutumn);
        sch.performAddEvent((SchedulerEvent)evtWinter, dWinter, periodWinter);
    }
    
    /**
     * Starts the scheduler service and add events into the scheduler vector if it is empty.
     * @exception InstanceAlreadyExistException The m-bean is already registered in the repository.
     * @exception InvalidPropertyValueException The value specified for a property is invalid.
     */
    public void performStartScheduler() throws 
    InstanceAlreadyExistException, InvalidPropertyValueException {
	
        // Add events into the scheduler vector.
        //
        if (sch.isListOfEventsEmpty() == true) {
            performAddSeasonEvent();
        }
	
        // Starts the scheduler service if it is not running.
        //
        sch.performStart();
    }
  
    /**
     * Stops the scheduler service and reset the season object.
     */
    public void performStopScheduler() {
	
        // Stops the scheduler service if it is running.
        //
        sch.performStop();
	  
        // Delete remaining events from the scheduler vector.
        //
        sch.performRemoveAllEvents();
		
        state = "initial state";
        temp = new Float(0);
    }
  
  
    // GETTERS AND SETTERS
    //--------------------
    /**
     * Returns the current value of the "State" property.
     */
    public String getState() {
        return state;
    }

    /**
     * Returns the current value of the "Temp" property.
     */
    public Float getTemp() {
        return temp;
    }
  
    /**
     * Returns the number of events registered into the scheduler vector.
     */
    public int getNumberOfEvents() {
        return sch.getNumberOfEvents();
    }
  
    /**
     * Tests if the <CODE>Scheduler</CODE> is active.
     */
    public boolean isActive() {
        return sch.isActive();
    }
  
    /**
     * Returns the flag that indicates if the scheduler notifies past events or not.
     */
    public boolean getNotifyPastEvents() {
        return sch.getNotifyPastEvents();
    }
  
    /**
     * Returns in an array all the events registered into the scheduler vector.
     */
    public SchedulerEvent[] getListOfEvents() {
        return sch.getListOfEvents();
    }
    
  
    // PRIVATE VARIABLES
    //------------------
    // M-bean properties.
    //
    private String		state = null;
    private Float			temp = null;
  
    // Private Variables.
    //
    private Scheduler		sch = null;

    private EventSpring	evtSpring = null;
    private EventSummer	evtSummer = null;
    private EventAutumn	evtAutumn = null;
    private EventWinter	evtWinter = null;
}
