// Copyright (c) 03/24/99, by Sun Microsystems, Inc.
// All rights reserved.

// "@(#)Weather.java 3.4 99/03/24 SMI"

// java import
import java.util.*;

// jaw import
//
import com.sun.jaw.reference.common.*;
import com.sun.jaw.reference.agent.cmf.*;
import com.sun.jaw.impl.agent.services.alarm.*;

/**
 * A simple definition of an m-bean.
 * The m-bean shows how to:
 *      - implement the AlarmClockListener interface
 *      - use the initCmf method to register the m-bean and start the alarm-clock
 *      - register itself to receive AlarmClockEvents
 *      - use the deleteCmf method to stop the alarm-clock and remove itself as a listener
 *
 * The object has one property:
 *      - "Temp" a read only property
 *      - "State" a read only property
 * 
 * @see  com.sun.jaw.impl.agent.services.alarm.AlarmClock
 */

public class Weather implements java.io.Serializable, AlarmClockListener {

    public Weather() {
        // Register a listener for receiving alarm-clock events.
        //
        alarm.addAlarmClockListener(this);
    }

    /**
     * Getter for the "State" property. The method
     * is compliant with the Bean design pattern.
     *
     * @return the current value of the "State" property.
     */
    public String getState() {
        return state;
    }

    /**
     * Getter for the "Temp" property. The method
     * is compliant with the Bean design pattern.
     *
     * @return the current value of the "Temp" property.
     */
    public Float getTemp() {
        return temp;
    }

    /**
     * Register the m-bean, add m-bean as a listener and start the clock-alarm.
     *
     * @param     agent  a reference to the core management framework.
     * @param     name  the object name to use for registering.
     *
     * @exception java.lang.IllegalAccessException
     *            The method cannot access the class definition of the specified m-bean.
     * @exception com.sun.jaw.reference.common.ServiceNotFoundException
     *            The requested service is not supported.
     * @exception com.sun.jaw.reference.common.InstanceAlreadyExistException
     *            The m-bean is already registered in the repository.
     * @exception IllegalArgumentException
     *            One of the parameters in the call to the method is invalid.
     */
    public void initCmf(Framework agent, ObjectName name, boolean db, ModificationList list)
        throws InstanceAlreadyExistException {

        // Creates an ObjectName if it was not given.
        // NOTE:
        //      The ObjectName can be null if the request was made
        //      throught the M-Let service.
        //
        if (name == null)
            name = new ObjectName(agent.getDomain(),this.getClass().getName());

        // Register the m-bean.
        //
        if (db) {
            agent.addDBObject(this, name);
        } else {
    	    agent.addObject(this, name);
        }

        // Sets the timeout value.
        //
        alarm.setTimeoutAsLong(timeout);

        // Starts the alarm-clock count.
        //
        alarm.performStart();
    }

    /**
     * Stop the alarm-clock and remove the m-bean as a listener.
     */
    public void deleteCmf() {
        // Stops the alarm-clock count.
        //
        alarm.performStop();

        // Remove a listener for alarm-clock events.
        //
        alarm.removeAlarmClockListener(this);

        // Set reference to alarm-clock to null.
        //
        alarm = null;	
    }

    /**
     * This handle will be called each time the alarm-clock has exceeded its timeout.
     */
    public void handleAlarmClock(AlarmClockEvent event) {
        // Set the "Temp" property between -30 to 30 degrees.
        //
        temp = new Float((Math.random() * 60.0) - 30.0);

        // Set the "State" property.
        //
        Float tempLow = new Float(15.0);
        Float tempHigh = new Float(25.0);
        if (temp.floatValue() < tempLow.floatValue()) {
            state = "Cold weather";
        } else if (temp.floatValue() > tempHigh.floatValue()) {
            state = "Hot weather";
        } else {
            state = "Nice weather";
        }

        // Re-start the alarm-clock count.
        //
        alarm.performStart();
    }

    // M-bean properties.
    //
    private String        state = "initial state";
    private Float 	temp = new Float((Math.random() * 60.0) - 30.0);

    // Private Variables.
    //
    private AlarmClock alarm = new AlarmClock();
    private Long timeout = new Long(2500);
}
