// Copyright (c) 03/12/99, by Sun Microsystems, Inc.
// All rights reserved.

// "@(#)Client.java 3.3 99/03/12 SMI"

import java.net.*;
import java.util.*;

// jaw import
//
import com.sun.jaw.reference.common.*;
import com.sun.jaw.impl.adaptor.rmi.*;
import com.sun.jaw.impl.agent.services.monitor.*;

/**
 * The Client implements a simple management application.
 * The example shows how the counter and gauge monitors can be used
 * to observe the property values of a specific m-beans properties and 
 * receive events according to changes in the values of these m-bean properties.
 * This client works in conjunction with the MbeanCreator manager.
 *
 * Prior to starting the client application, make sure an agent is running.
 *
 * @version     3.3     03/12/99 
 * @author      Sun Microsystems, Inc
 *
 * @see MbeanCreator
 * @see AskMe
 * @see com.sun.jaw.impl.agent.services.monitor.CounterMonitor
 * @see com.sun.jaw.impl.agent.services.monitor.GaugeMonitor
 * @see com.sun.jaw.impl.adaptor.rmi.AdaptorClient
 * @see com.sun.jaw.impl.base.Main
 */

public class Client implements Runnable {

    // CONSTRUCTOR
    //------------
    public Client() {
    }
  
    /**
     * It is possible to specify on the command line the implementation to use
     * for the adaptor function. 
     */
    public static void main(String argv[]) {
	
        System.exit( new Client().execute(argv) ) ;	
    }
  
    /**
     * When an object implementing interface Runnable is used to create a thread, 
     * starting the thread causes the object's run method to be called in that
     * separately executing thread.
     */
    public void run() {
	
        String[] s = new String[0];
        execute(s);
    }
  
  
    // PRIVATE METHODS
    //----------------
    private int execute(String argv[]) {
	
        try {
            // Set the host name of the remote agent.
            //
            String agentHost = InetAddress.getLocalHost().getHostName();
            if (argv.length >= 1) {
                agentHost = argv[0];
            }

            // Set the port number of the remote host.
            //
            int agentPort = 1099;
            if (argv.length >= 2) {
                agentPort = Integer.decode(argv[1]).intValue();
            }

            // Create an adaptor client to enable the client to manage the remote agent
            // and initializes the communications with the remote agent.
            //
            trace(">>> CREATE and INITIALIZE communication with the remote agent,");
            trace("HOST     = " + agentHost);
            trace("PORT     = " + agentPort);
            trace("PROTOCOL = RMI");
            trace("SERVER   = " + ServiceName.APT_RMI);
            AdaptorClient adaptor = new AdaptorClient();
            adaptor.connect(null, agentHost, agentPort, ServiceName.APT_RMI);
            trace("Communication ok\n");

            // Get the domain name from the remote agent.
            //
            String domain = adaptor.getDomain();

            // Creates an instance of the AskMe m-bean in the 
            // remote agent.
            //
            String askMeClass = "AskMe";
            ObjectName askMeName = new ObjectName(domain + ":" + askMeClass + ".id=1");
            trace(">>> CREATE an instance of an m-bean in the remote agent,");
            trace("M-BEAN      =" + askMeClass);
            trace("OBJECT NAME = " + askMeName);
            AskMeMO askMe = (AskMeMO) adaptor.cb_newMO(askMeClass, askMeName, null);
            trace("Creation ok\n");
      
            // Access the properties of the AskMe m-bean remotely
            // and get the initial property values.
            //
            trace(">>> GET the initial property values of an m-bean from the remote agent,");
            trace("OBJECT NAME = " + askMeName);
            int nbObjects = askMe.getNbObjects().intValue();
            int freeMem = askMe.getFreeMem().intValue();
            trace("PROPERTY \"NbObjects\" = " + nbObjects);
            trace("PROPERTY \"FreeMem\"   = " + freeMem);
            trace("Get ok\n");
      
            // Create a monitor listener.
            // This listener will be registered with the monitors.
            //
            ClientListener listen = new ClientListener();

            // Create a counter monitor m-bean.
            // 
            String objMonitorClass = "com.sun.jaw.impl.agent.services.monitor.CounterMonitor";
            ObjectName objMonitorName = new ObjectName(domain + ":com.sun.jaw.impl.agent.services.monitor.CounterMonitorMO.id=NbObject");
            trace(">>> CREATE an instance of a counter monitor in the remote agent,");
            trace("M-BEAN      =" + objMonitorClass);
            trace("OBJECT NAME = " + objMonitorName);
            CounterMonitorMO objMonitor = (CounterMonitorMO) adaptor.cb_newMO(objMonitorClass, objMonitorName, null);
            trace("Creation ok\n");

            // Initialize the comparison level and offset value for counter monitor.
            //
            Integer objHigh = new Integer(nbObjects + 5); // initial comparison level
            Integer offSet  = new Integer(5); // offset value

            // Set the values of the counter monitor object.
            // The counter monitor will check every second the value of the
            // "NbObjects" property for the AskMe m-bean.
            // A threshold reached monitor event will be emited if the 
            // "NbObjects" property value reached or exceeded the comparison level.
            // If the comparison level is reached or exceeded the comparison level
            // will be incremented by the offset value.
            //
            trace(">>> SET the property values of the counter monitor,");
      
            trace("PROPERTY \"ObservedObject\"    = " + askMeName);
            objMonitor.setObservedObject(askMeName);
      
            trace("PROPERTY \"ObservedProperty\"  = NbObjects");
            objMonitor.setObservedProperty("NbObjects");
      
            trace("PROPERTY \"NotifyOnOff\"       = true");
            objMonitor.setNotifyOnOff(Boolean.TRUE);
      
            trace("PROPERTY \"ComparisonLevel\"   = " + objHigh);
            objMonitor.setComparisonLevel(objHigh);
      
            trace("PROPERTY \"OffsetValue\"       = " + offSet);
            objMonitor.setOffsetValue(offSet);
      
            trace("PROPERTY \"GranularityPeriod\" = 1 second");
            objMonitor.setGranularityPeriod(new Integer(1000));
      
            objMonitor.performStart();
            trace("Set ok\n");
      
            // Register a monitor listener with the counter monitor,
            // enabling the client to receive monitor events emitted by the
            // counter monitor. 
            //
            trace(">>> ADD a monitor listener to the counter monitor,");
            trace("enabling the client to receive monitor events.\n");
            objMonitor.addMonitorListenerMO(listen);

            // Create a gauge monitor m-bean.
            //
            String memMonitorClass = "com.sun.jaw.impl.agent.services.monitor.GaugeMonitor";
            ObjectName memMonitorName = new ObjectName(domain + ":com.sun.jaw.impl.agent.services.monitor.GaugeMonitorMO.id=FreeMem");
            trace(">>> CREATE an instance of a gauge monitor in the remote agent,");
            trace("M-BEAN      =" + memMonitorClass);
            trace("OBJECT NAME = " + memMonitorName);
            GaugeMonitorMO memMonitor = (GaugeMonitorMO) adaptor.cb_newMO(memMonitorClass, memMonitorName, null);
            trace("Creation ok\n");
 
            // Initialize the threshold values for the gauge monitor.
            //
            Integer memHigh = new Integer(freeMem - 50000);
            Integer memLow = new Integer(freeMem - 150000);

            // Set the values of the gauge monitor object.
            // The gauge monitor will check every second the value of the
            // "FreeMem" property for the AskMe m-bean.
            // A threshold high monitor event will be emited if the "FreeMem"
            // property value reached or exceeded the high threshold value.
            // A threshold low monitor event will be emited if the "FreeMem"
            // property value reached or fell below the low threshold value.
            //
            trace(">>> SET the property values of the gauge monitor,");
      
            trace("PROPERTY \"ObservedObject\"     = " + askMeName);
            memMonitor.setObservedObject(askMeName);
      
            trace("PROPERTY \"ObservedProperty\"   = FreeMem");
            memMonitor.setObservedProperty("FreeMem");
      
            trace("PROPERTY \"NotifyLowOnOff\"     = true");
            memMonitor.setNotifyLowOnOff(Boolean.TRUE);
      
            trace("PROPERTY \"NotifyHighOnOff\"    = true");
            memMonitor.setNotifyHighOnOff(Boolean.TRUE);
      
            trace("PROPERTY \"ThresholdLowValue\"  = " + memLow);
            memMonitor.setThresholdLowValue(memLow);
      
            trace("PROPERTY \"ThresholdHighValue\" = " + memHigh);
            memMonitor.setThresholdHighValue(memHigh);
      
            trace("PROPERTY \"GranularityPeriod\"     = 1 second");
            memMonitor.setGranularityPeriod(new Integer(1000));
      
            memMonitor.performStart();
            trace("Set ok\n");
      
            // Register a monitor listener with the gauge monitor,
            // enabling the client to receive monitor events emitted by the
            // gauge monitor. 
            //
            trace(">>> ADD a monitor listener to the gauge monitor,");
            trace("enabling the client to receive monitor events.\n");
            memMonitor.addMonitorListenerMO(listen);

            // Wait before cleaning up and terminating.
            //
            trace("Now run the MbeanCreator client to create and delete m-beans.");
            trace("After a few seconds the client should start to receive monitor events");
            trace("and display the results in the client window.");
            trace("Press <Enter> to clean up and terminate...");
            System.in.read();

            // Remove the listener from the from CounterMonitor and GuageMonitor objects. 
            //
            trace(">>> REMOVE the monitor listener from the counter and gauge monitors.");
            objMonitor.removeMonitorListenerMO(listen);
            memMonitor.removeMonitorListenerMO(listen);
            trace("Removal ok\n");

            // Deletes an instance of the counter monitor in the 
            // remote agent.
            //
            trace(">>> DELETE an instance of a counter monitor in the remote agent,");
            trace("OBJECT NAME = " + objMonitorName);
            adaptor.deleteMO(objMonitorName);
            trace("Deletion ok\n");

            // Deletes an instance of the gauge monitor in the 
            // remote agent.
            //
            trace(">>> DELETE an instance of a gauge monitor in the remote agent,");
            trace("OBJECT NAME = " + memMonitorName);
            adaptor.deleteMO(memMonitorName);
            trace("Deletion ok\n");

            // Deletes an instance of AskMe in the 
            // remote object server.
            //
            trace(">>> DELETE an instance of an m-bean in the remote agent,");
            trace("OBJECT NAME = " + askMeName);
            adaptor.deleteMO(askMeName);
            trace("Deletion ok\n");
      
            // Terminate communication with the remote agent.
            //
            adaptor.disconnect();
            return 0;
        } catch(Exception e) {
            trace("Got an exception !");
            e.printStackTrace();
            return 1;
        }
    }
  
    /**
     * Display trace messages.
     */
    private void trace(String msg) {
        System.out.println(msg);
    }
}
