// Copyright (c) 03/12/99, by Sun Microsystems, Inc.
// All rights reserved.

// "@(#)Table.java 3.3 99/03/12 SMI"

// Java imports.
//
import java.io.*;
import java.sql.*;
import java.util.*;
import java.lang.*;

// Jaw imports.
//
import com.sun.jaw.reference.common.*;

/**
 * This class provides a basic service for managing database tables.
 *
 * @version     3.3     03/12/99
 * @author      Sun Microsystems, Inc
 */


public class Table {

    /**
     * Creates a Table to manage database tables.
     */
    public Table(Connection connection) {
        jdbcConnection = connection;
    }

    /**
     * Creates a database table.
     *
     * @param <VAR>tableName</VAR> The name of the database table to be created.
     * @param <VAR>tableStructure</VAR> The structure of the database table to be created.
     *
     * @exception SQLException A database-access error occured.
     */
    public void createTable(String tableName, String tableStructure)
        throws SQLException {

        trace("createTable: create the table=" + tableName);

        Statement statement = null;
        String create_clause = "CREATE TABLE ";

        try {
            statement = jdbcConnection.createStatement();
         
            create_clause += tableName + " (" + tableStructure + ")";
         
            trace("createTable: execute query=" + create_clause);
         
            statement.executeUpdate(create_clause);
        } finally {
            closeStatement(statement);
        }
      
        trace("createTable: ok !");
    }

    /**
     * Drops a database table.
     *
     * @param <VAR>tableName</VAR> The name of the database table to be droped.
     *
     * @exception SQLException A database-access error occured.
     */
    public void destroyTable(String tableName)
        throws SQLException {

        trace("destroyTable: drop the table=" + tableName);

        Statement statement = null;
        String drop_clause = "DROP TABLE ";

        try {
            statement = jdbcConnection.createStatement();
         
            drop_clause += tableName;
         
            trace("destroyTable: execute query=" + drop_clause);
         
            statement.executeUpdate(drop_clause);
        } finally {
            closeStatement(statement);
        }
      
        trace("destroyTable: ok !");
    }

    /**
     * Tests the existence of a database table.
     *
     * @param <VAR>tableName</VAR> The name of the database table.
     */
    public boolean tableExists(String tableName) {
   
        trace("tableExists: table exist ? " + tableName);
   
        ResultSet resultset = null;
        boolean found = false;

        try {
            DatabaseMetaData databasemetadata = jdbcConnection.getMetaData();
            if (databasemetadata == null) {
                return false;
            }

            for(resultset = databasemetadata.getTables(null, null, null, null); resultset.next();) {
                String table = resultset.getString("TABLE_NAME");
                if (table.equalsIgnoreCase(tableName)) {
                    found = true;
                    break;
                }
            }

        }
        catch(SQLException ee) {
            traceSqlException(ee);
        } finally {
            closeResult(resultset);
        }
      
        trace("tableExists: table exist=" + found);
      
        return found;
    }

    /**
     * Determines the size of a database table.
     *
     * @param <VAR>tableName</VAR> The name of the database table.
     *
     * @return The number of existing rows in the table.
     *
     * @exception SQLException A database-access error occured.
     */
    public int tableSize(String tableName)
        throws SQLException {
      
        trace("tableSize: size of table=" + tableName);
      
        int count = 0;
      
        Statement statement = null;
        ResultSet resultset = null;
      
        String select_clause = "SELECT count(*) FROM ";
      
        try {
            select_clause += tableName;
         
            statement = jdbcConnection.createStatement();

            trace("tableSize: execute query=" + select_clause);
         
            resultset = statement.executeQuery(select_clause);

            if (!resultset.next()) {
                return 0;
            }

            count = resultset.getInt(1);
            if (resultset.wasNull()) {
                return 0;
            }
        } finally {
            closeResult(resultset);
            closeStatement(statement);
        }
      
        trace("tableSize: size=" + count);
        trace("tableSize: ok !");
      
        return count;
    }

    /**
     * Displays the SQLException and additional error information.
     *
     * @param <VAR>exception</VAR> The SQLException to displayed.
     */
    public static void traceSqlException(SQLException exception) {
        Debug.println("\n *** SQL Exception ***");
        Debug.println(Debug.REP_DEBUG, exception.getMessage());
        while ((exception = exception.getNextException()) != null) {
            Debug.println("\n *** Additional SQL Exceptions ***");
            Debug.println(Debug.REP_DEBUG, exception.getMessage());
        }
    }

    /**
     * Close a ResultSet.
     *
     * @param <VAR>resultset</VAR> The ResultSet to be closed and resources released.
     */
    protected void closeResult(ResultSet resultset) {
        if (resultset == null) {
            return;
        }

        try {
            resultset.close();
        } catch(SQLException ee) {
            traceSqlException(ee);
        }
    }

    /**
     * Close a Statement.
     *
     * @param <VAR>statement</VAR> The Statement to be closed and resources released.
     */
    protected void closeStatement(Statement statement) {
        if (statement == null) {
            return;
        }

        try {
            statement.close();
        } catch(SQLException ee) {
            traceSqlException(ee);
        }
    }

    /**
     * Close a PreparedStatement.
     *
     * @param <VAR>preparedstatement</VAR> The PreparedStatement to be closed and resources released.
     */
    protected void closePreparedStatement(PreparedStatement preparedstatement) {
        if (preparedstatement == null) {
            return;
        }

        try {
            preparedstatement.close();
        } catch(SQLException ee) {
            traceSqlException(ee);
        }
    }

    /**
     * Displays debug information.
     *
     * @param <VAR>message</VAR> The message to displayed.
     */      
    protected void trace(String message) {
        Debug.print(Debug.REP_DEBUG, tableClass + "::" + message);
    }
 
    /* Connection to the specified database. */
    protected Connection jdbcConnection = null;
   
    /* Class name.*/
    protected String tableClass = "Table";
}
