// Copyright (c) 03/12/99, by Sun Microsystems, Inc.
// All rights reserved.

// "@(#)SerializationInputStream.java 3.4 99/03/12 SMI"

// Java imports.
//
import java.lang.*;
import java.io.*;
import java.util.*;

// Jaw imports.
//
import com.sun.jaw.reference.common.*;
import com.sun.jaw.reference.agent.cmf.*;

/**
 * This subclass of ObjectInputStream implement the resolveObject 
 * method to allow objects to be resolved from an alternate source.
 * All the instances of SerializationMarker will be replaced by 
 * an instance of Framework.
 *
 * @version     3.4     03/12/99
 * @author      Sun Microsystems, Inc
 *
 * @see SerializationMarker
 */


public class SerializationInputStream 
    extends ObjectInputStream {

    /**
     * Create an SerializationInputStream that reads from the 
     * specified InputStream and resolve objects.
     *
     * @exception StreamCorruptedException The version or magic number are incorrect.
     * @exception IOException An exception occurred in the underlying stream.
     */
    public SerializationInputStream(InputStream inputStream, Framework cmf)
        throws IOException, StreamCorruptedException {

        super(inputStream);
        try {
            super.enableResolveObject(true);
        } catch (SecurityException ee) {
            // Report the error ...
            System.err.println("Object replacement disabled !!");
            Debug.printException(ee);
        }
        this.cmf = cmf;
        Debug.println("Object replacement enabled !!");
    }

    /**
     * This method allows objects to be resolved with an alternate object.
     * It looks for the SerializationMarker object and 
     * replaces it with he appropriate Framework reference.
     *
     * @exception IOException An exception occurred in the underlying stream.
     */
    protected Object resolveObject(Object object)
        throws IOException {
        // Resolved all instances of the replacement object with the Framework object.
        //
        if (object instanceof SerializationMarker) {
            return this.cmf;
        }
        return object;
    }

    /**
     * This method allow classes to be fetched from an alternate source.
     * This method will be invoked only once for each unique
     * class in the stream.
    
     * @exception IOException An exception occurred in the underlying stream.
     * @exception java.lang.ClassNotFoundException Class of a serialized object
     *      cannot be found.
     */
    protected Class resolveClass(ObjectStreamClass objectStreamClass)
        throws IOException, ClassNotFoundException {

        // Try the primordial loader to load the class.
        // 
        try {
            return super.resolveClass(objectStreamClass);
        } catch (ClassNotFoundException ee) {
            // Drop through...
            Debug.printException(ee);
        }
      
        // Try the Framework loader to load the class.
        //
        ClassLoader cmfloader = null;
        if ((cmf != null) && ((cmfloader = cmf.getClassLoader()) != null)) {
            try {
                return cmfloader.loadClass(objectStreamClass.getName());
            } catch (ClassNotFoundException ee) {
                // Drop through...
                Debug.printException(ee);
            }
        }   

        // Try the static loader repository to load the class.
        // 
        return LoaderRepository.loadClass(objectStreamClass.getName());
    }

    /* The Framework used to replace the SerializationMarker. */
    private transient Framework cmf = null;
}
