// Copyright (c) 03/12/99, by Sun Microsystems, Inc.
// All rights reserved.

// "@(#)RegistryTable.java 3.3 99/03/12 SMI"

// Java imports.
//
import java.sql.*;
import java.util.*;
import java.lang.*;

/**
 * This class provides a basic service for managing a registry table in a database.
 * The registry manages the different domains in an agent. Each domain is
 * represented by a domain table in the database.
 *
 * The registry table structure:
 *
 *      Column Name             Column Type
 *
 *      domainName               VARCHAR
 *
 * @version     3.3     03/12/99
 * @author      Sun Microsystems, Inc
 *
 * @see DomainTable
 */


public class RegistryTable extends Table {

    /**
     * Creates a RegistryTable that manipulates registry tables in the
     * database. If the table doesn't exist in the database, it will be created.
     *
     * @param <VAR>connection</VAR> A session with a specific database.
     * @param <VAR>registryName</VAR> The name of the registry table.
     *
     * @exception SQLException A database-access error occured.
     */
    public RegistryTable(Connection connection, String registryName)
        throws SQLException {
      
        super(connection);      
        tableClass = "RegistryTable";
        this.registryName = registryName;
            
        if (!tableExists(registryName)) {
            createRegistryTable();
        }
      

    }

    /**
     * Creates a registry table in the specified database.
     *
     * @exception SQLException A database-access error occured.
     */
    protected void createRegistryTable()
        throws SQLException {

        trace("createRegistryTable: create registry table=" + registryName);
    
        // SQL types.
        //
        String varcharName = "VARCHAR";

        ResultSet resultset = null;      
      
        // Setup the SQL types.
        //
        try {
            DatabaseMetaData databasemetadata = jdbcConnection.getMetaData();
            if (databasemetadata != null) {
                trace("createRegistryTable: Check SQL type name: VARCHAR");
         
                resultset = databasemetadata.getTypeInfo();
                if (resultset != null) {
                    short datatype = 0;
                    while (resultset.next()) {
                        datatype = resultset.getShort("DATA_TYPE");
                        switch (datatype) {
                        case Types.CHAR:
                        case Types.VARCHAR:
                        case Types.LONGVARCHAR:
                            varcharName = resultset.getString("TYPE_NAME");
                            break;
                        }
                    }
                } // else use the default SQL type settings
            } // else use the default SQL type settings
        } catch (SQLException ee) {
            // Use the deafault SQL types settings.
            traceSqlException(ee);
        } finally {
            closeResult(resultset);
        }

        // Set the column types.
        //
        String varcharColumn = varcharName + "(" + VARCHAR_SIZE + ")";
         
        // Set the table structures.
        //
        // registry: name VARCHAR(X)
        String registryStructure = domainColumn 	   + " " + varcharColumn;

        trace("createRegistryTable: table structure=" + registryStructure);
      
        createTable(registryName, registryStructure);
      
        trace("createRegistryTable: ok !");
    }

    /**
     * Adds a new row to the registry table and creates the related domain table.
     *
     * @param <VAR>domainName</VAR> The name of the domain within which the managed object exists.
     *
     * @return A DomainTable representing the newly added domain.
     *
     * @exception SQLException A database-access error occured.
     */
    public DomainTable add(String domainName)
        throws SQLException {

        trace("add: domain=" + domainName);
      
        Statement statement = null;
      
        String insert_clause = "INSERT INTO ";
      
        try {
            insert_clause += registryName;
            insert_clause += " VALUES ('" + domainName + "')";
         
            statement = jdbcConnection.createStatement();

            trace("add: execute query=" + insert_clause);
         
            int count = statement.executeUpdate(insert_clause);
         
            trace("add: row count=" + count);
        } finally {
            closeStatement(statement);
        }
      
        trace("add: ok !");
      
        return new DomainTable(jdbcConnection, domainName);
    }

    /**
     * Removes existing rows from the registry table and drop related domain tables.
     *
     * @param <VAR>domainName</VAR> The name of the domain within which the managed object exists.
     *
     * @exception SQLException A database-access error occured.
     */
    public void delete(String domainName)
        throws SQLException {

        trace("delete: domain=" + domainName);
      
        Statement statement = null;
      
        String delete_clause = "DELETE FROM ";
        String where_clause = " WHERE ";

        try {      
            delete_clause += registryName;
            where_clause  += domainColumn + " = '" + domainName + "'";
            delete_clause += where_clause;
         
            statement = jdbcConnection.createStatement();
         
            trace("delete: execute query=" + delete_clause);
         
            int count = statement.executeUpdate(delete_clause);
         
            trace("delete: row count=" + count);
        } finally {
            closeStatement(statement);
        }
 
        trace("delete: ok !");
      
        new DomainTable(jdbcConnection, domainName).drop();
    }

    /**
     * Retrieves the domain table for the specified domain.
     *
     * @param <VAR>domainName</VAR> The name of the domain within which the managed object exists.
     *
     * @return A DomainTable representing the domain.
     *
     * @exception SQLException A database-access error occured.
     */
    public DomainTable get(String domainName)
        throws SQLException {
      
        trace("get: domain=" + domainName);
      
        return new DomainTable(jdbcConnection, domainName);
    }

    /**
     * Retrieves all the domain tables in the registry.
     *
     * @return A list of DomainTable representing the domains.
     *
     * @exception SQLException A database-access error occured.
     */
    public Vector getAll()
        throws SQLException {
      
        Vector result = new Vector();
      
        Statement statement = null;
        ResultSet resultset = null;

        String select_clause = "SELECT ";
      
        try {
            select_clause += domainColumn + " FROM " + registryName;
         
            statement = jdbcConnection.createStatement();
         
            trace("getAll: execute query=" + select_clause);
         
            resultset = statement.executeQuery(select_clause);

            for (resultset = statement.executeQuery(select_clause); resultset.next();) {
                String domain = resultset.getString(domainColumn);
                if (!resultset.wasNull()) {
                    try {
                        result.addElement(new DomainTable(jdbcConnection, domain));
                    } catch (SQLException ee) {
                        System.err.println("Failed to load the domain table <" + domain + "> : " + ee.getMessage());
                    }
                }
            }
        } finally {
            closeResult(resultset);
            closeStatement(statement);
        }      
      
        trace("getAll: ok !");
            
        return result;
    }

    /**
     * Tests if the row exists in the registry table. 
     *
     * @param <VAR>domainName</VAR> The name of the domain within which the managed object exists.
     *
     * @return True if the row exists in the registry table, else false.
     *
     * @exception SQLException A database-access error occured.
     */
    public boolean contains(String domainName)
        throws SQLException {

        trace("contains: domain=" + domainName);

        Statement statement = null;
        ResultSet resultset = null;

        String select_clause = "SELECT ";
        String where_clause = " WHERE ";
      
        try {
            select_clause += domainColumn + " FROM " + registryName;
            where_clause += domainColumn + " = '" + domainName + "'";
            select_clause += where_clause;
         
            statement = jdbcConnection.createStatement();
         
            trace("contains: execute query=" + select_clause);
         
            resultset = statement.executeQuery(select_clause);

            if (resultset.next()) {
                return true;
            }
        } finally {
            closeResult(resultset);
            closeStatement(statement);
            trace("contains: ok !");
        }
     
        return false;
    }

    /**
     * Determines the size of the registry table.
     *
     * @return The number of existing rows in the registry table.
     *
     * @exception SQLException A database-access error occured.
     */
    public int size()
        throws SQLException {
      
        return tableSize(registryName);
    }

    /**
     * Determines the size of all the domain tables in the registry table.
     *
     * @return The number of existing rows of all the domain tables in the registry table.
     *
     * @exception SQLException A database-access error occured.
     */
    public int sizeAll()
        throws SQLException {
      
        int total = 0;
        Vector domainTables = getAll();
      
        for (int i = 0; i < domainTables.size(); i++) {
            total += ((DomainTable) domainTables.elementAt(i)).size();
        }

        trace("size: size=" + total);
        trace("size: ok !");
      
        return total;
    }

    /**
     * Tests if there is any existing rows in the registry table.
     *
     * @return True if there are no existing rows in the registry table, else false.
     *
     * @exception SQLException A database-access error occured.
     */
    public boolean isEmpty()
        throws SQLException {
      
        trace("isEmpty: empty table ? " + registryName);
      
        return (size() == 0);
    }

    /**
     * Removes the registry table definition and all data with related domain tables.
     *
     * @exception SQLException A database-access error occured.
     */
    public void drop()
        throws SQLException {
      
        trace("drop: drop table=" + registryName);
      
        Vector domainTables = getAll();

        for (int i = 0; i < domainTables.size(); i++) {
            ((DomainTable) domainTables.elementAt(i)).drop();
        }      
           
        destroyTable(registryName);
      
        trace("drop: ok !");
    }

    /* Registry name. */
    private String registryName = null;   
   
    /* Column names. */
    private final String domainColumn = "domainName";
   
    /** The string column size. */
    public static int VARCHAR_SIZE = 256;
}
