// Copyright (c) 03/15/99, by Sun Microsystems, Inc.
// All rights reserved.

// "@(#)JDBCRepositorySrv.java 3.4 99/03/15 SMI"

// Java imports.
//
import java.io.*;
import java.net.*;
import java.sql.*;
import java.util.*;
import java.lang.*;
import java.lang.reflect.*;
 
// Jaw imports.
//
import com.sun.jaw.reference.query.*;
import com.sun.jaw.reference.common.*;
import com.sun.jaw.reference.agent.cmf.*;
import com.sun.jaw.reference.agent.services.*;
import com.sun.jaw.impl.agent.services.light.*;
import com.sun.jaw.impl.agent.services.persistent.*;


/**
 * This class provides an implementation of an object repository.
 * The repository contains a mixture of volatile and persistent m-beans.
 * The repository stores volatile m-beans in memory and persistent m-beans in 
 * a database. The database is accessed through the JDBC API.
 *
 * M-beans are stored in domain tables. The M-bean state is stored through
 * serialization. Domain tables are registered with a registry table,
 * that manages the different domain tables.
 *
 * The repository synchronize m-beans with the current framework. To enable
 * m-bean synchronization the framework should be the first object to register
 * with the repository and objects should not mark fields that reference the
 * framework with the transient keyword.
 *
 * @version     3.4     03/15/99
 * @author      Sun Microsystems, Inc
 *
 * @see com.sun.jaw.reference.agent.services.MoRepSrvIf
 * @see RegistryTable
 * @see DomainTable
 * @see SerializationConvertion
 */


public class JDBCRepositorySrv implements MoRepSrvIf {


    // MORepSrvIf INTERFACE METHODS
    //-----------------------------

    /**
     * Default constructor.
     */
    public JDBCRepositorySrv() {
        super();
        converter = new SerializationConvertion();
        exportList = new RepositorySrv();
    }

    /**
     * Indicates whether the repository supports filtering.
     * If the repository does not support filtering, 
     * the core management framework will perform filtering 
     * for the repository.
     *
     * @return True if the filtering is supported, false otherwise.
     *
     */
    public boolean isQuerySrv() {
        return false;
    }

    /**
     * Indicates whether the repository offers persistent storage.
     *
     * @return True if the repository offers persistent storage, false otherwise.
     *
     */
    public boolean isPersistent() {
        return true;
    }

    /**
     * Retrieves an m-bean from the repository.
     *
     * @param <VAR>name</VAR> The object name of the m-bean to be retrieved.
     *
     * @return The retrieved m-bean, or null if it could not be retrieved.
     */
    public Object retrieve(ObjectName name) {

        lntrace("retrieve: >>> name=" + name);
      
        // Check connection ?
        //
        isConnected();

        Object object = null;

        // Retrieve the m-bean from the list of exported objects.
        //
        if ((object = exportList.retrieve(name)) != null) {
            traceln("retrieve: >>> ok !");
            return object;
        }
      
        // Retrieve the m-bean from the database
        // and update the list of exported objects.
        //
        try {
            if ((object = getDomainEntry(name)) != null) {
                exportList.register(object, name);
            }
        } catch (InstanceAlreadyExistException ee) {
            // The m-bean has already been exported.
        } catch (SQLException ee) {
            System.err.println("Failed to restore the object <" + name + "> : " + ee.getMessage());
            Table.traceSqlException(ee);
            return null;
        } catch (IllegalArgumentException ee) {
            System.err.println("Failed to restore the object <" + name + "> : " + ee.getMessage());
            Debug.printException(ee);
            return null;
        }
      
        traceln("retrieve: >>> ok !");
      
        return object;
    }

    /**
     * Verifies whether an m-bean is registered with the repository.
     *
     * @param <VAR>name</VAR> The object name of the m-bean.
     *
     * @return True if the m-bean is registered with the repository, false 
     * otherwise.
     */
    public boolean contains(ObjectName name) {

        lntrace("contains: >>> name=" + name);

        // Check connection ?
        //
        isConnected();
    
        // Check the list of exported objects for the m-bean.
        //
        if (exportList.contains(name)) {
            traceln("contains: >>> ok ! =true");
            return true;
        }

        boolean contain = containsDomainEntry(name);

        traceln("contains: >>> ok ! =" + contain);

        return contain;
    }

    /**
     * Verifies whether a singleton m-bean is registered with the 
     * repository. A singleton m-bean has no
     * search key in its object name and is, therefore, the only instance of the 
     * class permitted in the domain.
     *
     * @param <VAR>object</VAR> The m-bean.
     *
     * @return  True if the m-bean is registered with the repository, 
     * false otherwise.
     */
    public boolean contains(Object object) {

        ObjectName name = null;
      
        try {
            name = new ObjectName(this.domain, object.getClass().getName());
        } catch(IllegalArgumentException ee) {
            throw new PersistentRepositoryException(ee.getClass().getName() + ": " + ee.getMessage());
        }
      
        return contains(name);
    }

    /**
     * Invoked by the core management framework to register a persistent 
     * m-bean with the repository.
     *
     * @param <VAR>object</VAR> The m-bean to be registered.
     * @param <VAR>name</VAR> The object name with which the m-bean is 
     * to be registered.
     *
     * @exception InstanceAlreadyExistException The m-bean is already
     * registered in the object repository.
     */
    public void registerDB(Object object, ObjectName name)
        throws InstanceAlreadyExistException {

        lntrace("registerDB: >>> name=" + name);

        register(object, name, true);
      
        traceln("registerDB: >>> ok !");
    }

    /**
     * Invoked by the core management framework to register a volatile m-bean 
     * with the repository.
     *
     * @param <VAR>object</VAR> The m-bean to be registered.
     * @param <VAR>name</VAR> The object name with which the m-bean is 
     * to be registered.
     *
     * @exception InstanceAlreadyExistException The m-bean is already
     * registered in the repository.
     */ 
    public void register(Object object, ObjectName name)
        throws InstanceAlreadyExistException {
      
        lntrace("register: >>> name=" + name);
      
        register(object, name, false);
      
        traceln("register: >>> ok !");
    }

    /**
     * Updates an m-bean registered with the repository.
     *
     * @param <VAR>object</VAR> The new instance of the m-bean.
     * @param <VAR>name</VAR> The object name of the m-bean to be updated.
     *
     * @exception InstanceNotFoundException The specified m-bean was not 
     * found in the repository.
     */
    public void update(Object object, ObjectName name)
        throws InstanceNotFoundException {

        lntrace("update: >>> name=" + name);
      
        // Check the repository for the m-bean.
        //
        if (!contains(name)) {
            throw new InstanceNotFoundException(name.toString());
        }
      
        // Update the m-bean in the database.
        //
        if (containsDomainEntry(name)) {
            try {
                updateDomainEntry(object, name);
            } catch(SQLException ee) {
                Table.traceSqlException(ee);
                throw new PersistentRepositoryException(ee.getClass().getName() + ": " + ee.getMessage());
            }
        }
            
        traceln("update: >>> ok !");
    }

    /**
     * Removes an m-bean from the repository.
     *  
     * @param <VAR>name</VAR> The object name of the m-bean to be removed. 
     *
     * @exception InstanceNotFoundException The specified m-bean was not 
     * found in the repository.
     */
    public void unregister(ObjectName name)
        throws InstanceNotFoundException {

        lntrace("unregister: >>> name=" + name);
      
        // Check the repository for the m-bean.
        //
        if (!contains(name)) {
            throw new InstanceNotFoundException(name.toString());
        }
      
        // Remove the m-bean from the list of exported objects.
        //
        try {
            exportList.unregister(name);
        } catch (InstanceNotFoundException ee) {
            // Try the persistent registry.
            // Drop through...
        }

        // Remove the m-bean from the database.
        //
        if (containsDomainEntry(name)) {
            try {
                deleteDomainEntry(name);
            } catch(SQLException ee) {
                Table.traceSqlException(ee);
                throw new PersistentRepositoryException(ee.getClass().getName() + ": " + ee.getMessage());
            }
        }
      
        traceln("unregister: >>> ok !");
    }

    /**
     * Removes a singleton m-bean from the repository. A singleton 
     * m-bean has no search key in its object name and is, therefore, the 
     * only instance of the class permitted in the domain.
     *
     * @param <VAR>object</VAR> The m-bean to be removed from the repository.
     *
     * @exception InstanceNotFoundException The specified m-bean was not 
     * found in the repository.
     */
    public void unregister(Object obj)
        throws InstanceNotFoundException {

        ObjectName name = null;
        try {
            name = new ObjectName(domain, obj.getClass().getName());
        } catch(IllegalArgumentException ee) {
            throw new PersistentRepositoryException(ee.getClass().getName() + ": " + ee.getMessage());
        }
      
        unregister(name);
    }

    /**
     * Gets handles on m-beans controlled by the repository.
     *
     * @param <VAR>name</VAR> An object name that specifies the m-beans to be
     * selected.
     * @param <VAR>query</VAR> A query to be applied to the selected m-beans.
     *
     * @return A list of named objects corresponding to the selected m-beans.
     */
    public Vector getObject(ObjectName name, QueryExp query) {

        lntrace("getObject: >>> name=" + name);

        // Check connection ?
        //
        isConnected();  
    
        if (name == null) {
            name = new ObjectName(":");
        }
      
        // Get the m-beans from the database and export them.
        //
        try {
            Vector database = getDomainEntries(name);
      
            for (int i = 0; i < database.size(); i++) {
                ObjectName objectName = (ObjectName) database.elementAt(i);
                if ((objectName != null) && !(exportList.contains(objectName))) {
                    try {
                        Object object = getDomainEntry(objectName);
                        if (object != null) {
                            exportList.register(object, objectName);
                        } else {
                            System.err.println("No object data for <" + objectName + ">");
                        }
                    } catch (InstanceAlreadyExistException oo) {
                        // The m-bean has already been exported.
                    } catch(SQLException oo) {
                        System.err.println("Failed to restore the object <" + objectName + "> : " + oo.getMessage());
                        Table.traceSqlException(oo);
                    } catch(IllegalArgumentException oo) {
                        System.err.println("Failed to restore the object <" + objectName + "> : " + oo.getMessage());
                        Debug.printException(oo);
                    }
                }
            }
        } catch(SQLException ee) {
            System.err.println("Failed to restore the object name(s) for <" + name + "> : " + ee.getMessage());
            System.err.println("Only objects exported to the JVM will be returned.");         
            Table.traceSqlException(ee);
        }
      
        traceln("getObject: >>> ok !");
      
        // Return the m-beans from the list of exported objects.
        //
        return exportList.getObject(name, query);
    }


    // GETTERS AND SETTERS
    //--------------------

    /**
     * Gets the number of m-beans registered with the repository.
     *    
     * @return The number of m-beans registered with the repository.
     *
     */
    public Integer getNbElements() {
   
        lntrace("getNbElements: count the registered objects.");
        
        // Count the m-beans in the database and the exported object list.
        //
        isConnected();
      
        int total = 0;
        try {
            total = registryTable.sizeAll() + exportList.getNbElements().intValue();
        } catch (SQLException ee) {
            Table.traceSqlException(ee);
            return new Integer(0);
        }
      
        // Synchronize the count for duplicates.
        //
        try {
            Vector duplicates = getDomainEntries(new ObjectName(":"));
            for (int i = 0; i < duplicates.size(); i++) {
                if (exportList.contains((ObjectName) duplicates.elementAt(i))) {
                    total--;
                }
            }
        } catch (SQLException ee) {
            Table.traceSqlException(ee);
            return new Integer(0);
        }

        traceln("getNbElements: >>> ok ! total=" + total);
      
        // Return the number of m-beans.
        //
        return new Integer(total);
    }

    /**
     * Gets the domain of the repository.
     *
     * @return The domain of the repository.
     */
    public String getDomain() {
        return domain;
    }

    /**
     * Sets the domain of the repository.
     */
    public void setDomain(String domain) {
        this.domain = domain;
    }


    /**
     * Configures the repository.
     * The configuration parameters must be specified in the following order:
     *  - the JDBC driver name (Mandatory)
     *  - the JDBC database connection URL (Mandatory)
     *  - the database user on whose behalf the connection is made (Optional)
     *  - the user password (Optional)
     *
     * Configures the repository as follows:
     *  - loads the JDBC driver
     *  - establish a database connection with auto commit enabled
     *  - create the main registry table
     *
     * NOTE: The repository must be configured before it can be used.
     *
     * @param <VAR>params</VAR> A vector containing the configuration parameters 
     * of the repository.
     *
     */
    public void setConfig(Vector params) {
   
        lntrace("setConfig: *** Start repository initialization ! ***");

        // The repository has not been configured.
        //
        configured = false;

        // The driver name and JDBC url are mandatory parameters.
        //
        if (params.size() < 2) {
            throw new IllegalArgumentException("JDBC driver and database connection URL are mandatory.");
        }

        // Mandatory: driver name.
        //
        jdbcDriverName = (String) params.elementAt(0);
        if ((jdbcDriverName == null) || (jdbcDriverName.length() == 0)) {
            throw new IllegalArgumentException("JDBC driver name is mandatory.");
        }
      
        // Mandatory: database url.
        //
        jdbcUrlName = (String) params.elementAt(1);
        if ((jdbcDriverName == null) || (jdbcDriverName.length() == 0)) {
            throw new IllegalArgumentException("JDBC database connection URL is mandatory.");
        }
      
        // Optional: database user on whose behalf the connection is made.
        //
        if (params.size() > 2) {
            jdbcUserName = (String) params.elementAt(2);
        }
      
        // Optional: user password.
        //
        if (params.size() > 3) {
            jdbcPassword = (String) params.elementAt(3);
        }
      
        // Start the database.
        //
        start();
      
        // The repository has been configured.
        //
        configured = true;

        traceln("setConfig: *** Repository initialization ok ! ***");
    }
 
    /**
     * Stop the repository and terminates the connection to the database.
     */  
    public void stop() {
        close(jdbcConnection);
        jdbcConnection = null;
    }

    protected void finalize() throws Throwable {
        stop();
    }


    // GENERAL METHODS
    //----------------

    /**
     * Store the object in the database, if required, and export it to
     * make it available to receive incoming calls.
     */
    private void register(Object object, ObjectName name, boolean db)
        throws InstanceAlreadyExistException {	

        // Set the reference to the Framework.
        //
        setFramework(object);

        // Check the repository for the m-bean.
        //
        if (contains(name)) {
            throw new InstanceAlreadyExistException(name.toString());
        }

        // Validate the registration.
        //
        if (!validRegistration(name)) {
            throw new IllegalArgumentException("Invalid registration: Singleton of this className exists or this m-bean is a singleton with instances of the className already exsisting=" + name);
        }
      
        // Store the object in the database, if persistent storage is required.
        //
        if (db) {
            try {
                addDomainEntry(object, name);
            } catch(SQLException ee) {
                Table.traceSqlException(ee);
                throw new PersistentRepositoryException(ee.getClass().getName() + ": " + ee.getMessage());
            } catch(IllegalArgumentException ee) {
                throw new PersistentRepositoryException(ee.getClass().getName() + ": " + ee.getMessage());
            }
        }

        // Export the object.
        //
        exportList.register(object, name);
    }
      
    /**
     * Validate the registration of a new m-bean.
     *
     * For a registration to be valid two things are checked:
     *    1) If the m-bean is a singleton, no other m-beans of this
     *       class are registered.
     *    2) If the m-bean is not a singleton, no singleton of this 
     *       class is regitered.
     */ 
    private boolean validRegistration(ObjectName name) {
   
        lntrace("validRegistration: >>> validate the registration of name=" + name);
   
        if (name.isPropertyListEmpty()) {
            if (exportList.getObject(name, null).size() != 0) {
                return false;
            } else if (containsClassName(name)) {
                return false;
            }
         
            traceln("validRegistration: >>> valid registration !");
         
            return true;
        }
      
        if (exportList.contains(new ObjectName(name.getDomain(), name.getClassName()))) {
            return false;
        } else if (containsSingleton(name)) {
            return false;
        }
      
        traceln("validRegistration: >>> valid registration !");
      
        return true;
    }

    /**
     * Set the the reference to the framework.
     */
    private void setFramework(Object object) {
        // This should be the first object to register with the
        // repository.
        // 
        if (object instanceof Framework) {
            converter.setFramework((Framework) object);
        }
    }

    /**
     * Parse an object name and retrieve host, post, service name.
     */
    private Hashtable parseObjectName(ObjectName name) {
        Hashtable values = new Hashtable(3);
      
        String domainName = name.getDomain();
        if (domainName.length() == 0) {
            domainName = this.domain;
        }
        values.put(DOMAIN, domainName);

        String className = name.getClassName();
        if (className.length() != 0) {
            values.put(CLASS, className);
        }

        if (!name.isPropertyListEmpty()) {
            values.put(PROPERTY, name.getPropertyListString());
        }
      
        return values;
    }

    /**
     * Instanciates a new object name.
     */
    private ObjectName newObjectName(String name) {
        ObjectName objname = null;
        if (name.length() == 0) {
            return null;
        }

        try {
            objname = new ObjectName(name);
        } catch (IllegalArgumentException ee) {
            System.err.println("Failed to recreate the object name <" + name + "> : " + ee.getMessage());
            return null;
        }
      
        return objname;
    }


    // GENERAL JDBC METHODS
    //---------------------

    /**
     * Start the database.
     */
    private void start() {
        // Load the JDBC driver.
        //
        trace("start: Load the JDBC driver=" + jdbcDriverName);
        if (!load(jdbcDriverName)) {
            throw new ServiceNotFoundException("Failed to load the JDBC driver=" + jdbcDriverName);
        }
      
        // Open a connection to the given database.
        //
        trace("start: Open a connection to the database=" + jdbcUrlName);
        stop(); // reset the connection
        if ((jdbcConnection = open(jdbcUrlName, jdbcUserName, jdbcPassword)) == null) {
            throw new PersistentRepositoryException("Cannot open a connection to the database=" + jdbcUrlName);
        }
           
        // Display the database access warnings.
        //
        checkForWarnings(jdbcConnection);

        // Create or open the registry table.
        //
        trace("start: Create the registry table=" + registryTableName);
        try {
            registryTable = new RegistryTable(jdbcConnection, registryTableName);
            commit();
        } catch(SQLException ee) {
            Table.traceSqlException(ee);
            throw new PersistentRepositoryException(ee.getClass().getName() + ": " + ee.getMessage());
        }
    }


    /**
     * Checks if the repository has been configured and establish a connection
     * to the database if required.
     */
    private void isConnected() {
        if (!configured) {
            throw new PersistentRepositoryException("Repository is not configured.");
        }
      
        // Open the connection to the database, if requiired.
        //
        if (jdbcConnection == null) {
            if ((jdbcConnection = open(jdbcUrlName, jdbcUserName, jdbcPassword)) == null) {
                throw new PersistentRepositoryException("Cannot open a connection to the database=" + jdbcUrlName);
            }
        }
    }

    /**
     * Loads the JDBC driver.
     */
    private boolean load(String driver) {
        try {
            // Some JVM's may not load the class until an instance is created.
            // To ensure the class is loaded create an instance at the same time.
            // eg. Class.forName(jdbcDriverName).newInstance();
            //
            Class.forName(jdbcDriverName); // JavaSoft recommended way
        } catch (ClassNotFoundException ee) {
            Debug.printException(ee);
            return false;
        }
        return true;
    }

    /**
     * Attempt to establish a connection to the given database.
     */
    private Connection open(String url, String user, String password) {
        Connection connection = null;
        try {
            connection =  DriverManager.getConnection(url, user, password);
        } catch (SQLException ee) {
            Table.traceSqlException(ee);
            return connection;
        }
        return connection;
    }

    /**
     * Terminate the connection to the database.
     */
    private void close(Connection connection) {
        if (connection == null) {
            return;
        }
    
        try {
            if (!connection.isClosed()) {
                connection.close();
            }
        } catch (SQLException ee) {
            Table.traceSqlException(ee);
        }
    }

    /**
     * Commits the changes made to the database,
     * if autocommit is disabled.
     */
    protected void commit()
        throws SQLException {

        if (jdbcConnection.getAutoCommit()) {
            return;
        }
        trace("commit: Commit changes made to the database.");
        jdbcConnection.commit();
    }
        
    /**
     * Display warnings reported by calls on this connection 
     * and clear the warnings.
     */   
    private void checkForWarnings(Connection connection) {
        try {
            SQLWarning sqlwarn = connection.getWarnings();
            for(; sqlwarn != null; sqlwarn = sqlwarn.getNextWarning()) {
                System.err.println("\n*** JDBC Warning *** \n");
                System.err.println("SQLState: " + sqlwarn.getSQLState());
                System.err.println("Message:  " + sqlwarn.getMessage());
                System.err.println("Vendor:   " + sqlwarn.getErrorCode() + "\n");
            }
            connection.clearWarnings();
        } catch (SQLException ee) {
            Table.traceSqlException(ee);
        }
    }


    // TABLE METHODS
    //--------------

    /**
     * Adds the m-bean to the database.
     */
    private void addDomainEntry(Object object, ObjectName name)
        throws SQLException {
      
        Hashtable values = parseObjectName(name);
        String domainName = (String) values.get(DOMAIN);
        String className = (String) values.get(CLASS);
        String propertyList = (String) values.get(PROPERTY);

        // Get the domain table from the registry table.
        //
        DomainTable domainTable = null;
        if (registryTable.contains(domainName)) {
            domainTable = registryTable.get(domainName);
        } else {
            domainTable = registryTable.add(domainName);
        }

        // Convert the m-bean to an array of object bytes and
        // add the new m-bean to the domain table.
        //
        domainTable.add(className, propertyList, converter.toBytes(object));

        // Commit the changes made.
        //
        commit();
    }

    /**
     * Updates the m-bean in the database.
     */
    private void updateDomainEntry(Object object, ObjectName name)
        throws SQLException {
   
        Hashtable values = parseObjectName(name);
        String domainName = (String) values.get(DOMAIN);
        String className = (String) values.get(CLASS);
        String propertyList = (String) values.get(PROPERTY);
      
        // Get the domain table from the registry table.
        //
        DomainTable domainTable = null;
        if (registryTable.contains(domainName)) {
            domainTable = registryTable.get(domainName);
        } else {
            return;
        }

        // Update the m-bean in the domain table.
        //
        domainTable.update(className, propertyList, converter.toBytes(object));

        // Commit the changes made.
        //
        commit();
    }

    /**
     * Removes the m-bean from the database.
     */
    private void deleteDomainEntry(ObjectName name)
        throws SQLException {

        Hashtable values = parseObjectName(name);
        String domainName = (String) values.get(DOMAIN);
        String className = (String) values.get(CLASS);
        String propertyList = (String) values.get(PROPERTY);

        // Get the domain table from the registry table.
        //
        DomainTable domainTable = null;
        if (registryTable.contains(domainName)) {
            domainTable = registryTable.get(domainName);
        } else {
            return;
        }

        // Delete the m-bean from the domain table.
        //
        domainTable.delete(className, propertyList);
        if (domainTable.isEmpty()) {
            registryTable.delete(domainName);
        }

        // Commit the changes made.
        //
        commit();
    }

    /**
     * Retrieves the m-bean from the database.
     */
    private Object getDomainEntry(ObjectName name)
        throws SQLException {
   
        Hashtable values = parseObjectName(name);
        String domainName = (String) values.get(DOMAIN);
        String className = (String) values.get(CLASS);
        String propertyList = (String) values.get(PROPERTY);

        // Get the domain table from the registry table.
        //
        DomainTable domainTable = null;
        if (registryTable.contains(domainName)) {
            domainTable = registryTable.get(domainName);
        } else {
            return null;
        }

        // Get the m-bean from the domain table.
        //
        byte[] objectData = domainTable.get(className, propertyList);
        if (objectData == null) {
            return null;
        }

        // Reconstruct the m-bean from the array of object bytes.
        //
        return converter.fromBytes(objectData);
    }
   
    /**
     * Tests if the m-bean is present in the database.
     */
    private boolean containsDomainEntry(ObjectName name) {
       
        Hashtable values = parseObjectName(name);
        String domainName = (String) values.get(DOMAIN);
        String className = (String) values.get(CLASS);
        String propertyList = (String) values.get(PROPERTY);
      
        try {
            // Get the domain table from the registry table..
            //
            DomainTable domainTable = null;
            if (registryTable.contains(domainName)) {
                domainTable = registryTable.get(domainName);
            } else {
                return false;
            }
      
            // Check the domain table for the m-bean.
            //
            return domainTable.contains(className, propertyList);
        } catch (SQLException ee) {
            Table.traceSqlException(ee);
            return false;
        }
    }

    /**
     * Tests if a singleton of this class is present in the database.
     */
    private boolean containsSingleton(ObjectName name) {
   
        Hashtable values = parseObjectName(name);
        String domainName = (String) values.get(DOMAIN);
        String className = (String) values.get(CLASS);
      
        try {
            // Get the domain table from the registry.
            //
            DomainTable domainTable = null;
            if (registryTable.contains(domainName)) {
                domainTable = registryTable.get(domainName);
            } else {
                return false;
            }
      
            // Check the domain table for a singelton.
            //
            return domainTable.contains(className, null);
        } catch (SQLException ee) {
            Table.traceSqlException(ee);
            return false;
        }
    }

    /**
     * Tests if a m-bean of this class is present in the database.
     */
    private boolean containsClassName(ObjectName name) {
   
        Hashtable values = parseObjectName(name);
        String domainName = (String) values.get(DOMAIN);
        String className = (String) values.get(CLASS);
      
        try {
            // Get the domain table from the registry.
            //
            DomainTable domainTable = null;
            if (registryTable.contains(domainName)) {
                domainTable = registryTable.get(domainName);
            } else {
                return false;
            }
      
            // Check the domain table for m-beans of this class.
            //
            return domainTable.contains(className, "");
        } catch (SQLException ee) {
            Table.traceSqlException(ee);
            return false;
        }
    }

    /**
     * Gets a list of ObjectNames present in the database.
     */
    private Vector getDomainEntries(ObjectName name)
        throws SQLException {
   
        Hashtable values = parseObjectName(name);
        String domainName = name.getDomain();
        String className = (String) values.get(CLASS);
        String propertyList = (String) values.get(PROPERTY);
      
        // Get the domain name(s).
        //
        Vector allDomains = null;
        if (domainName.length() == 0) {
            allDomains = registryTable.getAll();
        } else {
            allDomains = new Vector(1);
            allDomains.addElement(registryTable.get(domainName));
        }

        trace("getDomainEntries: >>> domains=" + allDomains.size());
      
        // Get the object name(s) of the m-bean(s) present in the domain table(s).
        //
        Vector result = new Vector();
        for (int i = 0; i < allDomains.size(); i++) {
            DomainTable domain = (DomainTable) allDomains.elementAt(i);
            Vector allEntries = domain.getObjectNameStrings(className, propertyList);
            for(int j = 0; j < allEntries.size(); j++) {
                ObjectName objName = newObjectName((String) allEntries.elementAt(j));
                if (objName != null) {
                    result.addElement(objName);
                }
            }
        }
      
        trace("getDomainEntries: >>> objects=" + result.size());
      
        // Return a list of object names.
        //
        return result;
    }


    // DEBUG METHODS
    //--------------
 
    /**
     * Displays the debug message.
     */ 
    private void trace(String message) {
        debug("JDBCRepositorySrv::" + message);
    }

    /**
     * Displays the debug message.
     */ 
    private void lntrace(String message) {
        debug("\nJDBCRepositorySrv::" + message);
    }

    /**
     * Displays the debug message.
     */ 
    private void traceln(String message) {
        debug("JDBCRepositorySrv::" + message + "\n");
    }

    /**
     * Displays the debug message.
     */ 
    private void debug(String message) {
        Debug.print(Debug.REP_DEBUG, message);
    }

    // PRIVATE VARIABLES
    //------------------

    /* Parse keys. */
    private final String DOMAIN   = "DOMAIN";
    private final String CLASS    = "CLASS";
    private final String PROPERTY = "PROPERTY";

    /* Registry table. */
    private RegistryTable registryTable = null;
    private final String registryTableName = "registry";

    /* Default domain name. */
    private String domain      = ServiceName.DOMAIN;;
   
    /* JDBC driver and url information. */
    private String jdbcDriverName = null;
    private String jdbcUrlName    = null;
    private String jdbcUserName   = null;
    private String jdbcPassword   = null;
   
    /* Connection to the specified database. */
    private Connection jdbcConnection = null;
   
    /* State variables. */
    private boolean configured = false;
   
    /* The reflexive converter used to covert objects to and from byte arrays. */
    private SerializationConvertion converter = null;
   
    /* List of objects exported to the JVM to make them available to receive
       incoming calls. */
    private RepositorySrv exportList = null;
}
