// Copyright (c) 03/12/99, by Sun Microsystems, Inc.
// All rights reserved.

// "@(#)DomainTable.java 3.3 99/03/12 SMI"

// Java imports.
//
import java.io.*;
import java.sql.*;
import java.util.*;
import java.lang.*;

/**
 * This class provides a basic service for managing a domain table in a database.
 * The domain table represents a domain in the agent.
 * Data is kept on all m-beans registered persistent in this domain.
 * 
 * The domain table structure:
 *
 *      Column Name             Column Type
 *
 *      className               VARCHAR
 *      keyProperties           VARCHAR
 *      mbeanData               BINARY
 *
 * @version     3.3     03/12/99
 * @author      Sun Microsystems, Inc
 */


public class DomainTable extends Table {

    /**
     * Creates a DomainTable that manipulates domain tables in the
     * database. If the table doesn't exist in the database, it will be created.
     *
     * @param <VAR>connection</VAR> A session with a specific database.
     * @param <VAR>domainName</VAR> The name of the domain table.
     *
     * @exception SQLException A database-access error occured.
     */
    public DomainTable(Connection connection, String domainName)
        throws SQLException {
      
        super(connection);
        tableClass = "DomainTable";
        this.domainName = domainName;
                  
        if (!tableExists(domainName)) {
            createDomainTable();
        }
    }

    /**
     * Creates a domain table in the specified database.
     *
     * @exception SQLException A database-access error occured.
     */
    protected void createDomainTable()
        throws SQLException {

        trace("createDomainTable: create domain table=" + domainName);
      
        // SQL types.
        //
        String varcharName = "VARCHAR";
        String binaryName  = "VARBINARY";      

        ResultSet resultset = null;
            
        // Setup the SQL types.
        //
        try {
            DatabaseMetaData databasemetadata = jdbcConnection.getMetaData();
            if (databasemetadata != null) {
                trace("createDomainTable: Check SQL types : VARCHAR, BINARY");
         
                resultset = databasemetadata.getTypeInfo();
                if (resultset != null) {
                    short datatype = 0;
                    while (resultset.next()) {
                        datatype = resultset.getShort("DATA_TYPE");
                        switch (datatype) {
                        case Types.CHAR:
                        case Types.VARCHAR:
                        case Types.LONGVARCHAR:
                            varcharName = resultset.getString("TYPE_NAME");
                            break;
                        case Types.BINARY:
                        case Types.VARBINARY:
                        case Types.LONGVARBINARY:
                            binaryName = resultset.getString("TYPE_NAME");
                            break;
                        }
                    }
                } // else use the default SQL type settings
            } // else use the default SQL type settings      
        } catch (SQLException ee) {
            // Use the deafault SQL types settings.
            traceSqlException(ee);
        } finally {
            closeResult(resultset);
        }


        // Set the column types.
        //
        String varcharColumn = varcharName + "(" + VARCHAR_SIZE + ")";
        String binaryColumn  = binaryName  + "(" + BINARY_SIZE + ")";
         
        // Set the domain table structure.
        // structure: class name VARCHAR(X), key properties VARCHAR(Y), object VARBINARY(Z)
        //
        String domainStructure = classColumn     + " " + varcharColumn + ", " +
            propertyColumn  + " " + varcharColumn + ", " + 
            objectColumn    + " " + binaryColumn;

        trace("createDomainTable: table structure=" + domainStructure);

        // Create the domain table.
        //
        createTable(domainName, domainStructure);
      
        trace("createDomainTable: ok !");
    }

    /**
     * Adds a new row to the domain table.
     *
     * @param <VAR>className</VAR> The class of object that the object represents.
     * @param <VAR>propertyList</VAR> The string representation of the list of key properties.
     * @param <VAR>objectData</VAR> The array of object bytes.
     *
     * @exception SQLException A database-access error occured.
     */
    public void add(String className, String propertyList, byte[] objectData)
        throws SQLException {

        trace("add: class name=" + className);
        trace("add: key properties=" + propertyList);
        trace("add: object data size=" + objectData.length);

        PreparedStatement preparedstatement = null;
      
        String insert_clause = "INSERT INTO ";
        String values_clause = " VALUES (?,?,?)";
      
        try {
            insert_clause += domainName + values_clause;
         
            preparedstatement = jdbcConnection.prepareStatement(insert_clause);
         
            preparedstatement.setString(1, className);
            if (propertyList == null) {
                preparedstatement.setNull(2, Types.VARCHAR);
            } else if (propertyList.length() != 0) {
                preparedstatement.setString(2, propertyList);
            }
            preparedstatement.setBytes(3, objectData);

            trace("add: execute query=" + insert_clause);
         
            int count = preparedstatement.executeUpdate();
         
            trace("add: row count=" + count);
        } finally {
            closePreparedStatement(preparedstatement);
        }
      
        trace("add: ok !");
    }

    /**
     * Changes data in existing rows of the domain table.
     *
     * @param <VAR>className</VAR> The class of object that the object represents.
     * @param <VAR>propertyList</VAR> The string representation of the list of key properties.
     * @param <VAR>objectData</VAR> The array of object bytes.
     *
     * @exception SQLException A database-access error occured.
     */
    public void update(String className, String propertyList, byte[] objectData)
        throws SQLException {

        trace("update: class name=" + className);
        trace("update: key properties=" + propertyList);
        trace("update: object data size=" + objectData.length);
  
        PreparedStatement preparedstatement = null;

        String update_clause = "UPDATE ";
        String set_clause = " SET ";
        String where_clause = " WHERE ";

        try {         
            update_clause += domainName;
            set_clause    += objectColumn + " = ?";
            where_clause  += classColumn + " = ?";
            if ((propertyList != null) && (propertyList.length() != 0)) {
                where_clause += " AND " + propertyColumn + " = ?";
            }
            update_clause += set_clause + where_clause;
         
            preparedstatement = jdbcConnection.prepareStatement(update_clause);
         
            preparedstatement.setBytes(1, objectData);
            preparedstatement.setString(2, className);
            if (propertyList != null) {
                preparedstatement.setString(3, propertyList);
            }

            trace("update: execute query=" + update_clause);
         
            int count = preparedstatement.executeUpdate();
         
            trace("update: row count=" + count);
        } finally {
            closePreparedStatement(preparedstatement);
        }
      
        trace("update: ok !");
    }

    /**
     * Removes existing rows from the domain table.
     *
     * @param <VAR>className</VAR> The class of object that the object represents.
     * @param <VAR>propertyList</VAR> The string representation of the list of key properties.
     *
     * @exception SQLException A database-access error occured.
     */
    public void delete(String className, String propertyList)
        throws SQLException {

        trace("delete: class name=" + className);
        trace("delete: key properties=" + propertyList);

        Statement statement = null;
      
        String delete_clause = "DELETE FROM ";
        String where_clause = " WHERE ";

        try {      
            delete_clause += domainName;
            where_clause  += classColumn + " = '" + className + "'";
            if (propertyList == null) {
                where_clause += " AND " + propertyColumn + " IS NULL";
            } else if (propertyList.length() != 0) {
                where_clause += " AND " + propertyColumn + " = '" + propertyList + "'";
            }
            delete_clause += where_clause;
         
            statement = jdbcConnection.createStatement();
         
            trace("delete: execute query=" + delete_clause);
         
            int count = statement.executeUpdate(delete_clause);
         
            trace("delete: row count=" + count);
        } finally {
            closeStatement(statement);
        }
      
        trace("delete: ok !");
    }

    /**
     * Retrieves the array of object bytes from an existing row in the domain table.
     *
     * @param <VAR>className</VAR> The class of object that the object represents.
     * @param <VAR>propertyList</VAR> The string representation of the list of key properties.
     *
     * @return The array of object bytes.
     *
     * @exception SQLException A database-access error occured.
     */
    public byte[] get(String className, String propertyList)
        throws SQLException {
 
        trace("get: class name=" + className);
        trace("get: key properties=" + propertyList);
   
        byte[] objectData = null;

        Statement statement = null;
        ResultSet resultset = null;
      
        String select_clause = "SELECT ";
        String where_clause = " WHERE ";
      
        try {
            select_clause += objectColumn + " FROM " + domainName;
            where_clause += classColumn + " = '" + className + "'";
            if (propertyList == null) {
                where_clause += " AND " + propertyColumn + " IS NULL";
            } else if (propertyList.length() != 0) {
                where_clause += " AND " + propertyColumn + " = '" + propertyList + "'";
            }
            select_clause += where_clause;
         
            statement = jdbcConnection.createStatement();

            trace("get: execute query=" + select_clause);
         
            resultset = statement.executeQuery(select_clause);

            if (!resultset.next()) {
                return null;
            }

            objectData = resultset.getBytes(objectColumn);
            if (resultset.wasNull()) {
                throw new SQLException("Could not retrieve object bytes");
            }
        } finally {
            closeResult(resultset);
            closeStatement(statement);
        }
      
        trace("get: object data size=" + objectData.length);
        trace("get: ok !");
      
        return objectData;
    }
 
    /**
     * Tests if the row exists in the domain table. 
     *
     * @param <VAR>className</VAR> The class of object that the object represents.
     * @param <VAR>propertyList</VAR> The string representation of the list of key properties.
     *
     * @return True if the row exists in the domain table, else false.
     *
     * @exception SQLException A database-access error occured.
     */  
    public boolean contains(String className, String propertyList)
        throws SQLException {

        trace("contains: class name=" + className);
        trace("contains: key properties=" + propertyList);

        Statement statement = null;
        ResultSet resultset = null;

        String select_clause = "SELECT ";
        String where_clause = " WHERE ";
      
        try {
            select_clause += classColumn + " FROM " + domainName;
            where_clause += classColumn + " = '" + className + "'";
            if (propertyList == null) {
                where_clause += " AND " + propertyColumn + " IS NULL";
            } else if (propertyList.length() != 0) {
                where_clause += " AND " + propertyColumn + " = '" + propertyList + "'";
            }
            select_clause += where_clause;
         
            statement = jdbcConnection.createStatement();
         
            trace("contains: execute query=" + select_clause);
         
            resultset = statement.executeQuery(select_clause);

            if (resultset.next()) {
                return true;
            }
        } finally {
            closeResult(resultset);
            closeStatement(statement);
            trace("contains: ok !");
        }
      
        return false;
    }

    /**
     * Retrieve all the rows satisfying the query and return a list of string
     * representations of object names.
     *
     * @param <VAR>className</VAR> The class of object that the object represents.
     * @param <VAR>propertyList</VAR> The string representation of the list of key properties.
     *
     * @return List of object name string representations.
     *
     * @exception SQLException A database-access error occured.
     */ 
    public Vector getObjectNameStrings(String className, String propertyList)
        throws SQLException {

        trace("getObjectNameStrings: class name=" + className);
        trace("getObjectNameStrings: key properties=" + propertyList);
      
        Statement statement = null;
        ResultSet resultset = null;
      
        String select_clause = "SELECT ";
        String where_clause = "";
      
        Vector result = new Vector();

        try {
            select_clause += classColumn + ", " + propertyColumn + " FROM " + domainName;
            if (className != null) {
                where_clause += " WHERE " + classColumn + " = '" + className + "'";
            }
            if (propertyList != null) {
                if (className != null) {
                    where_clause += " AND ";
                } else {
                    where_clause += " WHERE ";
                }
                where_clause += propertyColumn + " = '" + propertyList + "'";
            }
            select_clause += where_clause;
         
            statement = jdbcConnection.createStatement();
         
            trace("getObjectNameStrings: execute query=" + select_clause);
         
            for (resultset = statement.executeQuery(select_clause); resultset.next();) {

                String objectname = domainName;
            
                className = resultset.getString(classColumn);
                if (!resultset.wasNull()) {
                    objectname += ":" + className;
                }

                propertyList = resultset.getString(propertyColumn);
                if (!resultset.wasNull()) {
                    objectname += "." + propertyList;
                }

                result.addElement(objectname);
            }
        } finally {
            closeResult(resultset);
            closeStatement(statement);
        }

        trace("getObjectNameStrings: object name count=" + result.size());      
        trace("getObjectNameStrings: ok !");
      
        return result;
    }

    /**
     * Determines the size of the domain table.
     *
     * @return The number of existing rows in the domain table.
     *
     * @exception SQLException A database-access error occured.
     */ 
    public int size()
        throws SQLException {
      
        return tableSize(domainName);
    }
   
    /**
     * Tests if there is any existing rows in the domain table.
     *
     * @return True if there are no existing rows in the domain table, else false.
     *
     * @exception SQLException A database-access error occured.
     */ 
    public boolean isEmpty()
        throws SQLException {
      
        trace("isEmpty: empty table ? " + domainName);
      
        return (size() == 0);
    }
   
    /**
     * Removes the domain table definition and all data.
     *
     * @exception SQLException A database-access error occured.
     */ 
    public void drop()
        throws SQLException {
      
        trace("drop: drop table=" + domainName);
      
        destroyTable(domainName);
      
        trace("drop: ok !");
    }
   
    /* Domain table name. */
    private String domainName = null;
   
    /* Domain table column names. */
    private final String classColumn    = "className";
    private final String propertyColumn = "keyProperties";
    private final String objectColumn   = "mbeanData";

    /** The string column size. */
    public static int VARCHAR_SIZE = 256;
   
    /** The byte array column size. */
    public static int BINARY_SIZE  = 100000;
}
