// Copyright (c) 03/23/99, by Sun Microsystems, Inc.
// All rights reserved.

// "@(#)Agent.java 3.6 99/03/23 SMI"

// Java imports.
//
import	java.util.*;

// Jaw imports.
//
import  com.sun.jaw.impl.common.*;
import	com.sun.jaw.reference.common.*;
import	com.sun.jaw.reference.agent.cmf.*;
import	com.sun.jaw.reference.agent.services.*;
import  com.sun.jaw.impl.agent.services.mlet.*;
import	com.sun.jaw.impl.agent.services.jawdiscovery.*;

/**
 * This class provides an example of an agent that instanciates a framework
 * initialized with a repository offering persistence through the JDBC API.
 * The agent registers some services and adaptors at startup time with the
 * framework. If the agent is stopped and restarted the agent will recover
 * these services and adaptors and activate them.
 *
 * NOTE: By default the repository is configured to work with the
 * Cloudscape JBMS release 1.2.1. JDBC driver.
 * For futher information on the JBMS database go to the Cloudscape home page at: 
 *      http://www.cloudscape.com
 *
 * @version     3.6     03/23/99
 * @author      Sun Microsystems, Inc
 *
 * @see JDBCRepositorySrv
 */


public class Agent implements Runnable {

    // PUBLIC METHODS
    //---------------

    public static void main(String argv[]) {                 
        new Agent().execute(argv);
    }

    /**
     * When an object implementing interface Runnable is used to create a thread,
     * starting the thread causes the object's run method to be called in that
     * separately executing thread.
     */
    public void run() {
        execute(new String[0]);
    }


    // PRIVATE METHODS
    //----------------
  
    /**
     * Initialization phase of the agent.
     */
    private void execute(String argv[]) {
        MoRepSrvIf          rep;
        MetaDataSrvIf       mtdSrv;
        Class               repSrv;
        String              domain;
    
        Debug.setDefault(true);
        trace("Default debug is on");
    
        try {
            // Create the repository.
            //
            repSrv = Class.forName(repName);
            rep    = (MoRepSrvIf)repSrv.newInstance();

            // Setup the JDBC information and configure the repository with it.
            // NOTE: This part can differ between different JDBC driver 
            //       implementations. The default configuration is set for the
            //       Cloudscape JBMS release 1.2.1. JDBC driver.
            //
            // For more information on the JBMS database go to the 
            // Cloudscape home page:   http://www.cloudscape.com
            //
            setSystemProperty(jdbcSystemHome, dbDirectory);
      
            Vector	params = new Vector(2);
            params.insertElementAt(jdbcDriver, 0);
            params.insertElementAt(jdbcProtocol + jdbcName + jdbcAttributes, 1);

            try {
                rep.setConfig(params);
            } catch (ServiceNotFoundException oo) {
                System.err.println("Failed to configure the JDBC repository !");
                Debug.printException(oo);
                System.exit(1);
            }
            trace("Repository configuration ok !");

            // Create the Framework and initialize it with the 
            // newly created repository.
            //
            cmf    = new Framework((MoRepSrvIf)rep, null);
      
            // Retrieve the Framework domain.
            //
            domain = cmf.getDomain();
            trace("Framework domain=" + domain);

            // Add the MetaDataService.
            //
            try {
                mtdSrv = cmf.getMetaDataSrvIf();
                if (mtdSrv == null) {
                    addObject(mtdName, new ObjectName(domain + ":" + ServiceName.META), true);
                } else {
                    trace("Found a metadata service.");
                    cmf.setMetaDataSrvIf(mtdSrv);
                }
            } catch (ServiceNotFoundException oo) {
                System.err.println("Cannot check for metadata service");
                Debug.printException(oo);
            }

            // Add the MLet service.
            //
            String mletClass = "com.sun.jaw.impl.agent.services.mlet.MLetSrvMO";
            ObjectName mletName = new ObjectName(domain + ":" + mletClass);
            if (activateObject(mletName) == null) {
                addObject(mletClassName, mletName, true);
            }

            // Add the Launcher service.
            //
            String launchClass = "com.sun.jaw.impl.agent.services.mlet.LauncherSrvMO";
            ObjectName launcherName = new ObjectName(domain + ":" + launchClass);
            if (activateObject(launcherName) == null) {
                addObject(launcherClassName, launcherName, true);
            }

            // Add the HTTP adaptor service.
            //
            String httpClass = "com.sun.jaw.impl.adaptor.http.AdaptorServerImplMO";
            ObjectName httpName = new ObjectName(domain + ":" + ServiceName.APT_HTTP + ".protocol=http,port=8081");
            if (activateObject(httpName) == null) {
                addObject(adpNameHttp, httpName, true);
            }
      
            // Add the HTML adaptor service.
            //
            String htmlClass = "com.sun.jaw.impl.adaptor.html.AdaptorServerImplMO";
            ObjectName htmlName = new ObjectName(domain + ":" + htmlClass + ".protocol=html,port=8082");
            if (activateObject(htmlName) == null) {
                addObject(adpNameHtml, htmlName, true);
            }
      
            // Add the RMI adaptor service.
            //
            ObjectName rmiName= new ObjectName(domain + ":" + ServiceName.APT_RMI + ".protocol=rmi,port=1099");
            if (activateObject(rmiName) == null) {
                addObject(adpNameRmi, rmiName, true);
            }
      
            // Add the Discovery service.
            //
            ObjectName discoveryName = new ObjectName(domain + ":" + discoveryClassName);
            if (activateObject(discoveryName) == null) {
                addObject(discoveryClassName, discoveryName, true);
            }

            // Count the number of m-beans registered in the framework.
            //
            trace("Number of m-beans in repository=" + rep.getNbElements());
        } catch (Exception ee) {
            System.err.println("Exception ! " + ee.getMessage());
            Debug.printException(ee);
        }
    }
  
    /**
     * Register an m-bean persistent or non-persistent in the framework.
     */
    private Object addObject(String className, ObjectName localName, boolean db) {
        Object object = null;
        try {
            if (db) {
                trace("Adding " + className + " persistent to the Framework with the name=" + localName);
                object = cmf.newDBObject(className, localName, null);
            } else {
                trace("Adding " + className + " non-persistent to the Framework with the name=" + localName);
                object = cmf.newObject(className, localName, null);
            }
        } catch (Exception ee) {
            System.err.println("Failed to add the object to the Framework ! " + localName);
            Debug.printException(ee);
        }
        return object;
    }
  
    /**
     * Retrieve an m-bean from the framework.
     */
    private Object getObject(ObjectName localName) {
        Object object = null;
        try {
            trace("Get the m-bean=" + localName);
            Vector mbeans = cmf.getObject(localName, null);
            if (!mbeans.isEmpty()) {
                object = ((NamedObject)mbeans.firstElement()).getObject();
            }
        } catch (Exception ee) {
            System.err.println("Failed to get the object from the Framework ! " + localName);
            Debug.printException(ee);
        }
        return object;
    }
  
    /**
     * Retrieve an m-bean from the framework and
     * activate it, if it implements the ActivatableIf interface.
     */
    private Object activateObject(ObjectName localName) {
        Object object = getObject(localName);
        if ((object != null) && (object instanceof ActivatableIf)) {
            try {
                trace("Activate the m-bean=" + localName);
                ((ActivatableIf) object).performStart();
            } catch (Exception ee) {
                System.err.println("Failed to activate the object ! " + object);
                Debug.printException(ee);
            }
        }
        return object;
    }
  
    /**
     * Set system properties.
     */
    private void setSystemProperty(String property, String value) {
        Properties properties = System.getProperties();
        properties.put(property, value);
        System.setProperties(properties);
    }
 
    /**
     * Display debug information.
     */
    private void trace(String message) {
        Debug.print("Agent: " + message);
    }
  
 
    // PRIVATE VARIABLES
    //------------------
  
    /* JDBC information */
    private String jdbcDriver	= "COM.cloudscape.core.JDBCDriver";
    private String jdbcProtocol	= "jdbc:cloudscape:";
    private String jdbcName	= "Registry";    
    private String jdbcAttributes	= ";create=true" + ";autocommit=false";
    private String jdbcSystemHome = "cloudscape.system.home";
  
    /* Class names of services. */
    private String repName           	= "JDBCRepositorySrv";
    private String mtdName           	= "com.sun.jaw.impl.agent.services.light.MetaDataSrv";
    private String adpNameHttp       	= "com.sun.jaw.impl.adaptor.http.AdaptorServerImpl";
    private String adpNameHtml       	= "com.sun.jaw.impl.adaptor.html.AdaptorServerImpl";
    private String adpNameRmi        	= "com.sun.jaw.impl.adaptor.rmi.AdaptorServerImpl";
    private String mletClassName     	= "com.sun.jaw.impl.agent.services.mlet.MLetSrv";
    private String launcherClassName 	= "com.sun.jaw.impl.agent.services.mlet.LauncherSrv";
    private String discoveryClassName 	= "com.sun.jaw.impl.agent.services.jawdiscovery.DiscoveryResponder";
  
    /* Database directory */
    private String dbDirectory		= DefaultPaths.getTmpDir();
    
    /* Reference to Framework. */
    private Framework cmf;
}
