// Copyright (c) 03/12/99, by Sun Microsystems, Inc.
// All rights reserved.

// "@(#)Client.java 3.3 99/03/12 SMI"

// java import
//
import java.net.* ;
import java.util.* ;

// jaw import
//
import com.sun.jaw.reference.common.* ;
import com.sun.jaw.impl.adaptor.rmi.* ;


/**
 * The Client implements a simple management application
 * illustrating how to deal with exception handling.
 * 
 * The application performs the following tasks:
 *      - set up communication using the RMI client adaptor
 *      - instantiate a simple m-bean in a remote agent
 *      - access and modify properties of a remote m-bean
 * 
 * The application deals with exception handling and catchs 3 different exceptions:
 *      - "JawException" can occur when instantiating a simple m-bean 
 *        that is already registered in the agent for example.
 *        You can generate such an exception by starting the client application twice in succession.
 *      - "JawRuntimeException" can occur when trying to connect to an agent for example.
 *        You can generate such an exception by starting the client application without running agent.
 *      - "Exception".
 *
 * Prior to starting the client application, make sure an agent is running.
 * For instance you can use the sample agent provided with the example.
 * If you use a different agent, keep in mind that the client application uses the
 * RMI adaptor for communicating with the agent.
 *
 * In order to start the client, you need to invoke the following command:
 *      java Client [hostname]
 *        where hostname is the name of the machine where the agent is running.
 * By default it is the local host.
 *
 * The Client implements Runnable and can also be started using the Bootstrap service.
 *
 * @version     3.3     03/12/99
 * @author      Sun Microsystems, Inc
 *
 */

public class Client implements Runnable {

    // CONSTRUCTOR
    //------------
    public Client() {
    }
  
    /**
     * It is possible to specify on the command line the implementation to use
     * for the adaptor function. 
     * In particular, it is possible to specify the host name where the
     * agent is running.
     */
    public static void main(String argv[]) {
	
        System.exit( new Client().execute(argv) ) ;	
    }
  
    /**
     * When an object implementing interface Runnable is used to create a thread, 
     * starting the thread causes the object's run method to be called in that
     * separately executing thread.
     */
    public void run() {
	
        String[] s = new String[0];
        execute(s);
    }
  
  
    // PRIVATE METHODS
    //----------------
    private int execute(String argv[]) {
        
        try {
            String agentHost = InetAddress.getLocalHost().getHostName() ;
            if (argv.length >= 1) 
                agentHost = argv[0] ;
      
            System.out.println(">>> Connecting to " + agentHost + "...") ;

            // Set up the RMI AdaptorClient. 
            //
            AdaptorClient adaptor = new AdaptorClient() ;

            // Initialize communication with the remote RMI managed object server.
            //
            adaptor.connect(null, agentHost, 1099, ServiceName.APT_RMI) ;

            // Create an instance of the SimpleBean m-bean in the 
            // remote object server. 
            //
            System.out.println("\n>>> Creating new object...") ;
            String simpleClass = "SimpleBean" ;
            ObjectName simpleName = new ObjectName("defaultDomain:SimpleBean.id=1") ;
            SimpleBeanMO simple = (SimpleBeanMO)adaptor.cb_newMO(simpleClass, simpleName, null) ;
      
            // Access and modify the properties of the SimpleBean m-bean remotely.
            //
            System.out.println("simple.state     = " + simple.getState()) ;
            System.out.println("simple.nbChanges = " + simple.getNbChanges()) ;

            System.out.println("\n>>> Changing state...") ;
            simple.setState("New state") ;

            System.out.println("simple.state     = " + simple.getState()) ;
            System.out.println("simple.nbChanges = " + simple.getNbChanges()) ;

            System.out.println("\n>>> Resetting change counters...") ;
            simple.performReset() ;
      
            System.out.println("simple.state     = " + simple.getState()) ;
            System.out.println("simple.nbChanges = " + simple.getNbChanges()) ;

            // Terminates the communication with the remote managed object server.
            //
            adaptor.disconnect() ;
            return 0;
        }
        catch(JawException e) {
	  
            // InstanceAlreadyExistException can be thrown by the method cb_newMO.
            // LocalException can be thrown by the method cb_newMO.
            //
            System.out.println("\n>>> Got a jaw exception !") ;
            e.printStackTrace() ;
            return 1;
        }
        catch(JawRuntimeException e) {
	  
            // CommunicationException can be thrown by the method connect.
            //
            System.out.println("\n>>> Got a jaw runtime exception !") ;
            e.printStackTrace() ;
            return 1;
        }
        catch(Exception e) {
	  
            System.out.println("\n>>> Got a java exception !") ;
            e.printStackTrace() ;
            return 1;
        }
    }
}
