#
#ident  "@(#)test_accept_assoc.tcl	1.11 99/07/30 Copyright SMI"
#
# Copyright (c) 07/30/99 by Sun Microsystems, Inc. 
# All Rights Reserved
#
# This script file contains a simple example to demonstrate how to
# set up an association based on the TMNscript ASSOC commands
# and release the association based on the TMNscript RELEASE commands.
#
# Messages are send over the association based on the TMNscript GET commands.
#
# The script acts as both manager and agent. The addresses for the manager
# and the agent are set to be the same so that the messages are looped back.
#
# Mapping mode and ASN.1 mode are both enabled. Objects are expressed
# using human-readable names, and data values are expressed using the
# string representation provided by Solstice TMNscript.
#


#####################
# Define Procedures #
#####################

# Define a procedure to get the base directory of a package
#
proc pkgbasedir { pkgname } {
   if {[catch {exec pkgparam $pkgname BASEDIR} result]} {
        puts stderr "pkgparam $pkgname failed: $result"
        return "/opt"
   }
   return $result
}


# Define a procedure to send a GET request on whole subtree
#
proc sendGetReqOverConnection { session moc moi } {
   set req "mp::getReq -ses $session -scope {whole} -moc \{$moc\} -moi \{$moi\} -info {all}"
   puts stdout "Send GET REQUEST to the agent:\n $req \n"
   if [catch {eval $req} reqerror] {
        puts stdout "$reqerror \n"
        exit
   }
}


# Define a procedure to send GET response
#
proc sendGetRsp { session id moc moi info} {
   set rsp "mp::getRsp -ses $session -id $id -link last -moc \{$moc\} -moi \{$moi\} -error {} -info \{$info\}"
   puts stdout "Send GET RESPONSE to the manager:\n $rsp \n"
   if [catch {eval $rsp} rsperror] {
        puts stdout "$rsperror \n"
        exit
   }
}


###########################
# Load Metadata & Scripts #
###########################

puts stdout "\nLoad Metadata & Scripts"
puts stdout "---------------------------------------------------------------------------\n"

set basedir [pkgbasedir SUNWtmns2]
set compiler_dir $basedir/SUNWconn/TMNscript/example/compiler
set shared_dir $basedir/SUNWconn/TMNscript/example/tcl/shared
source $compiler_dir/mapper.tcl
source $compiler_dir/parser.tcl

set basedir [pkgbasedir SUNWtmns1]
source $basedir/SUNWconn/TMNscript/script/tmns_type.tcl


# Load the receive procedures

source $shared_dir/libreceive.tcl

# Load the ASSOC procedures

source $shared_dir/libassoc.tcl


##################################
# Initialization & Configuration #
##################################				 

puts stdout "\nInitialization & Configuration"
puts stdout "---------------------------------------------------------------------------\n"

# Set operating mode: mapper mode and ASN.1 mode enabled

set mp_mapperModeOn true
set mp_asn1ModeOn true

puts stdout "Mapping and ASN.1 modes enabled \n "

# Set diagnostic modes: trace mode disabled

set mp_misTraceOn false

puts stdout "Trace mode disabled \n"

# mp_acmString is by default enabled
 
puts stdout "Translation of association related information enabled \n"

# Disable support for universal types

mp::negotiate { XMP_DECODING off }

puts stdout "Support for universal types disabled \n"

# Set addresses for the manager and the agent to be the
# same, so that messages are looped back 

set manager {{psel rfc4} {ssel Prs} {tsel CMIP} {nsap ""}}
set agent {{psel rfc4} {ssel Prs} {tsel CMIP} {nsap ""}}

puts stdout "Address for manager application: $manager \n"
puts stdout "Address for agent application: $agent \n"

# Define a variable that contains the object class name

set moc system

puts stdout "Managed object class (moc): $moc \n"

# Define a variable that contains the object instance name

set moi "\{systemId \{[getHostName]\}\}"

puts stdout "Managed object instance (moi): $moi \n"

# Set the display of messages received (true/false)

set format_display false

#######################################
# Exercise ASSOC and RELEASE Commands #
#######################################

set nbIter 1
for { set i 0} {$i < $nbIter} {set i [expr $i +1]} {

puts stdout "\nExercise ASSOC Commands - Iteration $i"
puts stdout "---------------------------------------------------------------------------\n"


###################################################################
# PHASE 1 : Bind the application with manual association handling #
# -------							  #
###################################################################

# Bind the manager application with manual association handling enabled
#
set bound_session [manualAssoc $manager]

# Print the status of the bound session
#
sessionInfo $bound_session


#################################
# PHASE 2 : Set up associations #
# -------			#
#################################

# Send ASSOC request to the agent
# NOTE : A transient session is returned
#
set manager_transient_session [getSessionfromReq [sendAssocReq $bound_session]]

# Print the status of the transient session
#
sessionInfo $manager_transient_session

# Wait up to 10 seconds to receive the ASSOC indication
# NOTE : There should be a transient session in the confirmation
#
set ind [receiveOne $bound_session]
puts stdout "ASSOC INDICATION received by the agent:"
displayOne $ind

# Get the transient session from the indication
#
set agent_transient_session [getSessionfromInfo $ind]

# Print the status of the transient session
#
sessionInfo $agent_transient_session

# Get the invoke identifier contained in the indication
# in order to be able to send a response.
#
set id [getInvokeId $ind]

# Set the operation-specific information
#
set info " \{\{assoc-Result accept\} \{acse-Args \{\{application-Context $appcontext\}\}\}\}" 

# Send ASSOC response to the manager
# NOTE : Because of the association acceptance a connected session is returned
#
set agent_connected_session [sendAssocRsp $bound_session $id $info]

# Print the status of the connected session
#
sessionInfo $agent_connected_session

# Wait up to 10 seconds to receive the ASSOC confirmation
# NOTE : There should be a connected session in the confirmation
#
set cnf [receiveOne $bound_session]
puts stdout "ASSOC CONFIRMATION received by the manager:"
displayOne $cnf

# Get the connected session from the confirmation
#
set manager_connected_session [getSessionfromInfo $cnf]

# Print the status of the connected session
#
sessionInfo $manager_connected_session


#######################################################################
# TEST PHASE  : Send GET requests and responses over the associations #
# ----------							      #
#######################################################################

# Send GET request on whole subtree to the agent
#
sendGetReqOverConnection $manager_connected_session $moc $moi

# Wait up to 10 seconds to receive the GET indication
#
set ind [receiveOne $agent_connected_session]
puts stdout "GET INDICATION received by the agent:"
displayOne $ind

# Get the invoke identifier contained in the indication
# in order to be able to send a response.
#
set id [getInvokeId $ind]

# Set the operation-specific information
#
set info [list [lindex $moi 0] {systemTitle {{oid 1.2.3.4.5.6}}} {operationalState enabled} {usageState active} {administrativeState unlocked}]

# Send GET response to the manager
#
sendGetRsp $agent_connected_session $id $moc $moi $info

# Receive all GET confirmations
#
receiveAll $manager_connected_session


######################################
# PHASE 3 : Release the associations #
# -------			     #
######################################

# Send a RELEASE request to the agent 
#
sendReleaseReq $manager_connected_session

# Print the status of the session
#
sessionInfo $manager_connected_session

# Wait up to 10 seconds to receive the RELEASE indication
#
set ind [receiveOne $agent_connected_session]
puts stdout "RELEASE INDICATION received by the agent:"
displayOne $ind

# Print the status of the session
#
sessionInfo $agent_connected_session

# Get the invoke identifier contained in the indication
# in order to be able to send a response.
#
set id [getInvokeId $ind]

# Send RELEASE response to the manager
# NOTE: The connected session is removed automatically after 
#	the realeseRsp was send with "reason == normal".
#
sendReleaseRsp $agent_connected_session $id

# Wait up to 10 seconds to receive the RELEASE confirmation
# NOTE: The connected session will be destroyed automatically after
#	realeseCnf was received with "reason == normal". 
#
set cnf [receiveOne $manager_connected_session]
puts stdout "RELEASE CONFIRMATION received by the manager:"
displayOne $cnf


######################################
# PHASE 4 : Unbind the bound session #
# ------- 			     #
######################################

# Unbind the bound session
#
sessionUnbind $bound_session

}

##############
# That's all #
##############


