#ifdef _SUNOS
#pragma ident "@(#)tmns_interp.c	1.5 97/10/03 Copyright SMI"
#endif

/******************************************************************************
 * Copyright 10/03/97 Sun Microsystems, Inc. All Rights Reserved		      *
 *			All Rights Reserved				      *
 ******************************************************************************/

/******************************************************************************/
/*  File Name: 		tmns_interp.c                                         */
/*----------------------------------------------------------------------------*/
/*                                                                            */
/*  Description:                                                              */
/*  -----------                                                               */
/*       The current file contains the implementation of the C binding API    */
/*       for TMNscript.				              		      */
/*                                                                            */
/******************************************************************************/

/*----------------------------------------------------------------------------*/
/*                               C API INCLUDE	                	      */
/*----------------------------------------------------------------------------*/
#include "TMNsbind.h"

/*----------------------------------------------------------------------------*/
/*                               COMMON INCLUDE                      	      */
/*----------------------------------------------------------------------------*/
#include "tmns_interp.h"

/*----------------------------------------------------------------------------*/
/*                               STANDARD INCLUDE                             */
/*----------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/varargs.h>

/*----------------------------------------------------------------------------*/
/*                              TCL INCLUDE                                   */
/*----------------------------------------------------------------------------*/
#include <tcl.h>

/*----------------------------------------------------------------------------*/
/*                              TMNscript INCLUDE                             */
/*----------------------------------------------------------------------------*/
#include <TMNscript.h>

/*----------------------------------------------------------------------------*/
/*                              XOM/XMP    INCLUDE                            */
/*----------------------------------------------------------------------------*/
#include <xom.h>
#include <xmp.h>
#include <xmp_cmis.h>

/* 
* ------------------------------------------------------
* The macro map Tcl return codes on to tmns_code_t codes
* TCL_ERROR, TCL_BREAK, TCL_CONTINUE = TMNS_ERROR
* TCL_OK = TMNS_OK
* ------------------------------------------------------ 
*/
#define MAP(r) ( (r) ? TMNS_ERROR : TMNS_OK ) 

/*
* -------------------------------------------------- 
* The macro frees memory and set the pointer to NULL
* -------------------------------------------------- 
*/
#define FREE(ptr) \
	free(ptr); \
	ptr = NULL;

/*
* ---------------------
* TMNscript interpreter
* ---------------------
*/
static Tcl_Interp *tmnsInterp = NULL;

/* 
* -------------------
* TMNscript workspace
* ------------------- 
*/
extern OM_workspace ws;

/* 
* -----------------------------
* xom private objects hashtable
* -----------------------------
*/
extern Tcl_HashTable xomTable;

/*
* -----------------------
* Parser global variables
* -----------------------
*/
static char **splitArray;
static int splitCount = 0;

/*----------------------------------------------------------------------------*/
/*             DECLERATION OF DEBUG VARIABLE                                  */
/*----------------------------------------------------------------------------*/
int tmns_debug = FALSE;

/*
* -----------------------------
* External function definitions
* -----------------------------
*/


/*****************************************************************************
**  Function Name: tmns_init
**
**----------------------------------------------------------------------------
**
**  Description:
**  ------------
** 
**	Create and initialize the TMNscript interpreter 
**
*****************************************************************************/

tmns_return_t *tmns_init(void)
{
	int tcl_ret = TCL_OK; /* Holds Tcl function return values */
	tmns_return_t *set_ret; /* Holds tmns_setModestoDefault return value */

	/*
	* -----------------------------------------
	* Make sure only one interpreter is created 
	* -----------------------------------------
	*/
	if (tmnsInterp != NULL) {
		return tmns_buildResult(TMNS_INVALID_EXECUTION,"Interpreter already created");
	}

	/*
	* ---------------------
	* Initialize the parser
	* ---------------------
	*/
	splitArray = NULL;
	splitCount = 0;

	/*
	* --------------------------------------
	* Create a new Tcl interpreter structure
	* --------------------------------------
	*/
	tmnsInterp = Tcl_CreateInterp();
	if (tmnsInterp == NULL) { /* If create fails return an error */
		return tmns_buildResult(TMNS_PROCESSING_ERROR,"Failed to create the Tcl interpreter");
	}

	/*
	* ------------------------------
	* Initialize the Tcl interpreter
	* ------------------------------
	*/
	tcl_ret = Tcl_Init(tmnsInterp);
	if (tcl_ret != TCL_OK) { /* If initialization fails return error */
		Tcl_DeleteInterp(tmnsInterp);
		tmnsInterp = NULL;
		return tmns_buildResult(TMNS_PROCESSING_ERROR,"Failed to initialize the Tcl interpreter");
	}

	/*
	* ---------------------------------------------------------
	* Declare all the new commands to Tcl provided by TMNscript
	* ---------------------------------------------------------
	*/
	tcl_ret = Tmns_Init(tmnsInterp);
	if (tcl_ret != TCL_OK) {
		tmns_terminateInterp();
		return tmns_buildResult(TMNS_PROCESSING_ERROR,"Failed to load TMNscript extension");
	}

	/*
	* --------------------------------------------------------
	* Set the default operating modes and Check for any errors
	* --------------------------------------------------------
	*/
	set_ret = tmns_setModestoDefault();
	if (set_ret->code != TMNS_OK) {
		tmns_terminateInterp();
		return set_ret;
	}
	tmns_freeResultStructure(set_ret); /* Free the result if no error to report */

	/*
	* ----------------------------------
	* Return a successful initialization
	* ----------------------------------
	*/
	return tmns_buildResult(TMNS_OK,NULL);
}


/*****************************************************************************
**  Function Name: tmns_exe
**
**----------------------------------------------------------------------------
**
**  Description:
**  ------------
** 
**     Evaluates(execute) a command and return a result
**
**  Argument(s) :
**  -------------
**	usrArg = Contains the Tcl/TMNscript command the interpreter evaluates
**
*****************************************************************************/

tmns_return_t *tmns_exe(char *usrArg, ...)
{
	tmns_return_t *eval_result = NULL; /* Holds the result of evaluation */
	char *string_handle = NULL; /* Holds the copy of the input string */
	Tcl_DString argString; /* Holds the complete argument string */
	va_list argList; /* Holds the list of arguments */
	char *tmp = NULL; /* Keeps a temporary pointer to an argument */

	/*
	* ------------------------------------------
	* Check that the interpreter was initialized
	* ------------------------------------------
	*/
	if (tmnsInterp == NULL) {
		return tmns_buildResult(TMNS_INVALID_EXECUTION,"Interpreter not initialized");
	}

        /*
        * --------------
        * Validate input
        * --------------
        */
	if (usrArg == NULL) {
		return tmns_buildResult(TMNS_INVALID_EXECUTION,"No arguments");
	}

        /*
        * -------------------------------------------
        * Concatenate all the arguments into a string
        * -------------------------------------------
        */
	Tcl_DStringInit(&argString);
	Tcl_DStringAppend(&argString,usrArg,-1);

	va_start(argList,usrArg);
	tmp = va_arg(argList, char*);
	while (tmp != NULL) {
		Tcl_DStringAppend(&argString,tmp,-1);
		tmp = va_arg(argList, char*);
	}

	/*
	* -------------------------
	* Evaluate the user command
	* -------------------------
	*/
	eval_result = tmns_eval(Tcl_DStringValue(&argString));

	/*
	* --------
	* Cleaning
	* --------
	*/
	Tcl_DStringFree(&argString);
	va_end(argList);
	

	return eval_result; /* Return the result of the evaluation */
}

/*****************************************************************************
**  Function Name: tmns_getFd
**
**----------------------------------------------------------------------------
**
**  Description:
**  ------------
** 
**     Get the file descriptor associated with the session
**
**  Argument(s) :
**  -------------
**	session = Contains the session identifier
**
*****************************************************************************/

int tmns_getFd(char *session)
{
	/*
	* ------------------
	* Validate the input
	* ------------------
	*/
	if (session == NULL)
		return -1; /* Return -1 if error */

	return tmns_getFileDescriptor(session); /* Returns the C file discriptor for session */
}


/*****************************************************************************
**  Function Name: tmns_end
**
**----------------------------------------------------------------------------
**
**  Description:
**  ------------
** 
**	Terminates the TMNscript interpreter 
**
*****************************************************************************/

void tmns_end(void)
{
	tmns_terminateInterp(); /* Delete the interpreter and free resources */
	return;
}


/*****************************************************************************
**  Function Name: tmns_freeResult
**
**----------------------------------------------------------------------------
**
**  Description:
**  ------------
** 
**     Free memory allocated for the tmns_return_t structure
**
**  Argument(s) :
**  -------------
**      returnValue = Contains a valid pointer to a tmns_return_t structure
**
*****************************************************************************/

void tmns_freeResult(tmns_return_t *returnValue)
{
	/*
	* --------------------------------
	* Free the tmns_return_t structure
	* --------------------------------
	*/	
	tmns_freeResultStructure(returnValue); /* Free memory allocated for structure */

	return;
}

/*****************************************************************************
**  Function Name: tmns_freeSplitList
**
**----------------------------------------------------------------------------
**
**  Description:
**  ------------
** 
**     Free memory allocated by the tmns_splitList function
**
**  Argument(s) :
**  -------------
**      elementArray = Contains a valid pointer to a pointer of array
**
*****************************************************************************/

void tmns_freeSplitList(char ***elementArray)
{
	/*
	* --------------------------------
	* Free the tmns_return_t structure
	* --------------------------------
	*/	
	if ((*elementArray) != NULL)
		FREE((*elementArray));

	return;
}


/*****************************************************************************
**  Function Name: tmns_getArgValue
**
**----------------------------------------------------------------------------
**
**  Description:
**  ------------
** 
**     Finds the value of the specified argument in the string
**
**  Argument(s) :
**  -------------
**	usrArg = Contains a valid pointer to a string of arguments and values
**	key = Contains a valid pointer to an argument(key) to be searched for 
**
*****************************************************************************/

tmns_return_t *tmns_getArgValue(char *usrArg, char *key)
{
	/*
	* ------------------------------------------
	* Check that the interpreter was initialized
	* ------------------------------------------
	*/
	if (tmnsInterp == NULL) { /* Parser makes use of the interpreter */
		return tmns_buildResult(TMNS_INVALID_EXECUTION,"Interpreter not initialized");
	}

	/*
	* --------------
	* Validate input
	* --------------
	*/
	if (((usrArg == NULL) && (splitArray == NULL)) || (key == NULL)) {
		return tmns_buildResult(TMNS_INVALID_EXECUTION,"Argument(s) invalid");
	}

	return tmns_getValuefromList(usrArg,key); /* Return the argument value */
}

/*****************************************************************************
**  Function Name: tmns_splitList
**
**----------------------------------------------------------------------------
**
**  Description:
**  ------------
** 
**     Split a list of elements into an array of constituent elements
**
**  Argument(s) :
**  -------------
**	usrArg = Contains a valid pointer to a string of arguments and values
**	elementArray = Contains the address of an array of string pointers
**	elementCount = Number of elements in array
**
*****************************************************************************/

tmns_return_t *tmns_splitList(char *usrArg, char ***elementArray,int *elementCount)
{
	int split_ret = TMNS_OK; /* Holds the result of the TCL_SplitList */

	/*
	* ----------
	* Initialize
	* ----------
	*/	
	(*elementArray) = NULL;

	/*
	* ------------------------------------------
	* Check that the interpreter was initialized
	* ------------------------------------------
	*/
	if (tmnsInterp == NULL) {
		return tmns_buildResult(TMNS_INVALID_EXECUTION,"Interpreter not initialized");
	}

	/*
	* --------------
	* Validate input
	* --------------
	*/
	if (usrArg == NULL) {
		return tmns_buildResult(TMNS_INVALID_EXECUTION,"Argument(s) invalid");
	}

	/*
	* ----------------------------
	* Split the list into an array 
	* ----------------------------
	*/
	split_ret = MAP(Tcl_SplitList(tmnsInterp,usrArg,elementCount,elementArray));
	if (split_ret != TMNS_OK) {
		return tmns_buildResult(split_ret,tmnsInterp->result);
	}

	return tmns_buildResult(split_ret,NULL);
}



/*
* -----------------------------
* Internal function definitions
* -----------------------------
*/

/*****************************************************************************
**  Function Name: tmns_set
**
**----------------------------------------------------------------------------
**
**  Description:
**  ------------
** 
**     Sets a variable globally in Tcl
**
**  Argument(s) :
**  -------------
**	variable = Contains a valid pointer to a variable name
**	value = Contains a valid pointer to the value of the variable
**
*****************************************************************************/

tmns_return_t *tmns_set(char *variable, char *value)
{
	char *set_ret = NULL; /* Holds the result of the Tcl_SetVar */

	/*
	* ---------------------------------------------------
	* Set a Tcl variable globally and keep error messages
	* ---------------------------------------------------
	*/
	set_ret = Tcl_SetVar(tmnsInterp, variable, value, TCL_GLOBAL_ONLY|TCL_LEAVE_ERR_MSG);
	if (set_ret == NULL) { /* An error occured in setting the variable */
		return tmns_buildResult(TMNS_PROCESSING_ERROR, tmnsInterp->result);
	}

	/*
	* -------------------------------------
	* Check that variable was set correctly
	* -------------------------------------
	*/
	if (strcmp(value,set_ret) != 0) {
		return tmns_buildResult(TMNS_PROCESSING_ERROR,"Variable set incorrect by Tcl");
	}

	return tmns_buildResult(TMNS_OK, set_ret);	
}

/*****************************************************************************
**  Function Name: tmns_eval
**
**----------------------------------------------------------------------------
**
**  Description:
**  ------------
** 
**     Evaluate a command in Tcl with Tcl_Eval
**
**  Argument(s) :
**  -------------
**	evalString = Contains a valid pointer to a command
**
*****************************************************************************/

tmns_return_t *tmns_eval(char *evalString)
{
	tmns_code_t eval_ret; /* Holds the result of the TCL_Eval */
	
	/*
	* -----------------------------
	* Display the evaluation string
	* -----------------------------
	*/
	if (tmns_debug == TRUE) {
		printf("\n%s\n",evalString);
	}

	/*
	* ------------------------------
	* Evaluate the evaluation string
	* ------------------------------
	*/
	eval_ret = MAP(Tcl_Eval(tmnsInterp, evalString));

	return tmns_buildResult(eval_ret, tmnsInterp->result);
}

/*****************************************************************************
**  Function Name: tmns_terminateInterp
**
**----------------------------------------------------------------------------
**
**  Description:
**  ------------
** 
**     Delete the TMNscript interpreter and clean related allocated resources
**
*****************************************************************************/

void tmns_terminateInterp(void)
{
	if (!(Tcl_InterpDeleted(tmnsInterp))) { /* Clean only if interpreter was created */

		/* 
		* ------------------------------------
		* Clean-up memory allocated for parser
		* ------------------------------------
		*/
		if (splitArray != NULL) {
			FREE(splitArray);
			splitArray = NULL;
		}

		/* 
		* -------------------------------------------------------
		* Delete the TMNscript workspace and associated resources
		* -------------------------------------------------------
		*/
		mp_shutdown(ws);

		/* 
		* --------------------
		* Interpreter deletion 
		* --------------------
		*/
		Tcl_DeleteInterp(tmnsInterp);
		tmnsInterp = NULL;
	}
	return;
}

/*****************************************************************************
**  Function Name: tmns_getValuefromList
**
**----------------------------------------------------------------------------
**
**  Description:
**  ------------
** 
**     Find the value of the argument(key) from the argument list
**
**  Argument(s) :
**  -------------
**	listString = Contains a valid pointer to a list
**	argString = Contains a valid pointer to a search argument
**
*****************************************************************************/

tmns_return_t *tmns_getValuefromList(char *listString,char *argString)
{
	int split_ret = TMNS_ERROR; /* Holds the result of the Tcl_SpliList */
	tmns_return_t *retval = NULL; /* Holds the result send back to user */
	int count; /* Holds the current position in the array */
	int argFnd = 0; /* Holds the search result for the argument */
	char *val = NULL; /* Holds a pointer to the argument value */ 
	int valFnd = 0; /* Holds the search result for the argument value */

	/*
	* ------------------------------
	* Split the list into it's items
	* ------------------------------
	*/
	if (listString != NULL) { /* New list of arguments */

		/*
		* -------------------------------------
		* Free previous array and reset counter
		* -------------------------------------
		*/
		if (splitArray != NULL) {
			FREE(splitArray); /* Free allocated memory */
			splitArray = NULL;
			splitCount = 0;   /* Reset the array counter */
		}

		/* 
		* ------------------
		* Split the new list
		* ------------------
		*/
		split_ret = MAP(Tcl_SplitList(tmnsInterp,listString,&splitCount,&splitArray));
		if (split_ret != TMNS_OK) { /* Report Tcl error */
			return tmns_buildResult(split_ret,tmnsInterp->result);
		}
	}


	/*
	* ----------------------------------------------
	* Search for -mtype(message type) argument value 
	* ----------------------------------------------
	*/
	if (strcmp("-mtype",argString) == 0) {
		/*
		* ---------------------------------------
		* Message type always at beginnig of list
		* ---------------------------------------
		*/
		if ((splitArray[0][0] == 'm') && (splitArray[0][1] == 'p') && (splitArray[0][2] == ':') && (splitArray[0][3] == ':')) { /* Not a complete check for compatibility purposes */
			return tmns_buildResult(TMNS_OK,splitArray[0]);
		}
		return tmns_buildResult(TMNS_ERROR,"No message type");
	}


	/* 
	* -----------------------------------
	* Search for the argument in the list
	* -----------------------------------
	*/
	for (count = 0; count < splitCount; count++) {
		argFnd = tmns_findArgumentandValue(splitArray[count],argString,&valFnd,&val);
		if (argFnd == 1) {
		/* 
		* ------------------------------
		* A non TMNscript argument value
		* ------------------------------
		*/

			/* 
			* -------------------------------------
			* Valid value was found in lower levels
			* -------------------------------------
			*/
			if ((valFnd == 1) && (val == NULL))
				return tmns_buildResult(TMNS_ERROR,"No argument value");
			if ((val != NULL) && (valFnd == 1)) { /* Value from lower levels */
				retval = tmns_buildResult(TMNS_OK,val);
				FREE(val); /* Cleaning */
				return retval;
			}

		/* 
		* --------------------------
		* A TMNscript argument value
		* --------------------------
		*/

			/* 
			* --------------------
			* Test for end of list
			* --------------------
			*/
			if (count == (splitCount-1)) 
				return tmns_buildResult(TMNS_ERROR,"No argument value");

			/* 
			* ----------------------------------------------
			* Check the validity of TMNscript argument value 
			* ----------------------------------------------
			*/

			/*
			* -------------------
			* Unexpected argument
			* ------------------- 
			*/
			if (splitArray[count+1][0] == '-') {
				return tmns_buildResult(TMNS_ERROR,"No parameter given after argument");
			}

			/* 
			* -----------------------------------------------
			* Must be an argument or nill following the value
			* -----------------------------------------------
			*/
			if ((splitArray[count+2] != NULL)) {
				if ((splitArray[count+2][0] != '-'))
					return tmns_buildResult(TMNS_ERROR,"Invalid argument string");
			}
 
			return tmns_buildResult(TMNS_OK,splitArray[count+1]);
		}
	}

	return tmns_buildResult(TMNS_ERROR,"Argument not found");
}

/*****************************************************************************
**  Function Name: tmns_findArgumentandValue
**
**----------------------------------------------------------------------------
**
**  Description:
**  ------------
** 
**     Search for argument and value recursively in the lower layers of the list
**
**  Argument(s) :
**  -------------
**	element = Contains a valid pointer to a list
**	arg = Contains a valid pointer to a search argument
**	valFnd = Holds record if a value was found for the argument
**	val = Holds the argument value
**
*****************************************************************************/

int tmns_findArgumentandValue(char *element,char *arg,int *valFnd,char **val)
{
	char **argvptr; /* Holds the array returned by Tcl_SplitList */
	int argc = 0; /* Holds the number of elements in the array */
	int pos; /* Holds the current position in the array */
	int ret = TCL_OK; /* Holds the result of the Tcl_SplitList */
	int argFnd = 0; /* Holds record if the argument was found */

	/* 
	* --------------
	* Initialization
	* --------------
	*/
	(*val) = NULL;
	(*valFnd) = 0;
	argvptr = NULL;

	/* 
	* ------------------------------
	* Check if element is not a list
	* ------------------------------
	*/
	ret = Tcl_SplitList(tmnsInterp,element,&argc,&argvptr);
	if (ret == TCL_ERROR) { /* The list is invalid */
		return 0;
	}
	if ((argc == 1) && (element[0] != '{')) { /* Element is not a list */
		if (strcmp(arg,argvptr[0]) == 0) {
			FREE(argvptr); /* Cleaning */
			return 1; /* Return argument found to upper level */
		}

		FREE(argvptr); /* Cleaning */
		return argFnd;
	}

	/* 
	* -------------------------------------------------------
	* Search for the argument in the lower levels of the list
	* -------------------------------------------------------
	*/
	for (pos = 0; pos < argc; pos++) { /* Recurse into lower levels */
		argFnd = tmns_findArgumentandValue(argvptr[pos],arg,valFnd,val);
		if (argFnd == 1) { /* Argument found */
			/* 
			* ------------------------------
			* A non TMNscript argument value
			* ------------------------------
			*/
			if ((*valFnd) == 0) { /* Value not found */
				(*valFnd) = 1; /* The value must be on this level */
				/* 
				* ----------------------------------
				* Argument followed by value in list
				* ----------------------------------
				*/
				if (pos != (argc-1)) { /* Check for end of list */
					(*val) = tmns_buildString(argvptr[pos+1]);

					FREE(argvptr); /* Cleaning */
					return argFnd; /* Value found */
				}
			}

			FREE(argvptr); /* Cleaning */
			return argFnd; /* Value found in lower levels */
		}
	}

	FREE(argvptr); /* Cleaning */
	return argFnd; /* Argument not found */
}

/*****************************************************************************
**  Function Name: tmns_getFileDescriptor
**
**----------------------------------------------------------------------------
**
**  Description:
**  ------------
** 
**     Get the file descriptor for the session
**
**  Argument(s) :
**  -------------
**	tmnsSession = Contains a valid pointer to a session
**
*****************************************************************************/

int tmns_getFileDescriptor(char *tmnsSession) 
{
	Tcl_HashEntry *hash_entry; /* Holds tmnsSession entry in hash table */
	OM_private_object session_obj; /* Holds a session object */
	OM_exclusions exclusions; /* Holds the exclusions for om_get */
	OM_type types[2]; /* Holds the object types */
	OM_public_object copy; /* Holds a public object */
	OM_value_position total; /* Holds the number of descriptors */
	OM_return_code get_ret; /* Holds the om_get result */
	int fd ; /* Holds file descriptor of session */
	
        /*
        * ------------------------------------------------------------------
        *  Locate the entry corresponding to key(=tmnsSession) in the table
        * ------------------------------------------------------------------
        */
	hash_entry = Tcl_FindHashEntry(&xomTable, tmnsSession);
	if (hash_entry == NULL) { /* If the session was not find */
		return -1;
	}

	/*
	* ----------------------------------------
	* Get the session object from the hastable
	* ---------------------------------------- 
	*/
	session_obj = (OM_private_object) Tcl_GetHashValue(hash_entry);

        /*                               
        * ---------------------------------------------------
	* Extract the file descriptor from the session object
        * ---------------------------------------------------
        */

	/* 
	* -------------------------------------------------------------------
	* Excludes all attributes (except those spesified in types) and holds
	* handles to the original private subobjects
	* -------------------------------------------------------------------
	*/
   	exclusions = OM_EXCLUDE_ALL_BUT_THESE_TYPES | OM_EXCLUDE_SUBOBJECTS;

	/* 
	* --------------------------------------
	* Types of attributes to include in copy
	* -------------------------------------- 
	*/
	types[0] = MP_FILE_DESCRIPTOR ;
	types[1] = OM_NO_MORE_TYPES ;
	
	/* 
	* -----------------------------------------------------
	* Create a public object containing the file descriptor
	* ----------------------------------------------------- 
	*/
	get_ret = om_get(session_obj, exclusions, types, OM_FALSE, 0, 0, &copy, &total);
	if (get_ret != OM_SUCCESS) /* Return -1 if error */
		return -1;

	/* 
	* -----------------------------------------------------
	* Make a copy of the file descriptor from public object
	* ----------------------------------------------------- 
	*/
   	fd = copy[0].value.integer;

	/*
	* --------------------------
	* Return the file descriptor
	* --------------------------
	*/
	return fd;
}

/*****************************************************************************
**  Function Name: tmns_setModestoDefault
**
**----------------------------------------------------------------------------
**
**  Description:
**  ------------
** 
**     Sets the TMNscript operating modes to deafualt
**
*****************************************************************************/

tmns_return_t *tmns_setModestoDefault(void)
{
	tmns_return_t *set_ret = NULL; /* Holds the results of the set functions */

	/* 
	* --------------------------------
	* mapperModeOn disabled by default 
	* --------------------------------
	*/
	set_ret = tmns_set("mp_mapperModeOn","0"); /* false */
	if (set_ret->code == TMNS_PROCESSING_ERROR) {
		return set_ret;
	}
	tmns_freeResultStructure(set_ret); /* Free the result if no error to report */

	/* 
	* ------------------------------
	* asn1ModeOn disabled by default 
	* ------------------------------
	*/
	set_ret = tmns_set("mp_asn1ModeOn","0"); /* false */
	if (set_ret->code == TMNS_PROCESSING_ERROR) {
		return set_ret;
	}
	tmns_freeResultStructure(set_ret); /* Free the result if no error to report */

	/* 
	* ------------------------------
	* misTraceOn disabled by default 
	* ------------------------------
	*/
	set_ret = tmns_set("mp_misTraceOn","0"); /* false */
	if (set_ret->code == TMNS_PROCESSING_ERROR) {
		return set_ret;
	}
	tmns_freeResultStructure(set_ret); /* Free the result if no error to report */

	/* 
	* -------------------------------
	* errorModeOn enabled by default 
	* -------------------------------
	*/
	set_ret = tmns_set("mp_errorModeOn","1"); /* true */
	if (set_ret->code == TMNS_PROCESSING_ERROR) {
		return set_ret;
	}
	tmns_freeResultStructure(set_ret); /* Free the result if no error to report */

	/* 
	* ------------------------------------
	* internalStatusOn disabled by default 
	* ------------------------------------
	*/
	set_ret = tmns_set("mp_internalStatusOn","0"); /* false */
	return set_ret;
}

/*****************************************************************************
**  Function Name: tmns_buildString
**
**----------------------------------------------------------------------------
**
**  Description:
**  ------------
** 
**     Builds a character string containing a message
**
**  Argument(s) :
**  -------------
**	message = Contains a valid pointer to a message
**
*****************************************************************************/

char *tmns_buildString(char *message)
{
	char *msg_handle = NULL; /* Holds a pointer to a message string */

	/*
	* --------------
	* Validate input
	* --------------
	*/
	if (message == NULL) /* Return NULL if the input is invallid */
		return NULL;

	/* 
	* ---------------------------------------------
	* Allocate memory to hold a copy of the message
	* --------------------------------------------- 
	*/
	msg_handle = (char *) calloc(strlen(message) + 1,1);
	if (msg_handle != NULL)
		strcpy(msg_handle,message);

	return msg_handle; /* Return NULL if memory is insufficient */
}

/*****************************************************************************
**  Function Name: tmns_buildResult
**
**----------------------------------------------------------------------------
**
**  Description:
**  ------------
** 
**     Builds a character string containing a message
**
**  Argument(s) :
**  -------------
**	codeValue = Contains the result of processing
** 	message = Contains a valid pointer to a message
**
*****************************************************************************/

tmns_return_t *tmns_buildResult(tmns_code_t codeValue, char *message)
{
	tmns_return_t *result_handle = NULL; /* Hold a pointer to a tmns_return_t structure */

	/* 
	* ---------------------------------
	* Allocate memory for the structure
	* --------------------------------- 
	*/
	result_handle = (tmns_return_t *) calloc(1,sizeof(tmns_return_t));
	if (result_handle == NULL) { /* Return NULL if memory is insufficient */
		return NULL;
	}

	/* 
	* -------------------------------------
	* Give values to the tmns_return_t fields
	* -------------------------------------
	*/
	result_handle->code = codeValue; /* Success of the operation */
	if (message != NULL) { /* Result message string */
		result_handle->msg = tmns_buildString(message);
	}
	else { /* No message to report */
		result_handle->msg = tmns_buildString("");
	}

	return result_handle; /* return a tmns_return_t value */
}

/*****************************************************************************
**  Function Name: tmns_freeResultStructure
**
**----------------------------------------------------------------------------
**
**  Description:
**  ------------
** 
**     Free memory allocated for a tmns_return_t structure and related fields
**
**  Argument(s) :
**  -------------
**	result = Contains a valid pointer to a tmns_return_t structure
**
*****************************************************************************/

void tmns_freeResultStructure(tmns_return_t *result)
{
	/*
	* --------------
	* Validate input
	* --------------
	*/
	if (result == NULL)
		return;

	/*
	* ------------------
	* Free result string
	* ------------------
	*/
	if (result->msg != NULL)
		FREE(result->msg);

	/*
	* ----------------------------
	* Free tmns_return_t structure
	* ----------------------------
	*/
	FREE(result);

	return;
}
