#ifdef _SUNOS
#pragma ident "@(#)common.c	1.4 97/10/09 Copyright SMI"
#endif

/******************************************************************************
* Copyright 08/21/97 Sun Microsystems, Inc. All Rights Reserved		      *
*			All Rights Reserved				      *
******************************************************************************/

/******************************************************************************/
/*  File Name: 		common.c                                              */
/*----------------------------------------------------------------------------*/
/*                                                                            */
/*  Description:                                                              */
/*  -----------                                                               */
/*       The present file contains the implementation of the functions        */
/*       common to the example.			              		      */
/*                                                                            */
/******************************************************************************/

/*----------------------------------------------------------------------------*/
/*                               STANDARD INCLUDE                             */
/*----------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/*----------------------------------------------------------------------------*/
/*                               SYSTEM CALLS	                	      */
/*----------------------------------------------------------------------------*/
#include <sys/utsname.h>

/*----------------------------------------------------------------------------*/
/*                               COMMON INCLUDE                               */
/*----------------------------------------------------------------------------*/
#include "common.h"

/*
* -------------------------------------------------- 
* The macro frees memory and set the pointer to NULL
* -------------------------------------------------- 
*/
#define FREE(ptr) \
	free(ptr); \
	ptr = NULL;

/*
* ---------------------------------------------------
* Symbolic constants representing error message types
* ---------------------------------------------------
*/
#define EVALUATION_ERROR "Evaluation error:"
#define EXECUTION_ERROR "Execution error:"
#define PROCESSING_ERROR "Processing error:"

/*
* ------------------------------------------------------------------
* Global variable to enable the pretty printing of messages received
* ------------------------------------------------------------------
*/
int prettyprint = FALSE; /* By default disabled */


/*
* ------------------------------------------------------------------------
* This function loads Tcl\Tk scripts into the interpreter using the Tcl\Tk 
* command "source". 
* ------------------------------------------------------------------------
*/
void load(char *tclFile)
{
	tmns_return_t *result = NULL;

	/* 
	* Load the script into the interpreter
	*/
	result = evaluateResult(tmns_exe("source ",tclFile,NULL));

	/* 
	* Cleaning 
	*/
	tmns_freeResult(result);
}

/*
* ----------------------------------------------------------------------
* The funtion prints the constituent arguments of a message received
* using the tmns_splitList and tmns_freeSplitList.
* NOTE: These functions are very handy to retrieve the single message(s) 
*	contained in a synchronized linked reply. As all the message(s) are
*	received in a single list.
* ----------------------------------------------------------------------
*/
void prettyPrint(tmns_return_t *messageReceived)
{
	char **argumentArray;
	int argumentCount = 0;
	tmns_return_t *result = NULL;
	int loopCount;

	/* 
	* Brake the list up into its constituent arguments 
	*/
	result = evaluateResult(tmns_splitList(messageReceived->msg,&argumentArray,&argumentCount));
	/*
	* Print the heading
	*/
	printf("\n%40s\n","Message Received");
	printf("%40s\n","----------------");

	/*
	* Print the message type
	*/
	printf("%-15s%s\n","Type:",argumentArray[0]);
	printf("-----\n");

	/*
	* Print the column headings
	*/
	printf("%-15s%s\n","Argument","Value");
	printf("%-15s%s\n","--------","-----");

	/*
	* Print the constituent arguments and values of the message
	*/
	for (loopCount = 1; loopCount < argumentCount; loopCount += 2) {
		printf("%-15s%s\n",argumentArray[loopCount],argumentArray[loopCount+1]);

	}

	/*
	* Cleaning
	*/
	tmns_freeResult(result);
	tmns_freeSplitList(&argumentArray);
}


/*
* -------------------------------------------------------------------
* This function takes a message header and a tmns_return_t result and 
* displays it on the screen.
* -------------------------------------------------------------------
*/
void printMessage(char *header,tmns_return_t *result)
{
	printf("\n%s %s\n",header,result->msg);
}

/*
* --------------------------------------------------------------------------
* This function shows how you can use the Tcl/Tk commands with the CAPI.
* We assign a variable a value using the "set" command.
* The value itself is a Tcl\Tk list created with the "list" command.
* NOTE: The value returned by the interpreter in the tmns_return_t structure
*	contains the elements of the list and not the list representation
*	itself. If you want to make it a single list you will have to add
*	braces at the beginning and end of the string, the same as in Tcl\Tk.
* --------------------------------------------------------------------------
*/
tmns_return_t *setMoi(void)
{
        tmns_return_t *result = NULL;

	/*
	* Set the variable "moi" in the interpreter
	*/
	result = evaluateResult(tmns_exe("set moi [ list {systemId {",getHostName(),"}} ]",NULL));
 
	/*
	* Print the arguments of the list 
	*/
        printMessage("Managed object instance (moi):",result);

        return result;
}

/*
* ---------------------------------------------------------------------------
* This function takes a tmns_return_t structure returned and evaluates if the
* call was succesfull.If the call was not successful it will print out a
* corresponding error message and clean up before exiting the program, 
* otherwise it will return the result to the calling function.
* ---------------------------------------------------------------------------
*/
tmns_return_t *evaluateResult(tmns_return_t *result)
{
	/*
	* Check if the result was not succesfull
	*/
	if (result->code != TMNS_OK) { /* Error occured */
		/*
		* Print the type of error and error specific information
		*/
		switch (result->code) {
		case TMNS_ERROR :
			printMessage(EVALUATION_ERROR,result);
			break;
		case TMNS_INVALID_EXECUTION :
			printMessage(EXECUTION_ERROR,result);
			break;
		case TMNS_PROCESSING_ERROR :
			printMessage(PROCESSING_ERROR,result);
			break;
		}

		/*
		* Clean up and exit
		*/
		tmns_freeResult(result);
		tmns_end();
		exit(1); 
	}

	return result; /* no errors return result */
}

/*
* ---------------------------------------------------------------------------
* This function sets the interpreter configuration to the values specified in
* the tmns_configuration_t structure. It only sets a variable if the value 
* specified differs from the default value.
* ---------------------------------------------------------------------------
*/
void configureOperatingModes(tmns_configuration_t configure)
{
	tmns_return_t *result = NULL;

	/* 
	* Enables ASN.1 mode.
	*/
	if (configure.mp_asn1ModeOn == TRUE) { /* Disabled by default */
		result = evaluateResult(tmns_exe("set mp_asn1ModeOn true",NULL));
		printf("\nASN.1 mode enabled\n");
		tmns_freeResult(result);
	}

	/*
	* Disables the error mode.
	*/
	if (configure.mp_errorModeOn == FALSE) { /* Enabled by default */
		result = evaluateResult(tmns_exe("set mp_errorModeOn false",NULL));
		printf("\nerror mode enabled\n");
		tmns_freeResult(result);
	}

	/*
	* Enables the mapper mode.
	*/
	if (configure.mp_mapperModeOn == TRUE) { /* Disabled by default */
		result = evaluateResult(tmns_exe("set mp_mapperModeOn true",NULL));
		printf("\nmapper mode enabled\n");
		tmns_freeResult(result);
	}

	/*
	* Enables the trace mode.
	*/
	if (configure.mp_misTraceOn == TRUE) { /* Disabled by default */
		result = evaluateResult(tmns_exe("set mp_misTraceOn true",NULL));
		printf("\ntrace mode enabled\n");
		tmns_freeResult(result);
	}

	/*
	* Enables the status mode.
	*/
	if (configure.mp_internalStatusOn == TRUE) { /* Disabled by default */
		result = evaluateResult(tmns_exe("set mp_internalStatusOn true",NULL));
		printf("\nstatus mode enabled\n");
		tmns_freeResult(result);
	}

	/*
	* Disable the encoding/decoding of universal types.
	* NOTE: This should be disabled if the ASN.1 mode is enabled.
	*/
	if (configure.xmp_decoding == FALSE) { /* Enabled by default */
		result = evaluateResult(tmns_exe("mp::negotiate { XMP_DECODING off }",NULL));
		printf("\nSupport for universal types disabled\n");
		tmns_freeResult(result);
	}

}


/*
* ----------------------------------------------------------------------
* This function registers a user/application using the TMNscript command
* "mp::bind". 
* ----------------------------------------------------------------------
*/
char *bind(char *address)
{
	tmns_return_t *result = NULL;
	char *session = NULL;	

	/*
	* Register the user/application
	*/
	result = evaluateResult(tmns_exe("mp::bind ",address,NULL));
	printf("\nBind with automatic connection management\n");

	/* 
	* Make a copy of the session retuned
	*/
	session = buildMsg(result->msg);

	/* 
	* Cleaning 
	*/
	tmns_freeResult(result);

	return session; /* Return the session received */
}

/*
* ------------------------------------------------------------------------
* This function unregisters a user/application using the TMNscript command
* "mp::unbind".
* ------------------------------------------------------------------------
*/
void unbind(char *session)
{
	tmns_return_t *result = NULL;

	/*
	* Unregister the user/application
	*/
	result = evaluateResult(tmns_exe("mp::unbind ",session,NULL));
	printf("\nUnbind the session: %s\n",session);

	/* 
	* Cleaning 
	*/
	tmns_freeResult(result);
}

/*
* ---------------------------------------------------------------------------
* This function sends a GET request using the TMNscript command "mp::getReq".
* ---------------------------------------------------------------------------
*/
void sendGetAll(tmns_get_req_t getReq)
{
	tmns_return_t *result = NULL;

	/*
	* Enable the debug mode of the C API to see the messages send to the
	* interpreter.
	*/
	tmns_debug = TRUE;

	/* 
	* Send the GET request.
	* NOTE: We can still access the variables set in the interpreter
	* 	just like in Tcl\Tk.
	*	We use the "moi" variable previously set in the interpreter.
	*/
	printf("\nSend GET REQUEST to the agent:\n");
	result = evaluateResult(tmns_exe("mp::getReq -ses ",getReq.session," -at ",getReq.at," -scope ",getReq.scope," -moc ",getReq.moc," -moi $moi -info ",getReq.info,NULL));

	/*
	* Disable debug mode
	*/
	tmns_debug = FALSE;

	/* 
	* Cleaning 
	*/
	tmns_freeResult(result);
}

/*
* ----------------------------------------------------------------------------
* This function sends a GET response using the TMNscript command "mp::getRsp".
* ----------------------------------------------------------------------------
*/
void sendGetRsp(tmns_get_rsp_t getRsp)
{
	tmns_return_t *result = NULL;

	/*
	* Enable the debug mode of the C API to see the messages send to the
	* interpreter.
	*/
	tmns_debug = TRUE;

	/* 
	* Send the GET request.
	* NOTE: We can still access the variables set in the interpreter
	* 	just like in Tcl\Tk.
	*	In the INFO symbolic constant declared in the main program 
	* 	we access the "moi" variable set in the interpreter and 
	*	retrieve the value adding it to the list using Tcl/Tk commands.
	*/
	printf("\nSend GET RESPONSE to the manager:\n");
	result = evaluateResult(tmns_exe("mp::getRsp -ses ",getRsp.session," -id ",getRsp.id," -link ",getRsp.link," -moc ",getRsp.moc," -moi $moi -error ",getRsp.error," -info ",getRsp.info,NULL));

	/*
	* Disable debug mode
	*/
	tmns_debug = FALSE;

	/* 
	* Cleaning 
	*/
	tmns_freeResult(result);
}

/*
* ----------------------------------------------------------------------
* This function retrieves the value of the "-id" argument from a message 
* received using the tmns_getArgValue function.
* ----------------------------------------------------------------------
*/
char *getInvokeId(tmns_return_t *ind)
{
	tmns_return_t *result = NULL;
	char *id = NULL;

	/*
	* Retrieve the value of the corresponding key
	*/
	result = evaluateResult(tmns_getArgValue(ind->msg,"-id"));
	id = buildMsg(result->msg);

	/* 
	* Cleaning 
	*/
	tmns_freeResult(result);

	return id;	
}

/*
* --------------------------------------------------
* The function returns the name of the host machine.
* --------------------------------------------------
*/
char *getHostName(void) 
{
	struct utsname name;
	char *hostname = NULL;
	
	/* 
	* Get the host name
	*/
	if (uname(&name) > -1) {
	    hostname = buildMsg(name.nodename);
	}

	if (hostname == NULL) {
		return buildMsg("unknown");
	}

	return hostname;
}

/*
* --------------------------------------
* The function makes a copy of a string.
* --------------------------------------
*/
char *buildMsg(char *msg)
{
	char *buff = NULL;
	int buffsize = 0;

	buffsize = strlen(msg)+1;
	buff = (char *) calloc(buffsize,1);
	strcpy(buff,msg);

	return buff;
}

/*
* ------------------------------------------------------------------------------
* The function waits 10 seconds for a message and receives the incoming message.
* ------------------------------------------------------------------------------
*/
tmns_return_t *receiveOne(char *session)
{
	tmns_return_t *result = NULL;

	/*
	* Wait 10 seconds for a reply
	*/
	result = evaluateResult(tmns_exe("mp::wait 10 ",session,NULL));

	/* 
	* Cleaning 
	*/
	tmns_freeResult(result);

	/* 
	* Receive the incomming message
	*/
	result = evaluateResult(tmns_exe("mp::receive ",session,NULL));

	/*
	* Print the message received
	*/
	if (prettyprint) { /* Pretty print enabled */
		prettyPrint(result);
	} else {
		printMessage("Message Received:",result);	
	}

	return result;
}

/*
* -----------------------------------------------------------------------
* The function receives all the messages of an asynchronous linked reply.
* -----------------------------------------------------------------------
*/
void receiveAll(char *session)
{
	tmns_return_t *link = NULL;
	tmns_return_t *result = NULL;

	do {
		if (link != NULL) /* Cleaning */
			tmns_freeResult(link);

		result = receiveOne(session);

		/*
		* Retrieve the value of the "-link" argument from the message
		* received.
		*/
		link = evaluateResult(tmns_getArgValue(result->msg,"-link"));

		/* 
		* Cleaning 
		*/
		tmns_freeResult(result);
	} while (strcmp("last",link->msg) != 0);

	/* 
	* Cleaning 
	*/
	tmns_freeResult(link);

	return;
}
