#
# End of pp pre-processing

proc boks_admin_users {{list {}}} {
    set admins {}
    boks_priv {
	foreach u [split [read_file [boksdir etc]/admins] \n] {
	    if [string length $u] {
		lappend admins $u
	    }
	}
    }
    if ![string length $list] {
	set list [boks_listusers]
    }
    set result {}
    foreach u $list {	
	if {[lsearch $admins $u] > -1} {
	    lappend result $u
	}
    }
    return $result
}

# Return list of users who are allowed for subadmin based
# only on HOSTGROUPS and USERNAMES
# Accepts wildcard
#
proc boks_subadm_listusers {{wildcard *}} {
    set res {}
    foreach u [boks_listusers $wildcard] {
	foreach {h n} [split $u :] {}
	if {[boks_subadm_check HOSTGROUPS $h] && [boks_subadm_check USERNAMES $n]} {
	    lappend res $u
	}
    }
    return $res
}

#======================================================================
# Floding's additions to the sub administratio
#======================================================================

# Produces an error report on the arguments named in 'l'.
# Uses array 'u' to get the values corresponding to the symbols
# in 'l'.  'l' is a list of symbols as returned by boks_subadm_checkargs.
#
proc boks_subadm_err_report { l _u {textsym subadm_err_intro} {_x ""} } {
    upvar 1 $_u u
    if [string length $_x] {
	upvar 1 $_x x
	array set v [array get x]
    }
    set v(SUBADMIN) [boks_logname]
    set msg [descsubst [MSG_GET $textsym] v]
    append msg "\n\n"
    foreach i $l {
	if [info exists u($i)] {
	    append msg \
		[format "  %25s: %s\n\n" [MSG_GET subadm_err_sym_$i] $u($i)]
	} else {
	    append msg [format "  %s\n" [MSG_GET subadm_err_sym_$i]]
	}
    }
    return $msg
}

# Check arguments in 'chklist' and report any errors.
# Returns 0 if no errors were reported, 1 if there was any error.
#
proc boks_subadm_showerr {chklist {intro subadm_err_intro} {_t ""} } {
    set chk [eval boks_subadm_checkargs $chklist]

    if ![string length $chk] {
	return 0
    }
    if [string length $_t] {
	upvar 1 $_t t
    } else {
	set t ""
    }

    array set u $chklist

    set v(REPORT) [boks_subadm_err_report $chk u $intro t]

    ERROR subadm_error {} v errorpre.tmpl
    return 1
}

#######################################
# Routines for subadministration
#######################################

#######################################
# Global vars
#######################################

#######################################
# This one should maybe be in bokstab.tcl
#
# Array giving subadmin checks available,
# and what type the check is
# (range is number range, regexp is regular expression, yesno
# is simply 1 or 0 and that value is returned as result of check routines
# and in this case the val supplied to the ceck routines is just a dummy.
# accexp is a regexp, but any access methods are converted
# to upper case before comparison).
#
# Examples of what is in array elements
# range  : {{start stop} singlenumber single number {start stop}}
# regexp : {regexp regexp regexp}
# accexp : {accexp {accexp accexp !accexp} {accexp !accexp}}
#   (e.g. {{su:.*->.* !su:.*->(root|root@.*|#0|#0@.*)} rlogin:H1->H2})
# yesno  : 0
#
# UID		- allowed uid range
# UNIXGROUPS	- allowed group names
# USERNAMES	- allowed login names
# HOSTGROUPS	- allowed hostgroups to operate on
# USERCLASS	- allowed userclasses to operate on
# USER_USERCLASS - allowed userclasses to add users to
# ACCESSROUTES	- allowed access routes
# PSD_ALL	- allowed to admin PSD fully.
# PSD_USERS	- allowed to admin user & host PSDs only.
#
# Some reserved "yesno" symbols (not yet implemented, will always
# result in "no"):
#
# ONETIMEPSW	- allowed to administer one-time password functionality.
# PSW		- allowed to administer passwords.

array set boks_subadm_checklist {
    UID range
    UNIXGROUPS regexp
    USERNAMES regexp
    HOSTGROUPS regexp
    USERCLASS regexp
    USER_USERCLASS regexp
    ACCESSROUTES accexp
    BACKUP yesno
    INTEGRITYVIEW yesno
    INTEGRITYADM yesno
    LOGVIEW yesno
    LOGADM yesno
    UNLOCKDISPLAY yesno
    PSD_USERS yesno
    PSD_ALL yesno
}

######################################
# This one I don't know where to put
#
# Remembers if user is a subadmin after
# boks_subadm_is_subadm user call.
# Simply boks_subadm_is_subadm returns
# value of this var.
# 
set boks_admin_is_subadmin 0


#####################################
# Directly callable routines
#####################################


###################
# Get and set subadmin info for a user or delete subadmin
#
#   boks_subadm -get username arr
#
#   boks_subadm -set username arr
#
#   boks_subadm -delete username
# or
#   boks_subadm -list
#
# Return:
# 1 - success (for get, arr is filled in with info)
# 0 - $what not recognized or error from lower level routine.
#
proc boks_subadm {what args} {
    if ![string compare $what "-set"] {
	return [uplevel 1 boks_subadm_set [lindex $args 0] [lindex $args 1]]
    } elseif ![string compare $what "-get"] {
	return [uplevel 1 boks_subadm_get [lindex $args 0] [lindex $args 1]]
    } elseif ![string compare $what "-delete"] {
	return [boks_subadm_delete $args]
    } elseif ![string compare $what "-list"] {
	return [boks_subadm_list]
    }
    return 0
}


###################
# Source a subadmin config file given username.
# All the expected routine names are removed first.
#
# Return:
# 1 - ok, file sourced (If some routines where missing, the
#     check routines used will discover this and return 0 for
#     all those checks)
# 0 - file did not exist, or failed to source it. Fatal error.
#
proc boks_subadm_load_cfg {user} {
    global boks_subadm_checklist
    foreach l [array names boks_subadm_checklist] {
	@ rename boks_subadm_check_$l {}
    }
    set file [boksdir var]/subadm/$user.cfg
    if [catch {boks_priv {exec /bin/test -f $file}}] {
	return 0
    }
    @ boks_priv {source $file}
    if ${?} {
	return 0
    }
    return 1
}


###################
# Check if user is subadmin
# If called with a username it checks for the existance of
# the file $BOKS_var/subadm/$user.cfg, sets a global var
# accordingly and returns result.
# If called without a username, it returns value of the
# global var.
#
# The idea is that this routine is called with the username
# at startup, and without a name after that (speedup).
#
proc boks_subadm_is_subadm {{user ""}} {
    global boks_admin_is_subadmin
    if [string length $user] {
	# bypass for root on master
	if ![string compare $user [boks_localhost]:root] {
	    set boks_admin_is_subadmin 0
	    return 0
	}
	if ![catch {boks_priv {exec /bin/test -f [boksdir var]/subadm/$user.cfg}}] {
	    set boks_admin_is_subadmin 1
	} else {
	    set boks_admin_is_subadmin 0
	}
    }
    return $boks_admin_is_subadmin
}

#######################
# return mod timestamp for user's config file
# if file does not exist, return empty string
#
proc boks_subadm_get_timestamp { user } {
    @ boks_priv {file mtime [boksdir var]/subadm/$user.cfg}
    if ${?} {
	return ""
    }
    return ${@}
}

###################
# Setup subadm data for user. May cache if it is the same user
# and file has not changed.
# This is called as a request hook, and should return 0 if ok,
# !0 if not.
# caching removed, always do everything.
#
set boks_subadm_setup_request(user) ""
set boks_subadm_setup_request(timestamp) ""

proc boks_subadm_setup_request {arrayname} {
    global boks_subadm_setup_request
    upvar $arrayname data

    set user $data(user)
    boks_subadm_is_subadm $user
    boks_subadm_load_cfg $user
    set boks_subadm_setup_request(user) $user
    boks_subadm_set_if_state
    return 0
}

##################
# Setup array to be used in html pages in @IF statements
# This is to show/hide PSD admin for now.
#
# It sets up IF vars STATE_PSD_USERS, STATE_PSD_ALL,
# STATE_PSD_ALL_OR_USERS, STATE_IS_SUBADMIN and STATE_DSF_INSTALLED
# and is a bit hardcoded...
# (STATE_PSD_ALL_OR_USERS is set up because there is no or or @ELSE)
#

proc boks_subadm_set_if_state {} {
    global boks_subadm_state_array boksenv
    if [boks_subadm_is_subadm] {
	set u [boks_subadm_check PSD_USERS -]
	set a [boks_subadm_check PSD_ALL -]
	set boks_subadm_state_array(STATE_PSD_USERS) $u
	set boks_subadm_state_array(STATE_PSD_ALL) $a
	set boks_subadm_state_array(STATE_PSD_ALL_OR_USERS) [expr $u || $a]
	set boks_subadm_state_array(STATE_IS_SUBADMIN) 1
    } else {
	set boks_subadm_state_array(STATE_PSD_USERS) 1
	set boks_subadm_state_array(STATE_PSD_ALL) 1
	set boks_subadm_state_array(STATE_PSD_ALL_OR_USERS) 1
	set boks_subadm_state_array(STATE_IS_SUBADMIN) 0
    }
    # ok, so this one really belongs somewhere else...
    if ![info exists boks_subadm_state_array(STATE_DSF_INSTALLED)] {
	if [string length [array names boksenv *_DSF_*]] {
	    set boks_subadm_state_array(STATE_DSF_INSTALLED) 1
	} else {
	    set boks_subadm_state_array(STATE_DSF_INSTALLED) 0
	}
    }
    SET_IF_ARRAY boks_subadm_state_array state_
}

proc boks_psd_adm_setup_request {arrayname} {
    boks_psd_adm_set_if_state
    return 0
}

##################
# Setup array to be used in html pages in @IF statements
# This is to show/hide PSD admin on HPSSO for now.
#
# It sets up IF var STATE_HP_PSD_ADMIN
#
# Note: If no "psdadmin" symbol is defined, set to "1" so the menu
# item will be visible (this is not a safety check, after all)!
#
proc boks_psd_adm_set_if_state {} {
    global boks_subadm_state_array boksenv
    boks_keypkg_env -get o
    if [info exists o(psdadmin)] {
	set boks_subadm_state_array(STATE_HP_PSD_ADMIN) $o(psdadmin)
    } else {
	set boks_subadm_state_array(STATE_HP_PSD_ADMIN) 1
    }
    SET_IF_ARRAY boks_subadm_state_array state_
}

proc boks_subadm_delete_routes {list del} {
    foreach l $del {
	set i [lsearch $list $l]
	if { $i != -1 } {
	    set list [concat [lrange $list 0 [expr $i - 1]] [lrange $list [expr $i + 1] end]]
	}
    }
    return $list
}


###################
# Routine to call to check if subadmin action is allowed.
#
# Example:   boks_subadm_check UID 100
#
# For all but ACCESSROUTE it returns 1 if allowed, 0 otherwise
# For ACCESSROUTE it returns empty list if allowed, list of
# not allowed access routes otherwise (list for complex ones)
#
# Note: I added a dirty fix:  If ret is "" return 0.  I just couldn't
# make the routines write proper check-procs to the sub admin's file.
# /PF
#
proc boks_subadm_check {what val} {
    if ![boks_subadm_is_subadm] {
	return 1
    }
    if ![llength [info procs boks_subadm_check_$what]] {
	return 0
    }
    if ![string compare $what ACCESSROUTE] {
	return [boks_subadm_check_accessroute $val]
    }
    # bugfix. Empty userclass is stored as "-".
    # if checking with value from DB, val will be empty
    # wich does not match ^-$. Hardcoded extra check here
    # makes it backward compatible.
    # Alternative is to expand "" to "-" before calling test routines
    # (done in 4.3 in a few places)
    #
    if {![string compare $what USER_USERCLASS] && ![string length $val]} {
	set list [boks_subadm_check_$what $val getlist]
	if {[lsearch -exact $list "-"] != -1} {
	    return 1
	} else {
	    return 0
	}
    }
    set ret [boks_subadm_check_$what $val]
    if [string match "" $ret] { set ret 0 }
    return $ret
}

###############################################
# Higher(?) level test routine
#
# args are paired what val, e.g.:
#   boks_subadm_checkargs UID $uid UNIXGROUPS $group
#
# Return list of disallowed whats.

proc boks_subadm_checkargs {args} {
    if ![boks_subadm_is_subadm] {
	return {}
    }
    set ret {}
    foreach {what val} $args {
	if ![boks_subadm_check $what $val] {
	    lappend ret $what
	}
    }
    return $ret
}

###################
# Function to check gui format range for validity
# (syntax, leagal numbers and a valid start and end values for range)
#
# Returns illegal elements in gui fromat, but not always in the order
# they were entered.
# Empty list out means range was OK.
#
# Note! Empty list in gives empty list out.
#
proc boks_sanitycheck_range {range} {
    set isnumber {^-?[0-9]+$}
    set err {}
    foreach l [boks_subadm_guirange2range $range err] {
	if {[llength $l] == 2} {
	    set n1 [lindex $l 0]
	    set n2 [lindex $l 1]
	    if ![regexp $isnumber $n1] {
		lappend err [boks_subadm_range2guirange [list $l]]
	    } elseif ![regexp $isnumber $n2] {
		lappend err [boks_subadm_range2guirange [list $l]]
	    } elseif {$n1 >= $n2} {
		lappend err [boks_subadm_range2guirange [list $l]]
	    }
	} else {
	    if ![regexp $isnumber $l] {
		lappend err [boks_subadm_range2guirange [list $l]]
	    }
	}
    }
    return [join $err ,]
}

###################
# Access route matching route matching routine called by
# access route test routine in subadmin .cfg files.
# As regexp, but makes method part into uppercase before regexp test
#
# Returns result of regexp
#
proc accexp {expr route} {
    set expr [boks_UC_accessmethod $expr]
    set route [boks_UC_accessmethod $route]
    return [regexp -- "^$expr\$" $route]
}


#######################################
# End .cfg routines
#######################################


##################################
# Help functions below
##################################


###################
# Help for boks_subadm -list
#
# Get names of all .cfg files in  $BOKS_var/subadm
# Strip .cfg from name, sort list and return result
#
proc boks_subadm_list {} {
    set res {}
    foreach f [glob -nocomplain [boksdir var]/subadm/*.cfg] {
	lappend res [file rootname [file tail $f]]
    }
    return [lsort $res]
}

###################
# Help function for boks_subamd -get ...
#
# Given user and array name, return a filled in array with
# info on what checks are done for user
#
# Return:
# 1 - success and array filled in.
#     if any routines where missing, the corresponding array
#     element is empty
#     If file did not exist, all elements of array are empty.
# 0 - File existed but failed ot open it, file was opened,
#     but there was no #SUBADMIN on first line (customized)
#     or failed to source file. In these cases the array will be unset.
#
proc boks_subadm_get {user arr} {
    global boks_subadm_checklist
    upvar 1 $arr o

    set checklist [array names boks_subadm_checklist]
    set file [boksdir var]/subadm/$user.cfg
    @ unset o
    if [catch {boks_priv {exec /bin/test -f $file}}] {
	foreach l $checklist {
	    if [string compare $boks_subadm_checklist($l) yesno] {
		set o($l) {}
	    } else {
		set o($l) 0
	    }
	}
	return 1
    }
    @ boks_priv {open $file r}
    if ${?} {
	return 0
    }
    gets ${@} line
    close ${@}
    if [string compare "#SUBADMIN" $line] {
	return 0
    }
    foreach l [array names boks_subadm_checklist] {
	@ rename boks_subadm_check_$l {}
    }
    @ boks_priv {source $file}
    if ${?} {
	return 0
    }
    foreach {name type} [array get boks_subadm_checklist] {
	@ boks_subadm_check_$name dummy getlist
	if !${?} {
	    set o($name) ${@}
	} else {
	    if [string compare yesno $type] {
		set o($name) {}
	    } else {
		set o($name) 0
	    }
	}
    }
    return 1
}

###################
# Help function for boks_subadm -set ...
#
# Write out a subadmin config file for user givven info in arr
# If elements are missing from array, they are assumed to be empty
#
# Return:
# 1 - success
# 0 - failed to open or write file.
#
proc boks_subadm_set {user arr} {
    global boks_subadm_checklist
 
    upvar 1 $arr o
    set ofile [boksdir var]/subadm/$user.cfg
    set nfile [boksdir var]/subadm/$user.cfg.tmp
    @ boks_priv {open $nfile w}
    if ${?} {
	return 0
    }
    boks_uadm_set_admin_status $user 1
    set fp ${@}
    puts $fp "#SUBADMIN"
    puts $fp "# Subadmin values created from the menus, don't touch if you are going"
    puts $fp "# to use the menu system for adminstration of this user."
    puts $fp "# If you change something in the functions below, please delete the first"
    puts $fp "# line so the menu system will notice this."
    puts $fp "#"
    foreach {name type} [array get boks_subadm_checklist] {
	set l {}
	if [info exists o($name)] {
	    set l $o($name)
	}
	puts $fp "proc boks_subadm_check_$name \{val \{getlist \"\"\}\} \{"
	puts $fp "    if \[string length \$getlist\] \{"
	puts $fp "	return {$l}"
	puts $fp "    \}"
	if ![string compare $type yesno] {
	    puts $fp "    return {$l}"
	    puts $fp "\}\n"
	    continue
        }
	if [string compare $type accexp] {
	    puts -nonewline $fp "    if \{ 0 "
	} else {
	    puts $fp "    set ret 0"
	}
	foreach elem $l {
	    if ![string compare $type "range"] {
		if {[llength $elem] == 1} {
		    puts -nonewline $fp "|| \$val == $elem "
		} elseif {[llength $elem] == 2} {
		    puts -nonewline $fp "|| (\$val >= [lindex $elem 0] && \$val <= [lindex $elem 1]) "
		}
	    } elseif ![string compare $type "regexp"] {
		if {[string match ^* $elem] || [string match *$ $elem]} {
		    puts -nonewline $fp "|| \[regexp -- {$elem} \$val\] "
		} else {
		    puts -nonewline $fp "|| \[regexp -- {^$elem\$} \$val\] "
		}
	    } elseif ![string compare $type "accexp"] {
		foreach e $elem {
		    if ![string compare [string index $e 0] "!"] {
			set e [string range $e 1 end]
			puts $fp "    if \[accexp {$e} \$val\] \{ return 0 \}"
		    } else {
			puts $fp "    if \[accexp {$e} \$val\] \{ set ret 1 \}"
		    }
		}
	    }
	}
	if [string compare $type accexp] {
	    puts $fp "\} \{"
	    puts $fp "	return 1"
	    puts $fp "    \}"
	    puts $fp "    return 0"
	} else {
	    puts $fp "    return \$ret"
	}
	puts $fp "\}\n"
    }
    close $fp
    boks_priv {exec mv -f $nfile $ofile}
    return 1
}

###################
# Help function for boks_subadm -delete ...
#
# Remove a subadmin user (as subadmin user that is, not
# as a BoKS user)
#
proc boks_subadm_delete {user} {
    set file [boksdir var]/subadm/$user.cfg

    boks_uadm_set_admin_status $user 0
    @ boks_priv {exec rm -f $file}
    boks_log -type S -label subadm_del "User %s now has been deleted as Sub-Administrator" $user
    return 1
}

###################
# Help function for boks_subadm_check ACCESSROUTE ...
#
# Returns empty list if access route allowed,
# list of the disallowed ones otherwise (list for complex ones,
# e.g. rlogin,telnet:...)
#
proc boks_subadm_check_accessroute {croute} {
    set res {}
    foreach r [boks_split_accessroute $croute] {
	if ![boks_subadm_check_ACCESSROUTE $r] {
	    lappend res $r
	}
    }
    return $res
}


###################
# Help function for boks_subadm_check_accessroute
#
# Split a complex accessroute (e.g. rlogin,telnet:*->*)
# into a list with each access route (rlogin:*->* telnet:*->*)
#
# Return list or original route
#
proc boks_split_accessroute {route} {
    set l [split $route :]
    if {[llength $l] > 1} {
	set m [split [lindex $l 0] ,]
	if {[llength $m] > 1} {
	    set rest [lrange $l 1 end]
	    set res {}
	    foreach l $m {
		lappend res $l:[join $rest :]
	    }
	    return $res
	}
    }
    return $route
}

###################
# Help function for accexp
#
# Return the access route with everything before first ':'
# in upper case (the method part)
#
proc boks_UC_accessmethod {route} {
    set l [split $route :]
    if {[llength $l] > 1} {
	return [string toupper [lindex $l 0]]:[join [lrange $l 1 end] :]
    }
    return $route
}


#######################################
# End routines for subadministration
#######################################
