###########################################################################
#
# Creating PSD's
#
###########################################################################

proc boks_keypkg_create_user_creds { ca_dname ca_pin user country \
	organization organization_unit common_name serial uuid \
	admin avilogin keys private_symkey keylen validity pin boksuser \
	pcprot_dn pcprot_pin compat} {
    global env
    global BOKSTAB

    if [boks_keypkg_is_revoked_dname $ca_dname] {
	error psd_ca_revoked
    }
    if {$admin == 0} {
	set type USER
    } else {
	set type ADMIN
    }

    @ boks_keypkg_check_dname $country $organization $organization_unit \
	    $common_name $serial $type
    if ${?} {
	boks_keypkg_error ${@}
    }

    if {$compat} {
	debug 5 "Doing compat DN"
	set user_dn [boks_keypkg_set_dn $country $organization \
		$organization_unit $common_name $serial $type]
    } else {
	debug 5 "Doing new DN"
	set user_dn [boks_keypkg_set_dn $country $organization \
		$organization_unit $common_name {} {}]
    }

    if [string match "" $ca_dname] {
	error psd_no_ca_dname_given
    }

    if [string match "" $pin] {
	error psd_no_pin_given
    }

    if [string match "" $ca_pin] {
	error psd_no_ca_pin_given
    }

    debug 5 "Checking uuid"
    if $compat {
	if [string match "" $serial] {
	    error psd_no_serial_given
	}
	# disallow spaces
	if [string match "* *" $serial] {
	    error psd_illegal_serial
	}
    } else {
	if [string match "" $uuid] {
	    error psd_no_uuid_given
	}
	# disallow spaces
	if [string match "* *" $uuid] {
	    error psd_illegal_uuid
	}
    }
    #
    # Pass user and ca's password in environment for better safety
    #
    set env(DSC_USER_PASSWORD) $pin
    set env(DSC_CA_PASSWORD)   $ca_pin
    debug 5 "After env setting"
    #
    # Create user's key package
    #
    set keypkg_file [creddir USER keypkgs]/$user.kpg

    debug 5 "boks_exec keypkgadm -C -u $user -n $user_dn -a [PROT_ALGID] \
	    -e $keypkg_file -l $keylen -m BOKS -T $type"
    @ boks_exec keypkgadm -C -u $user -n $user_dn -a [PROT_ALGID] \
	    -e $keypkg_file -l $keylen -m BOKS -T $type
    if ${?} {
	error [boks_keypkg_CLI_code2label ${@}]
    }
    #
    # Insert CA's key as trusted
    #
    set ca_cert [creddir CA certs]/$ca_dname.crt
    debug 5 "boks_exec keypkgadm -A -u $user -t $ca_cert -m BOKS -T $type"
    @ boks_exec keypkgadm -A -u $user -t $ca_cert -m BOKS -T $type
    if ${?} {
	boks_keypkg_keypkgdel $type $user 1
	error [boks_keypkg_CLI_code2label ${@}]
    }
    #
    # Create certificate extensions, if any
    #
    debug 5 "Before match"
    if ![string match "" $uuid] {
	set sans(rfc822name) $uuid
	set subjAltNamesExt [cert_encode -type SubjAltNames extension sans]
	lappend args -c $subjAltNamesExt
    }

    set abi(aviBoKSRole) $type
    if {$avilogin != ""} {
	set abi(loginInfo) $avilogin
    }
    set aviBoKSExt [cert_encode -type AviBoKSInfo extension abi]
    lappend args -c $aviBoKSExt
    #
    # Create a certificate request
    #
    set certreq_file [creddir USER]/$user.crq
    lappend args -R -u $user -o $certreq_file
    debug 5 "boks_exec keypkgadm $args"
    eval @ boks_exec keypkgadm $args -m BOKS -T $type
    if ${?} {
	boks_keypkg_keypkgdel $type $user 1
	boks_priv {exec /bin/rm -f $certreq_file}
	error [boks_keypkg_CLI_code2label ${@}]
    }

    #
    # Create user's certificate
    #
    set cert_file [creddir USER certs]/${user_dn}.crt
    debug 5 "boks_exec dscertadm -C -u $user -g $ca_dname -t $validity \
	     -i $certreq_file -o $cert_file -m BOKS -T $type"
    @ boks_exec dscertadm -C -u $user -g $ca_dname -t $validity \
	     -i $certreq_file -o $cert_file -m BOKS -T $type
    if ${?} {
	boks_keypkg_keypkgdel $type $user 1
	boks_priv {exec /bin/rm -f $certreq_file}
	error [boks_keypkg_CLI_code2label ${@}]
    }
    @ boks_priv {exec /bin/rm -f $certreq_file}

    #
    # Add dsc-things
    #
    unset args
    lappend args -C -u $user -a 4 -c $user -m BOKS

    if {$keys != ""} {
	lappend args -k $keys
    }

    if !$private_symkey {
	lappend args "-o"
    }

    if {$keys != "" || $private_symkey} {
	if {$pcprot_dn != ""} {
	    set env(DSC_PCPROT_PASSWORD) $pcprot_pin
	    set pcprot_keypkg [creddir USER keypkgs]/$pcprot_dn.kpg
	    lappend args "-i" $pcprot_dn "-y" $pcprot_keypkg
	}
    }

    boks_keypkg_env -get v

    #
    # Check if user's pin is going to be inserted, and if so, determine 
    # by who.
    #
    if {[info exists v(pin_encryption)]} {
	if [string match "on" $v(pin_encryption)] {

	    #
	    # Find out PE and dump PE's cert to tmp file
	    #
    
	    # if not default ca, try to find pe associated with the ca
	    if {![info exists v(ca_sign)] || [string compare $ca_dname $v(ca_sign)]} {
		set pe [boks_keypkg_ca2special $ca_dname pe]
		if ![string length $pe] {
		    set pe $ca_dname
		}
	    } else {
		if {[info exists v(pe)] && [string length $v(pe)]} {
		    set pe $v(pe)
		} else {
		    set pe $ca_dname
		}
	    }

	    #
	    # Finally, add the argument
	    #
	    lappend args -b $pe
	}
    }

    debug 5 "dscadm $args"
    eval @ boks_exec dscadm $args
    if ${?} {
	debug 5 "Message:${@}, status:${?}"
	boks_keypkg_keypkgdel $type $user 1
	boks_priv {exec /bin/rm -f $cert_file}
	error [boks_keypkg_CLI_code2label ${@}]
    }

    boks_log -type S -label psd_log_add_psd "Created PSD for %s" $user

    #
    # Log in cred_list
    #
    #boks_keypkg_log -add [creddir USER]/cred_list $user $user_dn
    #
    # Log passwd in passwords (Passwd file is configurable)
    #
    if {[info exists v(passwd_file)]} {
	if [string match "on" $v(passwd_file)] {
	    boks_keypkg_log -add [creddir USER]/passwords $user $pin
	}
    }   

    #
    # Install certificate ?
    #
    if [string length $boksuser] {
	if $compat {
	    set key $serial
	} else {
	    set key $uuid
	}
	@ callboks master write o TAB=$BOKSTAB(CERT2USERMAP) KEY=$key \
		"NEWFIELDS=USER" \
		+USER=$boksuser
	if ${?} {
	    debug 5 "Could not write to CERT2USERMAP, KEY=$uuid"
	} else {
	    boks_log -type S -label psd_log_install_cert \
		    "Installed certificate mapping %s for %s" \
		    $uuid $boksuser
	}
    }

    return $pin
}

# Given filename and username, import file into PSD
# table. Also extract certificate from PSD and insert into CERT.
#
proc boks_keypkg_import_psd {file user type errout} {
    set F "boks_keypkg_import_psd"

    debug 5 "$F $file $user $type"
    upvar $errout errmsg

    if {[string compare $type PIN]} {
	@ boks_read_tab PSD $user FIELDS=USER
	if {!${?} && [string length ${@}]} {
	    set errmsg "$type already existed in PSD table"
	    return 0
	}
    }

    @ boks_read_tab CERT $user FIELDS=USER
    if {!${?} && [string length ${@}]} {
	set errmsg "$type already existed in CERT table"
	return 0
    }

    @ boks_priv {file exists $file}
    if ${?} {
	set errmsg "File no longer exists"
	return 0
    }

    if {![string compare $type PIN]} {
	set ofile [creddir USER keypkgs]/$user.kpg
	debug 9 "$F boks_exec keypgadm -I -u $user -e $file -o $ofile -T $type"
	@ boks_exec keypkgadm -I -u $user -e $file -o $ofile -T $type
    } else {
	debug 9 "$F boks_exec keypkgadm -I -m FILE -u $user -e $file -m BOKS \
		-o $user -T $type"
	@ boks_exec keypkgadm -I -m FILE -u $user -e $file -m BOKS -o $user \
		-T $type
    }

    if ${?} {
	debug 5 "Error:${@}"
	set errmsg ${@}
	return 0
    }

    if {![string compare $type PIN]} {
	@ boks_priv {boks_keypkg_read_file $ofile}
	debug 5 "Read psd:${@}"
    } else {
	@ boks_read_tab PSD $user FIELDS=PSD
    }
    if ${?} {
	boks_keypkg_keypkgdel $type $user 1
	set errmsg "Failed to create PSD"
	return 0
    }

    set psd ${@}
    # convert to hex
    set psd [b642hex $psd]

    @ psd_cert $psd
    if ${?} {	
	set errmsg "Failed to decode PSD (${@})"
	debug 5 "$errmsg"
	return 0
    }

    set cert [hex2b64 ${@}]

    # here is where we should determine type given certificate
    # and rewrite type in PSD table. I'll do it when I figure
    # out how.

    # dump cert to file and let dscertadm handle the rest
    set tmpfile [creddir USER]/cert.[pid]
    @ boks_priv {open $tmpfile w}
    if ${?} {
	boks_keypkg_keypkgdel $type $user 1
	set errmsg "Failed to create temp cert file"
	return 0
    }
    set fp ${@}
    puts $fp "$cert"
    close $fp

    debug 9 "$F boks_exec dscertadm -I -m FILE -i $tmpfile -m BOKS -o $user -T $type"
    @ boks_exec dscertadm -I -m FILE -i $tmpfile -m BOKS -o $user -T $type
    if ${?} {
	boks_keypkg_keypkgdel $type $user 1
	set errmsg ${@}
	@ boks_priv {/bin/rm -f $tmpfile}
	return 0
    }
    @ boks_priv {/bin/rm -f $tmpfile}
    return 1
}

proc boks_keypkg_create_spec_creds { ca_dname ca_pin country org org_unit \
	cn id pin keylen validity type install compat} {
    global env
    global BOKSTAB
    
    if [string compare $type CA] {
        set cred_dir [creddir USER]
	set install 0
    } else {
        set cred_dir [creddir CA]
    }

    @ boks_keypkg_check_dname $country $org $org_unit $cn {} $type
    if ${?} {
	boks_keypkg_error ${@}
    }

    if {[string compare $type "CA"] || $compat} {
	set dname [boks_keypkg_set_dn $country $org $org_unit $cn {} $type]
    } else {
	set dname [boks_keypkg_set_dn $country $org $org_unit $cn {} {}]
    }
    if [string match "" $ca_dname] {
	error psd_no_ca_dname_given
    }

    if {[string compare $type CA] && [string match "" $ca_pin]} {
	error psd_no_ca_pin_given
    }

    if [string match "" $pin] {
	if {$type == "CA"} {
	    error psd_no_pin_given
	} elseif {$type == "PIN"} {
	    error psd_no_pe_pin_given
	} else {
	    error psd_no_pcprot_pin_given
	}
    }

    if ![string compare $ca_dname SELF-SIGNED] {
	if [string compare $type CA] {
	    error psd_no_self_signed
	} else {
	    #
	    # Establish who really is ca
	    #
	    set ca_dname $dname
	    set ca_pin $pin
	}
    }
    if [boks_keypkg_is_revoked_dname $ca_dname] {
	error psd_ca_revoked
    }

    set keypkg_file $cred_dir/keypkgs/$dname.kpg
    #
    # Watch out for incidentally overwriting...
    #
    # Maybe FIXME, but these are on file anyway...
    if ![catch {boks_priv {file_exists $keypkg_file}}] {
	error psd_entity_already_has_keypkg
    }

    #
    # Pass user and ca's password in environment for better safety
    #
    set env(DSC_USER_PASSWORD) $pin
    set env(DSC_CA_PASSWORD)   $ca_pin

    debug 5 "boks_exec keypkgadm -C -u $dname -n $dname \
	    -e $keypkg_file -a [PROT_ALGID] \
             -l $keylen -d $cred_dir -m BOKS -T $type"
    @ boks_exec keypkgadm -C -u $dname -n $dname \
	    -e $keypkg_file -a [PROT_ALGID] \
             -l $keylen -d $cred_dir -m BOKS -T $type
    if ${?} {
	error [boks_keypkg_CLI_code2label ${@}]
    }

    #
    # Create certificate extensions
    #
    if {$id != ""} {
	set sans(rfc822name) $id
	set subjAltNamesExt [cert_encode -type SubjAltNames extension sans]
	lappend args -c $subjAltNamesExt
    }

    set abi(aviBoKSRole) $type
    set aviBoKSExt [cert_encode -type AviBoKSInfo extension abi]
    lappend args -c $aviBoKSExt

    #
    # Create a certificate request
    #
    lappend args -R -u $dname -e $keypkg_file
    set certreq_file $cred_dir/$dname.crq
    lappend args -o $certreq_file

    debug 5 "boks_exec keypkgadm $args -m BOKS -T $type"
    eval @ boks_exec keypkgadm $args -m BOKS -T $type
    if ${?} {
	boks_keypkg_keypkgdel $type $dname 1
	error [boks_keypkg_CLI_code2label ${@}]
    }

    #
    # Create certificate (Possibly self-signed...)
    #
    set cert_file $cred_dir/certs/$dname.crt

    unset args
    lappend args -C -u $dname -g $ca_dname -d $cred_dir -t $validity \
	    -i $certreq_file -o $cert_file

    if {$type == "CA" && !$compat} {
	lappend args -a
    }

    debug 5 "dscertadm $args -m BOKS -T $type"
	     
    eval @ boks_exec dscertadm $args -m BOKS -T $type
    if ${?} {
	debug 5 "Couldnt create the CA cert"
	boks_keypkg_keypkgdel $type $dname 1
	boks_priv {exec /bin/rm -f $certreq_file}
	error [boks_keypkg_CLI_code2label ${@}]
    }

    @ boks_priv {exec /bin/rm -f $certreq_file}

    boks_log -type S -label psd_log_add_psd "Created PSD for %s" $dname

    #
    # Install?
    #
    if {$install == 1} {
	boks_keypkg_select_ca $dname
    }

    #
    # Clean up
    #
#    @ boks_priv {exec /bin/rm -f $cred_dir/$dname.AVI}

    return 1
}

proc boks_keypkg_create_host_creds { ca_dname ca_pin ip country org orgunit \
	dnsname id pin keylen validity } {
    # Ex:ip = 172.5.16.10
    # Ex:dnsname=spirit.dynas.se

    global env
    global BOKSTAB

    set type HOST

    #
    # Check if compat mode = on
    #
    boks_keypkg_env -get v

    if {[info exists v(compat)] && $v(compat) == 1} {
      set host_dn [boks_keypkg_set_dn $country $org $orgunit $dnsname {} $type]
    } else {
	set host_dn [boks_keypkg_set_dn $country $org $orgunit $dnsname {} {}]
    }

    @ boks_keypkg_check_dname $country $org $orgunit $dnsname {} $type
    if ${?} {
	boks_keypkg_error ${@}
    }

    if [string match "" $ca_dname] {
	error psd_no_ca_dname_given
    }
    if [boks_keypkg_is_revoked_dname $ca_dname] {
	error psd_ca_revoked
    }

    if [string match "" $ca_pin] {
	error psd_no_ca_pin_given
    }

    if [string match "" $pin] {
	error psd_no_pin_given
    }

    #
    # Watch out for incidentally overwriting...
    #
    @ boks_read_tab PSD $ip FIELDS="TYPE"
    if {!${?} && [string length ${@}]} {
	error psd_host_already_has_keypkg
    }

    #
    # Pass user (host) and ca's password in environment for better safety
    #
    set env(DSC_USER_PASSWORD) $pin
    set env(DSC_CA_PASSWORD)   $ca_pin
    
    debug 5 "boks_exec keypkgadm -C -u $ip -n $host_dn -a [PROT_ALGID] \
	    -l $keylen -m BOKS -T $type"
    @ boks_exec keypkgadm -C -u $ip -n $host_dn -a [PROT_ALGID] \
	-l $keylen -m BOKS -T $type
    if ${?} {
	error [boks_keypkg_CLI_code2label ${@}]
    }
    
    #
    # Insert CA's key as trusted
    #
    set ca_cert [creddir CA certs]/$ca_dname.crt
    debug 5 "boks_exec keypkgadm -A -u $ip -t $ca_cert -m BOKS -T $type"
    @ boks_exec keypkgadm -A -u $ip -t $ca_cert -m BOKS -T $type
    if ${?} {
	boks_keypkg_keypkgdel $type $ip 1
	error [boks_keypkg_CLI_code2label ${@}]
    }
    #
    # Create certificate extensions
    #
    set abi(aviBoKSRole) HOST
    set aviBoKSExt [cert_encode -type AviBoKSInfo extension abi]
    lappend args -c $aviBoKSExt

    set alt(dnsname) $dnsname
    set alt(ipaddress) $ip
    set altNamesExt [cert_encode -type SubjAltNames extension alt]
    lappend args -c $altNamesExt
    #
    # Create a certificate request
    #
    set certreq_file [creddir USER]/$ip.crq
    lappend args -R -u $ip -d [creddir USER] -o $certreq_file
    debug 5 "keypkgadm $args -m BOKS -T $type"
    eval @ boks_exec keypkgadm $args -m BOKS -T $type
    if ${?} {
	boks_keypkg_keypkgdel $type $ip 1
	boks_priv {exec /bin/rm -f $certreq_file}
	error [boks_keypkg_CLI_code2label ${@}]
    }
    #
    # Create host's certificate
    #
    set cert_file [creddir USER certs]/${host_dn}.crt
    debug 5 "boks_exec dscertadm -C -u $ip -g $ca_dname -t $validity \
	    -i $certreq_file -o $cert_file -m BOKS -T $type"
    @ boks_exec dscertadm -C -u $ip -g $ca_dname -t $validity \
	    -i $certreq_file -o $cert_file -m BOKS -T $type
    if ${?} {
	boks_keypkg_keypkgdel $type $ip 1
	boks_priv {exec /bin/rm -f $certreq_file}
	error [boks_keypkg_CLI_code2label ${@}]
    }
	
    @ boks_priv {exec /bin/rm -f $certreq_file}

    #
    # Add dsc-things
    #
    debug 5 "boks_exec dscadm -C -u $ip -a 4 -o -c $ip -m BOKS"
    @ boks_exec dscadm -C -u $ip -a 4 -o -c $ip -m BOKS
    if ${?} {
	boks_keypkg_keypkgdel $type $ip 1
	boks_priv {exec /bin/rm -f $cert_file}
	error [boks_keypkg_CLI_code2label ${@}]
    }

    boks_log -type S -label psd_log_add_psd "Created PSD for %s" $ip


    #
    # Log in cred_list
    #
    # 
    # boks_keypkg_log -add [creddir USER]/cred_list $ip $host_dn

    return 1
}

###########################################################################
#
# Deleting PSD's
#
###########################################################################

# Delete the given user or hosts keypackage, also revoke the certificate
# Hosts should be given as ip address in dot-notation
#
# Type is HOST|USER|CA|PE|PCPROT
#
# user: remove user_creds/certs/<dname>.crt user_creds/keypkgs/<x>.crt
# host: remove user_creds/certs/<dname>.crt user_creds/keypkgs/<x>.crt
#	and BOKS_etc/keys/host.kpg (if same)
# ca: remove ca_creds/cert/<x>.crt ca_creds/keypkgs/<x>.kpg
# pe|pcprot: remove user_creds/cert/<x>.crt user_creds/keypkgs/<x>.kpg
#
proc boks_keypkg_keypkgdel {type x {noerror 0}} {

    global BOKSTAB

    set doerror 1
    if $noerror {
	set doerror 0
    }
    if [string match "" $x] {
	boks_keypkg_error "Nothing to delete!"
    }

    set kpgfile [creddir USER keypkgs]/$x.kpg
    set certfile [creddir USER certs]/$x.crt
    set cdir [creddir USER]
    if [string match $type CA] {
	set kpgfile [creddir CA keypkgs]/$x.kpg
	set certfile [creddir CA certs]/$x.crt
	set cdir [creddir CA]
    }
    

    @ boks_exec dscertadm -D -u $x -i $certfile -d $cdir -m BOKS -T $type

    if ${?} {
	if $doerror {
	    boks_keypkg_error "Could not delete cert:${@}"
	}
    }

    debug 5 "Will delete $x from PSD-tab"
    debug 5 "boks_exec keypkgadm -D -u $x -e $kpgfile -d $cdir -m BOKS -T $type"
    @ boks_exec keypkgadm -D -u $x -e $kpgfile -d $cdir -m BOKS -T $type
    # Don't treat fail as fatal any more. If you import a CA
    # you will normally only get a CERT, not a PSD entry.
#    if ${?} {
#	boks_keypkg_error "Could not delete psd:${@}"
#    }

    #
    # If CA, PE or PCPROT, and set in environment, remove this association
    #
    if {$type == "CA" || $type == "PCPROT" || $type == "PIN"} {
	boks_keypkg_env_del $type $x
    }

    boks_log -type S -label psd_log_del_psd "Removed PSD for %s" $x    

    return 1
}

