#
# $Id: icheck.tcl,v 1.1.6.1 1998/09/10 13:21:01 cvsusr Exp $
# Copyright  1996 Dynamic Software AB
#
# Support routines used by BoKS Integrity Check Administration.
# (Many of these belong in another file - move them later)

# TRUE if there exists a lock for `pathname'
# (should preferably check for aliveness too, but since this is
#  a task for clntd on the client I leave it for now)

#
proc boks_lock_test {pathname {host ""}} {
    set short [file tail $pathname]
    @ call_clntd read o $host 0 FILE=\$BOKS_var/LCK/$short
    if {!${?} && [info exists o(DATA)] && [string length $o(DATA)]} {
	return 1
    }
    debug 1 "Lockfile \$BOKS_var/LCK/$short does not exist on $host"
    return 0
}

# Return specified BoKS Integrity Check directory name.
#
proc boks_icheck_dir {name} {
    switch -- $name {
	data {
	    return [boksdir var]/bic
	}
	bin {
	    return [boksdir lib]
	}
    }
    return [boksdir $name]/bic
}

# Map name,<id> -> <cron-spec>
#     cron,<cron-spec> -> <id>

array set ICHECK_SCHEDS {
    name,monthly	{1 * *}
    name,weekly		{* * 0}
    name,biweekly	{* * 0,4}
    {cron,1 * *}	monthly
    {cron,* * 0}	weekly
    {cron,* * 0,4}	biweekly
}

array set ICHECK_DEFAULTS {
    cronminute	0
}

# Return a list of all hosts and hostgroups (sorted)
# which have bic configured in DB. ALL is never returned
#
proc boks_icheck_getconflist {} {
    set ret {}
    foreach n [lunique [boks_read_tab COPSSETUP * FIELDS=NAME]] {
	if [string compare $n ALL] {
	    lappend ret $n
	}
    }
    return [lsort $ret]
}

# Deletion of crontab entry for host
#
proc boks_icheck_delete_crontab_host { host } {
    @ call_clntd read o $host 0 FILE=crontab
    set line {}
    foreach line [split $o(DATA) \n] {
        if [string match {*boks_icheck*} $line] {
            @ call_clntd delline o $host 1 FILE=crontab LINE=$line
        }
    }
}
 
# Delete any crontab entry for host or hosts. This procedure
# is deciding if it's a host or a hostgroup.
#
proc boks_icheck_delete_crontab_hosts { hosts } {
    global BOKSTAB
    set host {}
    @ callboks master read o TAB=$BOKSTAB(HOST) KEY=$hosts
    if {!${?} && [info exists o(DATA)]} {
        boks_icheck_delete_crontab_host $hosts
    } else {
        foreach host [@ callboks master read o TAB=$BOKSTAB(HOSTGROUP) KEY=$hosts] {
            boks_icheck_delete_crontab_host $host
        }
    }
}


# Delete the config data in DB for host(groups) in conflist
# Always ignore ALL
#
proc boks_icheck_delconf { conflist } {
    global BOKSTAB
    foreach n $conflist {
	if [string compare $n ALL] {
	    @ callboks master delete o TAB=$BOKSTAB(COPSSETUP) \
		    FIRST=0 KEY=$n
	    boks_icheck_delete_crontab_hosts $n
	}
    }
}

#
# Configure BoKS Integrity checks. Modifies crontab (or whatever)
# Return 0 if all OK.
#
# boks_icheck_set_schedule how_often ?hour? ?spec? 
#
proc boks_icheck_set_schedule {how_often {hour 0} {spec {}} {host ""}} {
    global ICHECK_SCHEDS
    global ICHECK_DEFAULTS

    set minute $ICHECK_DEFAULTS(cronminute)
    set crontime {}
    switch $how_often {
	off {
	    set crontime {}
	}
	custom {
	    if {[llength $spec] > 4} {
		set crontime "$spec"
	    } else {
		set crontime "$minute $hour $spec"
	    }
	}
	default {
	    if [info exists ICHECK_SCHEDS(name,$how_often)] {
		set crontime "$minute $hour $ICHECK_SCHEDS(name,$how_often)"
	    } else {
		debug 1 "boks_icheck_set_schedule: unkown arg: $how_often"
		return 1
	    }
	}
    }	

    debug 1 "minute: $minute\nhour: $hour\ncrontime: $crontime"
    @ call_clntd read o $host 0 FILE=crontab
    set line {}
    foreach line [split $o(DATA) \n] {
	if [string match {*boks_icheck*} $line] {
	    @ call_clntd delline o $host 1 FILE=crontab LINE=$line
	}
    }
    if [string length $crontime] {
	@ call_clntd addline o $host 1 FILE=crontab \
		"LINE=$crontime [boks_icheck_dir bin]/boks_icheck"
	if ${?} {
	    return 1
	}
    }
    return 0
}

#
# Get scheduled time for Integrity checks
# reads from crontab (or whatever)
# Returns a 2-3 el. list: 
#   {<type> <hour> ?<cronspec>?}
#
proc boks_icheck_get_schedule {{host ""}} {

    global ICHECK_SCHEDS

    @ call_clntd read o $host 0 FILE=crontab
    set cronline {}
    foreach line [split $o(DATA) \n] {
	if [string match {*boks_icheck*} $line] {
	    set cronline [string trim $line]
	    break
	}
    }
    if {[string length $cronline] == 0} {
	return {off 0}
    } else {
	# Heuristic time..

	foreach {min hour day month dow cmd} [split $cronline] { }
	set idx "$day $month $dow"
	if [info exists ICHECK_SCHEDS(cron,$idx)] {
	    debug 1 "found: cron,$idx in table"
	    return [list $ICHECK_SCHEDS(cron,$idx) $hour]
	}
	return [list custom $hour [list $min $hour $day $month $dow] ] 
    }
}

# Return a list of all known checks (from $BIC_etc/checks.conf)
#
proc boks_icheck_get_checknames {} {
    set checks {}
    boks_priv {@ open [boks_icheck_dir etc]/checks.conf r}
    if !${?} {
	set f ${@}
	while {[gets $f line] >= 0} {
	    if {[regexp {^([^ #	]+)} $line word]} {
		lappend checks $word
	    }
	}
	close $f
    }
    return $checks
}

# Return state of checks in list (1 on, 0 off or does not exist)
#
proc boks_icheck_getstate {list {host ""}} {
    if [string length $host] {
	@ callboks servc bic_func o WHAT=getconfig HOST=$host
    } else {
	@ callboks servc bic_func o WHAT=getconfig
    }
    if {!${?} && [info exists o(CONFIG)]} {
	foreach line [split $o(CONFIG) \n] {
	    if {[regexp {^([^ #	]+)[ 	]+([^ 	]+)} $line word test state]} {
		set mode($test) $state
	    }
	}
    }
    set l {}
    foreach t $list {
	if {[info exists mode($t)] && [regexp {^[Oo][Nn]$} $mode($t)]} {
	    lappend l 1
	} else {
	    lappend l 0
	}
    }
    return $l
}



# Set state of tests to DB
#
proc boks_icheck_setstate {tests states {host "ALL"}} {
    global BOKSTAB
    @ callboks servc bic_func o WHAT=getconfig HOST=$host
    @ callboks master delete dummy TAB=$BOKSTAB(COPSSETUP) KEY=$host
    set now [clock seconds]

    if ![info exists o(CONFIG)] {
	# no previos info, write it directly
	foreach test $tests state $states {
	    set s off
	    if $state {
		set s on
	    }
	    @ callboks master write dummy TAB=$BOKSTAB(COPSSETUP) NEW=1 KEY=$host \
		"NEWFIELDS=LINE DATE" "+LINE=$test $s" "+DATE=$now"
	}
	return
    }
	
    foreach line [split $o(CONFIG) \n"] {
	if [regexp {^([^# 	]+)[ 	]+([^ 	]+)(.*)} $line w test state cmd] {
	    set cmd [string trim $cmd]
	    foreach t $tests s $states {
		if ![string compare $t $test] {
		    if $s {
			set state on
		    } else {
			set state off
		    }
		    break
		}
	    }
	    @ callboks master write dummy TAB=$BOKSTAB(COPSSETUP) NEW=1 KEY=$host \
		"NEWFIELDS=LINE DATE" "+LINE=$test $state $cmd" "+DATE=$now"
	}
    }
}


# Start integrity check "manually" by calling clntd
#
proc boks_icheck_start_right_now {{host ""}} {
    @ call_clntd startprog o $host 1 PROG=boks_icheck.nowait
    if [info exists o(ERROR)] {
	debug 1 "(1)ERROR=$o(ERROR)"
    }
    debug 1 "?=${?}"
    if {${?} != 0} {
	return 0
    }
    if [info exists o(ERROR)] {
	debug 1 "(2)ERROR=$o(ERROR)"
	return [expr $o(ERROR) != 0]
    }
    return 0
}

# -----------------------------
# Reports & exclusion stuff below. Needs commenting, rewriting and debugging

proc boks_icheck_db_path {{name {}} {host {}}} {
#    global boks_localhost
    if [string length $host] {
	return "[boks_icheck_dir data]/$host/$name"
    }
    return "[boks_icheck_dir data]/$name"
}

# Split a report line into its components (as a list)
#
proc boks_icheck_split_report_record {line} {
    regsub -all {([\{\}\]\[])} $line {\\\1} line
    set s {}
    foreach x $line { lappend s $x }
    return $s
}

# Join a "splitted" report record into a string 
#
proc boks_icheck_join_report_record {record} {
    set res {}
    foreach part $record {
	if [regexp {[ 	\\"]} $part] {
	    regsub -all {(["\\])} $part {\\\1} part
	    append res {"} $part {" }
	} else {
	    append res $part { }
	}
    }
    return [string trimright $res]
}

proc boks_icheck_load_report {name var} {
    upvar $var v
	
    set v {}
    boks_priv {@ open $name r}
    if !${?} {
	set f ${@}
	while {[gets $f line] >= 0} {
	    if ![regexp {^[# 	]$} $line] {
		lappend v [boks_icheck_split_report_record $line]
	    }
	}
	close $f
	return 0
    }
    return 1
}

proc boks_icheck_get_excluded {var} {
    upvar $var v

    set v {}
    set exclude_db [boks_icheck_db_path "exclude.dat"]
    return [boks_icheck_load_report $exclude_db v]
}
	
# Format a message from supplied report record.
proc boks_icheck_get_message_text {record} {
     set fmt [lindex $record 6]
     set args [lrange $record 7 end]
     return [eval format {$fmt} $args]
#    set cmd [list format $fmt]
#    foreach arg [lrange $record 7 end] {
#	lappend cmd $arg
#    }
#    return [eval $cmd]
}

#
# Compare two report records. Returns 0 if equal
#
proc boks_icheck_compare_records {record1 record2} {
    foreach {el1 el2} [list $record1 $record2] {
	set cmp [string compare $el1 $el2]
	if $cmp {
	    return $cmp
	}
    }
    return 0
}

#
# Remove 'line_to_rm' from the report file 'report'
#
proc boks_icheck_remove_report_record {report line_to_rm} {
    set v {}
    boks_priv {@ open $report r}
    if !${?} {
	set f ${@}
	while {[gets $f line] >= 0} {
	    if ![regexp {^[# 	]$} $line] {
		set this_line [boks_icheck_split_report_record $line]
		if [boks_icheck_compare_records $this_line $line_to_rm] {
		    lappend v $line
		}
	    } else {
		lappend v $line
	    }
	}
	close $f
	# Rewrite the file sans the possibly removed line
	boks_priv {@ open $report w}
	if !${?} {
	    set f ${@}
	    foreach l $v {
		puts $f $l
	    }
	    close $f
	    return 0
	}
	return 1
    }
}

#
# Append the report record 'rec' to the report file 'report'
#
proc boks_icheck_add_report_record {report rec} {
    boks_priv {@ open $report "a+"}
    if !${?} {
	set f ${@}
	puts $f [boks_icheck_join_report_record $rec]
	close $f
	return 0
    }
    return 1
}


#
# Remove a single record from the "exclusion DB"
#
proc boks_icheck_remove_excluded {line_to_rm} {
    set exclude_db [boks_icheck_db_path "exclude.dat"]

    debug 1 $line_to_rm
    return [boks_icheck_remove_report_record $exclude_db $line_to_rm]
}

#
# Append a single record from the "exclusion DB"
#
proc boks_icheck_add_excluded {rec} {
    set exclude_db [boks_icheck_db_path "exclude.dat"]

    return [boks_icheck_add_report_record $exclude_db $rec]
}

# Init db with ALL entries for bic if nothing is
# in the table at all.
#
proc boks_icheck_maybe_init {} {
    global BOKSTAB
    @ callboks master read o TAB=$BOKSTAB(COPSSETUP) KEY=* FIELDS=NAME
    if {${?} || ![info exists o(DATA)] || ![string length $o(DATA)]} {
	set tests [boks_icheck_get_checknames]
	# turn on all by default
	set states {}
	foreach t $tests {
	    lappend states 1
	}
	boks_icheck_setstate $tests $states
    }
}

# I think this module is self contained enough to
# do init here.
#
boks_icheck_maybe_init
