/*
 * Copyright (c) %G%, by Sun Microsystems, Inc.
 * All Rights Reserved.
 */

#ifndef	_SKI_H
#define	_SKI_H

#pragma ident	"%Z%%M%	%I%	%E% SMI"

#ifdef	__cplusplus
extern "C" {
#endif

#include <sys/types.h>
#include "ski_asn.h"

extern ski_oid *SKI_MD2_OID;
extern ski_oid *SKI_MD4_OID;
extern ski_oid *SKI_MD5_OID;
extern ski_oid *SKI_SHA_1_OID;
extern ski_oid *SKI_RSA_ENCR_OID;
extern ski_oid *SKI_MD2WithRSAEncryption_OID;
extern ski_oid *SKI_MD5WithRSAEncryption_OID;
extern ski_oid *SKI_DES_ECB_OID;
extern ski_oid *SKI_DES_CBC_OID;
extern ski_oid *SKI_DES3_CBC_OID;
extern ski_oid *SKI_DES_MAC_OID;
extern ski_oid *SKI_RC2_CBC_OID;
extern ski_oid *SKI_RC4_OID;
extern ski_oid *SKI_PbeWithMD2AndDES_CBC_OID;
extern ski_oid *SKI_PbeWithMD5AndDES_CBC_OID;

/*
 * Misc. AlgIDs
 */
extern struct ski_algid *SKI_RSA_ENCR_AID;
extern struct ski_algid *SKI_MD2WithRSAEncryption_AID;
extern struct ski_algid *SKI_MD5WithRSAEncryption_AID;
extern struct ski_algid *SKI_MD2_AID;
extern struct ski_algid *SKI_MD4_AID;
extern struct ski_algid *SKI_MD5_AID;
extern struct ski_algid *SKI_RC4_AID;

/* Algorithm types */
typedef enum {
	SKI_OTHER_ALG = -1,
	SKI_SYM_ENC_ALG,
	SKI_ASYM_ENC_ALG,
	SKI_HASH_ALG,
	SKI_SIG_ALG,
	SKI_KEYED_INTEGRITY_ALG
} ski_alg_type;

/* Parameter types */
typedef enum {
	SKI_PARM_OTHER = -1,
	SKI_PARM_ABSENT,
	SKI_PARM_INTEGER,
	SKI_PARM_OCTETSTRING,
	SKI_PARM_NULL,
	SKI_PARM_RC2_CBC,
	SKI_PARM_PBE
} ski_parm_type;

/* Algorithm table */
#define	SKI_NO_EXPORT_KEYSIZE_LIMIT	0
typedef struct ski_alg_list {
	ski_oid	*oid;
	char		*name;
	ski_alg_type	algType;
	ski_parm_type	parmType;
	size_t		keysize_limit;
} ski_alg_list;

/*
 * SKI function return values
 */

#define	SKI_OK				0
#define	SKI_EBUFSIZE			1
#define	SKI_ENOMEM			2	/* ENOMEM MUST be 2 */
#define	SKI_BAD_FILE			3
#define	SKI_FILE_NOT_FOUND		4
#define	SKI_FILE_IO_ERR			5
#define	SKI_BAD_PASSWD			6
#define	SKI_UNKNOWN_USER		7
#define	SKI_ALGORITHM_UNKNOWN		8
#define	SKI_ASN1_ENCODE_ERR		9
#define	SKI_ASN1_DECODE_ERR		10
#define	SKI_BAD_KEY			11
#define	SKI_KEYGEN_ERR			12
#define	SKI_ENCRYPT_ERR			13
#define	SKI_DECRYPT_ERR			14
#define	SKI_SIGN_ERR			15
#define	SKI_VERIFY_ERR			16
#define	SKI_DIGEST_ERR			17
#define	SKI_OUTPUT_FORMAT_ERR		18
#define	SKI_SYSTEM_ERR			19	/* General Errors */
#define	SKI_ATTRIBUTE_UNKNOWN		20
#define	SKI_SKILOGIN_ERR		21
#define	SKI_SKILOGOUT_ERR		22
#define	SKI_NO_SUCH_ENTRY		23
#define	SKI_ENTRY_ALREADY_EXISTS	24
#define	SKI_SKISERV_DECRYPT_ERR		25
#define	SKI_SKISERV_SIGN_ERR		26
#define	SKI_USER_DID_NOT_SKILOGIN	27
#define	SKI_SKISERV_CONNECT		28
#define	SKI_KEYPKG_NOT_FOUND		29
#define	SKI_TIME_INVALID		30
#define	SKI_UNTRUSTED_PUBLIC_KEY	31
#define	SKI_EPARM			32	/* EPARM MUST be 32 */
#define	SKI_BINARY_TO_RFC1421_ERR	33
#define	SKI_RFC1421_TO_BINARY_ERR	34
#define	SKI_RANDOM_NUM_ERR		35
#define	SKI_XFN_ERR			36
#define	SKI_CERT_CHAIN_ERR		37
#define	SKI_RDN_MISSING_EQUAL		38
#define	SKI_AVA_TYPE_MISSING		39
#define	SKI_AVA_VALUE_MISSING		40
#define	SKI_CERT_NOT_FOUND		41
#define	SKI_DN_NOT_FOUND		42
#define	SKI_CRITICAL_EXTNS_ERR		43
#define	SKI_ASN1_INIT_ERROR		44
#define	SKI_WRAP_ERROR			45
#define	SKI_UNWRAP_ERROR		46
#define	SKI_UNSUPPORTED_KEY_TYPE	47
#define	SKI_DH_PART1_ERR		48
#define	SKI_DH_PART2_ERR		49
#define	SKI_DOUBLE_ENCRYPT		50
#define	SKI_SKISERV_KEYPKG_UPDATE	51
#define	SKI_SKISERV_STAT_ERR		52
#define	SKI_GLOBAL_ERR			53
#define	SKI_TRUSTED_KEY_EXPIRED		54
#define	SKI_OPEN_ERR		55
#define	SKI_TOTAL_ERRNUM		56

/* flags for ski_encrypt, ski_decrypt, ski_sign, ski_verify, ski_digest */
#define	SKI_ADD_DATA	1
#define	SKI_END_DATA	2
#define SKI_DIGESTED_DATA 3  /*used in ski_verify where the data passed is 
				the digest and not the raw data*/

/* SKI Handle and status */
typedef struct ski_handle ski_handle_t;

/* SKI return variable */
typedef int SKI_STATUS;

/*
 * Parameter
 */

typedef struct ski_rsa_keygen_param_t {
	u_int modulusBits;
	uchar_t *publicExponent; /* const */
	size_t publicExponentLen;
} ski_rsa_keygen_param;

typedef struct ski_des_keygen_param_t {
	uchar_t *saltVal; /* const */
	size_t saltLen;
	char *passwd; /* const */
	int iterationCount;
} ski_des_keygen_param;

/*
 * PROTOTYPES should be set to one if and only if the compiler supports
 * function argument prototyping.
 * The following makes PROTOTYPES default to 1 if it has not already been
 * defined as 0 with C compiler flags.
 */
#ifndef	PROTOTYPES
#define	PROTOTYPES	1
#endif

/*
 * PROTO_LIST is defined depending on how PROTOTYPES is defined above.
 * If using PROTOTYPES, then PROTO_LIST returns the list, otherwise it
 * returns an empty list.
 */

#if PROTOTYPES
#define	PROTO_LIST(list) list
#else
#define	PROTO_LIST(list) ()
#endif

/*
 * SKI prototypes
 */

/* Init and Terminate a SKI session */
SKI_STATUS ski_init PROTO_LIST((
	ski_handle_t **,
	const char *,
	const char *,
	const u_int,
	const u_int,
	const char *));

SKI_STATUS ski_end PROTO_LIST((
	ski_handle_t *));

char *ski_strerror PROTO_LIST((
	const ski_handle_t *,
	const SKI_STATUS));	/* errno */

/* Key generation */
SKI_STATUS ski_gen_des_key PROTO_LIST((
	const ski_handle_t *,	/* IN:	ski handle */
	uchar_t **,		/* OUT: DES session key */
	ski_alg_params **));	/* OUT: IV */

SKI_STATUS ski_gen_des3_key PROTO_LIST((
	const ski_handle_t *,	/* IN:	ski handle */
	uchar_t **,		/* OUT: triple DES session key */
	ski_alg_params **));	/* OUT: IV */

SKI_STATUS ski_gen_rc2_key PROTO_LIST((
	const ski_handle_t *,	/* IN:  SKI handle */
	const size_t,		/* IN:  key length */
	const u_int,		/* IN:  effective key size in bits */
	uchar_t **,		/* OUT: RC2 session key */
	ski_alg_params **));	/* OUT: RC2 parameter */

SKI_STATUS ski_gen_rc4_key PROTO_LIST((
	const ski_handle_t *,	/* IN:	ski handle */
	const size_t,		/* IN:  key length in bytes */
	uchar_t **));		/* OUT: RC4 key */

SKI_STATUS ski_gen_rsa_keypair PROTO_LIST((
	const ski_handle_t *,		/* IN:	ski handle */
	const ski_rsa_keygen_param *,	/* IN:  keypair generation parameters */
	uchar_t **,			/* OUT: public key */
	size_t *,			/* OUT: public key length */
	uchar_t **,			/* OUT: private key */
	size_t *));			/* OUT: private key length */

/* crypto */
SKI_STATUS ski_digest PROTO_LIST((
	ski_handle_t *,			/* IN:	ski handle */
	const uchar_t *,		/* IN:  input data  */
	const size_t,			/* IN:  length of data in bytes */
	const int,			/* IN:  more input data flag */
	const ski_algid *,		/* IN:  digest algorithm */
	uchar_t **,			/* OUT: digest */
	size_t *));			/* OUT: length of digest */
SKI_STATUS ski_sign PROTO_LIST((
	ski_handle_t *,			/* IN:	ski handle */
	const uchar_t *,		/* IN:  data to be signed */
	const size_t,			/* IN:  data length */
	const int,			/* IN:  more input data flag */
	const ski_algid *,		/* IN:  signature key algorithm */
	const uchar_t *,		/* IN:  signature key */
	const size_t,			/* IN:  signature key length */
	const ski_algid *,		/* IN:  signature algorithm */
	uchar_t **, 			/* OUT: signature */
	size_t *));			/* OUT: signature length */
SKI_STATUS ski_verify PROTO_LIST((
	ski_handle_t *,			/* IN: ski handle */
	const uchar_t *, 		/* IN: data to be verified */
	const size_t,			/* IN: data length */
	const int,			/* IN: more input data flag */
	const ski_algid *,		/* IN: verification key algorithm */
	const uchar_t *,		/* IN: verification key */
	const size_t,			/* IN: verification key length */
	const ski_algid *,		/* IN: verification algorithm */
	const uchar_t *, 		/* IN: signature */
	const size_t));			/* IN: signature length */
SKI_STATUS ski_encrypt PROTO_LIST((
	ski_handle_t *,			/* IN:	ski handle */
	const uchar_t *,		/* IN:  input data */
	const size_t,			/* IN:  input data length */
	const int,			/* IN:	more input data flag */
	const ski_algid *,		/* IN:  encryption key algorithm */
	const uchar_t *,		/* IN:  encryption key */
	const size_t,			/* IN:  encryption key length */
	const ski_algid *,		/* IN:  encryption algorithm */
	uchar_t **,			/* OUT: ciphertext */
	size_t *));			/* OUT: ciphertext length */
SKI_STATUS ski_decrypt PROTO_LIST((
	ski_handle_t *,			/* IN:	ski handle */
	const uchar_t *,		/* IN:  ciphertext */
	const size_t,			/* IN:  ciphertext length */
	const int,			/* IN:  more input data flag */
	const ski_algid *,		/* IN:  decryption key algorithm */
	const uchar_t *,		/* IN:  decryption key */
	const size_t,			/* IN:  decryption key length */
	const ski_algid *,		/* IN:  decryption algorithm */
	uchar_t **,			/* OUT: cleartext */
	size_t *));			/* OUT: cleartext length */
SKI_STATUS ski_wrap_key PROTO_LIST((
	const ski_handle_t *,		/* IN:  ski handle */
	const uchar_t *,		/* IN:	key to be wrapped  */
	const size_t,			/* IN:	length of key to be wrapped */
	const ski_algid *,		/* IN:	wrapping key algorithm */
	const uchar_t *,		/* IN:	wrapping key */
	const size_t,			/* IN:	wrapping key length */
	const ski_algid *,		/* IN:	wrapping algorithm */
	uchar_t **,			/* OUT: wrapped key */
	size_t *));			/* IN/OUT: wrapped key length */
SKI_STATUS ski_unwrap_key PROTO_LIST((
	const ski_handle_t *,		/* IN:  ski handle */
	const uchar_t *,		/* IN:  wrapped key */
	const size_t,			/* IN:  wrapped key length */
	const ski_algid *,		/* IN:  unwrapping key algorithm */
	const uchar_t *,		/* IN:  unwrapping key */
	const size_t,			/* IN:  unwrapping key length */
	const ski_algid *,		/* IN:  unwrapping algorithm */
	uchar_t **,			/* OUT: unwrapped key */
	size_t *));			/* OUT: unwrapped key length */

/* certificate verification */
SKI_STATUS ski_verify_cert PROTO_LIST((
	const ski_handle_t *,		/* IN: ski handle */
	const ski_cert *, 		/* IN: certificate to be verified */
	const ski_pubkey_info *,	/* IN: public verification key */
	const int));			/* IN: flags (unused) */
SKI_STATUS ski_verify_cert_chain PROTO_LIST((
	const ski_handle_t *,		/* IN: ski handle */
	const ski_cert *, 	/* IN: certificate chain to be verified */
	const int,			/* IN: length of cert chain */
	const struct ski_tkey_list *,	/* IN: trusted key list */
	const int,			/* IN: flags (unused) */
	ski_cert **));		/* OUT: first expired certificate */
SKI_STATUS ski_verify_cert_est_chain PROTO_LIST((
	const ski_handle_t *,		/* IN: ski handle */
	const ski_cert *, 		/* IN: certificate to be verified */
	const struct ski_tkey_list *,	/* IN: trusted key list */
	const char **,			/* IN: CA Name list */
	const int,			/* IN: flags (unused) */
	ski_cert **,			/* OUT: first expired certificate */
	ski_cert **,			/* OUT: certificate chain */
	int *));			/* OUT: length of cert chain */

/* certificate chain establishment */
SKI_STATUS ski_get_cert_chain PROTO_LIST((
	const ski_handle_t *,	/* IN: ski handle */
	const ski_cert *,	/* IN: user certificate */
	const char **,		/* IN: CA name list */
	int flags,		/* IN: flags (unused) */
	ski_cert **,		/* OUT: certificate chain */
	int *));		/* OUT: length of cert chain */

/* I/O */
SKI_STATUS ski_set_keypkg PROTO_LIST((
	const ski_handle_t *,	/* IN: ski handle */
	const char *,		/* IN: keypkg filename or repository index */
	const ski_keypkg *));	/* IN: keypkg to be stored */
SKI_STATUS ski_get_keypkg PROTO_LIST((
	const ski_handle_t *,	/* IN:	ski handle */
	const char *,		/* IN:  keypkg_filename or repository index */
	ski_keypkg **));		/* OUT: keypkg */
SKI_STATUS ski_set_cert PROTO_LIST((
	const ski_handle_t *,	/* IN: ski handle */
	const char *,		/* IN: cert filename or repository index */
	const ski_cert *));	/* IN: certificate */
SKI_STATUS ski_get_cert PROTO_LIST((
	const ski_handle_t *,	/* IN:	ski handle */
	const char *,		/* IN:  certificate filename, rep index, DN */
	ski_cert **,		/* OUT: set of certificates */
	int *));		/* OUT: certificate set length */

/* generate random bytes */
SKI_STATUS ski_random PROTO_LIST((
	const u_short,		/* IN:  requested number of random bytes */
	uchar_t **));		/* OUT: random byte buffer */


/* Free */
void ski_free_keypkg PROTO_LIST((ski_keypkg **));
void ski_free_cert PROTO_LIST((ski_cert **));
void ski_free_cert_list PROTO_LIST((ski_cert **, int));
void ski_free_dn PROTO_LIST((ski_name **));

/* DN */
SKI_STATUS ski_str2dn PROTO_LIST((
	const ski_handle_t *, char *, ski_name **));
SKI_STATUS ski_dn2str PROTO_LIST((
	const ski_handle_t *, ski_name *, char **));

/* Supported algorithms */
SKI_STATUS ski_get_alglist PROTO_LIST((ski_alg_list **));

#ifdef	__cplusplus
}
#endif

#endif	/* _SKI_H */
