/*
 * Copyright 1995 by Sun Microsystems, Inc. All Rights Reserved.
 */

#pragma ident	"@(#)ipxtest.c 1.3 95/04/04 SMI"

#include "ipxtest.h"

void
main( int argc, char **argv)
{
	int pid;		/* process id */
	int counter = 0;
	int serverfd;		/* connection id for the server end */
	int clientfd;		/* connection id for the client end */
	struct t_bind servbind;	/* address associated with serv endpoint */
	struct t_bind clibind;	/* address associated with cli endpoint */
	struct t_call servcall;	/* structure storing server conn info */
	struct t_call clicall;	/* structure storing client conn info */
	struct t_info info;	/* structure storing the transport options */
	struct ipx_addr servaddr;	/* server's ipx connection address */
	struct ipx_addr cliaddr;	/* client's ipx connection address */
	char sendbuf[MAX_BUF_SIZE];	/* buffer for send operations */
	char recvbuf[MAX_BUF_SIZE];	/* buffer for recv operations */	


	printf("******************************************************\n");
	printf("*           IPX/SPX Diagnostic Program               *\n");
	printf("*       Copyright 1995 Sun Microsystems, Inc.        *\n");
	printf("*              All Rights Reserved                   *\n");
	printf("******************************************************\n");

	/* Create an endpoint for the server end */
	if (open_spx_endpoint(&serverfd, SPX_DEVICE, &info) != 0) {
		exit(1);
	}

	print_tinfo(&info);

	/* Bind the endpoint to an address */
	memset(&servaddr, 0, sizeof(struct ipx_addr));
	memset(&servbind, 0, sizeof(struct t_bind));
	servaddr.socket[0] = 0;
	servaddr.socket[1] = IPXTEST_SOCKET;
	if (bind_addr(serverfd, &servbind, &servaddr) != 0) {
		exit(1);
	}

	print_network_addr(servbind.addr.buf, SERVER);

	/* fork another process to act as an active end */
	if ((pid = fork()) > 0) {	/* pid > 0 means parent process */
		/* disregard the return code of the child process */
		signal(SIGCLD, SIG_IGN);

		/* accept an incoming connection request */
		memset(&servcall, 0, sizeof(struct t_call));
		if (passive_connect(serverfd, &servcall, &servaddr) != 0) {
			exit(1);
		}
		printf(">> Testing the IPX/SPX protocol stack ...\n");
		do {
			if (recv_spx(serverfd, recvbuf, MAX_BUF_SIZE) != 0) {
				exit(1);
			}
			counter++;
		} while (counter < ITERATION);

		/* Close the endpoint and exit */
		if (close_endpoint(serverfd) != 0) {
			exit(1);
		}
		printf(">> %d packets have been exchanged and verified\n", 
		    counter);
		printf("\nIPX/SPX protocol stack is validated\n");
		exit(0);
	}
	else {				/* pid == 0 means child process */
		if (close_endpoint(serverfd) != 0) {
			exit(1);
		}
		/* Create another endpoint for the active end */
		if (open_spx_endpoint(&clientfd, SPX_DEVICE, &info) != 0) {
			exit(1);
		}

		/* Bind the endpoint to an address */
		memset(&cliaddr, 0, sizeof(struct ipx_addr));
		memset(&clibind, 0, sizeof(struct t_bind));
		if (bind_addr(clientfd, &clibind, &cliaddr) != 0) {
			exit(1);
		}

		print_network_addr(clibind.addr.buf, CLIENT);

		/* Initiate a connection to the passive end parent process */
		memset(&clicall, 0, sizeof(struct t_call));
		if (active_connect(clientfd, &clicall, (struct ipx_addr *)
		    servbind.addr.buf) != 0) {
			exit(1);
		}
		do {
			if (send_spx(clientfd, sendbuf, MAX_BUF_SIZE) != 0) {
				exit(1);
			}
			counter++;
		} while (counter < ITERATION);
		
		sleep(5);
		/* Close the endpoint and exit */
		if (close_endpoint(clientfd) != 0) {
			exit(1);
		}
		exit(0);
	}
}

/*
 * int open_spx_endpoint(int *fd, char *path, struct t_info *tinfo)
 *
 * Create an endpoint for IPX/SPX. It takes a pseudo device path and
 * return the endpoint file descriptor.
 */
int
open_spx_endpoint(int *fd, char *path, struct t_info *tinfo)
{
	char sbuf[128];

	/* if t_open operation fails, then exit */
	if ((*fd = t_open(path, O_RDWR, tinfo)) == FAILED) {
		if (t_errno == TBADFLAG) {
			sprintf(sbuf, "t_open: failed due to invalid flag\n");
		}
		else {
			sprintf(sbuf, "t_open: failed to open %s\n", path);
		}
		t_error(sbuf);
		return(t_errno);
	}
	return 0;
}

/*
 * int bind_addr(int fd, struct t_bind *tbind, struct ipx_addr
 *     *connaddr)
 *
 * Bind an endpoint to an address
 */
int
bind_addr(int fd, struct t_bind *tbind, struct ipx_addr *connaddr)
{
	char sbuf[128];

	tbind->addr.buf = (char *) connaddr;
	tbind->addr.maxlen = sizeof(struct ipx_addr);
	tbind->addr.len = sizeof(struct ipx_addr);
	tbind->qlen = MAX_OUTSTANDING_CONN;
	if (t_bind(fd, tbind, tbind) == FAILED) {
		switch (t_errno) {
		case TACCES:
			sprintf(sbuf, "t_bind: failed due to TACCES\n");
			break; 
		case TBADADDR:
			sprintf(sbuf, "t_bind: failed due to TBADADDR\n");
			break;
		case TBADF:
			sprintf(sbuf, "t_bind: failed due to TBADF\n");
			break;
		case TBUFOVFLW:
			sprintf(sbuf, "t_bind: failed due to TBUFOVFLW\n");
			break; 
		case TNOADDR:
			sprintf(sbuf, "t_bind: failed due to TNOADDR\n");
			break;
		case TOUTSTATE:
			sprintf(sbuf, "t_bind: failed due to TOUTSTATE\n");
			break;
		default:
			sprintf(sbuf, "t_bind: failed due to TSYSERR\n");
			break;
		}
		t_error(sbuf);
		close_endpoint(fd);
		return(t_errno);
	}
	return 0;
}

/*
 * int passive_connect()
 *
 * Setup a connection in passive mode, meaning it waits for an incoming 
 * connection. Used on the receiving end of a connection.
 */
int
passive_connect(int fd, struct t_call *tcall, struct ipx_addr *connaddr)
{
	char sbuf[128];

	memset(tcall, 0 , sizeof(struct t_call));

	tcall->addr.len = sizeof(struct ipx_addr);
	tcall->addr.maxlen = sizeof(struct ipx_addr);
	tcall->addr.buf = (char *)connaddr;
	if (t_listen(fd, tcall) == FAILED) {
		switch (t_errno) {
		case TBADF:
			sprintf(sbuf, "t_listen: failed due to TBADF\n");
			break;
		case TBUFOVFLW:
			sprintf(sbuf, "t_listen: failed due to TBUFOVFLW\n");
			break;
		case TLOOK:
			sprintf(sbuf, "t_listen: failed due to TLOOK\n");
			break;
		case TNODATA:
			sprintf(sbuf, "t_listen: failed due to TNODATA\n");
			break;
		case TNOTSUPPORT:
			sprintf(sbuf, "t_listen: failed due to TNOTSUPPORT\n");
			break;
		default:
			sprintf(sbuf, "t_listen: failed due to TSYSERR\n");
			break;
		}
		t_error(sbuf);
		close_endpoint(fd);
		return(t_errno);
	}
	/* when a connection request is received, accept the connection */
	if (t_accept(fd, fd, tcall) == FAILED) {
		switch (t_errno) {
		case TACCES:
			sprintf(sbuf, "t_accept: failed due to TACCES\n");
			break;
		case TBADF:
			sprintf(sbuf, "t_accept: failed due to TBADF\n");
			break;
		case TBADDATA:
			sprintf(sbuf, "t_accept: failed due to TBADDATA\n");
			break;
		case TBADOPT:
			sprintf(sbuf, "t_accept: failed due to TBADOPT\n");
			break;
		case TBADSEQ:
			sprintf(sbuf, "t_accept: failed due to TBADSEQ\n");
			break;
		case TLOOK:
			sprintf(sbuf, "t_accept: failed due to TLOOK\n");
			break;
		case TNOTSUPPORT:
			sprintf(sbuf, "t_accept: failed due to TNOTSUPPORT\n");
			break;
		case TOUTSTATE:
			sprintf(sbuf, "t_accept: failed due to TOUTSTATE\n");
			break;
		default:
			sprintf(sbuf, "t_accept: failed due to TSYSERR\n");
			break;
		}
		t_error(sbuf);
		close_endpoint(fd);
		return(t_errno);
	}
	return 0;
}

/*
 * int active_connect()
 *
 * Setup a connection in active mode, meaning it initiates a connection.
 * Used in the requesting end of a connection.
 */
int
active_connect(int fd, struct t_call *tcall, struct ipx_addr *connaddr)
{
	char sbuf[128];

	memset(tcall, 0, sizeof(struct t_call));
	
	tcall->addr.len = sizeof(struct ipx_addr);
	tcall->addr.maxlen = sizeof(struct ipx_addr);
	tcall->addr.buf = (char *)connaddr;
	if (t_connect(fd, tcall, tcall) == FAILED) {
		switch (t_errno) {
		case TACCES:
			sprintf(sbuf, "t_connect: failed due to TACCES\n");
			break;
		case TBADADDR:
			sprintf(sbuf, "t_connect: failed due to TBADADDR\n");
			break;
		case TBADF:
			sprintf(sbuf, "t_connect: failed due to TBADF\n");
			break;
		case TBADDATA:
			sprintf(sbuf, "t_connect: failed due to TBADDATA\n");
			break;
		case TBADOPT:
			sprintf(sbuf, "t_connect: failed due to TBADOPT\n");
			break;
		case TBUFOVFLW:
			sprintf(sbuf, "t_connect: failed due to TBUFOVFLW\n");
			break;
		case TNODATA:
			sprintf(sbuf, "t_connect: failed due to TNODATA\n");
			break;
		case TLOOK:
			sprintf(sbuf, "t_connect: failed due to TLOOK\n");
			break;
		case TNOTSUPPORT:
			sprintf(sbuf, "t_connect: failed due to TNOTSUPPORT\n");
			break;
		case TOUTSTATE:
			sprintf(sbuf, "t_connect: failed due to TOUTSTATE\n");
			break;
		default:
			sprintf(sbuf, "t_connect: failed due to TSYSERR\n");
			break;
		}
		t_error(sbuf);
		close_endpoint(fd);
		return(t_errno);
	}
	return 0;
}

/*
 * int send_spx(int fd, char *buf, int bufsize)
 *
 * send packet to the server (blocking)
 */
int
send_spx(int fd, char *buf, int bufsize)
{
	char sbuf[128];

	memset(buf, 0, bufsize);

	if(t_snd(fd, buf, bufsize, NULL) == FAILED) {
		switch (t_errno) {
		case TBADDATA:
			sprintf(sbuf, "t_snd: failed due to TBADDATA\n");
			break;
		case TBADF:
			sprintf(sbuf, "t_snd: failed due to TBADF\n");
			break;
		case TFLOW:
			sprintf(sbuf, "t_snd: failed due to TFLOW\n");
			break;
		case TNOTSUPPORT:
			sprintf(sbuf, "t_snd: failed due to TNOTSUPPORT\n");
			break;
		default:
			sprintf(sbuf, "t_snd: failed due to TSYSERR\n");
			break;
		}
		t_error(sbuf);
		close_endpoint(fd);
		return(t_errno);
	}
	return 0;
}

/*
 * int recv_spx(int fd, char *buf, int bufsize)
 *
 * receive packet from client (blocking)
 */
int
recv_spx(int fd, char *buf, int bufsize)
{
	int flag;
	char sbuf[128];

	memset(buf, 0, bufsize);

	if(t_rcv(fd, buf, bufsize, &flag) == FAILED) {
		switch (t_errno) {
		case TBADF:
			sprintf(sbuf, "t_rcv: failed due to TBADF\n");
			break;
		case TLOOK:
			sprintf(sbuf, "t_rcv: failed due to TLOOK\n");
			break;
		case TNODATA:
			sprintf(sbuf, "t_rcv: failed due to TNODATA\n");
			break;
		case TNOTSUPPORT:
			sprintf(sbuf, "t_rcv: failed due to TNOTSUPPORT\n");
			break;
		default:
			sprintf(sbuf, "t_rcv: failed due to TSYSERR\n");
			break;
		}
		t_error(sbuf);
		close_endpoint(fd);
		return(t_errno);
	}
	return 0;
}

/*
 * int close_endpoint(int fd)
 *
 * close the endpoint
 */
int
close_endpoint(int fd)
{
	char sbuf[128];

	if (t_close(fd) == FAILED) {
		if (t_errno == TBADF) {
			sprintf(sbuf, "t_close: failed due to TBADF\n");
		}
		else {
			sprintf(sbuf, "t_close: failed due to TSYSERR\n");
		}
		t_error(sbuf);
		return(t_errno);
	}
	return 0;
}

/* 
 * print_network_addr(char *addr, char *name)
 * 
 * print out the network number to standard out
 */
void print_network_addr(char *addr, char *name)
{
    	int i;

	printf("******************************************************\n");
	printf("* Network Address Assigned to %s                 *\n", name);
	printf("******************************************************\n");
    	printf("Network Number: ");
    	for (i = 0; i < 4; i++)
        	printf("%02X", (unsigned char) *addr++);
    	printf("\n");
    	printf("Host Address:   ");
    	for (i = 0; i < 6; i++)
        	printf("%02X", (unsigned char) *addr++);
    	printf("\n");
	printf("Socket Number:  ");
	for (i = 0; i < 2; i++)
		printf("%02X", (unsigned char) *addr++);
	printf("\n");
	printf("\n");
}

/*
 * print_tinfo(struct t_info *tinfo)
 *
 * printout the content of tinfo
 */
void
print_tinfo(struct t_info *tinfo)
{
	printf("******************************************************\n");
	printf("* IPX/SPX Protocol via TLI Options                   *\n");
	printf("******************************************************\n");
	printf("Max size of IPX/SPX address: ");
	switch(tinfo->addr) {
	case -1:
		printf("No limit to the size\n");
		break;
	case -2:
		printf("No user access to the protocol addresses\n");
		break;
	default:
		printf("%d\n", tinfo->addr);
		break;
	}
	printf("Size of IPX/SPX Options: ");
	switch(tinfo->options) {
	case -1:
		printf("No limit to the size\n");
		break;
	case -2:
		printf("No user access to the options\n");
		break;
	default:
		printf("%d\n", tinfo->options);
		break;
	}
	printf("Max Size of Transport Service Data Unit: ");
	switch(tinfo->tsdu) {
	case 0:
		printf("No concept of TSDU\n");
		break;
	case -1:
		printf("No limit to the size\n");
		break;
	case -2:
		printf("Not supported\n");
		break;
	default:
		printf("%d\n", tinfo->tsdu);
		break;
	}
	printf("Max Size of Expedited Transport Service Data Unit: ");
	switch(tinfo->etsdu) {
	case 0:
		printf("No concept of ETSDU\n");
		break;
	case -1:
		printf("No limit to the size\n");
		break;
	case -2:
		printf("Not supported\n");
		break;
	default:
		printf("%d\n", tinfo->etsdu);
		break;
	}
	printf("Max Size of Transfer of User Data Along a Conn Request: ");
	switch(tinfo->connect) {
	case -1:
		printf("No limit to the size\n");
		break;
	case -2:
		printf("Not supported\n");
		break;
	default:
		printf("%d\n", tinfo->connect);
		break;
	}
	printf("Max Size of Transfer of User Data Along a Disconn Request: ");
	switch(tinfo->discon) {
	case -1:
		printf("No limit to the size\n");
		break;
	case -2:
		printf("Not supported\n");
		break;
	default:
		printf("%d\n", tinfo->discon);
		break;
	}
	printf("Service Type Provided: ");
	switch(tinfo->servtype) {
	case T_COTS:
		printf("Connection-oriented, w/o orderly release\n");
		break;
	case T_COTS_ORD:
		printf("Connection-oriented, with orderly release\n");
		break;
	case T_CLTS:
		printf("Connectionless service\n");
		break;
	default:
		printf("Unknown service\n");
		break;
	}
	printf("\n");
}
