/* @(#)hdlc.h 1.11 91/01/11 */


/* ------------------------------------------------------------------------- */
/* The HDLC protocol */


#define	IS_IFRAME(c)	((c & 1) == 0)
#define	IS_SFRAME(c)	((c & 3) == 1)
#define	IS_UFRAME(c)	((c & 3) == 3)

#define	FRAMETYPE(c)	(IS_IFRAME(c) ? HDLC_I : \
	(IS_SFRAME(c) ? (c & 0xF) : (c & 0xEF)))

/*
 * Frame types
 * Control fields with P/F and N(r), N(s) masked out
 *             8  7  6  5  4  3  2  1
 * I frame :    -N(r)-  P   -N(s)-  0
 * S frame :    -N(r)-  PF S2 S1 0  1
 * U frame :   M8 M7 M6 PF M4 M3 1  1
 */
#define	HDLC_I		(u_char)0x00	/* I - 2 way */
#define	HDLC_RR		(u_char)0x01	/* S - 2 way */
#define	HDLC_RNR	(u_char)0x05	/* S - 2 way */
#define	HDLC_REJ	(u_char)0x09	/* S - 2 way */
#define	HDLC_SREJ	(u_char)0x0d	/* S - 2 way */
#define	HDLC_UI		(u_char)0x03	/* U - 2 way */
#define	HDLC_SIM	(u_char)0x07	/* U - command only */
#define	HDLC_RIM	(u_char)0x07	/* U - response only */
#define	HDLC_DM		(u_char)0x0f	/* U - response only */
#define	HDLC_SARM	(u_char)0x0f	/* U - command only */
#define	HDLC_SABM	(u_char)0x2f	/* U - command only */
#define	HDLC_DISC	(u_char)0x43	/* U - command only */
#define	HDLC_RD		(u_char)0x43	/* U - response only */
#define	HDLC_UA		(u_char)0x63	/* U - response only */
#define	HDLC_SARME	(u_char)0x4f	/* U - command only */
#define	HDLC_SABME	(u_char)0x6f	/* U - command only */
#define	HDLC_SNRM	(u_char)0x83	/* U - command only */
#define	HDLC_FRMR	(u_char)0x87	/* U - response only */
#define	HDLC_XID	(u_char)0xaf	/* U - 2 way */
#define	HDLC_SNRME	(u_char)0xcf	/* U - command only */
#define	HDLC_TEST	(u_char)0xe3	/* U - 2 way */

#define PF_BIT		(u_char)0x10	/* p/f bit for normal mode */
#define PF_BITX		(u_char)0x01	/* p/f bit for extended mode */

#define	SABM(hl)	(hl->hl_param.hp_xcntl ? HDLC_SABME : HDLC_SABM)
#define	RSTATE(hl)	(hl->hl_lbusy ? HDLC_RNR : \
				(hl->hl_rejcond ? HDLC_REJ : HDLC_RR))


/* ------------------------------------------------------------------------- */
/* HDLC structures */

/*
 * Generic HDLC control information
 * Both normal and extended modes are munged into this common format
 */
struct hdlc_control {
	u_char	hc_frametype;	/* frame type, see above */
	char	hc_command;	/* 1 for command, 0 for response */
	char	hc_pf;		/* Poll/Final bit */
	u_short	hc_nr;		/* N(r) I & S only */
	u_short	hc_ns;		/* N(s) I only */
	u_char  hc_ssap;	/* llc-2 SSAP */
	u_char  hc_dsap;	/* llc-2 DSAP */
};

typedef struct hdlc_control llc2_control;

/*
 * Per-link information
 */
struct hdlc_link {
	struct ifnet hl_if;		/* ifnet structure */
	struct hdlc_param hl_param;	/* link parameters */

	u_short	hl_linkstate;		/* link state */

	/* link conditions
	 */
	u_char	hl_enable   : 1;	/* enabled by higher level */
	u_char	hl_needfinal: 1;	/* polled, need to send final */
	u_char	hl_rejcond  : 1;	/* REJ sent */
	u_char	hl_rbusy    : 1;	/* remote busy */
	u_char	hl_lbusy    : 1;	/* local busy */
	u_char	hl_needack  : 1;	/* need to ack recved frame */
	u_char	hl_timeout_running:1;
	u_char	hl_upper_stop : 1;	/* sent CTL_STOP_DATA to upper */
	u_char	hl_await : 1;		/* await timeout for state change */
	u_char	hl_free : 1;		/* free if llc2 context */
/* ICNCFIX by -TH- for ISO8882 conformance.  Add an additional flag here for
   passive mode. */
	u_char hl_passive : 1;		/* passive mode */
/* End of fix */

	/* primary state
	 */
	u_short	hl_NsSend;		/* seq of next transmitted frame */
	u_short	hl_NsRcv;		/* seq of next acknowledged frame */
	struct mbuf *hl_ohead;		/* head of output list */
	struct mbuf *hl_otail;		/* tail of output list */
	struct mbuf *hl_ahead;		/* head of un-ACKed list */
	struct mbuf *hl_atail;		/* tail of un-ACKed list */
	u_char	hl_pollcmd;		/* Poll frame ctrl field */
	u_char	hl_nlastpoll;		/* # of times last poll was sent */
	u_short	hl_hz;			/* timeout value */
	u_short	hl_t1timer;		/* T1 time left or zero */
	u_short	hl_t1cnt;		/* # of T1 ticks */

	/* secondary state
	 */
	u_short	hl_t2timer;		/* T2 time left or zero */
	u_short	hl_t3timer;		/* T3 time left or zero */
	u_short	hl_Vr;			/* receive state variable */
	u_short	hl_Vrlast;		/* last xmitted N(r) */
	struct mbuf *hl_frmr;		/* FRMR frame */

	/* miscellaneous
	 */
	u_short hl_sendsabms;
	u_short hl_tbusy;
	u_short hl_tbusycnt;
	u_short hl_treject;
	u_short hl_trejectcnt;
	u_short hl_tack;
	u_short hl_tackcnt;

	/* required by llc
	 */
	u_char 	hl_type;		/* LLC_TYPE2 or HDLC_TYPE */
	struct  sockaddr_802 hl_dst;	/* LLC2 l_sel and MAC address */
	struct  hdlc_link *hl_next; 	/* LLC2 connection list */
	struct  llc_client *hl_clp; 	/* LLC2 client pointer */

	/* back pointer to mbuf for freeing this block
	 */
	struct mbuf *mbufaddr;

	/* hashing parameters
	 */
	struct hdlc_link *hash_next;
	struct hdlc_link *hash_prev;
	int hash_val;
};

#define HDLC_TYPE	0
#define LLC_TYPE1	1
#define LLC_TYPE2	2
#define LLC_TEST	4
#define LLC_XID		8

#define IS_HDLCLINK(hl)  (hl->hl_type == HDLC_TYPE)
#define IS_LLC2LINK(hl)  (hl->hl_type & LLC_TYPE2)


/* ------------------------------------------------------------------------- */

#define NEWTIMEOUT1( hl )\
	{ hl->hl_t1timer = hl->hl_param.hp_t1; \
	  if ( ! hl->hl_timeout_running ) { \
                hl->hl_timeout_running= 1; \
		timeout( hdlc_timeout, (char *)hl, (int)hl->hl_hz ); }  }

#define OLDTIMEOUT2( hl )\
	{ if ( !hl->hl_t2timer ) \
		hl->hl_t2timer = hl->hl_param.hp_t2; \
          if ( ! hl->hl_timeout_running ) { \
             hl->hl_timeout_running= 1; \
	     timeout( hdlc_timeout, (caddr_t)hl, (int)hl->hl_hz ); }  }  

#define BUSYTIMEOUT( hl )\
	{ hl->hl_tbusy = hl->hl_param.hp_t4; hl->hl_tbusycnt = 0; \
          if ( ! hl->hl_timeout_running ) { \
             hl->hl_timeout_running= 1; \
	     timeout( hdlc_timeout, (caddr_t)hl, (int)hl->hl_hz ); }  }  

#define REJTIMEOUT( hl )\
	{ hl->hl_treject = hl->hl_param.hp_t5; hl->hl_trejectcnt = 0; \
          if ( ! hl->hl_timeout_running ) { \
             hl->hl_timeout_running= 1; \
	     timeout( hdlc_timeout, (caddr_t)hl, (int)hl->hl_hz ); }  }  

#define ACKTIMEOUT( hl )\
	{ hl->hl_tack = hl->hl_param.hp_t6; hl->hl_tackcnt = 0; \
          if ( ! hl->hl_timeout_running ) { \
             hl->hl_timeout_running= 1; \
	     timeout( hdlc_timeout, (caddr_t)hl, (int)hl->hl_hz ); }  }  
   
/* ------------------------------------------------------------------------- */
/*
 * Format of FRMR (frame reject) frames
 */
struct hdlc_frmr {			/* mod 8 sequences */
	u_char	frmr_addr;		/* HDLC address field */
	u_char	frmr_ctl;		/* FRMR and final bit */
	u_char	frmr_rctl;		/* rejected control field */
	u_char	frmr_state;		/* N(R), N(S), etc. */
	u_char	frmr_why;		/* w,x,y,z codes */
};
#define FRMR_SIZE	5


struct hdlc_xfrmr {			/* mod 128 sequences */
	u_char	xfrmr_addr;		/* HDLC address field */
	u_char	xfrmr_ctl;		/* FRMR and final bit */
	u_char	xfrmr_rctl1;		/* 1st byte rejected control field */
	u_char	xfrmr_rctl2;		/* 2nd byte rejected control field */
	u_char	xfrmr_state[2];		/* N(R), N(S), etc */
	u_char	xfrmr_why;		/* w,x,y,z codes */
};
#define XFRMR_SIZE	7


/*
 * Frame reject reason codes (xfrmr_why & frmr_why)
 */
#define	FRMR_W	0x01		/* control field not supported */
#define	FRMR_X	0x02		/* I field outside of I frame */
#define	FRMR_Y	0x04		/* I field too long */
#define	FRMR_Z	0x08		/* invalid N(r) */

/*
 * Flow control requirements
 */
#define HDLC_MTU	1024		    /* largest X.25 packet */
#define HDLC_MAX_PKTS	24	            /* max to be held for output */
#define HDLC_HI_WATER	(3*HDLC_MAX_PKTS/4) /* when to send CTL_STOP */
#define HDLC_LOW_WATER	(HDLC_MAX_PKTS/4)   /* when to send CTL_START */


/* ------------------------------------------------------------------------- */

/* HDLC tracing */

typedef enum {HT_RSP, HT_CMD} ht_command;

struct hdlc_trace_pci {
	int		xcntl;		/* Extended control */
	int		llclink;
	ht_command	command;
	struct sockaddr dstmac;		/* only for llc links */
};

#define hdlc_t_pdu(ifp, hl, m, direction) \
	{ if ((ifp)->if_flags & IFF_PROMISC) \
		r_hdlc_t_pdu(ifp, hl, m, direction); }


/* ------------------------------------------------------------------------- */

#ifdef KERNEL
extern int		nhdlcline;
extern struct ifqueue	hdlc_inq[];
extern struct hdlc_link	hdlclink[];

#ifdef DLCDEBUG
#define dprintf		if (dlcdebug >= 1) printf
#define dprint2		if (dlcdebug >= 2) printf
#define dprint3		if (dlcdebug >= 3) printf
#define dprint4		if (dlcdebug >= 4) printf
#define dprint5		if (dlcdebug >= 5) printf
#define HLUNIT		hl->hl_if.if_name, hl->hl_if.if_unit
#define IFUNIT		ifp->if_name, ifp->if_unit
extern	int		dlcdebug;
extern	int		hdlc_print_state();
extern	int		hdlc_print_frametype();
extern	int		hdlc_trace();

#endif
#endif

